﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Security.Permissions;
using System.Runtime.InteropServices;


namespace CaLib.User
{
	/// <summary>
	/// ClipboardViewerクラスは、NativeWindowの派生クラスとして宣言する。
	/// これにより、このクラスを“サブクラス化”できる。
	/// サブクラス化することで、元のウィンドウに送られてくるWindowsメッセージを
	/// サブクラス（ここではMyClipboardViewer）側で先に受け取ることができ、
	/// 元のウィンドウのメッセージ処理には手を加えずに、Windowsメッセージ
	/// （ここで利用するのはクリップボード関連のメッセージ）を処理できるようになる。
	/// </summary>
	/// <see cref="http://www.atmarkit.co.jp/fdotnet/dotnettips/848cbviewer/cbviewer.html"/>
	[PermissionSet(SecurityAction.Demand,Name = "FullTrust")]
	public class ClipboardViewer : System.Windows.Forms.NativeWindow
	{
		private IntPtr nextHandle;

		private System.Windows.Forms.Form parent;
		public event cbEventHandler ClipboardHandler;

		public ClipboardViewer(System.Windows.Forms.Form f)
		{
			f.HandleCreated	+= new EventHandler(this.OnHandleCreated);
			f.HandleDestroyed += new EventHandler(this.OnHandleDestroyed);
			this.parent = f;
		}

		internal void OnHandleCreated(object sender, EventArgs e)
		{
			AssignHandle(((System.Windows.Forms.Form)sender).Handle);

			// ビューアを登録
			nextHandle = WinApi.User32.SetClipboardViewer(this.Handle);
		}

		internal void OnHandleDestroyed(object sender, EventArgs e)
		{
			// ビューアを解除
			bool sts = WinApi.User32.ChangeClipboardChain(this.Handle, nextHandle);
			ReleaseHandle();
		}

		protected override void WndProc(ref System.Windows.Forms.Message msg)
		{
			switch (msg.Msg)
			{
				case (int)WinApi.Message.id.WM_DRAWCLIPBOARD:
					if (System.Windows.Forms.Clipboard.ContainsText())
					{
						// クリップボードの内容がテキストの場合のみ
						if (ClipboardHandler != null)
						{
							string buffer = null;
							try
							{
								buffer = System.Windows.Forms.Clipboard.GetText();
							}
							catch (ExternalException)
							{
								//クリップボードをクリアできませんでした。この例外は、通常、クリップボードが別のプロセスで使用されている場合に発生します。
							}

							if (buffer != null)
							{
								// クリップボードの内容を取得してハンドラを呼び出す
								ClipboardHandler(this, new ClipboardEventArgs(buffer));
							}
						}
					}
					if ((int)nextHandle != 0)
						WinApi.User32.SendMessage(nextHandle, msg.Msg, msg.WParam, msg.LParam);
					break;

				// クリップボード・ビューア・チェーンが更新された
				case (int)WinApi.Message.id.WM_CHANGECBCHAIN:
					if (msg.WParam == nextHandle)
					{
						nextHandle = (IntPtr)msg.LParam;
					}
					else if ((int)nextHandle != 0)
					{
						WinApi.User32.SendMessage(nextHandle, msg.Msg, msg.WParam, msg.LParam);
					}
					break;
			}
			base.WndProc(ref msg);
		}
	}
}
