require 'rubygems'
require 'citrus'
require 'test/unit'
BIAODIAN = File.join(File.dirname(__FILE__), '..', 'lib', 'biaodian')

class BiaodianTest < Test::Unit::TestCase
  def setup
    Citrus.load BIAODIAN
  end
  def assert_parse(string, root)
    match = Biaodian.parse(string, :root => root)
    assert(match)
    assert_equal(string, match)
  end
  def assert_parse_error(string, root)
    assert_raise Citrus::ParseError do
      Biaodian.parse(string, :root => root)
    end
  end

  def test_punctuation
    assert_parse('，', :punctuation)
    assert_parse('、', :punctuation)
    assert_parse('：', :punctuation)
    assert_parse('；', :punctuation)
    assert_parse_error('a', :punctuation)
    assert_parse_error('標点', :punctuation)
  end
  # text:  /[^《》〈〉【】「」。，、：；]+/u
  def test_text
    assert_parse('a', :text)  # TODO: 半角の排除
    assert_parse('標点', :text)
    assert_parse_error('、', :text)
#    assert_parse_error('？', :text)
    assert_parse_error('！', :text)
  end
  def test_book
    assert_parse('《book》', :book)
    assert_parse('《段注》', :book)
    assert_parse_error('《》', :book)
  end
  def test_chapter
    assert_parse('〈chap〉', :chapter)
    assert_parse('〈一篇上〉', :chapter)
    assert_parse_error('〈〉', :chapter)
    assert_parse_error('《book》', :chapter)
  end
  def test_person
    assert_parse('【person】', :person)
    assert_parse('【段玉裁】', :person)
    assert_parse_error('【】', :person)
  end
  def test_name
    assert_parse('《段注》', :name)
    assert_parse('〈一篇上〉', :name)
    assert_parse('【段玉裁】', :name)
    assert_parse_error('【】', :name)
  end
  def test_phrase
    assert_parse('《段注》', :phrase)
    assert_parse('《段注》曰', :phrase)
    assert_parse('《漢書》曰', :phrase)
    match = Biaodian.parse('《段注》曰', :root => :phrase)
    assert(match)
    assert_equal('《段注》曰', match)
    assert_equal('《段注》', match.name)
    assert_equal('曰', match.text)

  end
  def test_phrase_seq
    match = Biaodian.parse('《説文》，形書也', :root => :phrase_seq)
    assert(match)
    assert_equal('《説文》，形書也', match)
    assert_equal('《説文》', match.phrase)
    assert_equal('，', match.punctuation)
    assert_equal('，形書也', match.vphrase)
    assert_parse_error('《漢書》曰：', :phrase_seq)
  end
  def test_quotation
    assert_parse('「元元本本」', :quotation)
    assert_parse_error('「元元本本', :quotation)
    assert_parse_error('元元本本」', :quotation)
    assert_parse_error('「」', :quotation)
  end
  # TODO: 網羅的なテストの作成(失敗も含む)
  def test_sentence
    assert_parse('《漢書》曰:「元元本本」。', :sentence)

    assert_parse('《漢書》曰', :phrase)
    assert_parse('「元元本本，數始於一」', :quotation)
    assert_parse('《漢書》曰：「元元本本，數始於一」。', :sentence)
  end
  def test_sentence_seq
    assert_parse('《漢書》曰:「元元本本」。古義可互求焉。', :sentence_seq)
  end
end
