/*
 * Copyright (C) 2007 uguu at users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.deployer.server.core;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.regex.Pattern;

import jp.sourceforge.deployer.Deployer;
import jp.sourceforge.deployer.DeployerClassLoader;
import jp.sourceforge.deployer.DeployerListener;
import jp.sourceforge.deployer.FileMonitorFailException;

/**
 * <p>
 * {@link ApplicationServer}クラスを実行し、アプリケーションのデプロイを制御します。
 * </p>
 * 
 * @author $Author$
 * @version $Rev$ $Date$
 */
public final class DeployerServer implements Runnable, DeployerListener {

    /**
     * <p>
     * ファイル監視間隔。
     * </p>
     */
    private int                          _monitoringInterval;

    /**
     * <p>
     * クラスローダの破棄試行タイムアウト時間。
     * </p>
     */
    private int                          _tryDisposeTimeout;

    /**
     * <p>
     * 配置ディレクトリ。
     * </p>
     */
    private File                         _deployDir;

    /**
     * <p>
     * 作業ディレクトリ。
     * </p>
     */
    private File                         _workDir;

    /**
     * <p>
     * デプロイ対象であると認識するファイルの正規表現パターン。
     * </p>
     */
    private Pattern                      _filePattern;

    /**
     * <p>
     * 展開後におけるファイルの検索パス。
     * </p>
     */
    private String                       _fileDir;

    /**
     * <p>
     * 展開後におけるjarファイルの検索パス。
     * </p>
     */
    private String                       _jarDir;

    /**
     * <p>
     * サーバが実行中か停止後かを表すフラグ。
     * </p>
     */
    private boolean                      _executing = true;

    /**
     * <p>
     * デプロイされたアプリケーションを管理するマップ。
     * </p>
     */
    private Map<String, ApplicationInfo> _appMap    = new HashMap<String, ApplicationInfo>();

    /**
     * <p>
     * サーバ共通クラスローダー。
     * </p>
     */
    private ClassLoader                  _commonClassLoader;

    /**
     * <p>
     * サーバ実装クラスローダー。
     * </p>
     */
    private ClassLoader                  _serverClassLoader;

    /**
     * <p>
     * サーバ実装クラスのインスタンス。
     * </p>
     */
    private ApplicationServer            _server;

    /**
     * <p>
     * アプリケーションの様々な初期値を読み込んで初期化します。
     * </p>
     * 
     * @throws ClassNotFoundException
     *             サーバ・クラスを読み込むとき、クラスが存在しない場合。
     * @throws InstantiationException
     *             サーバ・クラスのインスタンス化に失敗した場合。
     * @throws IllegalAccessException
     *             クラスに不正なアクセスを行った場合。
     * @throws IOException
     */
    public DeployerServer() throws ClassNotFoundException, InstantiationException, IllegalAccessException, IOException {
        // 初期化します。
        ResourceBundle rb = ResourceBundle.getBundle("deployer-server");
        this._deployDir = new File(rb.getString("jp.sourceforge.deployer.server.deployDirectory")).getCanonicalFile();
        this._workDir = new File(rb.getString("jp.sourceforge.deployer.server.workDirectory")).getCanonicalFile();
        this._filePattern = Pattern.compile(rb.getString("jp.sourceforge.deployer.server.filePattern"));
        this._fileDir = rb.getString("jp.sourceforge.deployer.server.classLoader.fileDirectories");
        this._jarDir = rb.getString("jp.sourceforge.deployer.server.classLoader.jarDirectories");

        System.out.println("デプロイ・ディレクトリ[" + this._deployDir.getAbsolutePath() + "]");
        System.out.println("作業ディレクトリ[" + this._workDir.getAbsolutePath() + "]");
        System.out.println("ファイル・パターン[" + this._filePattern.toString() + "]");
        System.out.println("ファイル・ディレクトリ[" + this._fileDir + "]");
        System.out.println("ライブラリ・ディレクトリ[" + this._jarDir + "]");

        this._monitoringInterval = Integer.parseInt(rb.getString("jp.sourceforge.deployer.server.monitoringInterval"));
        this._tryDisposeTimeout = Integer.parseInt(rb.getString("jp.sourceforge.deployer.server.tryDisposeTimeout"));

        // サーバ共通クラスローダを取得する。
        this._commonClassLoader = this.getClass().getClassLoader().getParent();

        // デプロイヤー・クラスローダを取得する。
        ClassLoader deployerCL = this.getClass().getClassLoader();

        // サーバ実装クラスローダを構築する。
        File libDir = new File(rb.getString("jp.sourceforge.deployer.server.libDirectory")).getCanonicalFile();
        List<URL> libUrlList = this.getJarUrlList(libDir);
        libUrlList.add(libDir.toURL());
        this._serverClassLoader = new URLClassLoader(libUrlList.toArray(new URL[0]), deployerCL);

        // サーバをインスタンス化する。
        String serverClass = rb.getString("jp.sourceforge.deployer.server.serverClass");
        Class serverClazz = this._serverClassLoader.loadClass(serverClass);
        this._server = (ApplicationServer) serverClazz.newInstance();
    }

    /**
     * <p>
     * サーバ処理を開始します。
     * </p>
     * 
     * @throws IOException
     */
    public void start() throws IOException {
        Thread t = new Thread(this);
        t.start();

        System.out.println("[ENTER]を入力すると終了します。");
        System.in.read();

        this._executing = false;
    }

    /**
     * <p>
     * サーバのスレッド処理です。定期的にデプロイを監視します。
     * </p>
     */
    public void run() {
        // サーバにイベントを通知する。
        this._server.startup();

        // デプロイを監視する。
        Deployer deployer = new Deployer(this._deployDir, this._filePattern, this._workDir);
        deployer.addListener(this);
        while (this._executing) {
            try {
                deployer.monitor();
            } catch (FileMonitorFailException e) {
                e.printStackTrace();
            }
            try {
                Thread.sleep(this._monitoringInterval);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        // サーバにイベントを通知する。
        this._server.shutdown();
    }

    /**
     * <p>
     * 指定したディレクトリ以下の全てのjarファイルのパスを表すURLのリストを返します。
     * </p>
     * 
     * @param dir
     *            jarファイルを検索するディレクトリ。
     * @return 全てのjarファイルのパスを表すURLのリスト。
     * @throws MalformedURLException
     *             URLが不正である場合。
     */
    private List<URL> getJarUrlList(File dir) throws MalformedURLException {
        List<URL> urlList = new ArrayList<URL>();
        List<File> fileList = this.getJarFileList(dir);

        for (File file : fileList) {
            urlList.add(file.toURL());
        }

        return urlList;
    }

    /**
     * <p>
     * 指定したディレクトリ以下の全てのjarファイルのリストを返します。
     * </p>
     * 
     * @param dir
     *            jarファイルを検索するディレクトリ。
     * @return 全てのjarファイルのリスト。
     */
    private List<File> getJarFileList(File dir) {
        List<File> fileList = new ArrayList<File>();

        if (dir == null) {
            return fileList;
        }

        File[] files = dir.listFiles();
        if (files == null) {
            return fileList;
        }

        for (File file : files) {
            if (file.isFile() && file.getName().endsWith(".jar")) {
                fileList.add(file);
            } else if (file.isDirectory()) {
                fileList.addAll(this.getJarFileList(file));
            }
        }

        return fileList;
    }

    /**
     * <p>
     * デプロイが完了したときに呼び出されます。クラスローダーを初期化し、サーバにイベントを通知します。
     * </p>
     */
    public void deployEnd(Deployer deployer, File file, File destDirectory) {
        try {
            // アプリケーションを初期化。
            File[] classDirs = this.getClassDirectories(destDirectory, this._fileDir);
            File[] jarFiles = this.getJarFiles(destDirectory, this._jarDir);
            DeployerClassLoader cl = new DeployerClassLoader(classDirs, jarFiles, this._commonClassLoader);

            ApplicationInfo ai = new ApplicationInfoImpl(destDirectory, cl);

            this._appMap.put(file.getAbsolutePath(), ai);

            // サーバにイベントを通知。
            this._server.deploy(ai);

            // ログ。
            System.out.println("deployEnd: " + file.getAbsolutePath());
        } catch (MalformedURLException e) {
            e.printStackTrace();
        }
    }

    /**
     * <p>
     * 基準ディレクトリ、カンマ区切りの相対ディレクトリを用いて、ディレクトリの配列を構築し、返します。
     * </p>
     * 
     * @param baseDir
     *            基準ディレクトリ。
     * @param classDir
     *            カンマ区切りの相対ディレクトリの配列。
     * @return ディレクトリの配列。
     */
    private File[] getClassDirectories(File baseDir, String classDir) {
        String[] dirs = classDir.split(",");

        List<File> dirList = new ArrayList<File>();
        for (String d : dirs) {
            dirList.add(new File(baseDir, d));
        }

        return dirList.toArray(new File[0]);
    }

    /**
     * <p>
     * 基準ディレクトリ、カンマ区切りの相対ディレクトリを用いて、jarファイルの配列を構築し、返します。
     * </p>
     * 
     * @param baseDir
     *            基準ディレクトリ。
     * @param jarDir
     *            カンマ区切りの相対ディレクトリの配列。
     * @return jarファイルの配列。
     */
    private File[] getJarFiles(File baseDir, String jarDir) {
        String[] dirs = jarDir.split(",");

        List<File> jarFileList = new ArrayList<File>();
        for (String d : dirs) {
            jarFileList.addAll(this.getJarFileList(new File(baseDir, d)));
        }

        return jarFileList.toArray(new File[0]);
    }

    /**
     * <p>
     * ファイルがデプロイされると呼び出されます。
     * </p>
     */
    public void deployFile(Deployer deployer, File file, File destDirectory, File deployFile) {
        // ログ。
        System.out.println("deployFile: " + file.getAbsolutePath() + ", " + deployFile.getAbsolutePath());
    }

    /**
     * <p>
     * デプロイが開始したときに呼び出されます。
     * </p>
     */
    public void deployStart(Deployer deployer, File file) {
        // ログ。
        System.out.println("deployStart: " + file.getAbsolutePath());
    }

    /**
     * <p>
     * アンデプロイが完了したときに呼び出されます。クラスローダーを破棄し、サーバにイベントを通知します。
     * </p>
     */
    public void undeployEnd(Deployer deployer, File file) {
        // ログ。
        System.out.println("undeployEnd: " + file.getAbsolutePath());
    }

    /**
     * <p>
     * アンデプロイが完了したときに呼び出されます。
     * </p>
     */
    public void undeployStart(Deployer deployer, File file, File destDirectory) {
        // ログ。
        System.out.println("undeployStart: " + file.getAbsolutePath());

        // サーバにイベントを通知。
        ApplicationInfo ai = this._appMap.remove(file.getAbsolutePath());
        this._server.undeploy(ai);

        // アプリケーションを破棄。
        if (ai != null) {
            ai.getDeployerClassLoader().dispose(this._tryDisposeTimeout);
        }
    }

}
