//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dctoolwindowcelleditor.h"

#include "dceditablelabel.h"
#include "dceditabletreeview.h"
#include "dccelltypecombobox.h"
#include "dccell.h"
#include "dccreator.h"
#include "dcreceptor.h"
#include "dcaxon.h"
#include "dccellcode.h"
#include "dcaxonterminal.h"
#include "dccontainer.h"
#include "utils/dcresources.h"
#include "utils/dcqtitemmodel.h"
#include "utils/dcutil.h"
#include "codeeditor/dccellscriptseditorwindow.h"

#include <QGridLayout>
#include <QLineEdit>
#include <QTreeView>
#include <QSizePolicy>
#include <QIcon>

class ReceptorModel : public DCQtItemModel
{
    DCToolWindowCellEditor *d_owner;
public:
    ReceptorModel(DCToolWindowCellEditor *owner) : DCQtItemModel(QStringList("name")), d_owner(owner)
    {

    }

    virtual ~ReceptorModel()
    {
    }

    virtual bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole)
    {
        if (role != Qt::EditRole)
            return false;

        DCCell *cell = d_owner->getCell();
        if (!cell)
            return false;

        if (index.column() != 0)
        {
            return DCQtItemModel::setData(index, value, role);
        }

        DCQtItemModelItem *item = getItem(index);
        QString oldReceptorName = item->data(index.column()).toString();
        QString newReceptorName = value.toString().trimmed();
        if (newReceptorName.length() == 0)
        {
            return false;
        }

        if (oldReceptorName == newReceptorName)
        {
            //nothing actually changes
            return false;
        }

        if (!d_owner->getIsViewUpdating())
        {
            if (cell->getReceptor(newReceptorName))
            {
                //already exist
                QMessageBox::warning(NULL, tr("Rename receptor failed"), tr("The receptor name already exist"));
                return false;
            }

            bool result = item->setData(index.column(), newReceptorName);

            if (result)
            {
                DCCreator *creator = d_owner->getController();
                if (creator->doCommandRenameReceptorName(d_owner, cell, oldReceptorName, newReceptorName, true))
                {
                    emit dataChanged(index, index);
                }
                else
                {
                    item->setData(index.column(), oldReceptorName);
                }
            }
            return result;
        }
        else
        {
            return item->setData(index.column(), newReceptorName);
        }
    }

    virtual Qt::ItemFlags flags(const QModelIndex &index) const
    {
        if (!index.isValid())
            return 0;

        Qt::ItemFlags flags = Qt::ItemIsEnabled;

        if (index.column() == 0 && index.isValid() && index.parent().isValid() && !index.parent().parent().isValid())
        {
            flags |= Qt::ItemIsEditable | Qt::ItemIsSelectable;
        }

        return flags;
    }
};

class AxonTerminalModel : public DCQtItemModel
{
    DCToolWindowCellEditor *d_owner;
public:
    AxonTerminalModel(DCToolWindowCellEditor *owner) : DCQtItemModel(QStringList("name")), d_owner(owner)
    {

    }

    virtual ~AxonTerminalModel()
    {
    }

    virtual bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole)
    {
        if (role != Qt::EditRole)
            return false;

        DCCell *cell = d_owner->getCell();
        if (!cell)
            return false;

        if (index.column() != 0)
        {
            return DCQtItemModel::setData(index, value, role);
        }

        DCQtItemModelItem *item = getItem(index);
        QString oldReceptorName = item->data(index.column()).toString();
        QString newReceptorName = value.toString().trimmed();
        if (newReceptorName.length() == 0)
        {
            return false;
        }

        if (oldReceptorName == newReceptorName)
        {
            //nothing actually changes
            return false;
        }

        DCCell *axonTargetCell = NULL;
        if (!d_owner->getIsViewUpdating())
        {
            bool alreadyExist = false;
            DCAxon *axon = cell->getAxon();
            int num = axon->getNumberOfTerminals();
            for (int i = 0; i < num; i++)
            {
                DCReceptor *receptor = axon->getTerminalAt(i)->getTarget();
                if (receptor)
                {
                    DCCell *targetCell = receptor->getOwnerCell();
                    if (targetCell)
                    {
                        QString targetReceptorName = QString::fromStdString(targetCell->getReceptorName(receptor));
                        if (targetReceptorName == newReceptorName)
                        {
                            alreadyExist = true;
                            break;
                        }
                        else if (targetReceptorName == oldReceptorName)
                        {
                            axonTargetCell = targetCell;
                        }
                    }
                }
            }

            if (alreadyExist)
            {
                QMessageBox::warning(NULL, tr("Rename receptor failed"), tr("The receptor name already exist"));
                return false;
            }

            bool result = false;

            if (axonTargetCell)
            {
                result = item->setData(index.column(), newReceptorName);

                if (result)
                {
                    DCCreator *creator = d_owner->getController();
                    if (creator->doCommandRenameReceptorName(d_owner, axonTargetCell, oldReceptorName, newReceptorName, true))
                    {
                        emit dataChanged(index, index);
                    }
                    else
                    {
                        item->setData(index.column(), oldReceptorName);
                    }
                }
            }
            return result;
        }
        else
        {
            return item->setData(index.column(), newReceptorName);
        }
    }

    virtual Qt::ItemFlags flags(const QModelIndex &index) const
    {
        if (!index.isValid())
            return 0;

        Qt::ItemFlags flags = Qt::ItemIsEnabled;

        if (index.column() == 0 && index.isValid() && index.parent().isValid() && index.parent().parent().isValid() && !index.parent().parent().parent().isValid())
        {
            flags |= Qt::ItemIsEditable | Qt::ItemIsSelectable;
        }

        return flags;
    }

};

DCToolWindowCellEditor::DCToolWindowCellEditor(DCCreator *creator) :
    DCToolWindowBase("CELL:", creator), d_cell(NULL), d_isViewUpdating(false)
{
    d_layout = new QGridLayout;
#ifdef Q_WS_MAC
    d_layout->setSpacing(10);
#else
    d_layout->setSpacing(6);
#endif
    d_layout->setContentsMargins(0,0,0,0);
    d_layout->addWidget(new QLabel(tr("page")),0,0);
    d_layout->addWidget(new QLabel(tr("type")),1,0);
    d_layout->addWidget(new QLabel(tr("cell code")),2,0);
    d_layout->addWidget(new QLabel(tr("custom script")),3,0);

    d_detailButton = new QPushButton(tr("detail..."));
    d_detailButton->setStyleSheet("background-color: rgba(255,255,255,60);");

    d_layout->addWidget(d_detailButton,6,0,1,2);

    d_textPage = new DCEditableLabel(this, "");
    d_comboType = new DCCellTypeComboBox(creator, this);
    d_layout->addWidget(d_textPage, 0,1);
    d_layout->addWidget(d_comboType, 1,1);

    d_classButton = new QPushButton("...");
    d_customScriptButton = new QPushButton(tr("edit..."));

    d_layout->addWidget(d_classButton, 2,1);
    d_layout->addWidget(d_customScriptButton, 3,1);

    d_receptors = new DCEditableTreeView(this);
    d_receptors->setHeaderHidden(true);
    d_receptors->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    d_axonTerminals = new DCEditableTreeView(this);
    d_axonTerminals->setHeaderHidden(true);
    d_axonTerminals->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);


    d_receptorItemModel = new ReceptorModel(this);
    d_axonTerminalItemModel = new AxonTerminalModel(this);

    d_receptors->setModel(d_receptorItemModel);
    d_axonTerminals->setModel(d_axonTerminalItemModel);

    d_layout->addWidget(d_receptors, 4,0,1,2);
    d_layout->addWidget(d_axonTerminals, 5, 0, 1, 2);

    contentLayout()->addLayout(d_layout);

    connect(d_receptors, SIGNAL(collapsed(QModelIndex)), this, SLOT(slotReceptorTreeCollapsed(QModelIndex)));
    connect(d_receptors, SIGNAL(expanded(QModelIndex)), this, SLOT(slotReceptorTreeExpanded(QModelIndex)));
    connect(d_receptors, SIGNAL(clicked(QModelIndex)), this, SLOT(slotReceptorItemClicked(QModelIndex)));
    connect(d_receptors, SIGNAL(vscrollbarShown()), this, SLOT(adjustTreeColumnWidth()));
    connect(d_receptors, SIGNAL(vscrollbarHidden()), this, SLOT(adjustTreeColumnWidth()));
    connect(d_axonTerminals, SIGNAL(collapsed(QModelIndex)), this, SLOT(slotAxonTerminalTreeCollapsed(QModelIndex)));
    connect(d_axonTerminals, SIGNAL(expanded(QModelIndex)), this, SLOT(slotAxonTerminalTreeExpanded(QModelIndex)));
    connect(d_axonTerminals, SIGNAL(clicked(QModelIndex)), this, SLOT(slotAxonTerminalItemClicked(QModelIndex)));
    connect(d_axonTerminals, SIGNAL(vscrollbarShown()), this, SLOT(adjustTreeColumnWidth()));
    connect(d_axonTerminals, SIGNAL(vscrollbarHidden()), this, SLOT(adjustTreeColumnWidth()));

    connect(d_classButton, SIGNAL(clicked()), this, SLOT(slotCellCodeEditButtonPressed()));
    connect(d_customScriptButton, SIGNAL(clicked()), this, SLOT(slotCustomScriptEditButtonPressed()));
}

DCToolWindowCellEditor::~DCToolWindowCellEditor()
{
    d_receptors->disconnect(this);
    d_axonTerminals->disconnect(this);

    if (d_receptorItemModel)
        d_receptorItemModel->deleteLater();

    if (d_axonTerminalItemModel)
        d_axonTerminalItemModel->deleteLater();
}

void DCToolWindowCellEditor::setCell(DCCell *cell)
{
    if (d_cell)
        d_cell->disconnect(this);

    d_cell = cell;
    QString title = "CELL:";
    if (cell)
    {
        title.append(QString::fromStdString(cell->getName()));
        connect(cell, SIGNAL(destroyed(QObject*)), this, SLOT(slotCellDestroyed()));
        d_comboType->setEditingCell(cell);
    }

    setButtonedWindowTitle(title);

    updateView();
}

void DCToolWindowCellEditor::updateView()
{
    d_isViewUpdating = true;
    d_receptorItemModel->removeAllItems();
    d_axonTerminalItemModel->removeAllItems();

    DCContainer *container = getController()->getCurrentContainer();

    if (d_cell && container)
    {
        if (container->getIsScriptable(d_cell->getType()))
        {
            if (d_cell->getIsCellCodeAssgined())
            {
                d_classButton->setText(QString::fromStdString(d_cell->getCellCode()->getFQNName()));
            }
            else
            {
                d_classButton->setText("...");
            }
            d_classButton->setEnabled(true);

            d_customScriptButton->setText("Edit...");
            d_customScriptButton->setEnabled(true);
        }
        else
        {
            d_classButton->setText("N/A");
            d_classButton->setEnabled(false);
            d_customScriptButton->setText("N/A");
            d_customScriptButton->setEnabled(false);
        }

        d_comboType->updateSelection();
        if (container->getIsPluginType(d_cell->getType()))
        {
            d_comboType->setEnabled(false);
        }
        else
        {
            d_comboType->setEnabled(!d_cell->getIsCellCodeAssgined());
        }

        d_textPage->setText(QString::fromStdString(d_cell->getLocation()));

        d_receptorItemModel->insertString("receptors");
        d_receptorItemModel->insertColumns(1,2);
        d_receptorItemModel->setData(d_receptorItemModel->index(0,1), QVariant::fromValue(DCResources::addItemIcon()));

        d_axonTerminalItemModel->insertString("axonTerminals");
        d_axonTerminalItemModel->insertColumns(1,2);
        d_axonTerminalItemModel->setData(d_axonTerminalItemModel->index(0,1), QVariant::fromValue(DCResources::addItemIcon()));

        const TKReceptorMap *receptors = d_cell->getReceptors();
        TKReceptorMap::const_iterator it = receptors->begin();
        QModelIndex rroot = d_receptorItemModel->index(0,0);
        int i = 0;
        while( it != receptors->end())
        {

            QString receptorName = QString::fromStdString((*it).first);            
            d_receptorItemModel->insertString(receptorName,rroot);
            d_receptorItemModel->setData(rroot.child(i,2), QVariant::fromValue(DCResources::deleteItemIcon()));

            DCReceptor *receptor = dynamic_cast<DCReceptor*>((*it).second);
            DCAxonTerminal *terminal = receptor->getTarget();
            if (terminal)
            {
                DCCell *targetCell = dynamic_cast<DCCell*>(terminal->getOwner()->getOwner());
                if (targetCell)
                {
                    QString targetPath = DCUtil::getFQNPath(targetCell->getLocation(), targetCell->getName());
                    d_receptorItemModel->insertString(targetPath,rroot.child(i,0));
                }
            }

            ++it;
            ++i;
        }

        d_receptors->expand(rroot);
        for (int i = 0; i < d_receptorItemModel->rowCount(rroot); i++)
        {
            QModelIndex index = rroot.child(i,0);
            d_receptors->collapse(index);
            if (d_receptorItemModel->rowCount(index) > 0)
            {
                d_receptors->collapse(index.child(0,0));
            }
        }

        QModelIndex aroot = d_axonTerminalItemModel->index(0,0);
        DCAxon *axon = d_cell->getAxon();
        int acnt = axon->getNumberOfTerminals();
        for (int i = 0; i < acnt; i++)
        {
            DCReceptor *targetReceptor = NULL;
            DCCell *targetCell = NULL;
            DCAxonTerminal *tarminal = axon->getTerminalAt(i);
            if (tarminal)
            {
                targetReceptor = tarminal->getTarget();
                if (targetReceptor)
                {
                    targetCell = targetReceptor->getOwnerCell();
                }
            }

            if (targetCell)
            {
                QString targetPath = DCUtil::getFQNPath(targetCell->getLocation(), targetCell->getName());
                QString targetReceptorName = QString::fromStdString(targetCell->getReceptorName(targetReceptor));
                d_axonTerminalItemModel->insertString(targetPath,aroot);
                d_receptorItemModel->setData(aroot.child(i,2), QVariant::fromValue(DCResources::deleteItemIcon()));
                d_axonTerminalItemModel->insertString(targetReceptorName,aroot.child(i,0));
            }
            else
            {
                //TODO
            }
        }

        d_axonTerminals->expand(aroot);
        for (int i = 0; i < d_axonTerminalItemModel->rowCount(aroot); i++)
        {
            QModelIndex index = aroot.child(i,0);
            d_axonTerminals->collapse(index);
            if (d_axonTerminalItemModel->rowCount(index) > 0)
            {
                d_axonTerminals->collapse(index.child(0,0));
            }
        }

        resizeView();
    }
    d_isViewUpdating = false;
}

void DCToolWindowCellEditor::slotReceptorTreeCollapsed(const QModelIndex &index)
{
    (void)index;
    resizeView();
}

void DCToolWindowCellEditor::slotReceptorTreeExpanded(const QModelIndex &index)
{
    (void)index;
    resizeView();
}

void DCToolWindowCellEditor::slotReceptorItemClicked(const QModelIndex &index)
{
    switch (index.column())
    {
    case 1:
        qDebug() << "add item clicked..";
        if (!index.parent().isValid())
        {
            getController()->doCommandStartAddAxonTerminalFromReceptor(this, d_cell);
        }
        break;

    case 2:
        qDebug() << "delete item clicked..";
        if (index.parent().row() == 0)
        {
            int itemRowIndex = index.row();
            if (itemRowIndex >= 0)
            {
                QModelIndex itemIndex = d_receptorItemModel->index(itemRowIndex,0,index.parent());
                QString receptorName = d_receptorItemModel->data(itemIndex,Qt::DisplayRole).toString();
                if (receptorName.length()>0)
                {
                    d_receptors->setCurrentIndex(itemIndex);
                    getController()->doCommandRemoveAxonTerminal(this, d_cell, receptorName);
                }
            }
        }
        break;

    default:
        qDebug() << "other clicked..";
        break;

    }

}

void DCToolWindowCellEditor::slotAxonTerminalTreeCollapsed(const QModelIndex &index)
{
    (void)index;
    resizeView();
}

void DCToolWindowCellEditor::slotAxonTerminalTreeExpanded(const QModelIndex &index)
{
    (void)index;
    resizeView();
}

void DCToolWindowCellEditor::slotAxonTerminalItemClicked(const QModelIndex &index)
{
    switch (index.column())
    {
    case 1:
        qDebug() << "add item clicked..";
        if (!index.parent().isValid())
        {
            getController()->doCommandStartAddAxonTerminalFromAxon(this, d_cell);
        }
        break;

    case 2:
        qDebug() << "delete item clicked..";
        if (index.parent().row() == 0)
        {
            int itemIndex = index.row();
            if (itemIndex >= 0)
            {
                d_axonTerminals->setCurrentIndex(d_axonTerminalItemModel->index(itemIndex,0,index.parent()));
                getController()->doCommandRemoveAxonTerminal(this, d_cell, d_cell->getAxon()->getTerminalAt(itemIndex));
            }
        }
        break;

    default:
        qDebug() << "other clicked..";
        break;

    }
}

void DCToolWindowCellEditor::slotCellCodeEditButtonPressed()
{
    if (d_cell && d_cell->getCellCode())
    {
        DCCellScriptsEditorWindow::startEditing(d_cell, true);
    }
}

void DCToolWindowCellEditor::slotCustomScriptEditButtonPressed()
{
    if (d_cell)
    {
        DCCellScriptsEditorWindow::startEditing(d_cell, false);
    }
}

// TODO
// There may be better / proper way to do this.
// The goal is to adjust the window height to fit all inside components
// with its desigered size.
void DCToolWindowCellEditor::resizeView()
{
    int totalHeight = 0;
    int spacing = d_layout->verticalSpacing();
    for (int i = 0 ; i < d_layout->rowCount(); i++ )
    {
        int maxHeightInRow = 0;
        for (int j = 0; j < d_layout->columnCount(); j++)
        {
            QLayoutItem *item = d_layout->itemAtPosition(i,j);
            if (item)
            {
                QWidget *pWidget = item->widget();
                if (pWidget)
                {
                    QSize sh = pWidget->sizeHint();
                    pWidget->setFixedHeight(sh.height());
                    if (sh.height() > maxHeightInRow)
                    {
                        maxHeightInRow = sh.height();
                    }
                }
            }
        }
        totalHeight += maxHeightInRow  + spacing;
    }

    QSize s = size();
    int left,top,right,bottom;
    contentLayout()->getContentsMargins(&left,&top,&right,&bottom);
    s.setHeight(totalHeight + getTitleButtonHeight() + top + bottom);
    resize(s);

    adjustTreeColumnWidth();
}

void DCToolWindowCellEditor::adjustTreeColumnWidth()
{
    d_receptors->setColumnWidth(1, 18);
    d_receptors->setColumnWidth(2, 18);
    QScrollBar *sb = d_receptors->verticalScrollBar();
    if (sb && sb->isVisible())
        d_receptors->setColumnWidth(0, d_receptors->width()-44 - sb->width());
    else
        d_receptors->setColumnWidth(0, d_receptors->width()-42);

    d_axonTerminals->setColumnWidth(1, 18);
    d_axonTerminals->setColumnWidth(2, 18);
    d_axonTerminals->setColumnWidth(0, d_axonTerminals->width()-42);
    sb = d_axonTerminals->verticalScrollBar();
    if (sb && sb->isVisible())
        d_axonTerminals->setColumnWidth(0, d_axonTerminals->width()-44 - sb->width());
    else
        d_axonTerminals->setColumnWidth(0, d_axonTerminals->width()-42);
}

void DCToolWindowCellEditor::slotCellDestroyed()
{
    setCell(NULL);
}

