//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dctoolwindowbase.h"

const static QString DialogContentStyle =
        "* {color: white} "
        "QDialog {background-color: rgb(110,110,110);} "
        "QRadioButton {background-color: transparent;} "
        "QPlainTextEdit {background: black;} "
        "QGroupBox {border: 1px solid rgb(160,160,160); border-radius: 3px;} "
        "QTableView { background: black;} "
        "QPushButton { background-color: rgb(128,128,128);} "
        "QComboBox { background-color: rgb(80,80,80); "
        "       border: 1px  rgb(80,80,80);"
        "       border-radius: 3px;"
        "       padding: 1px 18px 1px 3px;"
        "       }"
        "QAbstractItemView {"
        "       background: rgb(80,80,80);"
        "       color: white;"
        "       selection-background-color: rgb(80,80,255);"
        "}"
        ;

DCToolWindowBase::DCToolWindowBase(const QString &windowTitle,DCCreator *creator)
    : QDialog(0,Qt::CustomizeWindowHint | Qt::FramelessWindowHint), d_collapsed(false), d_creator(creator)
{
    setWindowOpacity(0.8);
    setLayout(new QVBoxLayout);
    layout()->setContentsMargins(0,0,0,0);

    d_titleButton = new DCToolWindowTitleBar(windowTitle, this);
    d_titleButton->setStyleSheet("* {background-color: rgb(85, 85, 255);} ");
    connect(d_titleButton, SIGNAL(moved(int,int)), this, SLOT(slotMoveWindow(int,int)));
    connect(d_titleButton, SIGNAL(expandOrCollapse()), this, SLOT(slotExpandOrCollapse()));
    connect(d_titleButton, SIGNAL(dragMoveFinished()), this, SLOT(slotDragModeFinished()));
    d_titleButtonHeight = d_titleButton->geometry().height();
    setMinimumHeight(d_titleButtonHeight);
    d_titleButton->setMinimumHeight(d_titleButtonHeight);

    layout()->addWidget(d_titleButton);

    d_layout = new QVBoxLayout;
    d_layout->setContentsMargins(6,6,6,9);
    ((QVBoxLayout*)layout())->addLayout(d_layout);

    setStyleSheet(DialogContentStyle);

}

DCToolWindowBase::~DCToolWindowBase()
{
    this->disconnect();
}

void DCToolWindowBase::setButtonedWindowTitle(const QString &title)
{
    d_titleButton->setText(title);
}

void DCToolWindowBase::resizeEvent(QResizeEvent *event)
{
    (void)event;
}

void DCToolWindowBase::slotMoveWindow(int dx, int dy)
{
    QRect rect = geometry();
    move(rect.x() + dx, rect.y() + dy);
}

void DCToolWindowBase::slotExpandOrCollapse()
{
    d_collapsed = !d_collapsed;

    if (d_collapsed)
    {
        d_rect = QRect(geometry());
        resize(d_rect.width(), d_titleButtonHeight);
    }
    else
    {
        resize(d_rect.width(), d_rect.height());
    }
}

void DCToolWindowBase::slotDragModeFinished()
{
    emit dragMoveFinished();
}

DCToolWindowTitleBar::DCToolWindowTitleBar(const QString &windowTitle, QWidget *parent) : QPushButton(windowTitle, parent)
{
    setStyleSheet("background-color: rgb(54,72,237); color: white;");
}

DCToolWindowTitleBar::~DCToolWindowTitleBar()
{
    this->disconnect();
}

void DCToolWindowTitleBar::mousePressEvent(QMouseEvent *event)
{
    QPushButton::mousePressEvent(event);
    QPoint p = QCursor::pos();
    d_mousePrevX = p.x();
    d_mousePrevY = p.y();
    d_dragStartX = p.x();
    d_dragStartY = p.y();
    d_clickTime.start();
}

void DCToolWindowTitleBar::mouseMoveEvent(QMouseEvent *event)
{
    QPushButton::mouseMoveEvent(event);
    QPoint p = QCursor::pos();
    int dx = p.x() - d_mousePrevX;
    int dy = p.y() - d_mousePrevY;
    d_mousePrevX = p.x();
    d_mousePrevY = p.y();
    emit moved(dx,dy);
}

void DCToolWindowTitleBar::mouseReleaseEvent(QMouseEvent *event)
{
    QPushButton::mouseReleaseEvent(event);
    QPoint p = QCursor::pos();
    int dx = p.x() - d_dragStartX;
    int dy = p.y() - d_dragStartY;

    bool wasDrag = !(d_clickTime.elapsed() < 500 && dx * dx + dy * dy < 400);

    if (!wasDrag)
    {
        emit expandOrCollapse();
    }
    else
    {
        emit dragMoveFinished();
    }
}

