//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcvlayoutmodehandler.h"

#include "dcglvisualizerwidget.h"
#include "dcvcomponent.h"
#include "dcvcpage.h"
#include "dcvcaxon.h"
#include "dcscene.h"
#include "dccreator.h"
#include "dcglwidget.h"

#include "dccontainer.h"
#include "dccell.h"

#include "dialog/dcaddcelldialog.h"
#include "dialog/dcinputnewpagenamedialog.h"
#include "dialog/dcrenamecelldialog.h"
#include "codeeditor/dccellscriptseditorwindow.h"

#include <typeinfo>
#include <QList>

DCVLayoutModeHandler::DCVLayoutModeHandler(DCUIGraphicsScene *widget, DCCreator *creator, DCScene *scene) :
    DCVEventHandler(widget, creator, scene), d_inResigingDrag(false)
{

}

DCVLayoutModeHandler::~DCVLayoutModeHandler()
{

}

int DCVLayoutModeHandler::getHandlerType() const
{
    return DCScene::DCV_EDITMODE_LAYOUT;
}

void DCVLayoutModeHandler::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    bool shift = event->modifiers() & Qt::ShiftModifier;

    QPointF mousePos = event->scenePos();

    DCVComponent *obj = getView()->getObjectAt(mousePos.x(), mousePos.y());
    if (!obj)
        return;

    if (!obj->getIsSelectable())
        return;

    DCVCPage *page = dynamic_cast<DCVCPage*>(obj->getPageBelonging());
    if (!page)
        return;

    switch(getScene()->getPersMode())
    {
    case DCScene::DCV_PERSMODE_NAVIGATION:
        {
            if (obj != page)
            {
                float pageX, pageZ;
                getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), page, &pageX, &pageZ);
                getController()->selectPage(this, page, shift);
                getController()->selectCellObject(this, obj);
                if (obj->startDrag(pageX, 0, pageZ, false))
                {
                    getView()->requestRedraw(true);
                    d_draggedObject.assign(obj);
                    d_inResigingDrag = false;
                }
            }
            else
            {
                d_draggedObject.assign(obj);
                d_inResigingDrag = false;
            }
        }
        break;

    case DCScene::DCV_PERSMODE_PAGEEDIT:
        {
            float pageX, pageZ;
            float objX, objZ;
            getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), obj, &objX, &objZ);
            getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), page, &pageX, &pageZ);
            d_inResigingDrag = obj->isResizingArea(objX,0,objZ);
            getController()->selectCellObject(this, obj);
            if (obj->startDrag(pageX, 0, pageZ, d_inResigingDrag))
            {
                getView()->requestRedraw(true);
            }
            d_draggedObject.assign(obj);
        }
        break;
    }

    event->accept();
}

void DCVLayoutModeHandler::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
    DCVComponent *obj = d_draggedObject.ref();

    if (!obj)
        return;

    if (obj->getOwnerCell() && !obj->getOwnerCell()->getVComponent()->getIsSelectable())
        return;

    DCVComponent *page = obj->getPageBelonging();
    if (!page)
        return;

    switch(getScene()->getPersMode())
    {
    case DCScene::DCV_PERSMODE_NAVIGATION:
        {
            getController()->changePersMode(this, DCCreator::DC_PERSMODE_PAGEEDIT);
            d_draggedObject.assign(NULL);
        }

        break;

    case DCScene::DCV_PERSMODE_PAGEEDIT:
         {
            if (obj == page)
            {
                getController()->changePersMode(this, DCCreator::DC_PERSMODE_NAVIGATION);
                d_draggedObject.assign(NULL);
            }
            else
            {
                if (typeid(*obj) == typeid(DCVCAxon))
                {
                    getController()->doCommandStartAddAxonTerminalFromAxon(this, obj->getOwnerCell());
                }
                else
                {
                    DCCellScriptsEditorWindow::startEditing(d_draggedObject.ref()->getOwnerCell());
                }
            }
        }
        break;
    }

    event->accept();
}

void DCVLayoutModeHandler::mouseMoveEvent(QGraphicsSceneMouseEvent *event, int clickTimeElapsed)
{
    QPointF mousePos = event->scenePos();
    QPointF mousePrevPos = event->lastScenePos();

    // This method will be called continuously while the mouse is moving.
    // Some process we want to execute here may become heavyer and it will affect the performance.
    // To privent this to affect the performance, we only execute the potentially heavyer
    // process 12.5 times in a second in maximum.
    static int lastCheckTime = 0;
    bool canProcessHeavyTask = clickTimeElapsed - lastCheckTime < 0 || clickTimeElapsed - lastCheckTime >= 80;
    if (canProcessHeavyTask)
        lastCheckTime = clickTimeElapsed;


    if (event->buttons() == Qt::LeftButton)
    {
        //dragging
        bool shift = event->modifiers() & Qt::ShiftModifier;
        float dx = mousePos.x() - mousePrevPos.x();
        float dy = mousePos.y() - mousePrevPos.y();

        switch(getScene()->getPersMode())
        {
        case DCScene::DCV_PERSMODE_NAVIGATION:
            if (getScene()->getNavModeSelectedCellMovable() && d_draggedObject.ref() && d_draggedObject.ref()->getPageBelonging() && d_draggedObject.ref() != d_draggedObject.ref()->getPageBelonging())
            {
                float pageX, pageZ;
                getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), d_draggedObject.ref()->getPageBelonging(), &pageX, &pageZ);

                if (d_draggedObject.ref()->dragging(pageX, 0, pageZ, false))
                {
                    DCGLWidget::singleton()->setCursor(Qt::OpenHandCursor);
                    getView()->requestRedraw(false);
                }
            }
            else if (!shift)
            {
                getController()->rotateScene(this, dy / 5.0, dx / 5.0);
            }
            else
            {
                getController()->translateBrowsModeScene(this, dx, dy);
            }
            break;

        case DCScene::DCV_PERSMODE_PAGEEDIT:
            if (d_draggedObject.ref() && d_draggedObject.ref()->getPageBelonging())
            {
                if (d_draggedObject.ref() == d_draggedObject.ref()->getPageBelonging())
                {
                    getController()->translateEditModeScene(this, dx, dy);
                }
                else
                {
                    float pageX, pageZ;
                    getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), d_draggedObject.ref()->getPageBelonging(), &pageX, &pageZ);
                    if (d_draggedObject.ref()->dragging(pageX, 0, pageZ, d_inResigingDrag))
                    {
                        if (d_inResigingDrag)
                        {
                            DCGLWidget::singleton()->setCursor(Qt::SizeFDiagCursor);
                        }
                        else
                        {
                            DCGLWidget::singleton()->setCursor(Qt::OpenHandCursor);
                        }
                        getView()->requestRedraw(false);
                    }
                }
            }
            break;
        }
    }
    else
    {
        //mouse moving
        if (canProcessHeavyTask)
        {
            DCVComponent *obj = getView()->getObjectAt(mousePos.x(), mousePos.y());
            float objX, objZ;
            if (obj)
                getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), obj, &objX, &objZ);
            if (obj && obj->isResizingArea(objX,0,objZ))
            {
                DCGLWidget::singleton()->setCursor(Qt::SizeFDiagCursor);
            }
            else
            {
                DCGLWidget::singleton()->setCursor(Qt::ArrowCursor);
            }
        }
    }

    event->accept();
}

void DCVLayoutModeHandler::mouseReleaseEvent(QGraphicsSceneMouseEvent *event, bool wasDrag)
{
    bool shift = event->modifiers() & Qt::ShiftModifier;
    QPointF mousePos = event->scenePos();

    switch(getScene()->getPersMode())
    {
    case DCScene::DCV_PERSMODE_NAVIGATION:
    {
        DCVCPage *page = NULL;
        if (d_draggedObject.ref())
        {
            page = d_draggedObject.ref()->getPageBelonging();
            float pageX, pageZ;
            getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), page, &pageX, &pageZ);
            if (d_draggedObject.ref()->endDrag(pageX, 0, pageZ, d_inResigingDrag))
            {
                getView()->requestRedraw(true);
            }
        }

        if (!wasDrag)
        {
            if (page)
            {
                getController()->selectPage(this, page, shift);
            }
            else
            {
                getController()->unselectPageAll(this);
            }
        }
    }
        break;

    case DCScene::DCV_PERSMODE_PAGEEDIT:
        if (d_draggedObject.ref() && d_draggedObject.ref()->getPageBelonging())
        {
            float pageX, pageZ;
            getView()->toObjectCoordinateXZ(mousePos.x(), mousePos.y(), d_draggedObject.ref()->getPageBelonging(), &pageX, &pageZ);
            if (d_draggedObject.ref()->endDrag(pageX, 0, pageZ, d_inResigingDrag))
            {
                getView()->requestRedraw(true);
            }
        }
        break;
    }
    event->accept();
}

void DCVLayoutModeHandler::wheelEvent(QGraphicsSceneWheelEvent  *event)
{
    if (event->isAccepted()) return;

    getController()->changeSceneScale(this, getScene()->getScale() + event->delta()/5);

    event->accept();
}

void DCVLayoutModeHandler::keyPressEvent(QKeyEvent *event)
{
    switch(event->key())
    {
    case Qt::Key_Enter:
    case Qt::Key_Return:
    {
        switch(getScene()->getPersMode())
        {
        case DCScene::DCV_PERSMODE_NAVIGATION:
            if (getScene()->getSelectedPages().length() > 0)
            {
                getController()->changePersMode(this, DCCreator::DC_PERSMODE_PAGEEDIT);
            }
            break;

        case DCScene::DCV_PERSMODE_PAGEEDIT:
            if (getScene()->getSelectedCellObjects().length() == 1)
            {
                DCCellScriptsEditorWindow::startEditing(getScene()->getSelectedCellObjects().at(0)->getOwnerCell());
            }
            break;

        }
    }
        break;

    case Qt::Key_Escape:
    {
        switch(getScene()->getPersMode())
        {
        case DCScene::DCV_PERSMODE_NAVIGATION:
            getController()->unselectPageAll(this);
            break;

        case DCScene::DCV_PERSMODE_PAGEEDIT:
            getController()->changePersMode(this, DCCreator::DC_PERSMODE_NAVIGATION);
            break;
        }
    }
        break;
    }
}

void DCVLayoutModeHandler::keyReleaseEvent(QKeyEvent *event)
{
    (void)event;

}

void DCVLayoutModeHandler::showContextMenu(const QPoint &pos)
{
    QPoint globalPos = DCGLWidget::singleton()->mapToGlobal(pos);

    QAction *actionRenameCell = NULL;
    QAction *actinAddPage = NULL;
    QAction *actionEditPage = NULL;
    QAction *actionRemovePage = NULL;
    QAction *actionSavePage = NULL;
    QAction *actionSaveAll = NULL;
    QAction *actionEditCellCode = NULL;
    QAction *actionAddCell = NULL;
    QAction *actionRemoveCell = NULL;
    QAction *actionFinishEditMode = NULL;

    DCCell* selectedCell = NULL;
    DCVCPage* selectedPage = NULL;

    DCVComponent *obj = getView()->getObjectAt(pos.x(), pos.y());
    if (obj && obj->getIsSelectable())
    {
        selectedCell = obj->getOwnerCell();
        selectedPage = obj->getPageBelonging();
    }

    QMenu menu;
    switch(getScene()->getPersMode())
    {
    case DCScene::DCV_PERSMODE_NAVIGATION:
        actionAddCell = menu.addAction(tr("Add cell ..."));
        if (selectedCell)
        {
            actionRenameCell = menu.addAction(tr("Rename / move cell..."));
            menu.addSeparator();
            actionRemoveCell = menu.addAction(tr("Remove cell"));
        }
        menu.addSeparator();
        actinAddPage = menu.addAction(tr("Add page ..."));
        if (selectedPage)
        {
            actionEditPage = menu.addAction(tr("Edit page ..."));
            actionRemovePage = menu.addAction(tr("Remove page ..."));
            actionSavePage = menu.addAction(tr("Save page"));
            menu.addSeparator();
        }
        actionSaveAll = menu.addAction(tr("Save all"));

        break;

    case DCScene::DCV_PERSMODE_PAGEEDIT:

        actionAddCell = menu.addAction(tr("Add cell ..."));
        if (selectedCell)
        {
            actionRenameCell = menu.addAction(tr("Rename / move cell..."));
            actionEditCellCode = menu.addAction(tr("Edit cell code..."));
            menu.addSeparator();
            actionRemoveCell = menu.addAction(tr("Remove cell"));
        }
        menu.addSeparator();

        actionSavePage =  menu.addAction(tr("Save page"));
        menu.addSeparator();
        actionSaveAll = menu.addAction(tr("Save all"));
        menu.addSeparator();
        actionFinishEditMode = menu.addAction(tr("Finish edit mode"));

        break;
    }

    QAction* selectedItem = menu.exec(globalPos);
    if (selectedItem)
    {
        DCContainer *container = getController()->getCurrentContainer();
        if (selectedItem == actinAddPage)
        {
            QString sysFilePath = QString::fromStdString(container->getContainerRootPath());

            if (selectedPage)
            {
                sysFilePath = QFileInfo(sysFilePath + selectedPage->getLocationPath()).dir().absolutePath();
            }
            QString containerBasedPath = container->sysFilePathToContainerBasedPath(sysFilePath);

            DCInputNewPageNameDialog dialog(sysFilePath, tr("Add page"));
            if (dialog.exec())
            {
                QString pageName = dialog.getName();
                if (pageName.length() > 0)
                {
                    QString pathName = containerBasedPath;
                    if (containerBasedPath.length()>1)
                        pathName.append("/");
                    pathName.append(pageName);
                    getController()->doCommandAddPage(this, pathName);
                }
            }
        }
        else if (selectedItem == actionRenameCell)
        {
            if (selectedCell)
            {
                DCRenameCellDialog dialog(getController()->getCurrentContainer(), getController(), selectedCell);
                dialog.exec();

            }
        }
        else if (selectedItem == actionEditPage)
        {
            getController()->changePersMode(this, DCCreator::DC_PERSMODE_PAGEEDIT);
        }
        else if (selectedItem == actionRemovePage)
        {
            if (getScene()->getSelectedPages().length() > 0)
            {
                QList<DCVCPage*>  pages = getScene()->getSelectedPages();
                for (int i = 0; i < pages.length(); i++)
                {
                    getController()->doCommandRemovePage(this, pages.at(i));
                }
            }
        }
        else if (selectedItem == actionSavePage)
        {
            if (getScene()->getSelectedPages().length() > 0)
            {
                bool r = true;
                QList<DCVCPage*>  pages = getScene()->getSelectedPages();
                if (!getController()->savePage(pages.toSet()))
                {
                    r = false;
                }
                QMessageBox msgBox;
                if (r)
                {
                    msgBox.setText(tr("Page is saved successfully"));
                }
                else
                {
                    msgBox.setText(tr("Error!! Failed to save the page file"));
                }
                msgBox.exec();
            }
        }
        else if (selectedItem == actionSaveAll)
        {
            getController()->saveAll(true);

        }
        else if (selectedItem == actionEditCellCode)
        {
            if (selectedCell)
            {
                DCCellScriptsEditorWindow::startEditing(selectedCell);
            }
        }
        else if (selectedItem == actionAddCell)
        {
            if (selectedPage)
            {
                float pageX = 0;
                float pageY = 0;
                getView()->toObjectCoordinateXZ(pos.x(), pos.y(),selectedPage, &pageX, &pageY );
                DCAddCellDialog dialog(getController()->getCurrentContainer(), getController(), selectedPage->getLocationPath(), pageX, pageY);
                dialog.exec();
            }
        }
        else if (selectedItem == actionRemoveCell)
        {
            getController()->doCommandRemoveCell(this, getController()->getCurrentContainer(), selectedCell);
        }
        else if (selectedItem == actionFinishEditMode)
        {
            getController()->changePersMode(this, DCCreator::DC_PERSMODE_NAVIGATION);
        }
    }
    else
    {
        // nothing was chosen
    }
}

void DCVLayoutModeHandler::selectedPageChanged(const void *requester)
{
    (void)requester;

}
