/*
 *  cuberenderer.cpp
 *
 *  CubeRenderer class - rendering rounded cube
 *
 *  Copyright (C) 2010  Takuji Kawata
 */
//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//

#include "TKLog.h"
#include "dccuberenderer.h"
#include "dcglwidget.h"

#include <math.h>
#include <stdio.h>
#if !defined(Q_WS_MAC)
    #include <GL/gl.h>
#else
    #include <gl.h>
#endif

static const double PI = 3.14159265358979323846264338327950288419717;

typedef float Mat3x3_t[3][3];

// no transform
Mat3x3_t MATRIX_BASE = {
                 { 1, 0, 0 },
                 { 0, 1, 0 },
                 { 0, 0, 1 }};
// y rotate 90
Mat3x3_t MATRIX_Y90 = {
                 { 0, 0, 1},
                 { 0, 1, 0},
                 {-1, 0, 0}};
// y rotate 180
Mat3x3_t MATRIX_Y180 = {
                 {-1, 0, 0},
                 { 0, 1, 0},
                 { 0, 0,-1}};
// y rotate 270
Mat3x3_t MATRIX_Y270 = {
                 { 0, 0,-1},
                 { 0, 1, 0},
                 { 1, 0, 0}};
// z rotate 90
Mat3x3_t MATRIX_Z90 = {
                 { 0,-1, 0},
                 { 1, 0, 0},
                 { 0, 0, 1}};
// z rotate 180
Mat3x3_t MATRIX_Z180 = {
                 {-1, 0, 0},
                 { 0,-1, 0},
                 { 0, 0, 1}};
// y roate 90 + x rotate 90
Mat3x3_t MATRIX_Y90X90 = {
                 { 0,-1, 0},
                 { 0, 0, 1},
                 {-1, 0, 0}};
// y roate 90 + x rotate 180
Mat3x3_t MATRIX_Y90X180 = {
                 { 0, 0,-1},
                 { 0,-1, 0},
                 {-1, 0, 0}};
// x rotate 90
Mat3x3_t MATRIX_X90 = {
                 { 1, 0, 0},
                 { 0, 0, 1},
                 { 0,-1, 0}};
// x rotate 270
Mat3x3_t MATRIX_X270 = {
                 { 1, 0, 0},
                 { 0, 0,-1},
                 { 0, 1, 0}};
// x rotate 90 + y roate 90
Mat3x3_t MATRIX_X90Y90 = {
                 { 0, 0, 1},
                 {-1, 0, 0},
                 { 0,-1, 0}};

// x rotate 90 + y roate 180
Mat3x3_t MATRIX_X90Y180 = {
                 {-1, 0, 0},
                 { 0, 0,-1},
                 { 0,-1, 0}};
// x rotate 90 + y roate 270
Mat3x3_t MATRIX_X90Y270 = {
                 { 0, 0,-1},
                 { 1, 0, 0},
                 { 0,-1, 0}};



void translate(const float verticesIn[3], float verticesOut[3], const Mat3x3_t matrix)
{
    verticesOut[0] = verticesIn[0] * matrix[0][0] + verticesIn[1] * matrix[1][0] + verticesIn[2] * matrix[2][0];
    verticesOut[1] = verticesIn[0] * matrix[0][1] + verticesIn[1] * matrix[1][1] + verticesIn[2] * matrix[2][1];
    verticesOut[2] = verticesIn[0] * matrix[0][2] + verticesIn[1] * matrix[1][2] + verticesIn[2] * matrix[2][2];
}


DCCubeRenderer::DCCubeRenderer(float size, float height, float radius, int r_step, bool cache)
    : m_size(size), m_height(height), m_radius(radius), m_step(r_step), m_cache(cache), m_shapeId(0), m_shapeUpdated(true)
{
    for (int i = 0; i < 6; i++)
    {
        m_textures[i] = 0;
    }

    m_edgeColor[0] = 0;
    m_edgeColor[1] = 0;
    m_edgeColor[2] = 0;
    m_edgeColor[3] = 1;

    m_faceColor[0] = 1;
    m_faceColor[1] = 1;
    m_faceColor[2] = 1;
    m_faceColor[3] = 1;

    m_text1 = "";
    m_text2 = "";
    m_widget = DCGLWidget::singleton();
}

DCCubeRenderer::~DCCubeRenderer()
{
    deleteTextures();
    if (m_shapeId)
    {
        glDeleteLists(m_shapeId,1);
    }
}

void DCCubeRenderer::draw()
{
    if (m_shapeUpdated)
    {
        if (m_shapeId != 0)
        {
            glDeleteLists(m_shapeId, 1);
            m_shapeId = 0;
        }
        deleteTextures();
        buildTextures();
    }

    if (m_shapeId)
    {
        glCallList(m_shapeId);
    }
    else
    {
        if (m_cache)
        {
            m_shapeId = glGenLists(1);
            if (m_shapeId <= 0)
            {
                // ERROR for creating glGenList
                m_shapeId = 0;
                buildShape();
                TKLog::printf(TKLog::WARNING, "FAILED for create displaylist in DCCubeRender::draw(). Out-of-memory?");
            }

            glNewList(m_shapeId, GL_COMPILE);
            buildShape();
            glEndList();

            glCallList(m_shapeId);
        }
        else
        {
            buildShape();
        }
    }

    m_shapeUpdated = false;
}

void DCCubeRenderer::setEdgeColor(GLfloat r, GLfloat g, GLfloat b)
{
    m_edgeColor[0] = r;
    m_edgeColor[1] = g;
    m_edgeColor[2] = b;
    m_shapeUpdated = true;
}

void DCCubeRenderer::setFaceColor(GLfloat r, GLfloat g, GLfloat b)
{
    m_faceColor[0] = r;
    m_faceColor[1] = g;
    m_faceColor[2] = b;
    m_shapeUpdated = true;
}

void DCCubeRenderer::setTextLabel(const QString &text1, const QString &text2, const QString &text3)
{
    QString newText1 = " " + text1 + " ";
    QString newText2 = " " + text2 + " ";

    if (newText1 != m_text1 || newText2 != m_text2 || text3 != m_text3)
    {
        m_text1 = newText1;
        m_text2 = newText2;
        m_text3 = text3;
        m_shapeUpdated = true;
    }
}

void DCCubeRenderer::setSize(float size_)
{
    if (m_size != size_)
    {
        m_size = size_;
        m_shapeUpdated = true;
    }
}

void DCCubeRenderer::setHeight(float height)
{
    if (m_height != height)
    {
        m_height = height;
        m_shapeUpdated = true;
    }
}

bool DCCubeRenderer::buildShape()
{
    renderTopAndBottomFaces();
    renderSideFaces();

    if (m_radius == 0)
    {
        return true;
    }

    renderEdges();
    renderJoints();

    return true;
}

void DCCubeRenderer::buildTextures()
{
    if (m_widget)
    {
        //texture for top and bottom
        {
            int texSize = 128 * (m_size < 1 ? 1 : m_size);
            QPixmap image(texSize,texSize);
            image.fill(QColor(m_edgeColor[0]*255, m_edgeColor[1]*255, m_edgeColor[2]*255));
            QPainter painter;
            painter.begin(&image);

            float r = texSize * m_radius / m_size;
            painter.setPen(Qt::NoPen);
            painter.setBrush(QBrush(QColor(m_faceColor[0]*255, m_faceColor[1]*255, m_faceColor[2]*255)));
            painter.drawRoundedRect(0,0, texSize, texSize, r,r);

            float text1Rate = m_text3.length() > 0 ? 0.3 : 0.4;
            float text2Rate = m_text3.length() > 0 ? 0.4 : 0.6;

            QFont font;
            font.setPointSize(22);
            font.setItalic(true);
            painter.setPen(Qt::black);
            painter.setFont(font);
            QRectF rect = painter.boundingRect(0,0,image.width(), image.height()*text1Rate, Qt::AlignCenter | Qt::TextDontClip, m_text1);
            if (rect.width() > image.rect().width())
            {
                font.setPointSizeF(22*(image.rect().width()/rect.width()));
                painter.setFont(font);
            }
            painter.drawText(0,0,image.width(),image.height()*text1Rate, Qt::AlignCenter | Qt::TextDontClip, m_text1);


            font.setPointSize(48);
            font.setItalic(false);
            painter.setPen(Qt::black);
            painter.setFont(font);
            rect = painter.boundingRect(0,0,image.width(), image.height()*text2Rate, Qt::AlignCenter | Qt::TextDontClip, m_text2);
            if (rect.width() > image.rect().width())
            {
                font.setPointSizeF(48*(image.rect().width()/rect.width()));
                painter.setFont(font);
            }
            painter.drawText(0, image.height()*text1Rate, image.width(), image.height()*text2Rate, Qt::AlignCenter | Qt::TextDontClip, m_text2);

            if (m_text3.length() > 0)
            {
                font.setPointSize(22);
                font.setItalic(false);
                painter.setPen(Qt::black);
                painter.setFont(font);
                rect = painter.boundingRect(0,0,image.width(), image.height()*0.2, Qt::AlignCenter | Qt::TextDontClip, m_text3);
                if (rect.width() > image.rect().width())
                {
                    font.setPointSizeF(22*(image.rect().width()/rect.width()));
                    painter.setFont(font);
                }
                painter.drawText(0, image.height()*0.7, image.width(), image.height()*0.2, Qt::AlignCenter | Qt::TextDontClip, m_text3);
            }

            GLuint texture = m_widget->bindTexture(image, GL_TEXTURE_2D,GL_RGBA,QGLContext::LinearFilteringBindOption|QGLContext::InvertedYBindOption);

            for (int i = 0; i < 2; i++)
            {
                m_textures[i] = texture;
            }
            glBindTexture(GL_TEXTURE_2D, 0);
        }

        //texture for side faces
        {
            int texSize_w = 128;
            int texSize_h = 128;
            if (m_size < m_height)
            {
                texSize_w = 128 * (m_size < 1 ? 1 : m_size);
                texSize_h = 128 * m_height / m_size;
            }
            else
            {
                texSize_h = 128 * (m_height < 1 ? 1 : m_height);
                texSize_w = 128 * m_size / m_height;
            }

            QPixmap image(texSize_w,texSize_h);
            image.fill(QColor(m_edgeColor[0]*255, m_edgeColor[1]*255, m_edgeColor[2]*255));
            QPainter painter;
            painter.begin(&image);

            float r_w = texSize_w * m_radius / m_size;
            float r_h = texSize_h * m_radius / m_height;
            painter.setPen(Qt::NoPen);
            painter.setBrush(QBrush(QColor(m_faceColor[0]*255, m_faceColor[1]*255, m_faceColor[2]*255)));
            painter.drawRoundedRect(0,0, texSize_w, texSize_h, r_w, r_h);

            float text1Rate = m_text3.length() > 0 ? 0.3 : 0.4;
            float text2Rate = m_text3.length() > 0 ? 0.4 : 0.6;

            QFont font;
            font.setPointSize(22);
            font.setItalic(true);
            painter.setPen(Qt::black);
            painter.setFont(font);
            QRectF rect = painter.boundingRect(0,0,image.width(), image.height()*text1Rate, Qt::AlignCenter | Qt::TextDontClip, m_text1);
            if (rect.width() > image.rect().width())
            {
                font.setPointSizeF(22*(image.rect().width()/rect.width()));
                painter.setFont(font);
            }
            painter.drawText(0,0,image.width(),image.height()*text1Rate, Qt::AlignCenter | Qt::TextDontClip, m_text1);


            font.setPointSize(48);
            font.setItalic(false);
            painter.setPen(Qt::black);
            painter.setFont(font);
            rect = painter.boundingRect(0,0,image.width(), image.height()*text2Rate, Qt::AlignCenter | Qt::TextDontClip, m_text2);
            if (rect.width() > image.rect().width())
            {
                font.setPointSizeF(48*(image.rect().width()/rect.width()));
                painter.setFont(font);
            }
            painter.drawText(0, image.height()*text1Rate, image.width(), image.height()*text2Rate, Qt::AlignCenter | Qt::TextDontClip, m_text2);

            if (m_text3.length() > 0)
            {
                font.setPointSize(22);
                font.setItalic(false);
                painter.setPen(Qt::black);
                painter.setFont(font);
                rect = painter.boundingRect(0,0,image.width(), image.height()*0.2, Qt::AlignCenter | Qt::TextDontClip, m_text3);
                if (rect.width() > image.rect().width())
                {
                    font.setPointSizeF(22*(image.rect().width()/rect.width()));
                    painter.setFont(font);
                }
                painter.drawText(0, image.height()*0.7, image.width(), image.height()*0.2, Qt::AlignCenter | Qt::TextDontClip, m_text3);
            }

            GLuint texture = m_widget->bindTexture(image, GL_TEXTURE_2D,GL_RGBA,QGLContext::LinearFilteringBindOption|QGLContext::InvertedYBindOption);

            for (int i = 2; i < 6; i++)
            {
                m_textures[i] = texture;
            }
            glBindTexture(GL_TEXTURE_2D, 0);
        }
    }
}

void DCCubeRenderer::deleteTextures()
{
    GLuint deletedTextureIds[6];

    for (int i = 0; i < 6; i++)
    {
        if (m_textures[i] != 0)
        {
            bool alreadyDeleted = false;
            for (int j = 0; j < i; j++)
            {
                if (deletedTextureIds[j] == m_textures[i])
                {
                    alreadyDeleted = true;
                    break;
                }
            }
            if (!alreadyDeleted)
                glDeleteTextures(1, &m_textures[i]);
            deletedTextureIds[i] = m_textures[i];
            m_textures[i] = 0;
        }
    }
}

void DCCubeRenderer::renderTopAndBottomFaces()
{
    float vertices[4][3];
    float normals[4][3];

    float vertexOut[3];
    float normalOut[3];

    float sh = m_size / 2.0 - m_radius;
    float hh = m_height / 2.0;

    Mat3x3_t *trans[4] = {&MATRIX_BASE, &MATRIX_Z180};

    normals[0][0]   = 0;    normals[0][1]   = 1;    normals[0][2]   = 0;
    vertices[0][0]  = -sh;  vertices[0][1]  = hh;   vertices[0][2]  = -sh;

    normals[1][0]   = 0;    normals[1][1]   = 1;    normals[1][2]   = 0;
    vertices[1][0]  = -sh;  vertices[1][1]  = hh;   vertices[1][2]  = sh;

    normals[2][0]   = 0;    normals[2][1]   = 1;    normals[2][2]   = 0;
    vertices[2][0]  =  sh;  vertices[2][1]  = hh;   vertices[2][2]  = sh;

    normals[3][0]   = 0;    normals[3][1]   = 1;    normals[3][2]   = 0;
    vertices[3][0]  = sh;   vertices[3][1]  = hh;   vertices[3][2]  = -sh;

    for (int i = 0; i < 2; i++)
    {
        const float texcords[4][2] = {{0,1}, {0,0}, {1,0}, {1,1}};

        if (m_textures[i] != 0)
        {
            glColor4fv(m_faceColor);
            glBindTexture(GL_TEXTURE_2D ,m_textures[i]);
        }

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

        glBegin(GL_QUADS);

        for (int j = 0; j < 4; j++)
        {
            translate(vertices[j], vertexOut, *trans[i]);
            translate(normals[j], normalOut, *trans[i]);

            glTexCoord2fv(texcords[j]);
            glNormal3fv(normalOut);
            glVertex3fv(vertexOut);
        }
        glEnd();

        if (m_textures[i] != 0)
            glBindTexture(GL_TEXTURE_2D, 0);
    }
}

void DCCubeRenderer::renderSideFaces()
{
    float vertices[4][3];
    float normals[4][3];

    float vertexOut[3];
    float normalOut[3];

    float sh0 = m_size / 2.0;
    float sh  = m_size / 2.0 - m_radius;
    float hh = m_height / 2.0 - m_radius;

    Mat3x3_t *trans[4] = {&MATRIX_BASE, &MATRIX_Y90, &MATRIX_Y180,&MATRIX_Y270};

    normals[0][0]   = 0;    normals[0][1]   = 0;    normals[0][2]   = 1.0;
    vertices[0][0]  = sh;   vertices[0][1]  = hh;   vertices[0][2]  = sh0;

    normals[1][0]   = 0;    normals[1][1]   = 0;    normals[1][2]   = 1.0;
    vertices[1][0]  = -sh;  vertices[1][1]  = hh;   vertices[1][2]  = sh0;

    normals[2][0]   = 0;    normals[2][1]   = 0;    normals[2][2]   = 1.0;
    vertices[2][0] = - sh;  vertices[2][1]  = -hh;  vertices[2][2]  = sh0;

    normals[3][0]   = 0;   normals[3][1]    = 0;    normals[3][2]   = 1.0;
    vertices[3][0]  = sh;   vertices[3][1]  = -hh;  vertices[3][2]  = sh0;

    for (int i = 0; i < 4; i++)
    {
        const float texcords[4][2] = {{1,1}, {0,1}, {0,0}, {1,0}};


        if (m_textures[i+2] != 0)
        {
            glColor4fv(m_faceColor);
            glBindTexture(GL_TEXTURE_2D ,m_textures[i+2]);
        }

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

        glBegin(GL_QUADS);

        for (int j = 0; j < 4; j++)
        {
            translate(vertices[j], vertexOut, *trans[i]);
            translate(normals[j], normalOut, *trans[i]);

            glTexCoord2fv(texcords[j]);
            glNormal3fv(normalOut);
            glVertex3fv(vertexOut);
        }
        glEnd();

        if (m_textures[i+2] != 0)
            glBindTexture(GL_TEXTURE_2D, 0);
    }
}

void DCCubeRenderer::renderEdges()
{
    float vertex[3];
    float normal[3];

    float vertexOut[3];
    float normalOut[3];

    float sh = m_size / 2.0 - m_radius;
    float hh = m_height / 2.0 - m_radius;

    float rad_step = PI / 2.0 / m_step;

    //horizontal part
    int n = 0;
    float side = 1;
    for (int i = 0; i < 8; i ++)
    {
        Mat3x3_t *trans[4] = {&MATRIX_BASE, &MATRIX_Y90, &MATRIX_Y180,&MATRIX_Y270};
        glBegin(GL_TRIANGLE_STRIP);
        glColor4fv(m_edgeColor);
        for (int j = 0 ; j <= m_step ; j++)
        {
            float xn = cos(rad_step * j);
            float yn = sin(rad_step * j);
            float xp = m_radius * cos(rad_step * j);
            float yp = m_radius * sin(rad_step * j);

            normal[0] = xn; normal[1] = yn; normal[2] = 0;
            vertex[0] = (sh + xp) * side;
            vertex[1] = (hh + yp) * side;
            vertex[2] = sh;
            translate(vertex, vertexOut, *trans[n]);
            translate(normal, normalOut, *trans[n]);
            glNormal3fv(normalOut);
            glVertex3fv(vertexOut);

            normal[0] = xn; normal[1] = yn; normal[2] = 0;
            vertex[2] = - sh;
            translate(vertex, vertexOut, *trans[n]);
            translate(normal, normalOut, *trans[n]);
            glNormal3fv(normalOut);
            glVertex3fv(vertexOut);
        }
        glEnd();
        n++;
        if (n==4)
        {
            n = 0;
            side = -1;
        }
    }

    //vertical part
    for (int i = 0; i < 4; i ++)
    {
        glBegin(GL_TRIANGLE_STRIP);
        Mat3x3_t *trans[4] = {&MATRIX_BASE, &MATRIX_Y90, &MATRIX_Y180,&MATRIX_Y270};
        glBegin(GL_TRIANGLE_STRIP);
        glColor4fv(m_edgeColor);
        for (int j = 0 ; j <= m_step ; j++)
        {
            float xn = cos(rad_step * j);
            float zn = sin(rad_step * j);
            float xp = m_radius * cos(rad_step * j);
            float zp = m_radius * sin(rad_step * j);

            normal[0] = xn; normal[1] = 0; normal[2] = zn;
            vertex[0] = sh + xp;
            vertex[1] = - hh;
            vertex[2] = sh + zp;
            translate(vertex, vertexOut, *trans[i]);
            translate(normal, normalOut, *trans[i]);
            glNormal3fv(normalOut);
            glVertex3fv(vertexOut);

            normal[0] = xn; normal[1] = 0; normal[2] = zn;
            vertex[1] = hh;
            translate(vertex, vertexOut, *trans[i]);
            translate(normal, normalOut, *trans[i]);
            glNormal3fv(normalOut);
            glVertex3fv(vertexOut);
        }
        glEnd();
    }
}

void DCCubeRenderer::renderJoints()
{
    float vertex[3];
    float normal[3];

    float vertexOut[3];
    float normalOut[3];

    float sh = m_size / 2.0 - m_radius;
    float hh = m_height / 2.0 - m_radius;

    float rad_step = PI / 2.0 / m_step;

    int n = 0;
    float side = 1;
    for (int i = 0; i < 8; i ++)
    {
        glBegin(GL_TRIANGLE_STRIP);
        Mat3x3_t *trans[4] = {&MATRIX_BASE, &MATRIX_Y90, &MATRIX_Y180,&MATRIX_Y270};
        glColor4fv(m_edgeColor);
        glBegin(GL_TRIANGLE_STRIP);
        for (int j = 0 ; j < m_step ; j++)
        {
            float x0n = cos(rad_step * j);
            float y0n = sin(rad_step * j);

            float x1n = cos(rad_step * (j+1));
            float y1n = sin(rad_step * (j+1));

            for (int k = 0; k <= m_step ; k++)
            {
                normal[0] = x0n * cos(rad_step * k) * side;
                normal[1] = y0n * side;
                normal[2] = x0n * sin(rad_step * k);
                vertex[0] = sh * side + m_radius * normal[0];
                vertex[1] = hh * side + m_radius * normal[1];
                vertex[2] = sh + m_radius * normal[2];

                translate(vertex, vertexOut, *trans[n]);
                translate(normal, normalOut, *trans[n]);
                glNormal3fv(normalOut);
                glVertex3fv(vertexOut);

                normal[0] = x1n * cos(rad_step * k) * side;
                normal[1] = y1n * side;
                normal[2] = x1n * sin(rad_step * k);
                vertex[0] = sh * side + m_radius * normal[0];
                vertex[1] = hh * side + m_radius * normal[1];
                vertex[2] = sh + m_radius * normal[2];

                translate(vertex, vertexOut, *trans[n]);
                translate(normal, normalOut, *trans[n]);
                glNormal3fv(normalOut);
                glVertex3fv(vertexOut);
            }

        }
        glEnd();
        n++;
        if (n==4)
        {
            n = 0;
            side = -1;
        }
    }
}


