//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcvcaxonterminal.h"

#include "dcvcomponent.h"
#include "dcaxon.h"
#include "dcreceptor.h"
#include "dcaxonterminal.h"

#include <QtOpenGL>

static const double PI = 3.14159265358979323846264338327950288419717;

DCVCAxonTerminal::DCVCAxonTerminal(DCAxonTerminal *owner)
    : d_owner(owner), d_isRendered(false)
{
}

void DCVCAxonTerminal::prepareChildrenForDraw(bool isAnimationInterval)
{
    (void)isAnimationInterval;
    d_isRendered = false;
}

void DCVCAxonTerminal::drawChildren(bool isAnimationInterval)
{
    (void)isAnimationInterval;
}

void DCVCAxonTerminal::drawChildrenForSelection(QList<DCVComponent *> *itemList)
{
    (void)itemList;
}

void DCVCAxonTerminal::renderOwnShape(bool isAnimationInterval, bool renderAsWireframe)
{
    (void)isAnimationInterval; (void)renderAsWireframe;

    if (d_owner->getIsViewAxonPointUpdated() && d_owner->getIsViewReceptorPointUpdated() && !d_isRendered)
    {
        //
        const float *axonPointMatrix = d_owner->getViewAxonPointMatrix();
        const float *receptorPointMatrix = d_owner->getViewReceptorPointMatrix();

        float dx = axonPointMatrix[12] - receptorPointMatrix[12];
        float dy = axonPointMatrix[13] - receptorPointMatrix[13];
        float dz = axonPointMatrix[14] - receptorPointMatrix[14];

        float receptorX = receptorPointMatrix[12];
        float receptorY = receptorPointMatrix[13];
        float receptorZ = receptorPointMatrix[14];

        DCReceptor *receptor = d_owner->getTarget();
        if (receptor)
        {
            float ox,oy,oz;
            receptor->getViewReceptorOffsetPoint(dx,dy,dz, &ox, &oy, &oz);
            receptorX = receptorPointMatrix[0] * ox + receptorPointMatrix[4] * oy + receptorPointMatrix[8] * oz + receptorPointMatrix[12];
            receptorY = receptorPointMatrix[1] * ox + receptorPointMatrix[5] * oy + receptorPointMatrix[9] * oz + receptorPointMatrix[13];
            receptorZ = receptorPointMatrix[2] * ox + receptorPointMatrix[6] * oy + receptorPointMatrix[10] * oz + receptorPointMatrix[14];
        }

        //render terminal
        glMatrixMode(GL_MODELVIEW);
        glPushMatrix();
        glLoadIdentity();
        glLineWidth(2);
        glColor4f(0.2f, 1.0f, 0.4f,1.0f);
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        glEnable( GL_LINE_SMOOTH );
        glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );

        glBegin(GL_LINES);
        glVertex3f(axonPointMatrix[12], axonPointMatrix[13], axonPointMatrix[14]);
        glVertex3f(receptorX, receptorY, receptorZ);
        glEnd();

        //arrow
        float ang1 = atan2(dz,dx)/ PI * 180;
//        float ang2 = atan2(dy,dx)/ PI * 180;

        glTranslatef(receptorX,receptorY,receptorZ);
        glRotatef(-ang1,0,1,0);
//        glRotatef(ang2,1,0,0);
        glBegin(GL_LINE_STRIP);
        glVertex3f(0.08f,0,0.04f);
        glVertex3f(0,0,0);
        glVertex3f(0.08f,0,-0.04f);
        glEnd();

        glPopMatrix();
        glDisable(GL_BLEND);
        d_isRendered = true;
        d_owner->resetViewPoinMatrices();
    }
}

void DCVCAxonTerminal::translate()
{

}

DCVComponent::DCVVisibility DCVCAxonTerminal::getIsVisible() const
{
    DCAxon *axon = (DCAxon*)d_owner->getOwner();
    DCReceptor *receptor = d_owner->getTarget();
    return ((axon && axon->getIsTerminalVisible()) || (receptor && receptor->getIsTerminalVisible())) ?
                DCV_VISIBLE_FULL : DCV_VISIBLE_NONE;
}

DCAxonTerminal* DCVCAxonTerminal::getOwnerTerminal() const
{
    return d_owner;
}

DCVComponent* DCVCAxonTerminal::getAxonComponent() const
{
    DCAxon *axon = (DCAxon*)d_owner->getOwner();
    if (axon)
    {
        return axon->getVComponent();
    }
    return NULL;
}

DCVComponent* DCVCAxonTerminal::getReceptorComponent() const
{
    DCReceptor *receptor = d_owner->getTarget();
    if (receptor)
    {
        return receptor->getVComponent();
    }
    return NULL;
}

DCVCPage* DCVCAxonTerminal::getPageBelonging() const
{
    return NULL;
}

DCCell* DCVCAxonTerminal::getOwnerCell() const
{
    DCAxon *axon = (DCAxon*)d_owner->getOwner();
    if (axon)
    {
        return axon->getOwnerCell();
    }
    return  NULL;
}

bool DCVCAxonTerminal::startDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return false;
}

bool DCVCAxonTerminal::dragging(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return false;
}

bool DCVCAxonTerminal::endDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return false;
}
