//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 1/6/2012.
//
#include "DNContainerBuilder.h"

#include "TKLock.h"
#include "TKContainer.h"
#include "DNUtils.h"
#include "DNDirectory.h"
#include "DNFileList.h"
#include "DNXML.h"
#include "DNXMLElement.h"
#include "TKLog.h"
#include "DNUtils.h"
#include "TKCell.h"
#include "TKDebug.h"
#include "DNGlobal.h"

DNContainerBuilder::DNContainerBuilder(TKContainer *container) : mContainer(container)
{
    
}

DNContainerBuilder::~DNContainerBuilder()
{
    
}

bool DNContainerBuilder::buildContainerFromXHTML(const char* docRoot)
{
    mLock.lock();
    
    mContainer->setContainerRootPath(std::string(docRoot));

    DNDirectory directory(docRoot);
    const DNFileList *flist = directory.getFileList("html|htm|xml|xhtml");

    //parse XHTML files
    while(flist)
    {
        std::string p = flist->getFilePath();
        parseXHTML(docRoot, p.c_str());

        flist = flist->next;
    }

    //build cells
    while(!mPendingCellDefinitions.empty())
    {
        PendingCellDefinition *cellInfo = mPendingCellDefinitions.front();
        TKCell *newCell = NULL;
        if (cellInfo->useCellCodeClass)
        {
            TKCellCode *cellCode = mContainer->getCellCode(cellInfo->cellCodeClassOrType);
            if (cellCode)
            {
                newCell = mContainer->addCell(cellInfo->location, cellInfo->cellName, cellCode, cellInfo->customScript);
            }
            else
            {
                std::string message = std::string("Failed to construct cell: ").append(cellInfo->location).append("#").append(cellInfo->cellName);
                message.append("\nCellCode '").append(cellInfo->cellCodeClassOrType).append("' couldn't be found.");
                dnNotifyError("Initialization failed", message);
            }
        }
        else
        {
            newCell = mContainer->addCell(cellInfo->location, cellInfo->cellName, cellInfo->cellCodeClassOrType, cellInfo->customScript);
        }

        if (newCell == NULL)
        {
            std::string message = std::string("Failed to construct cell: ").append(cellInfo->location).append("#").append(cellInfo->cellName);
            dnNotifyError("Initialization failed", message);
        }
        mPendingCellDefinitions.pop();
    }
    
    //build connections
    while(!mPendingConnections.empty())
    {
        PendingCellConnection *connInfo = mPendingConnections.front();
        TKCell *orgCell = mContainer->getCell(connInfo->originCellName);
        TKCell *tgtCell = mContainer->getCell(connInfo->targetCellName);
        
        if (orgCell && tgtCell)
        {
            if (!orgCell->connectTo(connInfo->connectionName, tgtCell))
            {
                std::string message = std::string("Failed to make connection. Receptor name for the target cell may be duplicated. (").append(connInfo->connectionName).append(") cell:").append(connInfo->originCellName).append(" - > ").append(connInfo->targetCellName);
                dnNotifyError("Initialization failed", message);
            }
        }
        else
        {
            std::string message = std::string("Failed to make connection (").append(connInfo->connectionName).append(") cell:").append(connInfo->originCellName).append(" - > ").append(connInfo->targetCellName);
            dnNotifyError("Initialization failed", message);
        }
        mPendingConnections.pop();
    }
    mLock.unlock();

    return dnGlobal()->isErrorStatusNormal();
}

void DNContainerBuilder::defineCellWithoutCellCodeClass(const char *path, std::string name, std::string type, std::string customScript)
{
    DEBUG_TRACE("\n===== Define cell (no CellCode class) ================ \nName:%s/%s\nType:%s\nCustom script:\n%s\n", path,name.c_str(), type.c_str(),customScript.c_str());

    PendingCellDefinition *newCellDefine = new PendingCellDefinition();
    newCellDefine->useCellCodeClass = false;
    newCellDefine->location = path;
    newCellDefine->cellName = name;
    newCellDefine->cellCodeClassOrType = type;
    newCellDefine->customScript = customScript;

    mPendingCellDefinitions.push(newCellDefine);
}

void DNContainerBuilder::defineCellWithCellCodeClass(const char *path, std::string name, std::string cellcode, std::string customScript)
{
    DEBUG_TRACE("\n===== Define cell (CellCode class attached)=========== \nName:%s/%s\nClass:%s\nCustom script:\n%s\n", path,name.c_str(), cellcode.c_str(),customScript.c_str());

    PendingCellDefinition *newCellDefine = new PendingCellDefinition();
    newCellDefine->useCellCodeClass = true;
    newCellDefine->location = path;
    newCellDefine->cellName = name;
    newCellDefine->cellCodeClassOrType = getFQNString(path, cellcode.c_str());
    newCellDefine->customScript = customScript;

    mPendingCellDefinitions.push(newCellDefine);
}

void DNContainerBuilder::defineCellCodeClass(const char *path, std::string name, std::string type, std::string cellScript)
{
    std::string fqnName = getFQNString(path, name.c_str());

    DEBUG_TRACE("\n===== Define cell code class ===== \nClass:%s\nAPI:%s\n%s\n", fqnName.c_str(), type.c_str(), cellScript.c_str());

    mContainer->addCellCode(fqnName, type, cellScript);
}

void DNContainerBuilder::defineConnection(const char *path, std::string origine, std::string destination, std::string name)
{
    PendingCellConnection *newConnection = new PendingCellConnection();
    newConnection->connectionName = name;
    newConnection->originCellName = getFQNString(path, origine.c_str());
    newConnection->targetCellName = getFQNString(path, destination.c_str());
    mPendingConnections.push(newConnection);
}

void DNContainerBuilder::parseXHTML(const char *docRoot, const char *path)
{
    mContainer->beganParsePage(docRoot, path);
    DNXML *xml = DNXML::createXMLFromFile(docRoot, path);
    DNXMLElement *element = xml->getRoot();
    
    std::string  cellName;
    std::string  cellCodeClassName;
    std::string  cellAPIType;
    std::string  script;

    int defineDepth = -1;

    while(element)
    {
        std::string define = element->getAttributeValue("define");
        if (define.length() > 0)
        {
            if (defineDepth > 0)
            {
                std::string message = std::string("Syntax error while parsing ").append(path).append(". A defnie element seems to be nested.");
                dnNotifyError("Initialization failed", message);
            }
            else
            {
                if (define == "cell")
                {
                    cellName = element->getAttributeValue("name");
                    defineDepth = element->depth;
                }
                else if (define == "cellcode")
                {
                    cellCodeClassName = element->getAttributeValue("name");
                    cellAPIType = element->getAttributeValue("type");
                    defineDepth = element->depth;
                }
                else
                {
                    std::string message = std::string("Syntax error while parsing ").append(path).append(". defnie='").append("' isn't correct.");
                    dnNotifyError("Initialization failed", message);
                }
            }
        }

        std::string parameter = element->getAttributeValue("parameter");
        if (parameter.length() > 0)
        {
            if (parameter == "script")
            {
                script = element->text;
            }
            else if (parameter == "cellcode")
            {
                cellCodeClassName = element->getAttributeValue("href");
                cellAPIType = element->getAttributeValue("type");
                if (cellCodeClassName.length() > 0 && cellAPIType.length() > 0)
                {
                    std::string message = std::string("Syntax error while parsing ").append(path).append(". CellCode:").append(cellName);
                    message.append("\n'type' can't be defined here if you use CellCode class.");
                    dnNotifyError("Initialization failed", message);
                }
            }
            else if (parameter== "connection")
            {
                std::string target = element->getAttributeValue("href");
                std::string rname = element->getAttributeValue("receptor");
                if (rname.length()==0)
                {
                    rname = cellName;
                }
                defineConnection(path, cellName, target, rname);
            }
            else
            {
                std::string message = std::string("Syntax error while parsing ").append(path).append(". parameter='").append(parameter).append("' isn't correct.");
                dnNotifyError("Initialization failed", message);
            }
        }

        //go to next element
        //
        if (element->inner)
        {
            // go inside
            element = element->inner;
        }
        else
        {
            // go next 
            do {
                if (!element->next)
                {
                    element = element->outer;
                }
            } while(element && element->next == NULL && element != xml->getRoot());
            
            if (!element)
                break;
            element = element->next;

            if (defineDepth >= 0 && (element == NULL || element->depth <= defineDepth))
            {
                defineDepth = -1;
                if (cellName.length()>0)
                {
                    if (cellCodeClassName.length()>0)
                    {
                        defineCellWithCellCodeClass(path, cellName, cellCodeClassName, script);
                    }
                    else
                    {
                        if (cellAPIType.length() > 0)
                        {
                            defineCellWithoutCellCodeClass(path, cellName, cellAPIType, script);
                        }
                        else
                        {
                            std::string message = std::string("Syntax error while parsing ").append(path).append(". cellName:'").append(cellName).append("'\nFaield to determin the cellcode.\nThe cellcode isn't defined for the cell?");
                            dnNotifyError("Initialization failed", message);
                        }
                    }

                    cellName = "";
                    cellCodeClassName = "";
                    cellAPIType = "";
                    script = "";
                }
                else
                {
                    defineCellCodeClass(path, cellCodeClassName, cellAPIType, script);
                    cellCodeClassName = "";
                    cellAPIType = "";
                    script = "";
                }
            }
        }
    }
    if (xml)
        delete xml;
    
    mContainer->endedParsePage(docRoot, path);
}
