//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Oct-2, 2012.
//
#include "dccontainersaver.h"

#include "dccontainer.h"
#include "dcscene.h"
#include "dcvcpage.h"
#include "dcvccellcode.h"
#include "dccell.h"
#include "dccellcode.h"
#include "dcreceptor.h"
#include "dcaxon.h"
#include "dcaxonterminal.h"
#include "dcvpagecomponent.h"
#include "dcscene.h"
#include "utils/dcutil.h"

#include <QDomDocument>
#include <QMap>
#include <QFile>
#include <QDir>
#include <QCryptographicHash>
#include <QByteArray>

static QDomDocument* constructDocumentForPage(DCContainer *container, const DCVCPage *page);

DCContainerSaver::DCContainerSaver(DCContainer *container) : d_container(container)
{

}

DCContainerSaver::~DCContainerSaver()
{

}

QString DCContainerSaver::calculateHashForPage(const DCVCPage *page)
{
    QCryptographicHash hash(QCryptographicHash::Md5);
    QDomDocument *pDoc1 = constructDocumentForPage(d_container, page);
    if (pDoc1)
    {
        hash.addData(pDoc1->toString(-1).toLocal8Bit());
        delete pDoc1;
    }

    QDomDocument doc2;
    QDomElement doc2Root = doc2.createElement("denncoCreator");
    page->saveAttributesToXML(&doc2, &doc2Root);
    doc2.appendChild(doc2Root);
    hash.addData(doc2.toString(-1).toLocal8Bit());

    return QString(hash.result().toHex());
}

bool DCContainerSaver::saveAll(const QString& containerRootPath)
{
    const QMap<QString,DCVCPage*> *pages = d_container->getScene()->getPages();
    QMapIterator<QString, DCVCPage*> i(*pages);
    bool r = true;
    while (i.hasNext())
    {
        i.next();
        DCVCPage *page = i.value();
        if (!saveForPage(containerRootPath, page))
        {
            r = false;
        }
    }
    return true;
}

bool DCContainerSaver::saveForPage(const QString& containerRootPath, DCVCPage *page)
{
    QString pageFilePath = containerRootPath;
    pageFilePath.append(page->getLocationPath());

    QDomDocument *doc = constructDocumentForPage(d_container, page);
    if (!doc)
        return false;

    QFile file(pageFilePath);
    QFileInfo pathInfo = QFileInfo(pageFilePath);
    QDir dir = pathInfo.dir();
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    bool r = false;
    if (file.open(QFile::WriteOnly | QFile::Truncate))
    {
        QTextStream out(&file);
        doc->save(out, 4);
        r = true;
    }
    file.close();

    delete doc;

    if (r)
    {
        r = d_container->getScene()->saveSceneForPage(containerRootPath, page);
    }

    if (r)
    {
        page->updateSavedState(d_container);
    }

    return r;
}

QDomDocument* constructDocumentForPage(DCContainer *conteinr, const DCVCPage *page)
{
    QDomDocument *doc = new QDomDocument("html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\"");
    doc->createProcessingInstruction("xml", "version=\"1.0\" encoding=\"utf-8\"");

    QDomElement root = doc->createElementNS("http://www.w3.org/1999/xhtml","html");
    root.setAttribute("lang", "en");
    QDomElement body = doc->createElement("body");
    root.appendChild(body);

    const QList<DCVPageComponent*>* vcells = page->getCells();
    for (int i = 0; i < vcells->length(); i++)
    {
        DCCell *cell = vcells->at(i)->getOwnerCell();
        QString cellName = QString::fromStdString(cell->getName());

        QDomElement aCellTag = doc->createElement("a");
        aCellTag.setAttribute("define", "cell");
        aCellTag.setAttribute("name", cellName);

        QDomElement aCellCodeTag = doc->createElement("a");
        aCellCodeTag.setAttribute("parameter", "cellcode");
        if (cell->getIsCellCodeAssgined())
        {
            aCellCodeTag.setAttribute("href", QString::fromStdString(cell->getCellCode()->getFQNName()));
        }
        else
        {
            aCellCodeTag.setAttribute("type", cell->getType());
        }
        aCellTag.appendChild(aCellCodeTag);
        DCAxon *axon = cell->getAxon();
        for (int j = 0; j < axon->getNumberOfTerminals(); j++)
        {
            DCAxonTerminal *terminal = axon->getTerminalAt(j);
            DCReceptor *receptor = terminal->getTarget();
            if (receptor)
            {
                DCCell *targetCell = receptor->getOwnerCell();
                if (targetCell)
                {
                    QString connectionPath = QString::fromStdString(targetCell->getLocation());
                    connectionPath.append("#");
                    connectionPath.append(QString::fromStdString(targetCell->getName()));
                    QString receptorName = QString::fromStdString(targetCell->getReceptorName(receptor));

                    QDomElement aConnectionTag = doc->createElement("a");
                    aConnectionTag.setAttribute("parameter", "connection");
                    aConnectionTag.setAttribute("href", connectionPath);
                    aConnectionTag.setAttribute("receptor", receptorName);
                    aCellTag.appendChild(aConnectionTag);
                }
            }
        }
        QDomElement preScriptTag = doc->createElement("pre");
        preScriptTag.setAttribute("parameter", "script");
        QDomCDATASection scriptCData = doc->createCDATASection(conteinr->readCustomScriptFromWorkFile(cell));
        preScriptTag.appendChild(scriptCData);
        aCellTag.appendChild(preScriptTag);
        body.appendChild(aCellTag);
    }

    const QList<DCVPageComponent*>* vcellcodeclasses = page->getCellCodeClasses();
    for (int i = 0; i < vcellcodeclasses->length(); i++)
    {
        DCCellCode *cellcodeclass = dynamic_cast<DCVCCellCode*>(vcellcodeclasses->at(i))->getOwnerCellCodeClass();
        QString cellCodeClassName = DCUtil::getNameFromFQNPath(QString::fromStdString(cellcodeclass->getFQNName()));

        QDomElement aCellCodeTag = doc->createElement("a");
        aCellCodeTag.setAttribute("define", "cellcode");
        aCellCodeTag.setAttribute("name", cellCodeClassName);
        aCellCodeTag.setAttribute("type", QString::fromStdString(cellcodeclass->getCellAPIName()));
        QDomElement preScriptTag = doc->createElement("pre");
        preScriptTag.setAttribute("parameter", "script");
        QDomCDATASection scriptCData = doc->createCDATASection(conteinr->readCellCodeScriptFromFile(cellcodeclass));
        preScriptTag.appendChild(scriptCData);
        aCellCodeTag.appendChild(preScriptTag);
        body.appendChild(aCellCodeTag);
    }

    doc->appendChild(root);

    return doc;
}
