//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dccontainer.h"

#include "dccell.h"
#include "dccellcode.h"
#include "dcaxon.h"
#include "dcaxonterminal.h"
#include "dcreceptor.h"
#include "dcscene.h"
#include "dccontent.h"
#include "dccreator.h"
#include "dcvpagecomponent.h"
#include "utils/dccomponentutil.h"
#include "dctreeviewwidget.h"
#include "dnplugininfo.h"

#include "DNFileList.h"
#include "DNDirectory.h"
#include "utils/dcutil.h"

#include <QMutexLocker>
#include <QDir>
#include <QDirIterator>


//static
TKContainer* TKContainer::createContainer()
{
    TKContainer *container = new DCContainer();
    container->init();
    return container;
}

DCContainer::DCContainer() : d_content(NULL), d_factoryCachedLocation("")
{
    d_scene = new DCScene(this);

    d_workDirRoot = QDir::homePath() + "/.denncoCreator/work";
    d_workDirCellRoot = d_workDirRoot + "/cells";
    d_workDirCellCodeRoot = d_workDirRoot + "/cellcodes";

}

DCContainer::~DCContainer()
{
    if (d_scene)
    {
        delete d_scene;
        d_scene = NULL;
    }
}

bool DCContainer::getIsModified() const
{
    if (d_scene)
    {
        return d_scene->getIsModified();
    }
    else
    {
        return false;
    }
}

bool DCContainer::getIsScriptableCell(const DCCell *cell) const
{
    return getIsScriptable(cell->getType());
}

void DCContainer::setContent(DCContent *content)
{
    d_content = content;
}

void DCContainer::unselectCellObjectAll()
{
    for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it ) {
        DCCell *cell = dynamic_cast<DCCell*>(it->second);
        if (cell)
            cell->getVComponent()->setSelected(false,true);
    }
}

QList<DCVComponent*> DCContainer::getSelectedCellObjects() const
{
    QList<DCVComponent*> list;

    for ( TKCellMap::const_iterator it = mCells.begin(); it != mCells.end(); ++it ) {
        DCCell *cell = dynamic_cast<DCCell*>(it->second);
        if (cell)
        {
            DCVComponent *object = NULL;

            object = cell->getVComponent();
            if (object && object->getIsSelected())
            {
                list.push_back(object);
            }

            //axon
            DCAxon *axon = cell->getAxon();
            object = axon->getVComponent();
            if (object && object->getIsSelected())
            {
                list.push_back(object);
            }

            //terminals
            int n = axon->getNumberOfTerminals();
            for (int i = 0; i < n; i++)
            {
                object = axon->getTerminalAt(i)->getVComponent();
                if (object && object->getIsSelected())
                {
                    list.push_back(object);
                }
            }

            //receptors
            const TKReceptorMap *receptors = cell->getReceptors();
            TKReceptorMap::const_iterator it = receptors->begin();
            while( it != receptors->end())
            {
                object = ((DCReceptor*)(*it).second)->getVComponent();
                if (object && object->getIsSelected())
                {
                    list.push_back(object);
                }
                ++it;
            }
        }
    }

    return list;
}

TKCell* DCContainer::addCell(std::string theLocation, std::string theName, std::string type, std::string customScript)
{
    QMutexLocker lock(d_scene->getSceneLock());

    DCCell *cell = dynamic_cast<DCCell*>(TKContainer::addCell(theLocation, theName, type, customScript));

    saveCustomScriptToWorkFile(cell, customScript);

    return cell;
}

TKCell*  DCContainer::addCell(std::string theLocation, std::string theName, TKCellCode *cellCode, std::string customScript)
{
    QMutexLocker lock(d_scene->getSceneLock());

    DCCell *cell = dynamic_cast<DCCell*>(TKContainer::addCell(theLocation, theName, cellCode, customScript));
    if (cell)
        saveCustomScriptToWorkFile(cell, customScript);

    return cell;
}

TKCellCode* DCContainer::addCellCode(std::string theName, std::string theAPIType, std::string code)
{
    QMutexLocker lock(d_scene->getSceneLock());

    DCCellCode *cellCode = dynamic_cast<DCCellCode*>(TKContainer::addCellCode(theName, theAPIType, code));
    if (cellCode)
        saveClassScriptToWorkFile(cellCode, code);

    return cellCode;
}

TKCell* DCContainer::cellFactory(std::string location, std::string name, std::string type, bool canInterfaceIn, bool canInterfaceOut)
{
    if (location != d_factoryCachedLocation || d_factoryCachedPageObject.ref() == NULL)
    {
        d_factoryCachedPageObject.assign(d_scene->getPage(location));
        if (d_factoryCachedPageObject.ref() == NULL)
        {
            d_factoryCachedPageObject.assign(d_scene->addPage(location));
        }
    }
    d_factoryCachedLocation = location;

    TKCell *cell = DCComponentUtil::createCell(this, dynamic_cast<DCVCPage*>(d_factoryCachedPageObject.ref()), location, name, type, canInterfaceIn, canInterfaceOut);
    cell->init();

    return cell;
}

TKCell* DCContainer::pluginCellFactory(std::string location, std::string fullName, std::string type, std::string pluginName, std::string pluginValue, bool canInterfaceIn, bool canInterfaceOut)
{
    return cellFactory(location, fullName, type, canInterfaceIn, canInterfaceOut );
}


TKAxon* DCContainer::axonFactory(TKCell *theOwner)
{
    return DCComponentUtil::createAxon((DCCell*)theOwner);
}

TKReceptor* DCContainer::receptorFactory(TKCell *theOwner)
{
    return DCComponentUtil::createReceptor((DCCell*)theOwner);
}

TKAxonTerminal* DCContainer::axonTerminalFactory(TKAxon *theOwner)
{
    return DCComponentUtil::createAxonTerminal((DCAxon*)theOwner);
}

TKCellCode* DCContainer::cellCodeFactory(std::string name, std::string cellapi, std::string code)
{
    DCVCPage *page = NULL;
    if (name != TKContainer::CELLCODENAME_EMPTY)
    {
        std::string location = name;
        if (location.find("#") != std::string::npos)
        {
            location = location.substr(0, location.find("#"));
        }
        if (location != d_factoryCachedLocation || d_factoryCachedPageObject.ref() == NULL)
        {
            d_factoryCachedPageObject.assign(d_scene->getPage(location));
            if (d_factoryCachedPageObject.ref() == NULL)
            {
                d_factoryCachedPageObject.assign(d_scene->addPage(location));
            }
        }
        d_factoryCachedLocation = location;
        page = dynamic_cast<DCVCPage*>(d_factoryCachedPageObject.ref());
    }
    return DCComponentUtil::createCellCode(this, page, name, cellapi);
}

void DCContainer::beganParsePage(const char *docRoot, const char *path)
{
    (void)docRoot;

    QDir workCellDir(d_workDirCellRoot + QString::fromLocal8Bit(path));
    QDir workCellCodeDir(d_workDirCellCodeRoot + QString::fromLocal8Bit(path));
    if (!workCellDir.exists())
    {
        workCellDir.mkpath(workCellDir.absolutePath());
    }
    if (!workCellCodeDir.exists())
    {
        workCellCodeDir.mkpath(workCellCodeDir.absolutePath());
    }

    std::string location = path;
    d_factoryCachedPageObject.assign(d_scene->getPage(location));
    if (d_factoryCachedPageObject.ref() == NULL)
    {
        d_factoryCachedPageObject.assign(d_scene->addPage(location));
    }
    d_factoryCachedLocation = location;
}

void DCContainer::endedParsePage(const char *docRoot, const char *path)
{
    (void)docRoot; (void)path;
}


void DCContainer::beganBuildContainer()
{
    DCUtil::removeDirRecursive(d_workDirRoot);
}

void DCContainer::endedBuildContainer()
{


}

bool DCContainer::removeCellCode(DCCellCode *cellcode)
{
    for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it )
    {
        DCCell *cell = dynamic_cast<DCCell*>(it->second);
        if (cell->getCellCode() == cellcode)
        {
            cell->setCellCode(mEmptyCellClass, cell->getCustomScript().toLocal8Bit().data());
        }
    }

    bool found = false;
    for ( TKCellCodeMap::iterator it = mCellCodes.begin(); it != mCellCodes.end(); ++it )
    {
        if (it->second == cellcode)
        {
            mCellCodes.erase(it);
            found = true;
            break;
        }
    }

    if (cellcode->getPageBelonging())
    {
        cellcode->getPageBelonging()->unregisterCellCodeClass(cellcode->getVComponent());
    }
    delete cellcode;

    return found;
}

bool DCContainer::removeCell(DCCell *cell)
{
    for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it )
    {
        if (it->second == cell)
        {
            mCells.erase(it++);
            if (cell->getPageBelonging())
            {
                cell->getPageBelonging()->unregisterCell(cell->getVComponent());
            }
            delete cell;
            return true;
        }
    }
    return false;
}

bool DCContainer::saveCustomScriptToWorkFile(const DCCell *cell, std::string customScript)
{
    QFile workfile(getWorkFilePathForCustomScript(cell));
    QFileInfo fileInfo(workfile);
    if (!fileInfo.absoluteDir().exists())
    {
        QDir dir;
        dir.mkpath(fileInfo.absolutePath());
    }

    workfile.open(QIODevice::WriteOnly);
    QTextStream out(&workfile);
    out << QString::fromStdString(customScript);
    workfile.close();
    return true;
}

bool DCContainer::saveClassScriptToWorkFile(const DCCellCode *cellCode, std::string code)
{
    QString qNamePath = QString::fromStdString(cellCode->getFQNName());
    QString name = DCUtil::getNameFromFQNPath(qNamePath);

    if (name.length() == 0)
        return false;

    QFile workfile(getWorkFilePathForCellCode(cellCode));
    QFileInfo fileInfo(workfile);
    if (!fileInfo.absoluteDir().exists())
    {
        QDir dir;
        dir.mkpath(fileInfo.absolutePath());
    }
    workfile.open(QIODevice::WriteOnly);
    QTextStream out(&workfile);
    out << QString::fromStdString(code);
    workfile.close();
    return true;
}

QString DCContainer::readCustomScriptFromWorkFile(const DCCell *cell)
{
    QFile file(getWorkFilePathForCustomScript(cell));

    if (!file.exists())
    {
        return "";
    }

    file.open(QIODevice::ReadOnly);
    QTextStream in(&file);
    QString out = in.readAll();
    file.close();

    return out;
}

QString DCContainer::readCellCodeScriptFromFile(const DCCellCode *cellCode)
{
    QString qNamePath = QString::fromStdString(cellCode->getFQNName());
    QString name = DCUtil::getNameFromFQNPath(qNamePath);

    if (name.length() == 0)
        return false;

    QFile file(getWorkFilePathForCellCode(cellCode));
    if (!file.exists())
    {
        return "";
    }

    file.open(QIODevice::ReadOnly);
    QTextStream in(&file);
    QString out = in.readAll();
    file.close();

    return out;
}

QString DCContainer::sysFilePathToContainerBasedPath(const QString &sysFilePath)
{
    QString containerPath = QString::fromStdString(getContainerRootPath());
    return "/" + QDir(containerPath).relativeFilePath(sysFilePath);
}

QString DCContainer::containerBasedPathToSysFilePath(const QString &containerBasedPath)
{
    QString containerPath = QString::fromStdString(getContainerRootPath());
    return containerPath + containerBasedPath;
}

bool DCContainer::renameCell(DCCell *cell, const QString &newName)
{
    QString oldFqnName = DCUtil::getFQNPath(cell->getLocation(), cell->getName());
    QString oldName = QString::fromStdString(cell->getName());

    if (oldName == newName)
        return true;

    QString customScript = readCustomScriptFromWorkFile(cell);
    QFile workFileOld(getWorkFilePathForCustomScript(cell));

    cell->changeName(newName);
    if (!saveCustomScriptToWorkFile(cell, customScript.toStdString()))
    {
        cell->changeName(oldName);
        return false;
    }
    workFileOld.remove();
    mCells.erase(oldFqnName.toStdString());

    QString newFqnName = DCUtil::getFQNPath(cell->getLocation(), cell->getName());
    std::pair<TKCellMap::iterator, bool> r = mCells.insert(TKCellMap::value_type(newFqnName.toStdString(), cell));

    return r.second;
}

bool DCContainer::moveCell(DCCell *cell, const QString& pageNewContainerBasedPathName)
{
    QString oldFqnName = DCUtil::getFQNPath(cell->getLocation(), cell->getName());
    QString oldContainerPath = QString::fromStdString(cell->getLocation());

    if (oldContainerPath == pageNewContainerBasedPathName)
        return true;

    QString customScript = readCustomScriptFromWorkFile(cell);
    QFile workFileOld(getWorkFilePathForCustomScript(cell));

    cell->changePath(pageNewContainerBasedPathName);
    if (!saveCustomScriptToWorkFile(cell, customScript.toStdString()))
    {
        cell->changePath(oldContainerPath);
        return false;
    }
    workFileOld.remove();
    mCells.erase(oldFqnName.toStdString());

    QString newFqnName = DCUtil::getFQNPath(pageNewContainerBasedPathName.toStdString(), cell->getName());
    std::pair<TKCellMap::iterator, bool> r = mCells.insert(TKCellMap::value_type(newFqnName.toStdString(), cell));
    return r.second;
}

bool DCContainer::renameCellCodeClass(DCCellCode *cellcode, const QString &newName)
{
    QString oldFqnName = QString::fromStdString(cellcode->getFQNName());
    QString oldPath = DCUtil::getContainerBasedPathFromFQNPath(oldFqnName);
    QString oldName = DCUtil::getNameFromFQNPath(oldFqnName);

    if (oldName == newName)
        return true;

    QFile oldFile(getWorkFilePathForCellCode(cellcode));
    QString script = readCellCodeScriptFromFile(cellcode);

    cellcode->changeName(newName);
    if (!saveClassScriptToWorkFile(cellcode, script.toStdString()))
    {
        cellcode->changeName(oldName);
        return false;
    }
    oldFile.remove();
    mCellCodes.erase(oldFqnName.toStdString());

    std::pair<TKCellCodeMap::iterator, bool> r = mCellCodes.insert(TKCellCodeMap::value_type(cellcode->getFQNName(), cellcode));

    return r.second;
}

bool DCContainer::moveCellCodeClass(DCCellCode *cellcode, const QString& pageNewContainerBasedPathName)
{
    QString oldFqnName = QString::fromStdString(cellcode->getFQNName());
    QString oldPath = DCUtil::getContainerBasedPathFromFQNPath(oldFqnName);
    QString oldName = DCUtil::getNameFromFQNPath(oldFqnName);

    if (oldPath == pageNewContainerBasedPathName)
        return true;

    QFile oldFile(getWorkFilePathForCellCode(cellcode));
    QString script = readCellCodeScriptFromFile(cellcode);

    cellcode->changePath(pageNewContainerBasedPathName);
    if (!saveClassScriptToWorkFile(cellcode, script.toStdString()))
    {
        cellcode->changePath(oldPath);
        return false;
    }
    oldFile.remove();
    mCellCodes.erase(oldFqnName.toStdString());

    std::pair<TKCellCodeMap::iterator, bool> r = mCellCodes.insert(TKCellCodeMap::value_type(cellcode->getFQNName(), cellcode));

    return r.second;
}

DCVCPage* DCContainer::movePage(const QString &oldContainerBasedPathName, const QString &newContainerBasedPathName)
{
    bool r = false;
    DCVCPage *newPage = NULL;
    DCScene *scene = getScene();
    if (scene)
    {
        DCVCPage *oldPage = getScene()->getPage(oldContainerBasedPathName.toStdString());
        newPage = getScene()->addPage(newContainerBasedPathName.toStdString());

        if (oldPage && newPage)
        {
            r = oldPage->moveComponentsTo(newPage);
        }
        if (r)
        {
            QString sysOldFilePath = containerBasedPathToSysFilePath(oldContainerBasedPathName);
            QString sceneSettingFilePath = scene->getSceneSettingFilePathForPage(oldPage);
            if (!DCTreeViewWidget::removeFile(sysOldFilePath))
            {
                r = false;
            }
            QDir dir;
            dir.remove(sceneSettingFilePath);

            if (!scene->removePage(oldPage))
            {
                r = false;
            }
        }
    }

    return r ? newPage : NULL;
}


QList<QString> DCContainer::getAvailableCellTypes() const
{
    QList<QString> list;

    list.append(QString::fromStdString(TKContainer::CELLTYPE_JSBASIC));
    list.append(QString::fromStdString(TKContainer::CELLTYPE_BASICSTORAGE));
    list.append(QString::fromStdString(TKContainer::CELLTYPE_IN));
    list.append(QString::fromStdString(TKContainer::CELLTYPE_OUT));
    list.append(QString::fromStdString(TKContainer::CELLTYPE_PLUGIN_IN));
    list.append(QString::fromStdString(TKContainer::CELLTYPE_PLUGIN_OUT));

    return list;
}

QList<QString> DCContainer::getAvailableScriptableCellTypes() const
{
    QList<QString> list;

    list.append(QString::fromStdString(TKContainer::CELLTYPE_JSBASIC));
    list.append(QString::fromStdString(TKContainer::CELLTYPE_BASICSTORAGE));

    return list;
}

bool DCContainer::getIsScriptable(const QString& type) const
{
    std::string t = type.toStdString();

    if (t.length() == 0)
    {
        return true;
    }

    if (t == TKContainer::CELLTYPE_JSBASIC)
    {
        return true;
    }
    else if (t == TKContainer::CELLTYPE_BASICSTORAGE)
    {
        return true;
    }
    else if (t == TKContainer::CELLTYPE_IN)
    {
        return false;
    }
    else if (t == TKContainer::CELLTYPE_OUT)
    {
        return false;
    }
    else if (t == TKContainer::CELLTYPE_PLUGIN_IN)
    {
        return false;
    }
    else if (t == TKContainer::CELLTYPE_PLUGIN_OUT)
    {
        return false;
    }

    return false;
}


bool DCContainer::getIsReceptorAvailable(const QString& type) const
{
    std::string t = type.toStdString();

    if (t == TKContainer::CELLTYPE_JSBASIC)
    {
        return true;
    }
    else if (t == TKContainer::CELLTYPE_BASICSTORAGE)
    {
        return true;
    }
    else if (t == TKContainer::CELLTYPE_IN)
    {
        return false;
    }
    else if (t == TKContainer::CELLTYPE_OUT)
    {
        return true;
    }
    else if (t == TKContainer::CELLTYPE_PLUGIN_IN)
    {
        return false;
    }
    else if (t == TKContainer::CELLTYPE_PLUGIN_OUT)
    {
        return true;
    }

    return false;
}

QString DCContainer::getWorkFilePathForCustomScript(const DCCell *cell) const
{
    QDir workdir(d_workDirCellRoot + QString::fromStdString(cell->getLocation()));
    return workdir.absoluteFilePath(QString::fromStdString(cell->getName()) + ".js");
}

QString DCContainer::getWorkFilePathForCellCode(const DCCellCode *cellCode) const
{
    QString qNamePath = QString::fromStdString(cellCode->getFQNName());
    QString path = DCUtil::getContainerBasedPathFromFQNPath(qNamePath);
    QString name = DCUtil::getNameFromFQNPath(qNamePath);

    QDir workdir(d_workDirCellCodeRoot + path);
    return workdir.absoluteFilePath(name + ".js");
}

bool DCContainer::getIsPluginType(const QString &type) const
{
    std::string t = type.toStdString();

    if (t == TKContainer::CELLTYPE_PLUGIN_IN)
    {
        return true;
    }
    else if (t == TKContainer::CELLTYPE_PLUGIN_OUT)
    {
        return true;
    }

    return false;
}

QList<QString> DCContainer::getAvailablePluginLibraries() const
{
    QList<QString> list;

    for ( TKCellMap::const_iterator it = mCells.begin(); it != mCells.end(); ++it )
    {
        DCCell *cell = dynamic_cast<DCCell*>(it->second);
        if (cell)
        {
            if (getIsPluginType(cell->getType()))
            {
                DNPluginInfo info = DNPluginInfo::create(cell->getName());
                if (!list.contains(QString::fromStdString(info.pluginName)))
                {
                    list.append(QString::fromStdString(info.pluginName));
                }
            }
        }
    }
    return list;
}

QString DCContainer::createPluginCellName(const QString &name, const QString& type, const QString &libraryName)
{
    QString cellName = name;

    if (type == QString::fromStdString(TKContainer::CELLTYPE_PLUGIN_IN))
    {
        cellName += "@@";
        cellName += libraryName;
    }
    else if (type == QString::fromStdString(TKContainer::CELLTYPE_PLUGIN_OUT))
    {
        cellName += "@";
        cellName += libraryName;
    }
    return cellName;
}
