//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dceditcommands.h"

#include "dccreator.h"
#include "dcscene.h"

#include "dccontainer.h"
#include "dccell.h"
#include "dccellcode.h"
#include "dcaxon.h"
#include "dcaxonterminal.h"
#include "dcreceptor.h"
#include "dcvpagecomponent.h"
#include "dcvccellcode.h"
#include "utils/dcskeltoncreatorutil.h"

#include "dctreeviewwidget.h"

// -----------------------------
DCStartAddAxonTerminalCommandFromAxon::DCStartAddAxonTerminalCommandFromAxon(const void *requester, DCCreator *creator, DCCell *ownerCell)
    : DCCommand(requester, creator), d_ownerCell(ownerCell)
{

}

void DCStartAddAxonTerminalCommandFromAxon::redo()
{
    DCAxon *axon = d_ownerCell->getAxon();
    bool r = getScene()->startTerminalEditForAxon(getRequester(), axon);
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);

}

void DCStartAddAxonTerminalCommandFromAxon::undo()
{
    getScene()->finishTerminalEdit(getRequester());
}


// -----------------------------
DCStartAddAxonTerminalCommandFromReceptor::DCStartAddAxonTerminalCommandFromReceptor(const void *requester, DCCreator *creator, DCCell *ownerCell, const QString &receptorName)
    : DCCommand(requester, creator), d_ownerCell(ownerCell), d_receptorName(receptorName)
{

}

void DCStartAddAxonTerminalCommandFromReceptor::redo()
{
    DCReceptor *receptor = d_ownerCell->createReceptor(d_receptorName);
    bool r = getScene()->startTerminalEditForReceptor(getRequester(), receptor);
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCStartAddAxonTerminalCommandFromReceptor::undo()
{
    getScene()->finishTerminalEdit(getRequester());
    d_ownerCell->removeReceptor(d_receptorName);
}


// -----------------------------
DCCommitAddAxonTerminalCommand::DCCommitAddAxonTerminalCommand(const void *requester, DCCreator *creator, DCAxon *axon, DCCell *receptorCell, const QString &receptorName)
    : DCCommand(requester, creator), d_receptorCell(receptorCell),d_receptor(NULL), d_receptorName(receptorName), d_isAdded(false)
{
    d_axonCell = axon->getOwnerCell();
}

DCCommitAddAxonTerminalCommand::DCCommitAddAxonTerminalCommand(const void *requester, DCCreator *creator, DCCell *axonCell, DCReceptor *receptor)
    : DCCommand(requester, creator), d_axonCell(axonCell), d_receptor(receptor), d_receptorName(""), d_isAdded(false)
{
    d_receptorCell = d_receptor->getOwnerCell();
}

void DCCommitAddAxonTerminalCommand::redo()
{
    bool r = false;
    if (!d_receptor)
    {
        d_isAdded = d_axonCell->connectTo(d_receptorName.toStdString(), d_receptorCell);
        r = getScene()->finishTerminalEdit(getRequester());
    }
    else
    {
        TKAxonTerminal *terminal = d_axonCell->getAxon()->addTerminal();
        if (terminal)
        {
            d_receptor->setTarget(terminal);
            d_isAdded = true;
        }
        r = getScene()->finishTerminalEdit(getRequester());
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCCommitAddAxonTerminalCommand::undo()
{
    if (!d_isAdded)
        return;

    if (!d_receptor)
    {
        DCAxon *axon = dynamic_cast <DCAxon*>(d_axonCell->getAxon());
        DCReceptor *receptor = d_receptorCell->getReceptor(d_receptorName);
        if (axon && receptor)
        {
            DCAxonTerminal *terminal = receptor->getTarget();
            d_receptorCell->removeReceptor(d_receptorName);
            axon->removeAxonTerminal(terminal);
        }
    }
    else
    {
        DCAxon *axon = dynamic_cast <DCAxon*>(d_axonCell->getAxon());
        QString receptorName = QString::fromStdString(d_receptorCell->getReceptorName(d_receptor));
        DCAxonTerminal *terminal = d_receptor->getTarget();

        d_receptorCell->removeReceptor(receptorName);
        axon->removeAxonTerminal(terminal);
    }

}


DCRemoveAxonTerminalCommand::DCRemoveAxonTerminalCommand(const void *requester, DCCreator *creator, DCCell *axonCell, DCAxonTerminal *terminal)
    : DCCommand(requester, creator), d_cell(axonCell), d_axonTerminal(terminal), d_receptorName(""), d_isDeleted(false)
{

}

DCRemoveAxonTerminalCommand::DCRemoveAxonTerminalCommand(const void *requester, DCCreator *creator, DCCell *receptorCell, const QString &receptorName)
    : DCCommand(requester, creator), d_cell(receptorCell), d_axonTerminal(NULL), d_receptorName(receptorName), d_isDeleted(false)
{

}

void DCRemoveAxonTerminalCommand::redo()
{
    bool r = false;
    if (d_axonTerminal)
    {
        //delete from axon terminal
        DCReceptor *receptor = d_axonTerminal->getTarget();
        DCAxon *axon = dynamic_cast <DCAxon*>(d_axonTerminal->getOwner());
        if (receptor && axon)
        {
            DCCell *targetCell = receptor->getOwnerCell();
            QString receptorName = QString::fromStdString(targetCell->getReceptorName(receptor));
            d_isDeleted = targetCell->removeReceptor(receptorName);
            if (d_isDeleted)
                r = axon->removeAxonTerminal(d_axonTerminal);
        }
    }
    else
    {
        //delete from receptor name
        DCReceptor *receptor = d_cell->getReceptor(d_receptorName);
        DCAxonTerminal *terminal = receptor->getTarget();
        DCAxon *axon = dynamic_cast<DCAxon*>(terminal->getOwner());
        d_isDeleted = d_cell->removeReceptor(d_receptorName);
        if (d_isDeleted)
            r = axon->removeAxonTerminal(terminal);
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRemoveAxonTerminalCommand::undo()
{
    qDebug() << "TODO undo DCDeleteAxonTermianlCommand";
    //TODO
}

/*--------------------------------*/
DCAssignCellCodeClassToCellCommand::DCAssignCellCodeClassToCellCommand(const void *requester, DCCreator *creator, DCCell *cell, DCCellCode *cellCode)
    : DCCommand(requester, creator), d_cell(cell), d_cellCode(cellCode), d_prevClass(NULL)
{

}

void DCAssignCellCodeClassToCellCommand::redo()
{
    bool r = false;
    if (d_cell)
    {
        d_prevClass = d_cell->getCellCode();
        r = d_cell->setCellCode(d_cellCode, NULL);
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCAssignCellCodeClassToCellCommand::undo()
{
    if (d_cell && d_prevClass)
    {
        d_cell->setCellCode(d_prevClass, NULL);
    }

}


/*--------------------------------*/
DCUnassignCellCodeClassFromCellCommand::DCUnassignCellCodeClassFromCellCommand(const void *requester, DCCreator *creator, DCCell *cell)
    : DCCommand(requester, creator), d_cell(cell), d_prevClass(NULL)
{

}

void DCUnassignCellCodeClassFromCellCommand::redo()
{
    bool r = false;
    if (d_cell)
    {
        d_prevClass = d_cell->getCellCode();
        r = d_cell->setCellCode(getContainer()->getEmptyCellCodeClass(), NULL);
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCUnassignCellCodeClassFromCellCommand::undo()
{
    if (d_cell && d_prevClass)
    {
        d_cell->setCellCode(d_prevClass, NULL);
    }

}


/*--------------------------------*/

DCAddCellCodeClassCommand::DCAddCellCodeClassCommand(const void *requester, DCCreator *creator, DCContainer *container, const QString &name, const QString &type)
    : DCCommand(requester, creator), d_container(container), d_name(name), d_type(type)
{

}

void DCAddCellCodeClassCommand::redo()
{
    bool r = d_container->addCellCode(d_name.toStdString(), d_type.toStdString(), DCSkeltonCreatorUtil::getDefaultCellCodeClassScript().toStdString()) != NULL;
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCAddCellCodeClassCommand::undo()
{
}

/*--------------------------------*/

DCMoveCellCodeClassCommand::DCMoveCellCodeClassCommand(const void *requester, DCCreator *creator, const QString &name, const QString &type)
    : DCCommand(requester, creator), d_name(name), d_type(type)
{
}

DCMoveCellCodeClassCommand::~DCMoveCellCodeClassCommand()
{
}

void DCMoveCellCodeClassCommand::redo()
{

}

void DCMoveCellCodeClassCommand::undo()
{

}


/*--------------------------------*/

DCChangeCellCodeClassTypeCommand::DCChangeCellCodeClassTypeCommand(const void *requester, DCCreator *creator, DCCellCode *cellCode, const QString& newType)
    : DCCommand(requester, creator), d_cellCode(cellCode), d_type(newType)
{

}

void DCChangeCellCodeClassTypeCommand::redo()
{
    bool r = false;
    if (getContainer()->getIsScriptable(d_type))
    {
        d_cellCode->changeType(d_type);
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCChangeCellCodeClassTypeCommand::undo()
{

}


/*--------------------------------*/

DCRemoveCellCodeClassCommand::DCRemoveCellCodeClassCommand(const void *requester, DCCreator *creator, DCContainer *container, DCCellCode *cellCode)
    : DCCommand(requester, creator), d_container(container)
{
    d_cellCodes.append(cellCode);
}

DCRemoveCellCodeClassCommand::DCRemoveCellCodeClassCommand(const void *requester, DCCreator *creator, DCContainer *container, const QList<DCCellCode *> &cellcodes)
    : DCCommand(requester, creator), d_container(container), d_cellCodes(cellcodes)
{
}

DCRemoveCellCodeClassCommand::~DCRemoveCellCodeClassCommand()
{
}

void DCRemoveCellCodeClassCommand::redo()
{
    bool r = false;

    for (int i = 0; i < d_cellCodes.length(); i++)
    {
        if (!d_container->removeCellCode(d_cellCodes.at(i)))
        {
            r = false;
        }
    }

    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRemoveCellCodeClassCommand::undo()
{

}

/*--------------------------------*/

DCAddCellCommand::DCAddCellCommand(const void *requester, DCCreator *creator, DCContainer *container, const QString &path, const QString &name, const QString &type, float pageX, float pageY)
    : DCCommand(requester, creator), d_container(container), d_path(path), d_name(name), d_type(type), d_pageX(pageX), d_pageY(pageY)
{

}

void DCAddCellCommand::redo()
{
    bool r = false;

    DCCell *cell = NULL;
    if (d_container->getIsScriptable(d_type))
        cell = dynamic_cast<DCCell*>(d_container->addCell(d_path.toStdString(), d_name.toStdString(), d_type.toStdString(), DCSkeltonCreatorUtil::getDefaultCustomScript().toStdString()));
    else
        cell = dynamic_cast<DCCell*>(d_container->addCell(d_path.toStdString(), d_name.toStdString(), d_type.toStdString(), ""));

    if (cell)
    {
        r = true;
        cell->setViewPageX(d_pageX);
        cell->setViewPageY(d_pageY);
        cell->getVComponent()->setSelected(true,true);
        getController()->selectPage(this, cell->getPageBelonging(), false);
        if (getController()->getPersMode() != DCCreator::DC_PERSMODE_PAGEEDIT)
        {
            getController()->changePersMode(this, DCCreator::DC_PERSMODE_PAGEEDIT);
        }
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCAddCellCommand::undo()
{

}

/*--------------------------------*/

DCChangeCellTypeCommand::DCChangeCellTypeCommand(const void *requester, DCCreator *creator, DCCell* cell, const QString &newType)
    : DCCommand(requester, creator), d_cell(cell), d_type(newType)
{

}

void DCChangeCellTypeCommand::redo()
{
    bool r = false;
    if (d_cell)
    {
        if (!getContainer()->getIsScriptable(d_type))
        {
            d_cell->setCellCode(getContainer()->getEmptyCellCodeClass(), NULL);
            d_cell->saveCustomScript("");
        }

        if (d_cell->getIsCellCodeAssgined())
        {
            //TODO
        }
        else
        {
            d_cell->changeType(d_type);
            r = true;
        }
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCChangeCellTypeCommand::undo()
{

}

/*--------------------------------*/

DCRenameCellCommand::DCRenameCellCommand(const void *requester, DCCreator *creator, DCCell *cell, const QString &newName)
    : DCCommand(requester, creator), d_cell(cell), d_name(newName)
{
    d_location = QString::fromStdString(cell->getLocation());
}

DCRenameCellCommand::DCRenameCellCommand(const void *requester, DCCreator *creator, DCCell *cell, const QString &newLocation, const QString &newName)
    : DCCommand(requester, creator), d_cell(cell), d_location(newLocation), d_name(newName)
{

}

DCRenameCellCommand::~DCRenameCellCommand()
{

}

void DCRenameCellCommand::redo()
{
    bool r = false;
    if (getContainer())
    {
        if (d_location != QString::fromStdString(d_cell->getLocation()))
        {
            DCVCPage *page = getScene()->getPage(d_location.toStdString());
            if (page)
            {
                d_cell->getVComponent()->changePageBelonging(page);
                r = getContainer()->moveCell(d_cell, d_location);
            }
        }
        else
        {
            r = true;
        }
        if (r)
            r = getContainer()->renameCell(d_cell, d_name);
    }

    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRenameCellCommand::undo()
{

}

/*--------------------------------*/

DCMoveCellCommand::DCMoveCellCommand(const void *requester, DCCreator *creator, DCCell *cell, DCVCPage *newPage)
    : DCCommand(requester, creator), d_cell(cell), d_page(newPage)
{
}

DCMoveCellCommand::~DCMoveCellCommand()
{

}

void DCMoveCellCommand::redo()
{
    bool r = false;

    if (d_page && d_cell)
    {
        d_cell->getVComponent()->changePageBelonging(d_page);
        r = true;
    }

    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCMoveCellCommand::undo()
{

}

/*--------------------------------*/

DCRemoveCellCommand::DCRemoveCellCommand(const void *requester, DCCreator *creator, DCContainer *container, DCCell* cell)
    : DCCommand(requester, creator), d_container(container)
{
    d_cells.append(cell);
}

DCRemoveCellCommand::DCRemoveCellCommand(const void *requester, DCCreator *creator, DCContainer *container, const QList<DCCell*> &cells)
    : DCCommand(requester, creator), d_container(container), d_cells(cells)
{
}

DCRemoveCellCommand::~DCRemoveCellCommand()
{

}

void DCRemoveCellCommand::redo()
{
    bool r = true;
    for (int i = 0; i < d_cells.length(); i++)
    {
        if (!d_container->removeCell(d_cells.at(i)))
        {
            r = false;
        }
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRemoveCellCommand::undo()
{

}

/*--------------------------------*/
DCAddPageCommand::DCAddPageCommand(const void *requester, DCCreator *creator, const QString &containerBasedPath)
    : DCCommand(requester, creator), d_path(containerBasedPath), d_page(NULL)
{

}

void DCAddPageCommand::redo()
{
    if (!getController())
        return;

    DCTreeViewWidget::selectWhenFilePathAdded(getContainer()->containerBasedPathToSysFilePath(d_path));
    d_page = getScene()->addPage(d_path.toStdString());
    bool r = getController()->savePage(d_page);
    getController()->selectPage(getRequester(), d_page, false);
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCAddPageCommand::undo()
{

}

/*--------------------------------*/
DCMovePageCommand::DCMovePageCommand(const void *requester, DCCreator *creator, const QString &oldContainerBasedPathName, const QString &newContainerBasedPathName)
    : DCCommand(requester, creator), d_oldPath(oldContainerBasedPathName), d_newPath(newContainerBasedPathName)
{
}

DCMovePageCommand::~DCMovePageCommand()
{
    qDebug() << "DCRenamePageCommand removed..";
}

void DCMovePageCommand::redo()
{
    bool r = false;

    DCVCPage *oldPage = getScene()->getPage(d_oldPath.toStdString());
    if (getContainer() && oldPage && !getScene()->getIsPageExist(d_newPath.toStdString()))
    {
        DCTreeViewWidget::selectWhenFilePathAdded(getContainer()->containerBasedPathToSysFilePath(d_newPath));
        DCVCPage *newPage = getContainer()->movePage(d_oldPath, d_newPath);
        getController()->selectPage(getRequester(), newPage, false);

        if (newPage)
        {
            r = getController()->saveAll(false);
        }
        else
        {
            //TODO show error message?
            DCTreeViewWidget::selectWhenFilePathAdded("");
        }
    }
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCMovePageCommand::undo()
{

}


/*--------------------------------*/
DCRemovePageCommand::DCRemovePageCommand(const void *requester, DCCreator *creator, DCVCPage *page)
    : DCCommand(requester, creator), d_page(page)
{

}

DCRemovePageCommand::~DCRemovePageCommand()
{
    qDebug() << "DCRemovePageCommand removed..";
}

void DCRemovePageCommand::redo()
{
    bool r = false;

    if (!getContainer())
    {
        setCommandResult(DCCommand::DCCOMMAND_FAILED);
        return;
    }

    const QList<DCVPageComponent*> *vcells = d_page->getCells();
    const QList<DCVPageComponent*> *vcellCodeClasses = d_page->getCellCodeClasses();

    QList<DCCellCode*> cellcodes;
    for (int i = 0; i < vcellCodeClasses->length(); i++)
    {
        DCCellCode *cellCode = dynamic_cast<DCVCCellCode*>(vcellCodeClasses->at(i))->getOwnerCellCodeClass();
        if (cellCode)
            cellcodes.append(cellCode);
    }

    if (cellcodes.length()>0)
    {
        DCCommand *command = new DCRemoveCellCodeClassCommand(getRequester(), getController(), getContainer(), cellcodes);
        d_undoStack.push(command);

        if (command->getCommandResult() != DCCommand::DCCOMMAND_SUCCEEDED)
            r = false;
    }

    QList<DCCell*> cells;
    for (int i = 0; i < vcells->length(); i++)
    {
        DCCell *cell = vcells->at(i)->getOwnerCell();
        if (cell)
            cells.append(cell);
    }

    if (cells.length()>0)
    {
        DCCommand *command = new DCRemoveCellCommand(getRequester(), getController(), getContainer(), cells);
        d_undoStack.push(command);

        if (command->getCommandResult() != DCCommand::DCCOMMAND_SUCCEEDED)
            r = false;
    }

    QString filePath = QString::fromStdString(getContainer()->getContainerRootPath());
    filePath.append(d_page->getLocationPath());

    QString sceneSettingFilePath = getScene()->getSceneSettingFilePathForPage(d_page);

    QDir dir;
    if (!dir.remove(filePath))
    {
        r = false;
    }
    dir.remove(sceneSettingFilePath);

    if (!getScene()->removePage(d_page))
    {
        r = false;
    }

    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRemovePageCommand::undo()
{

}

/*--------------------------------*/
DCAddDirectoryCommand::DCAddDirectoryCommand(const void *requester, DCCreator *creator, const QString &sysFilePath)
    : DCCommand(requester, creator), d_path(sysFilePath)
{

}

void DCAddDirectoryCommand::redo()
{
    bool r = DCTreeViewWidget::addDirectory(d_path);
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCAddDirectoryCommand::undo()
{

}

/*--------------------------------*/
DCRenameDirectoryCommand::DCRenameDirectoryCommand(const void *requester, DCCreator *creator, const QString &sysOldFilePath, const QString &sysNewFilePath)
    : DCCommand(requester, creator), d_oldPath(sysOldFilePath), d_newPath(sysNewFilePath)
{

}

void DCRenameDirectoryCommand::redo()
{
    DCTreeViewWidget::selectWhenFilePathAdded(d_newPath);
    bool r = moveComponentsToDirectory(d_oldPath, d_newPath);
    if (r)
    {
        DCTreeViewWidget::removeDirectory(d_oldPath);
    }
    else
    {
        //TODO show error message?
        DCTreeViewWidget::selectWhenFilePathAdded("");
    }
    getController()->saveAll(false);
}

void DCRenameDirectoryCommand::undo()
{

}

bool DCRenameDirectoryCommand::moveComponentsToDirectory(const QString &orgSysDirectory, const QString &destSysDirectory)
{    
    if (QDir(destSysDirectory).exists())
        return false;

    QDir dir;
    if (!dir.mkpath(destSysDirectory))
        return false;

    bool r = true;
    QDir orgDir(orgSysDirectory);
    QFileInfoList list = orgDir.entryInfoList(QDir::NoDot | QDir::NoDotDot | QDir::AllEntries);
    for (int i = 0; r && i < list.size(); i++)
    {
        QFileInfo fileInfo = list.at(i);
        if (fileInfo.isDir())
        {
            QDir d(fileInfo.absoluteFilePath());
            QString newDir = destSysDirectory + "/" + d.dirName();
            r = moveComponentsToDirectory(fileInfo.absoluteFilePath(), newDir);
        }
        else
        {
            QString newFile = destSysDirectory + "/" + fileInfo.fileName();
            QDir newFileDir(newFile);
            if (!newFileDir.exists())
            {
                QString oldFileContainerBasedPath = getContainer()->sysFilePathToContainerBasedPath(fileInfo.absoluteFilePath());
                QString newFileContainerBasedPath = getContainer()->sysFilePathToContainerBasedPath(newFile);
                if (getScene()->getIsPageExist(oldFileContainerBasedPath))
                {
                    r = getContainer()->movePage(oldFileContainerBasedPath, newFileContainerBasedPath) != NULL;
                }
                else
                {
                    QFile oldFile(fileInfo.absoluteFilePath());
                    if (oldFile.exists())
                    {
                        r = oldFile.copy(newFile);
                        if (r)
                        {
                            oldFile.remove();
                        }
                    }
                }
            }
        }
    }
    return r;
}


/*--------------------------------*/
DCRemoveDirectoryCommand::DCRemoveDirectoryCommand(const void *requester, DCCreator *creator, const QString &sysFilePath)
    : DCCommand(requester, creator), d_path(sysFilePath)
{

}

void DCRemoveDirectoryCommand::redo()
{
    bool r = DCTreeViewWidget::removeDirectory(d_path);
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRemoveDirectoryCommand::undo()
{

}

/*--------------------------------*/
DCRenameReceptorNameCommand::DCRenameReceptorNameCommand(const void *requester, DCCreator *creator, DCCell *cell, const QString &oldName, const QString &newName)
    : DCCommand(requester, creator), d_cell(cell), d_oldName(oldName), d_newName(newName)
{

}

void DCRenameReceptorNameCommand::redo()
{
    bool r = d_cell->renameReceptor(d_oldName, d_newName);
    setCommandResult(r ? DCCommand::DCCOMMAND_SUCCEEDED : DCCommand::DCCOMMAND_FAILED);
}

void DCRenameReceptorNameCommand::undo()
{

}
