//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#ifndef DCSCENE_H
#define DCSCENE_H

#include <QObject>
#include <QString>
#include <QMap>
#include <QList>
#include <QMutex>

class DCContent;
class DCVComponent;
class DCVCPage;
class DCVCEditModeCursor;
class DCContainer;
class DCCellCode;
class DCCell;
class DCAxon;
class DCReceptor;

class DCScene : public QObject
{
    Q_OBJECT

public:
    enum DCVPersMode { DCV_PERSMODE_NAVIGATION, DCV_PERSMODE_PAGEEDIT };
    enum DCVEditMode { DCV_EDITMODE_LAYOUT, DCV_EDITMODE_TERMINAL_FROM_AXON, DCV_EDITMODE_TERMINAL_FROM_RECEPTOR};

private:
    DCContainer             *d_owner;
    QMap<QString,DCVCPage*> d_pages;
    static bool             sShouldContinueAnimation;

    DCVPersMode             d_vpMode;
    DCVEditMode             d_veMode;

    enum DCSceneTransition { DC_TRANS_NONE,
                             DC_TRANS_TO_PAGEEDIT,
                             DC_TRANS_TO_NAVIGATION
                           };
    DCSceneTransition       d_transition;

    float                   d_scale;
    float                   d_xAngle;
    float                   d_yAngle;
    float                   d_centerX;
    float                   d_centerBrowsModeY;
    float                   d_centerEditModeY;

    float                   d_navigationModeSelectedPageAlpha;
    float                   d_navigationModeAboveAlpha;
    float                   d_navigationModeBelowAlpha;
    float                   d_pageEditModeSelectedPageAlpha;

    bool                    d_navigationModeSelectedCellMovable;
    bool                    d_navigationModeOtherCellSelectable;
    bool                    d_navigationModeOtherCellVisible;
    bool                    d_navigationModeOtherCellRenderFull;

    bool                    d_pageEditModeOtherSamePageRenderFull;
    bool                    d_pageEditModeLinkedOtherPageVisible;
    bool                    d_pageEditModeLinkedOtherPageRenderFull;
    bool                    d_pageEditModeOtherOtherPageVisible;
    bool                    d_pageEditModeOtherOtherPageRenderFull;

    DCVCEditModeCursor      *d_terminalEditCursor;
    DCAxon                  *d_editAxon;
    DCReceptor              *d_editReceptor;

    DCCell                  *d_editCellCodeCell;

public:
    DCScene(DCContainer *owner);
    virtual ~DCScene();

    void initMode(DCVPersMode persMode, DCVEditMode editMode);
    //
    static QMutex*          getSceneLock();
    static bool             getShouldContinueAnimation();

    DCContainer*            getContainer() const { return d_owner; }
    bool                    getIsModified() const;
    float                   getScale() const { return d_scale; }
    float                   getXAngle() const { return d_xAngle; }
    float                   getYAngle() const { return d_yAngle; }
    float                   getCenterX() const { return d_centerX; }
    float                   getCenterBrowsModeY() const { return d_centerBrowsModeY; }
    float                   getCenterEditModeY() const { return d_centerEditModeY; }
    int                     getNumberOfSelectedPages() const;
    QList<DCVCPage*>        getSelectedPages() const;
    QList<DCVComponent*>    getSelectedCellObjects() const;
    DCVPersMode             getPersMode() const { return d_vpMode; }
    DCVEditMode             getEditMode() const { return d_veMode; }
    bool                    getIsInModeTransition() const { return d_transition != DC_TRANS_NONE; }

    float                   getNavModeSelectedPageAlpha() const { return d_navigationModeSelectedPageAlpha; }
    float                   getNavModeAboveAlpha() const { return d_navigationModeAboveAlpha; }
    float                   getNavModeBelowAlpha() const { return d_navigationModeBelowAlpha; }
    float                   getPageModeSelectedAlpha() const { return d_pageEditModeSelectedPageAlpha; }

    bool                    getNavModeSelectedCellMovable() const { return d_navigationModeSelectedCellMovable; }
    bool                    getNavModeOtherCellSelectable() const { return d_navigationModeOtherCellSelectable; }
    bool                    getNavModeOtherCellVisible() const { return d_navigationModeOtherCellVisible; }
    bool                    getNavModeOtherCellRenderFull() const { return d_navigationModeOtherCellRenderFull; }

    bool                    getPageModeOtherSamePageRenderFull() const { return d_pageEditModeOtherSamePageRenderFull; }
    bool                    getPageModeLinkedOtherPageVisible() const { return d_pageEditModeLinkedOtherPageVisible; }
    bool                    getPageModeLinkedOtherPageRenderFull() const { return d_pageEditModeLinkedOtherPageRenderFull; }
    bool                    getPageModeOtherOtherPageVisible() const { return d_pageEditModeOtherOtherPageVisible; }
    bool                    getPageModeOtherOtherPageRenderFull() const { return d_pageEditModeOtherOtherPageRenderFull; }

    DCVCEditModeCursor *    getTerminalEditCursor() const { return d_terminalEditCursor; }
    DCAxon*                 getEditAxon() const { return d_editAxon; }
    DCReceptor*             getEditReceptor() const { return d_editReceptor; }
    QString                 getTerminalEditCursorLocationString() const;
    DCVComponent*           getTerminalEditCursorDropTarget() const;

    DCCell*                 getEditCellCodeCell() const { return d_editCellCodeCell; }

    const QMap<QString,DCVCPage*>* getPages() const { return &d_pages; }
    DCVCPage*               getPage(const std::string& location) const;
    bool                    getIsPageExist(const std::string& location) const;
    bool                    getIsPageExist(const QString& location) const;
    QString                 getSceneSettingFilePathForPage(DCVCPage *page) const;
    QString                 getSceneSettingFilePathForPage(const QString& containerRootPath, const DCVCPage *page) const;

    //
    void                    setNavModeSelectedPageAlpha(const void *requester, float alpha);
    void                    setNavModeAboveAlpha(const void *requester, float alpha);
    void                    setNavModeBelowAlpha(const void *requester, float alpha);
    void                    setPageModeSelectedAlpha(const void *requester, float alpha);

    void                    setNavModeSelectedCellMovable(const void *requester, bool b);
    void                    setNavModeOtherCellSelectable(const void *requester, bool b);
    void                    setNavModeOtherCellVisible(const void *requester, bool b);
    void                    setNavModeOtherCellRenderFull(const void *requester, bool b);

    void                    setPageModeOtherSamePageRenderFull(const void *requester, bool b);
    void                    setPageModeLinkedOtherPageVisible(const void *requester, bool b);
    void                    setPageModeLinkedOtherPageRenderFull(const void *requester, bool b);
    void                    setPageModeOtherOtherPageVisible(const void *requester, bool b);
    void                    setPageModeOtherOtherPageRenderFull(const void *requester, bool b);

    void draw(bool isAnimationInterval);
    void drawForSelection(QList<DCVComponent*> *itemList);
    bool doAnimation();
    void updateVisiblity();

    DCVCPage* addPage(const std::string& location);
    bool removePage(DCVCPage *page);

    void setScale(const void *requester, float scale);
    void setAngle(const void *requester, float xangle, float yangle);
    void setBrowsModeCenter(const void *requester, float cx, float cy);
    void setPageModeCenter(const void *requester, float cx, float cy);

    bool selectPage(const void *requester, const DCVCPage *page, bool multipleSelection);
    bool unselectPage(const void *requester, const DCVCPage *page);
    bool selectPage(const void *requester, const QString &locationPath, bool multipleSelection);
    bool unselectPage(const void *requester, const QString &locationPath);
    bool unselectPageAll(const void *requester);

    void selectCellObject(const void *requester, DCVComponent *object, bool multipleSelection = false);
    void unselectCellObject(const void *requester, DCVComponent *object);
    void unselectCellObjectAll(const void *requester);

    bool changePersMode(const void *requester, DCVPersMode mode);
    bool startTerminalEditForAxon(const void *requester, DCAxon *ownerAxon);
    bool startTerminalEditForReceptor(const void *requester, DCReceptor *ownerReceptor);
    bool finishTerminalEdit(const void *requester);

    bool loadSceneAll();
    bool loadSceneForPage(DCVCPage *page);

    bool saveSceneForPage(const QString& containerRootPath, const DCVCPage *page);

    static void     lockScene();
    static void     unlockScene();
    static void     setShouldContinueAnimation(bool shouldContinue);

signals:
    void selectedPageChanged(const void *requester);
    void selectedCellObjectChanged(const void *requester);
    void viewScaleChanged(const void *requester);
    void viewAngleChanged(const void *requester);
    void viewCenterChanged(const void *requester);
    void viewSettingChanged(const void *requester);
    void viewPersModeChanged(const void* requester);
    void viewEditModeChanged(const void *requester);

    void modeInitialized();

    void terminalEditCursorBelongingPageChagend(DCVCPage *newPage);
    void terminalEditCursorDropTargetChanged(DCVComponent *newTarget);

private slots:
    void slotTerminalEditCursorBelongingPageChagend(DCVCPage *newPage);
    void slotTerminalEditCursorDropTargetChanged(DCVComponent *newTarget);

};

#endif // DCSCENE_H
