//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "TKLog.h"

#include "dcglvisualizerwidget.h"
#include "dcvcomponent.h"
#include "dcvpagecomponent.h"
#include "dcvcpage.h"

#include "dccreator.h"
#include "dcscene.h"
#include "dcanimationtimer.h"
#include "dccell.h"
#include "dcaxon.h"
#include "dcreceptor.h"
#include "utils/dccomponentutil.h"
#include "dcglwidget.h"

#include "dctoolwindowviewcontrol.h"
#include "dctoolwindowcelleditor.h"
#include "dctoolwindowterminalfromaxoneditor.h"
#include "dctoolwindowterminalfromreceptoreditor.h"

#include "eventhandler/dcveventhandler.h"
#include "utils/dcveventhandlerutil.h"
#include "codeeditor/dccellscriptseditorwindow.h"

#include <QtGui>
#include <QObject>
#include <QGLFormat>
#include <QtOpenGL>
#include <QDebug>
#if !defined(Q_WS_MAC)
    #include <GL/glu.h>
#else
    #include <glu.h>
#endif
#include <typeinfo>

#ifndef GL_MULTISAMPLE
#define GL_MULTISAMPLE  0x809D
#endif

static QGraphicsView *s_uiView = NULL;


DCGLVisualizerWidget::DCGLVisualizerWidget(DCCreator *creator, QWidget *parent) : QGraphicsView(parent)
{
    QGraphicsScene *scene = new DCUIGraphicsScene(creator, parent);
    setScene(scene);

    setViewport(DCGLWidget::singleton());
    setViewportUpdateMode(QGraphicsView::FullViewportUpdate);

    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(const QPoint&)),scene, SLOT(showContextMenu(const QPoint&)));

    s_uiView = this;
}

DCGLVisualizerWidget::~DCGLVisualizerWidget()
{
}

QPointF DCGLVisualizerWidget::globalPointToLocal(const QPoint &point)
{
    if (s_uiView && s_uiView->scene())
    {
        return s_uiView->mapToScene(s_uiView->mapFromGlobal(point));
    }
    return QPoint();
}

bool toolWindowPositionLessThan(const DCGraphicsProxyWidget *t1, const DCGraphicsProxyWidget *t2)
{
    if (t1->geometry().right() < t2->geometry().left())
    {
        return true;
    }
    else if (t2->geometry().right() < t1->geometry().left())
    {
        return false;
    }

    if (t1->pos().y() < t2->pos().y())
    {
        return true;
    }
    else if (t1->pos().y() == t2->pos().y())
    {
        if (t1->getPosPriority() < t2->getPosPriority())
        {
            return true;
        }
    }
    return false;
}

void DCGLVisualizerWidget::resizeEvent(QResizeEvent *event) {
    if (scene())
    {
        scene()->setSceneRect(QRect(QPoint(0, 0), event->size()));
    }
    QGraphicsView::resizeEvent(event);
}

void DCGLVisualizerWidget::setupViewport(QWidget *viewport)
{
    QGLWidget *glWidget = dynamic_cast<QGLWidget*>(viewport);
    glWidget->setAutoFillBackground(false);
    glWidget->setMouseTracking(true);

    if (glWidget)
        glWidget->updateGL();
}

void DCGLVisualizerWidget::setVisible(bool visible)
{
    QGraphicsView::setVisible(visible);
}

int DCGraphicsProxyWidget::getPosPriority() const
{
    DCToolWindowBase *window = dynamic_cast<DCToolWindowBase*>(widget());
    if (window)
    {
        return window->getPosPriority();
    }
    return 999;
}

DCGraphicsProxyWidget::DCGraphicsProxyWidget(DCSticky sticky, QGraphicsItem *parent, Qt::WindowFlags wFlags)
    : QGraphicsProxyWidget(parent, wFlags), d_sticky(sticky), d_hasTargetPos(false), d_targetX(0), d_targetY(0),d_embeddedToolWindow(NULL)
{
}

DCGraphicsProxyWidget::~DCGraphicsProxyWidget()
{
    this->disconnect();
}

void DCGraphicsProxyWidget::setTargetPos(float targetX, float targetY)
{
    d_targetX = targetX;
    d_targetY = targetY;
    d_hasTargetPos = true;
}

void DCGraphicsProxyWidget::doAnimation()
{
    const static int MINSPEED = 60;

    if (!d_embeddedToolWindow)
    {
        d_embeddedToolWindow = dynamic_cast<DCToolWindowBase*>(widget());
        if (!d_embeddedToolWindow)
            return;
    }

    if (d_hasTargetPos)
    {
        QPointF p = pos();
        float dx = d_targetX - p.x();
        float dy = d_targetY - p.y();
        float d = sqrt(dx * dx + dy * dy);
        if (d < MINSPEED)
        {
            d_hasTargetPos = false;
            setPos(d_targetX, d_targetY);
        }
        else
        {
            float speed = MINSPEED + d/3;
            float sr = speed / d;
            moveBy(dx * sr, dy * sr);
        }
    }

    d_embeddedToolWindow->doAnimation();
}

QVariant DCGraphicsProxyWidget::itemChange(GraphicsItemChange change, const QVariant &value)
{
    DCUIGraphicsScene *dcGraphicScene = dynamic_cast<DCUIGraphicsScene*>(scene());

    if (change == ItemPositionChange && dcGraphicScene) {
        QRectF rect = boundingRect();
        QPointF pos = value.toPointF();

        bool adjusted = false;

        QRectF sceneRect = dcGraphicScene->sceneRect();

        if (d_sticky == DC_STICKYWIDGET_LEFT && pos.x() + rect.left() > dcGraphicScene->getStickyAreaLeftWidth())
        {
            d_sticky = DC_STICKYWIDGET_NONE;
        }
        else if (d_sticky == DC_STICKYWIDGET_RIGHT && pos.x() + rect.width() < sceneRect.right() - dcGraphicScene->getStickyAreaRightWidth())
        {
            d_sticky = DC_STICKYWIDGET_NONE;
        }

        if (pos.x() + rect.left() < sceneRect.left() + 2)
        {
            d_sticky = DC_STICKYWIDGET_LEFT;
            adjusted = true;
            pos.setX(sceneRect.left() - rect.left() + 2);
        }
        else if (pos.x() + rect.right() >= sceneRect.right() - 2)
        {
            d_sticky = DC_STICKYWIDGET_RIGHT;
            adjusted = true;
            pos.setX(sceneRect.right() - rect.right() - 2);
        }

        if (pos.y() + rect.top() < sceneRect.top() + 2)
        {
            adjusted = true;
            pos.setY(sceneRect.top() - rect.top() + 2);
        }
        else if (pos.y() + rect.bottom() >= sceneRect.bottom() - 2)
        {
            adjusted = true;
            pos.setY(sceneRect.bottom() - rect.bottom() - 2);
        }

        if (adjusted)
        {
            widget()->move(pos.x(), pos.y());
        }
        return pos;
    }
    return QGraphicsProxyWidget::itemChange(change, value);
}

void DCGraphicsProxyWidget::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    QGraphicsProxyWidget::resizeEvent(event);
}


DCUIGraphicsScene::DCUIGraphicsScene(DCCreator *creator, QWidget *parent)
    : QGraphicsScene(parent), d_creator(creator), d_scene(NULL), d_eventHandler(NULL), d_isPaintForAnimation(false), d_requestRedraw(false),
      d_inResigingDrag(false), d_dragStartX(0), d_dragStartY(0),
      d_requestLayoutToolWindows(false), d_stickyAreaLeftWidth(10), d_stickyAreaRightWidth(10),
      d_terminalFromAxonToolWindow(NULL),d_terminalFromReceptorToolWindow(NULL)
{
    connect(d_creator, SIGNAL(sceneChanged(const void*, DCScene*)), this, SLOT(sceneChanged(const void*, DCScene*)));
    connect(d_creator, SIGNAL(sceneSelectedPageChanged(const void*, const DCScene*)), this, SLOT(sceneSelectedPageChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(sceneSelectedCellChanged(const void*, const DCScene*)), this, SLOT(sceneSelectedCellChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(sceneViewAngleChanged(const void*,const DCScene*)), this, SLOT(sceneAngleChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(sceneViewCenterChanged(const void*,const DCScene*)), this, SLOT(sceneCenterChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(sceneViewScaleChanged(const void*, const DCScene*)), this, SLOT(sceneScaleChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(sceneViewSettingChanged(const void*, const DCScene*)), this, SLOT(sceneSettingChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(sceneViewEditModeChanged(const void*,const DCScene*)), this, SLOT(sceneEditModeChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(destroyed(QObject*)), this, SLOT(creatorDestroyed()));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand *)), this, SLOT(creatorCommandExecuted()));
    connect(DCAnimationTimer::instance(), SIGNAL(goNextFrame()), this, SLOT(doAnimation()));
    connect(this, SIGNAL(sceneRectChanged(QRectF)), this, SLOT(resizeEvent(QRectF)));
    d_viewControlTool = new DCToolWindowViewControl(d_creator);
    addToolWindow(d_viewControlTool);
}

DCUIGraphicsScene::~DCUIGraphicsScene()
{
    if (d_creator)
    {
        d_creator->disconnect(this);
    }
    DCAnimationTimer::instance()->disconnect(this);

    foreach (QGraphicsItem *item, items())
    {
        removeItem(item);
    }

    while (d_cellEditToolWindowPool.length() > 0)
    {
        DCToolWindowBase *window = d_cellEditToolWindowPool.takeFirst();
        delete window;
    }

    while (d_cellEditToolWindowList.length() > 0)
    {
        DCToolWindowBase *window = d_cellEditToolWindowList.takeFirst();
        removeToolWindow(window);
        delete window;
    }

    if (d_terminalFromAxonToolWindow)
    {
        removeToolWindow(d_terminalFromAxonToolWindow);
        delete d_terminalFromAxonToolWindow;
    }
}


void DCUIGraphicsScene::addToolWindow(DCToolWindowBase *window, int x, int y, DCGraphicsProxyWidget::DCSticky sticky )
{
    DCGraphicsProxyWidget *proxy = new DCGraphicsProxyWidget(sticky);
    proxy->setWidget(window);
    proxy->setZValue(1e30);
    proxy->setCacheMode(QGraphicsItem::DeviceCoordinateCache);  // QGraphicsItem::ItemCoordinateCache
    proxy->setFlag(QGraphicsItem::ItemIsMovable);
    proxy->setFlag(QGraphicsItem::ItemIsSelectable, true);
    proxy->setFlag(QGraphicsItem::ItemIsFocusable, true);

    proxy->setPos(x,y);
    addItem(proxy);
    connect(window, SIGNAL(dragMoveFinished()), this, SLOT(requestLayoutToolWindows()));
    connect(proxy, SIGNAL(heightChanged()), this, SLOT(requestLayoutToolWindows()));
    connect(proxy, SIGNAL(widthChanged()), this, SLOT(requestLayoutToolWindows()));

    requestLayoutToolWindows();
}

void DCUIGraphicsScene::removeToolWindow(DCToolWindowBase *window)
{
    for (int i = items().length() - 1; i >= 0; i-- )
    {
        DCGraphicsProxyWidget *proxy = dynamic_cast<DCGraphicsProxyWidget*>(items().at(i));
        if (proxy && proxy->widget() == window)
        {
            proxy->setWidget(NULL);
            removeItem(proxy);
            delete proxy;
            break;
        }
    }
    requestLayoutToolWindows();
}

void DCUIGraphicsScene::layoutToolWindows()
{
    QList<DCGraphicsProxyWidget*> stickyListLeft;
    QList<DCGraphicsProxyWidget*> stickyListRight;

    foreach (QGraphicsItem *item, items())
    {
        DCGraphicsProxyWidget *proxy = dynamic_cast<DCGraphicsProxyWidget*>(item);
        if (proxy)
        {
            switch(proxy->getSticky())
            {
            case DCGraphicsProxyWidget::DC_STICKYWIDGET_LEFT:
                stickyListLeft.append(proxy);
                break;
            case DCGraphicsProxyWidget::DC_STICKYWIDGET_RIGHT:
                stickyListRight.append(proxy);
                break;
            default:
                break;
            }
        }
    }
    qSort(stickyListLeft.begin(), stickyListLeft.end(), toolWindowPositionLessThan);
    qSort(stickyListRight.begin(), stickyListRight.end(), toolWindowPositionLessThan);

    QRectF srect = sceneRect();
    QPointF pos(10, 10);
    float xmax = 10;
    for (int i = 0; i < stickyListLeft.length(); i++)
    {
        DCGraphicsProxyWidget *item = stickyListLeft.at(i);
        const QRectF rect = item->boundingRect();
        float tx = pos.x() - rect.x();
        float ty =  pos.y() - rect.y();
        if (ty + rect.height() > srect.height())
        {
            pos.setX(xmax + 10);
            pos.setY(10);
            tx = pos.x() - rect.x();
            ty =  pos.y() - rect.y();
        }
        item->setTargetPos(tx, ty);
        if (tx + rect.width() > xmax)
            xmax = tx + rect.width();
        pos += QPointF(0, 10 + rect.height());
    }
    d_stickyAreaLeftWidth = xmax;

    pos.setX(srect.width() - 10);
    pos.setY(srect.height() - 10);
    float xmin = srect.width() - 10;
    for (int i = stickyListRight.length() - 1; i >= 0 ; i--)
    {
        DCGraphicsProxyWidget *item = stickyListRight.at(i);
        const QRectF rect = item->boundingRect();
        float tx = pos.x() - rect.width() - rect.x();
        float ty = pos.y() - rect.height() - rect.y();
        if (ty < 0)
        {
            pos.setX(xmin - 10);
            pos.setY(srect.height() - 10);
            tx = pos.x() - rect.width() - rect.x();
            ty = pos.y() - rect.height() - rect.y();
        }
        item->setTargetPos(tx, ty);
        if (tx < xmin)
            xmin = tx;
        pos -= QPointF(0, 10 + rect.height());
    }
    d_stickyAreaRightWidth = srect.width() - xmin;

    d_requestLayoutToolWindows = false;
}

void DCUIGraphicsScene::creatorDestroyed()
{
    d_creator = NULL;
}

void DCUIGraphicsScene::sceneChanged(const void *requester, DCScene *scene)
{
    (void)requester;

    DCScene::lockScene();
    if (d_eventHandler)
    {
        delete d_eventHandler;
        d_eventHandler = NULL;
    }
    if (d_scene)
    {
        d_scene->disconnect(d_viewControlTool);
        d_scene->disconnect(this);
    }
    d_scene = scene;
    if (d_scene)
    {
        d_viewControlTool->setDCScene(d_scene);
        connect(d_scene, SIGNAL(viewPersModeChanged(const void*)), d_viewControlTool, SLOT(changeMode(const void*)));
        connect(d_scene, SIGNAL(modeInitialized()), this, SLOT(initMode()));
        connect(d_scene, SIGNAL(terminalEditCursorBelongingPageChagend(DCVCPage*)), this, SLOT(sceneTerminalEditPageBelongingChanged(DCVCPage*)));
        connect(d_scene, SIGNAL(terminalEditCursorDropTargetChanged(DCVComponent*)), this, SLOT(sceneTerminalEditDropTargetChanged(DCVComponent*)));
    }

    while (!d_cellEditToolWindowList.isEmpty())
    {
        DCToolWindowBase *toolWindow = d_cellEditToolWindowList.takeFirst();
        removeToolWindow(toolWindow);
    }

    DCScene::unlockScene();
    updateView(false);
}

void DCUIGraphicsScene::sceneAngleChanged(const void* requester, const DCScene *scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    d_requestRedraw = true;
}

void DCUIGraphicsScene::sceneCenterChanged(const void *requester, const DCScene *scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    d_requestRedraw = true;
}

void DCUIGraphicsScene::sceneScaleChanged(const void* requester, const DCScene *scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    d_requestRedraw = true;
}

void DCUIGraphicsScene::sceneSettingChanged(const void* requester, const DCScene *scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    d_requestRedraw = true;
}

void DCUIGraphicsScene::sceneSelectedPageChanged(const void* requester, const DCScene *scene)
{
    if (d_scene != scene)
        return;

    if (d_eventHandler)
        d_eventHandler->selectedPageChanged(requester);

    d_requestRedraw = true;
}


void DCUIGraphicsScene::sceneSelectedCellChanged(const void* requester, const DCScene *scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    switch (scene->getEditMode())
    {
    case DCScene::DCV_EDITMODE_LAYOUT:
        updateCellEditorToolWindows();
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_AXON:
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_RECEPTOR:
        break;

    }

    d_requestRedraw = true;
}

void DCUIGraphicsScene::sceneEditModeChanged(const void *requester, const DCScene *scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    if (d_eventHandler && scene->getEditMode() == d_eventHandler->getHandlerType() &&  d_scene == d_eventHandler->getScene())
        return;

    if (d_eventHandler)
    {
        delete d_eventHandler;
        d_eventHandler = NULL;
    }

    switch (scene->getEditMode())
    {
    case DCScene::DCV_EDITMODE_LAYOUT:
        closeTerminalFromAxonToolWindow();
        closeTerminalFromReceptorToolWindow();
        showCellEditorToolWindows();
        d_eventHandler = DCVEventHandlerUtil::createLayoutModeHandler(this, d_creator, d_scene);
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_AXON:
        closeAllCellEditorToolWindows();
        closeTerminalFromReceptorToolWindow();
        showTerminalFromAxonToolWindow();
        d_eventHandler = DCVEventHandlerUtil::createTerminalFromAxonModeHandler(this, d_creator, d_scene, scene->getEditAxon(), scene->getTerminalEditCursor());
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_RECEPTOR:
        closeAllCellEditorToolWindows();
        closeTerminalFromAxonToolWindow();
        showTerminalFromReceptorToolWindow();
        d_eventHandler = DCVEventHandlerUtil::createTerminalFromReceptorModeHandler(this, d_creator, d_scene, scene->getEditReceptor(), scene->getTerminalEditCursor());
        break;

    }
    requestRedraw(false);
}

void DCUIGraphicsScene::sceneTerminalEditPageBelongingChanged(DCVCPage *page)
{
    (void)page;

    switch (d_scene->getEditMode())
    {
    case DCScene::DCV_EDITMODE_LAYOUT:
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_AXON:
        updateTerminalFromAxonToolWindow();
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_RECEPTOR:
        updateTerminalFromReceptorToolWindow();
        break;

    }

    d_requestRedraw = true;
}

void DCUIGraphicsScene::sceneTerminalEditDropTargetChanged(DCVComponent *dropTarget)
{
    (void)dropTarget;

    switch (d_scene->getEditMode())
    {
    case DCScene::DCV_EDITMODE_LAYOUT:
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_AXON:
        updateTerminalFromAxonToolWindow();
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_RECEPTOR:
        updateTerminalFromReceptorToolWindow();
        break;

    }

    d_requestRedraw = true;
}

void DCUIGraphicsScene::setupProjection()
{
    if (!d_creator || !d_scene)
        return;

    float _width = width();
    float _height = height();
    float _scale = d_scene->getScale();
    glViewport(0, 0, _width, _height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-_width / _scale, _width / _scale, -_height / _scale, _height / _scale, -100.0, 100.0);

    setupCamera();

    glGetDoublev(GL_PROJECTION_MATRIX, d_projectionMatrix);
    glGetIntegerv(GL_VIEWPORT, d_viewportMatrix);

}

void DCUIGraphicsScene::setupCamera()
{
    glTranslatef(- d_scene->getCenterX(),  d_scene->getCenterEditModeY(), 0);
    glRotatef(d_scene->getXAngle(),1,0,0);
    glRotatef(d_scene->getYAngle(),0,1,0);
    glTranslatef(0, - d_scene->getCenterBrowsModeY(), 0);
}

void DCUIGraphicsScene::updateView(bool isAnimationInterval)
{
    DCScene::lockScene();

    DCGLWidget::singleton()->makeCurrent();

    d_isPaintForAnimation = isAnimationInterval;
    d_requestRedraw = false;

    update();
}

void DCUIGraphicsScene::requestRedraw(bool immediate)
{
    if (immediate)
        updateView(false);
    else
        d_requestRedraw = true;
}

DCVComponent* DCUIGraphicsScene::getObjectAt(int viewX, int viewY, DCVComponent *priorityObject)
{
    if (!d_creator || !d_scene)
        return NULL;

    DCScene::lockScene();

    DCGLWidget::singleton()->makeCurrent();

    float _width = width();
    float _height = height();
    float _scale = d_scene->getScale();

    GLuint selectionBuffer[512];
    glSelectBuffer(512, selectionBuffer);
    glRenderMode(GL_SELECT);
    glInitNames();
    glPushName(0);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    int viewportMat[4];
    memcpy(viewportMat, d_viewportMatrix, sizeof(viewportMat));
    gluPickMatrix(viewX,viewportMat[3] - viewY, 4,4,viewportMat);
    glOrtho(-_width / _scale, _width / _scale, -_height / _scale, _height / _scale, -100.0, 100.0);

    setupCamera();

    QList<DCVComponent *> itemList;
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    d_scene->drawForSelection(&itemList);
    glPopMatrix();

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();

    glFlush();
    glMatrixMode(GL_MODELVIEW);

    int items = glRenderMode(GL_RENDER);
    DCVComponent *selectedObject = NULL;
    DCVComponent *selectedPage = NULL;
    int idx = 0;
    unsigned int minDepthPage = UINT_MAX;
    unsigned int minDepthObject = UINT_MAX;
    if (priorityObject)
    {
        for (int i = 0; i < items; i++)
        {
            unsigned int nameLen = selectionBuffer[idx];
            unsigned int min = selectionBuffer[idx + 1];
            unsigned int nameId = selectionBuffer[idx + 3];

            if (nameLen == 1 && nameId < (unsigned int) itemList.length())
            {
                DCVComponent *obj = itemList[nameId];
                if (priorityObject == obj)
                {
                    selectedObject = obj;
                    break;
                }
                if (obj->getPageBelonging() == obj)
                {
                    if (min < minDepthPage)
                    {
                        minDepthPage = min;
                        selectedPage = obj;
                    }
                }
                else
                {
                    if (obj->getIsSelectable() &&  min < minDepthObject)
                    {
                        minDepthObject = min;
                        selectedObject = obj;
                    }
                }
            }
            idx += nameLen + 3;
        }
    }
    else
    {
        for (int i = 0; i < items; i++)
        {
            unsigned int nameLen = selectionBuffer[idx];
            unsigned int min = selectionBuffer[idx + 1];
            unsigned int nameId = selectionBuffer[idx + 3];

            if (nameLen == 1 && nameId < (unsigned int) itemList.length())
            {
                DCVComponent *obj = itemList[nameId];
                if (obj->getPageBelonging() == obj)
                {
                    if (min < minDepthPage)
                    {
                        minDepthPage = min;
                        selectedPage = obj;
                    }
                }
                else
                {
                    if (obj->getIsSelectable() &&  min < minDepthObject)
                    {
                        minDepthObject = min;
                        selectedObject = obj;
                    }
                }
            }
            idx += nameLen + 3;
        }
    }

    DCScene::unlockScene();

    return selectedObject ? selectedObject : selectedPage;
}

//slot
void DCUIGraphicsScene::initMode()
{
    sceneEditModeChanged(NULL, d_scene);
}

//slot
void DCUIGraphicsScene::doAnimation()
{
    if (d_requestLayoutToolWindows)
    {
        layoutToolWindows();
    }

    foreach (QGraphicsItem *item, items())
    {
        DCGraphicsProxyWidget *proxy = dynamic_cast<DCGraphicsProxyWidget*>(item);
        if (proxy)
        {
            proxy->doAnimation();
        }
    }

    if (!d_creator || !d_scene)
        return;

    if (DCScene::getShouldContinueAnimation())
    {
        bool r = d_scene->doAnimation();
        DCScene::setShouldContinueAnimation(r);
        updateView(true);
    }
    else if (d_requestRedraw)
    {
        updateView(false);
    }
}


void DCUIGraphicsScene::drawBackground(QPainter *painter, const QRectF &rect)
{
    (void)rect;
    painter->beginNativePainting();
    setStates();

    paintGL();

    defaultStates();
    painter->endNativePainting();
}

void DCUIGraphicsScene::setStates()
{
    glPushAttrib(GL_ALL_ATTRIB_BITS);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

    initializeGL();
    glBindTexture(GL_TEXTURE_2D, 0);

}

void DCUIGraphicsScene::defaultStates()
{
    glPopAttrib();

    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();

}

void DCUIGraphicsScene::initializeGL()
{
    glClearColor(0.0, 0.0, 0.0, 1.0);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_MULTISAMPLE);
    glEnable(GL_CULL_FACE);
    glEnable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
    glDisable(GL_LIGHT0);
    glDisable(GL_NORMALIZE);
}

void DCUIGraphicsScene::paintGL()
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    setupProjection();

    DCScene::lockScene();
    glMatrixMode(GL_MODELVIEW);
    if (d_scene)
    {
        glPushMatrix();
        d_scene->draw(d_isPaintForAnimation);
        glPopMatrix();
    }
    d_isPaintForAnimation = false;
    glFlush();
    DCScene::unlockScene();
}

void DCUIGraphicsScene::resizeEvent(const QRectF & rect)
{
    (void)rect;

    updateView(false);
    requestLayoutToolWindows();
}

void DCUIGraphicsScene::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    QGraphicsScene::mousePressEvent(event);
    if (event->isAccepted()) return;

    if (!d_creator || !d_scene || d_scene->getIsInModeTransition())
        return;

    QPointF mousePos = event->scenePos();
    d_dragStartX = mousePos.x();
    d_dragStartY = mousePos.y();

    d_clickTime.start();

    if (d_eventHandler)
        d_eventHandler->mousePressEvent(event);
}

void DCUIGraphicsScene::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
    QGraphicsScene::mousePressEvent(event);
    if (event->isAccepted()) return;

    if (!d_creator || !d_scene || d_scene->getIsInModeTransition())
        return;

    if (d_eventHandler)
        d_eventHandler->mouseDoubleClickEvent(event);
}

void DCUIGraphicsScene::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    QGraphicsScene::mouseMoveEvent(event);
    if (event->isAccepted()) return;

    if (!d_creator || !d_scene || d_scene->getIsInModeTransition())
        return;

    if (d_eventHandler)
        d_eventHandler->mouseMoveEvent(event, d_clickTime.elapsed());
}

void DCUIGraphicsScene::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    QGraphicsScene::mouseReleaseEvent(event);
    if (event->isAccepted()) return;
    if (!d_creator || !d_scene || d_scene->getIsInModeTransition())
        return;

    QPointF mousePos = event->scenePos();

    int dx = mousePos.x() - d_dragStartX;
    int dy = mousePos.y() - d_dragStartY;

    bool wasDrag = !(d_clickTime.elapsed() < 500 && dx * dx + dy * dy < 400);

    DCGLWidget::singleton()->setCursor(Qt::ArrowCursor);

    if (d_eventHandler)
        d_eventHandler->mouseReleaseEvent(event, wasDrag);
}

void DCUIGraphicsScene::wheelEvent(QGraphicsSceneWheelEvent  *event)
{
    QGraphicsScene::wheelEvent(event);

    if (!d_creator || !d_scene)
        return;

    if (d_eventHandler)
        d_eventHandler->wheelEvent(event);
}

void DCUIGraphicsScene::keyPressEvent(QKeyEvent *event)
{
    QGraphicsScene::keyPressEvent(event);
    if (event->isAccepted()) return;

    if (!d_creator || !d_scene)
        return;

    if (d_eventHandler)
        d_eventHandler->keyPressEvent(event);
}

void DCUIGraphicsScene::keyReleaseEvent(QKeyEvent *event)
{
    QGraphicsScene::keyReleaseEvent(event);
    if (event->isAccepted()) return;

    if (d_eventHandler)
        d_eventHandler->keyReleaseEvent(event);
}

void DCUIGraphicsScene::showContextMenu(const QPoint &pos)
{
    if (!d_creator || !d_scene)
        return;

    if (d_eventHandler)
    {
        d_eventHandler->showContextMenu(pos);
    }
}

void DCUIGraphicsScene::menuActionEditThisPage()
{
    if (!d_creator || !d_scene)
        return;

    d_creator->changePersMode(this, DCCreator::DC_PERSMODE_NAVIGATION);
}

void DCUIGraphicsScene::menuActionEditCellCode()
{
    if (!d_creator || !d_scene)
        return;

    DCCell *cell = NULL;
    bool isSingleCellSelection = false;
    QList<DCVComponent*>  list = d_scene->getSelectedCellObjects();
    for (int i = 0; i < list.size(); i++)
    {
        if (cell && cell != list.at(i)->getOwnerCell())
        {
            isSingleCellSelection = false;
            break;
        }
        else if (cell == NULL)
        {
            cell = list.at(i)->getOwnerCell();
        }
    }
    DCCellScriptsEditorWindow::startEditing(cell);
}

void DCUIGraphicsScene::menuActionFinishEditMode()
{
    if (!d_creator || !d_scene)
        return;

    d_creator->changePersMode(this, DCCreator::DC_PERSMODE_PAGEEDIT);
}

void DCUIGraphicsScene::menuActionEditReceptor()
{
    if (!d_creator || !d_scene)
        return;

    //TODO
}

void DCUIGraphicsScene::menuActionEditAxonTerminal()
{
    if (!d_creator || !d_scene)
        return;

    //TODO
}

void DCUIGraphicsScene::menuActionSaveChangeAll()
{
    if (!d_creator || !d_scene)
        return;

    d_creator->saveAll(true);
}

void DCUIGraphicsScene::creatorCommandExecuted()
{
    switch (d_scene->getEditMode())
    {
    case DCScene::DCV_EDITMODE_LAYOUT:
        updateCellEditorToolWindows();
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_AXON:
        break;

    case DCScene::DCV_EDITMODE_TERMINAL_FROM_RECEPTOR:
        break;

    }

    d_requestRedraw = true;
}

bool DCUIGraphicsScene::toObjectCoordinateXZ(float viewX, float viewY, DCVComponent *object, float *pLocalX, float *pLocalZ)
{
    double localX0, localY0, localZ0;
    double localX1, localY1, localZ1;

    gluUnProject(viewX,  d_viewportMatrix[3] - viewY, 0, object->getModelViewMatrix(), d_projectionMatrix, d_viewportMatrix, &localX0, &localY0, &localZ0);
    gluUnProject(viewX,  d_viewportMatrix[3] - viewY, 1, object->getModelViewMatrix(), d_projectionMatrix, d_viewportMatrix, &localX1, &localY1, &localZ1);
    double dx = localX1 - localX0;
    double dy = localY1 - localY0;
    double dz = localZ1 - localZ0;
    if (dy == 0)
    {
        *pLocalX = 0;
        *pLocalZ = 0;
        return false;
    }
    else
    {
        *pLocalX = -(dx * localY0 - dy * localX0)/dy;
        *pLocalZ = -(dz * localY0 - dy * localZ0)/dy;
    }
    return true;
}

void DCUIGraphicsScene::showCellEditorToolWindows()
{
    updateCellEditorToolWindows();
}

void DCUIGraphicsScene::updateCellEditorToolWindows()
{
    QList<DCVComponent*> selectedList = d_scene->getSelectedCellObjects();

    for (int i = d_cellEditToolWindowList.length() - 1; i >= 0 ; i--)
    {
        DCToolWindowCellEditor *window = d_cellEditToolWindowList.at(i);
        if (window->getCell() == NULL || !window->getCell()->getVComponent()->getIsSelected())
        {
            removeToolWindow(window);
            d_cellEditToolWindowList.takeAt(i);
            d_cellEditToolWindowPool.push_back(window);
        }
        else
        {
            window->updateView();
            requestLayoutToolWindows();
        }
    }

    for (int i = 0; i < selectedList.length(); i++)
    {
        bool found = false;
        for (int j = 0 ; j < d_cellEditToolWindowList.length(); j++)
        {
            if (d_cellEditToolWindowList.at(j)->getCell() == selectedList.at(i)->getOwnerCell())
            {
                found = true;
                break;
            }
        }
        if (!found)
        {
            DCToolWindowCellEditor *window = NULL;
            if (d_cellEditToolWindowPool.length() > 0)
            {
                window = d_cellEditToolWindowPool.takeFirst();
            }
            else
            {
                window = new DCToolWindowCellEditor(d_creator);
            }
            addToolWindow(window);
            window->setCell(selectedList.at(i)->getOwnerCell());
            d_cellEditToolWindowList.append(window);
        }
    }
}

void DCUIGraphicsScene::closeAllCellEditorToolWindows()
{
    while (d_cellEditToolWindowList.length() > 0)
    {
        DCToolWindowCellEditor *window = d_cellEditToolWindowList.takeFirst();
        removeToolWindow(window);
        d_cellEditToolWindowPool.push_back(window);
    }
    requestLayoutToolWindows();
}

void DCUIGraphicsScene::showTerminalFromAxonToolWindow()
{
    if (d_terminalFromAxonToolWindow == NULL)
    {
        d_terminalFromAxonToolWindow = new DCToolWindowTerminalFromAxon(d_creator);
    }
    updateTerminalFromAxonToolWindow();
    addToolWindow(d_terminalFromAxonToolWindow);
}

void DCUIGraphicsScene::updateTerminalFromAxonToolWindow()
{
    if (d_terminalFromAxonToolWindow)
    {
        d_terminalFromAxonToolWindow->updateData(d_scene->getEditAxon()->getOwnerCell(),
                                                 d_scene->getTerminalEditCursorLocationString(),
                                                 d_scene->getTerminalEditCursorDropTarget() ? d_scene->getTerminalEditCursorDropTarget()->getOwnerCell() : NULL);
    }
}

void DCUIGraphicsScene::closeTerminalFromAxonToolWindow()
{
    if (d_terminalFromAxonToolWindow)
    {
        removeToolWindow(d_terminalFromAxonToolWindow);
    }
}

void DCUIGraphicsScene::showTerminalFromReceptorToolWindow()
{
    if (d_terminalFromReceptorToolWindow == NULL)
    {
        d_terminalFromReceptorToolWindow = new DCToolWindowTerminalFromReceptor(d_creator);
    }
    updateTerminalFromReceptorToolWindow();
    addToolWindow(d_terminalFromReceptorToolWindow);
}

void DCUIGraphicsScene::updateTerminalFromReceptorToolWindow()
{
    if (d_terminalFromReceptorToolWindow)
    {
        DCReceptor *receptor = d_scene->getEditReceptor();
         if (receptor)
           d_terminalFromReceptorToolWindow->updateData(receptor->getOwnerCell(),
                                                         receptor,
                                                         d_scene->getTerminalEditCursorLocationString(),
                                                         d_scene->getTerminalEditCursorDropTarget() ? d_scene->getTerminalEditCursorDropTarget()->getOwnerCell() : NULL);
    }
}

void DCUIGraphicsScene::closeTerminalFromReceptorToolWindow()
{
    if (d_terminalFromReceptorToolWindow)
    {
        d_terminalFromReceptorToolWindow->resetData();
        removeToolWindow(d_terminalFromReceptorToolWindow);
    }
}
