//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#ifndef DCVCPAGE_H
#define DCVCPAGE_H

#include "dcvcomponent.h"

class DCVPageComponent;
class DCContainer;

class DCVCPage : public DCVComponent
{
    QString d_locationPath;
    float   d_height;
    float   d_width;
    float   d_depth;
    bool    d_shouldUpdateShape;
    float   d_colorR;
    float   d_colorG;
    float   d_colorB;
    float   d_colorA;

    QList<DCVPageComponent*> d_cells;
    QList<DCVPageComponent*> d_cellCodeClasses;

    DCVPageComponent *d_editCursor;

    QString d_hash;

public:
    DCVCPage(const QString &locationPath);
    virtual ~DCVCPage();

    virtual DCVCPage *      getPageBelonging() const { return const_cast<DCVCPage*>(this);}
    virtual DCCell*         getOwnerCell() const { return NULL; }
    virtual DCVVisibility   getIsVisible() const { return d_colorA == 0 ?  DCV_VISIBLE_NONE : DCVComponent::getIsVisible(); }
    virtual bool            isResizingArea(float x, float y, float z) const { (void)x; (void)y; (void)z; return false; }
    QString                 getLocationPath() const { return d_locationPath; }
    bool                    getIsModified(DCContainer *container) const;
    inline float            getWidth() const { return d_width; }
    inline float            getDepth() const { return d_depth; }

    const QList<DCVPageComponent*>* getCells() const { return &d_cells; }
    const QList<DCVPageComponent*>* getCellCodeClasses() const { return &d_cellCodeClasses; }

    // register / unregister cell to this page.
    // They are called from DCVCCell.
    // Other classes shouldn't call this directly.
    // To change the page a cell belonging,
    // DCMoveCellCommand() should be used instead.
    void registerCell(DCVPageComponent *cell);
    void unregisterCell(DCVPageComponent *cell);

    // register / unregister cellcode to this page.
    // They are called from DCVCCellCode.
    // Other classes shouldn't call this directly.
    // To change the page a cell code class belonging,
    // DCMoveCellCodeClassCommand should be used instead.
    void registerCellCodeClass(DCVPageComponent *cellCodeClass);
    void unregisterCellCodeClass(DCVPageComponent *cellCodeClass);

    // register / unregister cellcode to this page.
    // They are called from DCVCEditModeCursor.
    // Other classes shouldn't call this directly.
    // To change the page a cell code class belonging,
    // DCVCEditModeCursor::changePageBelonging() should be called instead.
    void registerEditCursor(DCVPageComponent *ecursor);
    void unregisterEditCursor();

    /**
     * @brief Move cells and cell codes belong to this page to newPage.
     * @param newPage : the target page cells and cell codes are moving into.
     *
     * This will move cells, cell codes and connections belong to this page to the page passed in the parameter.
     * This expected to be called from DCContainer.
     * This shouldn't be called directly.
     * To change the path / file name for a container file, DCMovePageCommand should be used.
     *
     * @return true for succuss, false for failure.
     */
    bool moveComponentsTo(DCVCPage *newPage);

    void updateVisiblity(bool linkedCellVisible, bool linkedCellRenderFull, bool linkedCellSelectable, bool otherCellVisible, bool otherCellRenderFull, bool otherCellSelectable);

    virtual void    prepareChildrenForDraw(bool isAnimationInterval);
    virtual void    drawChildren(bool isAnimationInterval);
    virtual void    drawChildrenForSelection(QList<DCVComponent*> *itemList);
    virtual void    translate();
    virtual void    setSelected(bool selected, bool updateChildren);
    virtual void    setVisible(DCVVisibility visibleSelf, DCVVisibility visibleChildren);
    virtual void    renderOwnShape(bool isAnimationInterval, bool renderAsWireframe);

    virtual bool    startDrag(float x, float y, float z, bool isResizingDrag);
    virtual bool    dragging(float x, float y, float z, bool isResizingDrag);
    virtual bool    endDrag(float x, float y, float z, bool isResizingDrag);
    virtual void    updateShape() {}

    virtual void    saveAttributesToXML(QDomDocument *document, QDomElement* element) const;
    virtual void    loadAttributesFromXML(QDomElement element);

    void setHeight(float height);
    void setWidth(float width);
    void setDepth(float depth);
    void setColor(float r, float g, float b, float a);

    void updateSavedState(DCContainer *container);
};

#endif // DCVCPAGE_H
