//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcvccell.h"

#include "dccell.h"
#include "dccellcode.h"
#include "dcaxon.h"
#include "dcreceptor.h"
#include "dccontainer.h"
#include "dcscene.h"
#include "dnplugininfo.h"

#include "dccuberenderer.h"
#include "dcselectionrenderer.h"

#include <QtOpenGL>


static const double CUBE_ROUNDRADIUS = 0.05;
static const double PI = 3.14159265358979323846264338327950288419717;

DCVCCell::DCVCCell(DCCell *owner, DCVCPage *page, float size, float height)
    : DCVPageComponent(page), d_owner(owner), d_shouldUpdateShape(true),
      d_dragOffsetX(0), d_dragOffsetY(0), d_dragOffsetZ(0), d_draggingOriginalSize(0), d_isDragging(false), d_isResizingDrag(false), d_draggingOriginalAxonLength(0), d_emptyCellCode(0)
{
    DCContainer *container = dynamic_cast<DCContainer*> (d_owner->getContainer());
    if (container)
    {
        d_shape = new DCCubeRenderer(size,height, CUBE_ROUNDRADIUS,3, true);
        d_shape->setEdgeColor(0.12f,0.12f,0.12f);
        d_shouldUpdateShape = true;
        d_emptyCellCode = container->getEmptyCellCodeClass();
    }

    d_selectionRectRenderer = new DCSelectionRenderer(CUBE_ROUNDRADIUS,CUBE_ROUNDRADIUS);
}

DCVCCell::~DCVCCell()
{
    if (d_shape)
        delete d_shape;

    if (d_selectionRectRenderer)
        delete d_selectionRectRenderer;


}

void DCVCCell::changePageBelonging(DCVCPage *page)
{
    if (page)
    {
        page->registerCell(this);
    }
    else
    {
        getPageBelonging()->unregisterCell(this);
    }

    DCContainer *container = dynamic_cast<DCContainer*> (d_owner->getContainer());
    if (container)
    {
        if (getPageBelonging())
        {
            container->moveCell(getOwnerCell(), page->getLocationPath());
        }
        else
        {
            //TODO
        }
    }
}

void DCVCCell::prepareChildrenForDraw(bool isAnimationInterval)
{
    const TKReceptorMap *receptors = d_owner->getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        ((DCReceptor*)(*it).second)->getVComponent()->prepareForDraw(isAnimationInterval);
        ++it;
    }
    d_owner->getAxon()->getVComponent()->prepareForDraw(isAnimationInterval);

    float s = d_owner->getViewSize();
    float ew = fabs(getPageX()) * 2 + s + 0.75;
    float ed = fabs(getPageY()) * 2 + s + 0.75;

    DCVCPage *page = getPageBelonging();
    if (ew > page->getWidth())
    {
        page->setWidth(ew);
    }

    if (ed > page->getDepth())
    {
        page->setDepth(ed);
    }
}

void DCVCCell::drawChildren(bool isAnimationInterval)
{
    const TKReceptorMap *receptors = d_owner->getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        ((DCReceptor*)(*it).second)->getVComponent()->draw(isAnimationInterval);
        ++it;
    }
    d_owner->getAxon()->getVComponent()->draw(isAnimationInterval);
}

void DCVCCell::drawChildrenForSelection(QList<DCVComponent *> *itemList)
{
    const TKReceptorMap *receptors = d_owner->getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        ((DCReceptor*)(*it).second)->getVComponent()->draw(false);
        ++it;
    }
    d_owner->getAxon()->getVComponent()->drawForSelection(itemList);
}

void DCVCCell::translate()
{
    glTranslatef(getPageX() , d_owner->getViewHeight()/2.0, getPageY());
}

void DCVCCell::setSelected(bool selected, bool updateChildren)
{
    DCVComponent::setSelected(selected, updateChildren);

    const TKReceptorMap *receptors = d_owner->getReceptors();
    if (updateChildren)
    {
        TKReceptorMap::const_iterator it = receptors->begin();
        while( it != receptors->end())
        {
            ((DCReceptor*)(*it).second)->getVComponent()->setSelected(selected, true);
            ++it;
        }
        d_owner->getAxon()->getVComponent()->setSelected(selected, updateChildren);
    }
    d_shouldUpdateShape = true;
}

void DCVCCell::setVisible(DCVVisibility visibleSelf, DCVVisibility visibleChildren)
{
    DCVComponent::setVisible(visibleSelf, visibleChildren);

    const TKReceptorMap *receptors = d_owner->getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        ((DCReceptor*)(*it).second)->getVComponent()->setVisible(visibleChildren,visibleChildren);
        ++it;
    }
    d_owner->getAxon()->getVComponent()->setVisible(visibleSelf,visibleChildren);
}

void DCVCCell::setSelectable(bool selectable)
{
    DCVComponent::setSelectable(selectable);

    const TKReceptorMap *receptors = d_owner->getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        ((DCReceptor*)(*it).second)->getVComponent()->setSelectable(selectable);
        ++it;
    }
    d_owner->getAxon()->getVComponent()->setSelectable(selectable);
}

void DCVCCell::renderOwnShape(bool isAnimationInterval, bool renderAsWireframe)
{
    if (renderAsWireframe)
    {
        //render bounding box
        float hwidth = d_owner->getViewSize()/2.0;
        float hheight = d_owner->getViewHeight()/2.0;
        glLineWidth(1);
        glColor4f(1.0, 1.0, 1.0,0.5);
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        glEnable( GL_LINE_SMOOTH );
        glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );
        glBegin(GL_LINE_LOOP);
        glVertex3f(hwidth, hheight, hwidth);
        glVertex3f(-hwidth, hheight, hwidth);
        glVertex3f(-hwidth, hheight, -hwidth);
        glVertex3f(hwidth, hheight, -hwidth);
        glEnd();
        glBegin(GL_LINE_LOOP);
        glVertex3f(hwidth, -hheight, hwidth);
        glVertex3f(-hwidth, -hheight, hwidth);
        glVertex3f(-hwidth, -hheight, -hwidth);
        glVertex3f(hwidth, -hheight, -hwidth);
        glEnd();
        glBegin(GL_LINES);
        glVertex3f(hwidth, -hheight, hwidth);
        glVertex3f(hwidth, hheight, hwidth);
        glVertex3f(-hwidth, -hheight, hwidth);
        glVertex3f(-hwidth, hheight, hwidth);
        glVertex3f(-hwidth, -hheight, -hwidth);
        glVertex3f(-hwidth, hheight, -hwidth);
        glVertex3f(hwidth, -hheight, -hwidth);
        glVertex3f(hwidth, hheight, -hwidth);
        glEnd();
        glDisable(GL_BLEND);
    }
    else
    {
        if (d_shouldUpdateShape)
        {
            d_shape->setSize(d_owner->getViewSize());
            d_shape->setHeight(d_owner->getViewHeight());

            QString label1;
            if (d_owner->getCellCode() && d_owner->getCellCode()!= d_emptyCellCode)
            {
                label1 = QString::fromStdString(d_owner->getCellCode()->getFQNName());
                label1 = label1.mid(label1.lastIndexOf("#") + 1);
            }
            else
            {
                label1 = "type : " + d_owner->getType();
            }

            QString label2 = QString::fromStdString(d_owner->getName());
            QString label3 = "";
            DCContainer *container = dynamic_cast<DCContainer*> (d_owner->getContainer());
            if (container)
            {
                if (container->getIsPluginType(d_owner->getType()))
                {
                    DNPluginInfo info = DNPluginInfo::create(d_owner->getName());
                    if (info.isValid)
                    {
                        label2 = QString::fromStdString(info.pluginValueName);
                        label3 = " (" + QString::fromStdString(info.pluginName) +") ";
                    }
                }
            }
            d_shape->setTextLabel(label1, label2, label3);

            d_shouldUpdateShape = false;
        }
        d_shape->draw();

        if (getIsSelected())
        {
            d_selectionRectRenderer->setRectSize(d_owner->getViewSize());
            glPushMatrix();
            glTranslatef(0,d_owner->getViewHeight()/2,0);
            d_selectionRectRenderer->draw(isAnimationInterval && (d_isDragging && !d_isResizingDrag));
            glPopMatrix();
            DCScene::setShouldContinueAnimation(true);
        }
    }
}

void DCVCCell::updateShape()
{
    d_shouldUpdateShape = true;
}

bool DCVCCell::startDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    d_dragOffsetX = getPageX() - x;
    d_dragOffsetY = getPageY() - z;

    d_isResizingDrag = isResizingDrag;
    d_isDragging = true;

    if (isResizingDrag)
    {
        d_draggingOriginalSize = d_owner->getViewSize();
        d_draggingOriginalAxonLength = d_owner->getAxon()->getViewLength();
    }

    return true;
}

bool DCVCCell::dragging(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z;

    if (isResizingDrag && d_isResizingDrag)
    {
        float dx = d_dragOffsetX - getPageX() + x;
        float dy = d_dragOffsetY - getPageY() + z;
        float length = sqrt(dx * dx + dy * dy);
        float dlength = dx < 0 && dy < 0 ? - length : length;
        float newSize = d_draggingOriginalSize + dlength;
        d_owner->setViewSize(newSize);
        float newAxonLength = d_draggingOriginalAxonLength + dlength/2;
        if (newAxonLength < d_owner->getViewSize()/2)
            newAxonLength = d_owner->getViewSize()/2;
        d_owner->getAxon()->setViewLength(newAxonLength);
    }
    else
    {
        setPageX(d_dragOffsetX + x);
        setPageY(d_dragOffsetY + z);
    }

    d_isDragging = true;

    return true;
}

bool DCVCCell::endDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    d_isDragging = false;
    return true;
}

bool DCVCCell::isResizingArea(float x, float y, float z) const
{
    (void)x; (void)y; (void)z;

    d_selectionRectRenderer->setRectSize(d_owner->getViewSize());
    return d_selectionRectRenderer->isResizingArea(x, z);
}

void DCVCCell::saveAttributesToXML(QDomDocument *document, QDomElement* element) const
{
    element->setAttribute("x", getPageX());
    element->setAttribute("y", getPageY());
    element->setAttribute("size", d_owner->getViewSize());
    element->setAttribute("height", d_owner->getViewHeight());

    const TKReceptorMap *receptors = d_owner->getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        QDomElement receptorNode = document->createElement("receptor");
        receptorNode.setAttribute("name", QString::fromStdString((*it).first));
        ((DCReceptor*)(*it).second)->getVComponent()->saveAttributesToXML(document, &receptorNode);
        element->appendChild(receptorNode);
        ++it;
    }

    QDomElement axonNode = document->createElement("axon");
    d_owner->getAxon()->getVComponent()->saveAttributesToXML(document, &axonNode);
    element->appendChild(axonNode);
}

void DCVCCell::loadAttributesFromXML(QDomElement element)
{
    if (element.hasAttribute("x"))
        setPageX(element.attribute("x", "0").toFloat());

    if (element.hasAttribute("y"))
        setPageY(element.attribute("y", "0").toFloat());

    if (element.hasAttribute("size"))
        d_owner->setViewSize(element.attribute("size", "1.0").toFloat());

    if (element.hasAttribute("height"))
        d_owner->setViewHeight(element.attribute("height", "1.0").toFloat());


    const TKReceptorMap *receptors = d_owner->getReceptors();
    QDomNodeList children = element.childNodes();
    for (int i = 0; i < children.size(); i++)
    {
        if (!children.at(i).isElement())
            continue;

        QDomElement e = children.at(i).toElement();
        QString tName = e.tagName();
        if (tName == "axon")
        {
            d_owner->getAxon()->getVComponent()->loadAttributesFromXML(e);
        }
        else if(tName == "receptor")
        {
            QString rName = e.attribute("name", "");
            TKReceptorMap::const_iterator it = receptors->find(rName.toStdString());
            if (it != receptors->end())
            {
                ((DCReceptor*)(*it).second)->getVComponent()->loadAttributesFromXML(e);
            }
        }
    }
}

