//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dctreeviewwidget.h"

#include "dccreator.h"
#include "dcscene.h"
#include "dcvcpage.h"
#include "dcvccell.h"
#include "dccontainer.h"
#include "command/dccommand.h"

#include "dialog/dcinputnewpagenamedialog.h"
#include "dialog/dcaddcellcodeclassdialog.h"
#include "dialog/dcaddcelldialog.h"

#include "uieditor/dcuieditor.h"
#include "propertyeditor/dcpropertyeditor.h"

#include <QList>
#include <QItemSelectionModel>
#include <QModelIndexList>

static DCTreeViewWidget *s_treeViewWidget = NULL;

static bool IsIndexUnderContainer(const QFileSystemModel *model, QModelIndex index, bool includesContainerDir = false)
{
    bool r = false;

    if (!includesContainerDir)
    {
        if (index.isValid())
            index = index.parent();
    }

    while (index.isValid())
    {
        if (model->fileName(index) == "Container"
                && index.parent().isValid()
                && index.parent() == model->index(model->rootPath()))
        {
            r = true;
        }
        index = index.parent();
    }
    return r;
}

class FileIconProvider : public QFileIconProvider
{
    QIcon pageIcon;
    QIcon folderIcon;
public:
    FileIconProvider() :
        pageIcon(":/pageIcon.png"),
        folderIcon(":/folderIcon.png")
    {
    }

    virtual ~FileIconProvider()
    {
    }

    virtual QIcon icon(const QFileInfo &info) const
    {
        if (info.completeSuffix() == "xhtml")
        {
            return pageIcon;
        }
        else if (info.isDir())
        {
//            return folderIcon;
        }

        return QFileIconProvider::icon(info);

    }

};

DCTreeViewModel::DCTreeViewModel(DCCreator *creator)
    : d_creator(creator)
{
    setIconProvider(new FileIconProvider);
}

DCTreeViewModel::~DCTreeViewModel()
{

}

bool DCTreeViewModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    bool r = false;
    if (role == Qt::EditRole)
    {
        QString oldFilePath = filePath(index);
        QString oldFileName = fileName(index);
        QString baseDirPath = oldFilePath.left(oldFilePath.length()-oldFileName.length());

        if (QFileInfo(oldFilePath).isDir() && IsIndexUnderContainer(this, index))
        {
            QString newFilePath = baseDirPath + value.toString();
            d_creator->doCommandRenameDirectory(this, oldFilePath, newFilePath);
            r = true;
        }
    }
    else
    {
        r = QFileSystemModel::setData(index, value, role);
    }
    return r;
}

DCTreeViewWidget::DCTreeViewWidget(QWidget *parent, DCCreator *creator) :
    QTreeView(parent), d_creator(creator), d_fileSystemModel(creator), d_scene(NULL), d_inSelectionChange(false), d_pendingSelectionPath("")
{
    setModel(&d_fileSystemModel);
    d_fileSystemModel.setReadOnly(false);
    connect(d_creator, SIGNAL(contentRootPathChanged(const void*, QString)), this, SLOT(contentRootPathChanged(const void*, QString)));
    connect(d_creator, SIGNAL(sceneChanged(const void*, DCScene*)), this, SLOT(sceneChanged(const void*, DCScene*)));
    connect(d_creator, SIGNAL(sceneSelectedPageChanged(const void*, const DCScene*)), this, SLOT(selectedPageChanged(const void*, const DCScene*)));
    connect(d_creator, SIGNAL(destroyed(QObject*)), this, SLOT(creatorDestroyed()));

    connect(&d_fileSystemModel, SIGNAL(directoryLoaded(QString)), this, SLOT(directoryLoaded(QString)));
    connect(&d_fileSystemModel, SIGNAL(rowsRemoved(const QModelIndex, int, int)), this, SLOT(rowsRemoved()));
    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(const QPoint&)),this, SLOT(doContextMenu(const QPoint&)));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand*)), this, SLOT(commandExecuted(const QUndoCommand*)));
    s_treeViewWidget = this;
}

DCTreeViewWidget::~DCTreeViewWidget()
{
    if (d_creator)
    {
        d_creator->disconnect(this);
    }
}

//static
bool  DCTreeViewWidget::addDirectory(const QString &path)
{
    bool r = false;

    if (s_treeViewWidget)
    {
        QModelIndex index = s_treeViewWidget->d_fileSystemModel.index(path);
        if (index.isValid())
        {
            QString newDirName = "NewFolder";
            int i = 0;
            while(QDir(path + "/" + newDirName).exists())
            {
                i++;
                newDirName = "";
                QTextStream(&newDirName) << "NewFolder(" << i << ")";
            }
            QModelIndex newDirIndex = s_treeViewWidget->d_fileSystemModel.mkdir(index, newDirName);
            if (newDirIndex.isValid())
            {
                r = true;
                s_treeViewWidget->setCurrentIndex(newDirIndex);
            }
        }
    }
    else
    {
        Q_ASSERT(0);
    }
    return r;
}

//static
bool DCTreeViewWidget::removeDirectory(const QString &path)
{
    bool r = false;

    if (s_treeViewWidget)
    {
        QDir  dir(path);

        QFileInfoList list = dir.entryInfoList(QDir::NoDot | QDir::NoDotDot | QDir::AllEntries);

        for (int i = 0; i < list.length(); i++)
        {
            QFileInfo info = list.at(i);
            if (info.isDir())
            {
                removeDirectory(info.absoluteFilePath());
            }
            else
            {
                QModelIndex index = s_treeViewWidget->d_fileSystemModel.index(info.absoluteFilePath());
                if (index.isValid())
                {
                    //hack for prevening auto selection happing by removing row.
                    s_treeViewWidget->d_inSelectionChange = true;
                    s_treeViewWidget->d_fileSystemModel.remove(index);
                }
            }
        }

        QModelIndex index = s_treeViewWidget->d_fileSystemModel.index(path);
        if (index.isValid())
        {
            //hack for prevening auto selection happing by removing row.
            s_treeViewWidget->d_inSelectionChange = true;
            r = s_treeViewWidget->d_fileSystemModel.rmdir(index);
        }
    }
    else
    {
        Q_ASSERT(0);
    }
    return r;
}

//static
bool DCTreeViewWidget::removeFile(const QString &path)
{
    bool r = false;

    if (s_treeViewWidget)
    {
        QModelIndex index = s_treeViewWidget->d_fileSystemModel.index(path);
        if (index.isValid() && s_treeViewWidget->d_fileSystemModel.fileInfo(index).isFile())
        {
            //hack for prevening auto selection happing by removing row.
            s_treeViewWidget->d_inSelectionChange = true;
            r = s_treeViewWidget->d_fileSystemModel.remove(index);
        }
    }
    else
    {
        Q_ASSERT(0);
    }
    return r;
}

//static
void DCTreeViewWidget::selectWhenFilePathAdded(const QString &path)
{
    if (s_treeViewWidget)
    {
        s_treeViewWidget->d_pendingSelectionPath = path;
    }
}


void DCTreeViewWidget::doContextMenu(const QPoint &pos)
{
    QPoint globalPos = mapToGlobal(pos);

    QFileSystemModel *fileModel = dynamic_cast<QFileSystemModel*>(model());

    QMenu menu;
    bool showMenu = false;

    QAction *addDictionaryAction = NULL;
    QAction *addPageAction = NULL;
    QAction *addCellAction = NULL;
    QAction *addCellCodeClassAction = NULL;
    QAction *renamePageAction = NULL;
    QAction *removeDirectoryAction = NULL;
    QAction *removePageAction = NULL;

    bool isDir = true;
    bool isPage = false;

    QModelIndex index;

    QModelIndexList indexes = selectedIndexes();
    if (indexes.length() == 0)
    {
        addDictionaryAction = menu.addAction(tr("Add folder"));
        index = fileModel->index(fileModel->rootPath() + "/Container");
        showMenu = true;
    }
    else if (indexes.length() > 0)
    {        
        index = indexes.at(0);
        isDir = d_fileSystemModel.isDir(index);
        isPage = !isDir && d_fileSystemModel.fileInfo(index).completeSuffix() == "xhtml";

        if (isDir)
        {
            addDictionaryAction = menu.addAction(tr("Add folder"));
            if (IsIndexUnderContainer(&d_fileSystemModel, index, true))
            {
                addPageAction = menu.addAction(tr("Add page..."));
            }
            if (fileModel->flags(index) & Qt::ItemIsEditable && !index.child(0,0).isValid())
            {
                removeDirectoryAction = menu.addAction(tr("Remove"));
            }
            showMenu = true;
        }
        else
        {
            if (isPage)
            {
                addCellAction = menu.addAction(tr("Add cell ..."));
                addCellCodeClassAction = menu.addAction(tr("Add cell code class ..."));
                menu.addSeparator();
                renamePageAction = menu.addAction(tr("Rename"));
                removePageAction = menu.addAction(tr("Remove..."));
                menu.addSeparator();
            }
            addDictionaryAction = menu.addAction(tr("Add folder"));
            showMenu = true;
        }
    }

    if (showMenu)
    {
        DCContainer *container = d_creator->getCurrentContainer();

        if (container)
        {
            QString containerBasedPath;

            if (index.isValid())
            {
                containerBasedPath = container->sysFilePathToContainerBasedPath(fileModel->filePath(index));
            }

            QAction* selectedItem = menu.exec(globalPos);
            if (selectedItem)
            {
                if (selectedItem == addDictionaryAction)
                {
                    if (index.isValid())
                        d_creator->doCommandAddDirectory(this, fileModel->filePath(index));
                }
                else if (selectedItem == removeDirectoryAction)
                {
                    if (index.isValid())
                        d_creator->doCommandRemoveDirectory(this, fileModel->filePath(index));
                }
                else if (selectedItem == addPageAction)
                {
                    if (index.isValid())
                    {
                        if (containerBasedPath.length() >= 1)
                        {
                            DCInputNewPageNameDialog dialog(fileModel->filePath(index), tr("Add new page"));
                            if (dialog.exec())
                            {
                                QString pageName = dialog.getName();
                                if (pageName.length() > 0)
                                {
                                    QString pathName = containerBasedPath;
                                    if (containerBasedPath.length()>1)
                                        pathName.append("/");
                                    pathName.append(pageName);
                                    d_creator->doCommandAddPage(this, pathName);
                                }
                            }
                        }
                    }
                }
                else if (selectedItem == renamePageAction)
                {
                    if (index.isValid())
                    {
                        if (containerBasedPath.length() >= 1)
                        {
                            QFileInfo fileInfo = fileModel->filePath(index);
                            QString oldName = fileInfo.fileName().left(fileInfo.fileName().length() - fileInfo.completeSuffix().length() - 1);
                            DCInputNewPageNameDialog dialog(fileInfo.absolutePath(), tr("Rename"), oldName);
                            if (dialog.exec())
                            {
                                QString pageName = dialog.getName();
                                if (pageName.length() > 0)
                                {
                                    QString pathName = container->sysFilePathToContainerBasedPath(fileInfo.absolutePath() + "/" + pageName);
                                    d_creator->doCommandMovePage(this, containerBasedPath, pathName);
                                }
                            }
                        }
                    }
                }
                else if (selectedItem == removePageAction)
                {
                    if (index.isValid())
                    {
                        bool doExecute = true;
                        DCVCPage *page = d_scene->getPage(containerBasedPath.toStdString());
                        if (page->getCells()->length() > 0 || page->getCellCodeClasses()->length() > 0)
                        {
                            QMessageBox::StandardButton button = QMessageBox::warning(this,
                                                 tr("Remove file"),
                                                 tr("This Page has cells / cellcodes. Those cells / cell codes will be removed"),
                                                 QMessageBox::Cancel | QMessageBox::Ok, QMessageBox::Cancel);

                            doExecute = (button == QMessageBox::Ok);
                        }
                        if (doExecute)
                        {
                            d_creator->doCommandRemovePage(this, page);
                        }
                    }
                }
                else if (selectedItem == addCellAction )
                {
                    if (index.isValid())
                    {
                        DCAddCellDialog dialog(container, d_creator, containerBasedPath,0,0);
                        dialog.exec();
                    }
                }
                else  if (selectedItem == addCellCodeClassAction)
                {
                    if (index.isValid())
                    {
                        DCAddCellCodeClassDialog dialog(d_creator, containerBasedPath);
                        dialog.exec();
                    }
                }

            }
        }
    }
}


void DCTreeViewWidget::showEvent(QShowEvent *event)
{
    setColumnHidden(1,true);
    setColumnHidden(2,true);
    setColumnHidden(3,true);

    QTreeView::showEvent(event);
}

void DCTreeViewWidget::mouseDoubleClickEvent(QMouseEvent *event)
{
    QModelIndex index = currentIndex();
    if (index.isValid())
    {
        bool isDir = d_fileSystemModel.isDir(index);
        bool isPage = false;
        bool isHtml = false;
        bool isProprty = false;

        if (!isDir)
        {
            if (d_fileSystemModel.fileInfo(index).completeSuffix() == "xhtml")
            {
                isPage = true;
                isHtml = true;
            }
            else if (d_fileSystemModel.fileInfo(index).completeSuffix() == "htm")
            {
                isHtml = true;
            }
            else if (d_fileSystemModel.fileInfo(index).completeSuffix() == "html")
            {
                isHtml = true;
            }
            else if (d_fileSystemModel.fileName(index) == "property.xml"
                     && index.parent() == d_fileSystemModel.index(d_fileSystemModel.rootPath()))
            {
                isProprty = true;
            }
        }

        if (!IsIndexUnderContainer(&d_fileSystemModel, index))
        {
            if (isHtml)
            {
                DCUIEditor *editor = DCUIEditor::getEditor();
                editor->startEditing(d_fileSystemModel.filePath(index));
            }
            else if (isProprty)
            {
                DCPropertyEditor *editor = DCPropertyEditor::getEditor();
                editor->startEditing(d_fileSystemModel.filePath(index));
            }
            event->accept();
            return;
        }

        if (isDir)
        {
            QTreeView::mouseDoubleClickEvent(event);
        }
        else if (isPage)
        {
            d_creator->changePersMode(this, DCCreator::DC_PERSMODE_PAGEEDIT);
            event->accept();
        }
    }
    else
    {
        QTreeView::mouseDoubleClickEvent(event);
    }
}

void DCTreeViewWidget::rowsInserted(const QModelIndex &parent, int start, int end)
{
    (void)parent; (void)start; (void)end;

    if (d_pendingSelectionPath.length() > 0)
    {
        QModelIndex index = d_fileSystemModel.index(d_pendingSelectionPath);
        if (index.isValid())
        {
            clearSelection();
            d_pendingSelectionPath = "";
            QItemSelectionModel *selection = selectionModel();
            selection->select(index, QItemSelectionModel::Select);
        }
    }
}

void DCTreeViewWidget::rowsRemoved()
{
    //hack for prevening auto selection happing by removing row.
    d_inSelectionChange = false;
}

void DCTreeViewWidget::creatorDestroyed()
{
    d_creator = NULL;
}

void DCTreeViewWidget::contentRootPathChanged(const void *requester, QString rootPath)
{
    (void)requester;

    QModelIndex idx = d_fileSystemModel.setRootPath(rootPath);
    QStringList nameFilters;
    nameFilters << "*.xhtml";
    nameFilters << "property.xml";
    nameFilters << "data.db";
    nameFilters << "*.html";
    nameFilters << "*.htm";
    d_fileSystemModel.setNameFilters(nameFilters);
    d_fileSystemModel.setNameFilterDisables(false);
    setRootIndex(idx);

    sortByColumn(0, Qt::DescendingOrder);
}

void DCTreeViewWidget::sceneChanged(const void *requester, DCScene*scene)
{
    (void)requester;

    d_scene = scene;
}

void DCTreeViewWidget::selectedPageChanged(const void *requester, const DCScene*scene)
{
    (void)requester;

    if (d_scene != scene)
        return;

    if (requester == this)
    {
        return;
    }

    if (d_scene)
    {
        d_inSelectionChange = true;
        clearSelection();
        QList<DCVCPage*> list = d_scene->getSelectedPages();
        QItemSelectionModel *selection = selectionModel();
        QString rootPath = d_fileSystemModel.rootPath();
        for (int i = 0; i < list.length(); i++)
        {
            DCVCPage *page = list.at(i);
            QString path = rootPath + QString("/Container") + page->getLocationPath();
            QModelIndex idx = d_fileSystemModel.index(path);
            selection->select(idx, QItemSelectionModel::Select);
        }
        d_inSelectionChange = false;
    }
}

void DCTreeViewWidget::selectionChanged(const QItemSelection &selected, const QItemSelection &deselected)
{
    QTreeView::selectionChanged(selected, deselected);

    if (d_inSelectionChange)
    {
        return;
    }

    if (d_creator)
    {
        QModelIndexList selectedList = selected.indexes();
        QModelIndexList deselectedList = deselected.indexes();
        int stripLen = d_fileSystemModel.rootPath().length();

        for (int i = 0; i < selectedList.length(); i++)
        {
            QString fpath = d_fileSystemModel.filePath(selectedList.at(i));
            if (fpath.length() > stripLen)
            {
                 fpath = fpath.mid(stripLen);
                 if (fpath.indexOf("/Container") == 0)
                 {
                     fpath = fpath.mid(10);
                     d_creator->selectPage(this, fpath, true);
                 }
            }
        }

        for (int i = 0; i < deselectedList.length(); i++)
        {
            QString fpath = d_fileSystemModel.filePath(deselectedList.at(i));
            if (fpath.length() > stripLen)
            {
                 fpath = fpath.mid(stripLen);
                 if (fpath.indexOf("/Container") == 0)
                 {
                     fpath = fpath.mid(10);
                     d_creator->unselectPage(this, fpath);
                 }
            }
        }

        if (d_creator->getPersMode() == DCCreator::DC_PERSMODE_PAGEEDIT)
        {
            d_creator->changePersMode(this, DCCreator::DC_PERSMODE_NAVIGATION);
        }
    }
}

void DCTreeViewWidget::commandExecuted(const QUndoCommand *command)
{
    const DCCommand *dccommand = dynamic_cast<const DCCommand*>(command);
    if (dccommand && dccommand->getRequester() == this)
    {
        sortByColumn(0, Qt::DescendingOrder);
    }
}

void DCTreeViewWidget::directoryLoaded(const QString &directory)
{
    (void)directory;

    expandAll();
}
