//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Oct-19, 2012.
//
#include "dcrenamecelldialog.h"

#include "dccontainer.h"
#include "dccreator.h"
#include "dccell.h"
#include "dcscene.h"
#include "dcvcpage.h"
#include "command/dccommand.h"
#include "utils/dcqtitemmodel.h"
#include "dcsinglecolumntableview.h"
#include "DNUtils.h"
#include "dnplugininfo.h"

#include <QLineEdit>
#include <QTableView>
#include <QHeaderView>

  DCRenameCellDialog::DCRenameCellDialog(DCContainer *container, DCCreator *creator, DCCell *cell, QWidget *parent) :
      QDialog(parent), d_container(container), d_creator(creator), d_cell(cell), d_pluginLibraryNameComboBox(NULL)
{
    setWindowTitle(tr("Rename cell"));

    d_textField = new QLineEdit;
    if (d_container->getIsPluginType(cell->getType()))
    {
        d_pluginLibraryNameComboBox = new QComboBox;
        d_pluginLibraryNameComboBox->setEditable(true);
        d_pluginLibraryNameComboBox->addItems(d_container->getAvailablePluginLibraries());

        DNPluginInfo info = DNPluginInfo::create(cell->getName());
        d_textField->setText(QString::fromStdString(info.pluginValueName));
        d_pluginLibraryNameComboBox->setEditText(QString::fromStdString(info.pluginName));
    }
    else
    {
        d_textField->setText(QString::fromStdString(cell->getName()));
    }

    d_table = new DCSingleColumnTableView();
    QStringList headers;
    headers << "page";
    d_tableModel = new DCQtItemModel(headers);
    d_tableModel->setReadOnly(0,true);

    DCScene *scene = container->getScene();
    const QMap<QString,DCVCPage*> *pages = scene->getPages();
    QMapIterator<QString, DCVCPage*> i(*pages);
    int row = 0;
    int selection = -1;
    QString path = QString::fromStdString(cell->getLocation());
    while (i.hasNext())
    {
        i.next();
        QString item = i.key();
        d_tableModel->insertString(item);
        if (item == path) selection = row;
        row++;
    }
    d_table->setModel(d_tableModel);

    setLayout(new QVBoxLayout);
    d_table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

    d_okButton = new QPushButton(tr("OK"));
    d_cancelButton = new QPushButton(tr("Cancel"));
    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addItem(new QSpacerItem(10,10, QSizePolicy::Expanding));
    buttonLayout->addWidget(d_cancelButton);
    buttonLayout->addWidget(d_okButton);
    d_okButton->setEnabled(false);
    d_cancelButton->setAutoDefault(true);
    d_cancelButton->setDefault(true);

    QGridLayout *glayout = new QGridLayout;
    glayout->addWidget(new QLabel(tr("name")),0,0);
    glayout->addWidget(d_textField,0,1);
    if (d_pluginLibraryNameComboBox)
    {
        glayout->addWidget(new QLabel(tr("plugin")),1,0);
        glayout->addWidget(d_pluginLibraryNameComboBox,1,1);
    }
    ((QVBoxLayout*)layout())->addLayout(glayout);
    layout()->addWidget(d_table);
    d_statusText = new QLabel;
    d_message = new QLabel;
    d_message->setStyleSheet("color : red;");
    layout()->addWidget(d_statusText);
    layout()->addWidget(d_message);
    ((QVBoxLayout*)layout())->addLayout(buttonLayout);

    connect(d_textField, SIGNAL(textChanged(QString)), this, SLOT(textInputChanged(QString)));
    connect(d_okButton, SIGNAL(clicked()), this, SLOT(okButtonClicked()));
    connect(d_cancelButton, SIGNAL(clicked()), this, SLOT(cancelButtonClicked()));
    connect(d_table,SIGNAL(selectionChangedSignal(QItemSelection,QItemSelection)), this, SLOT(listSelectionChanged(QItemSelection,QItemSelection)));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand*)), this, SLOT(commandExecuted(const QUndoCommand*)));
    connect(d_pluginLibraryNameComboBox, SIGNAL(editTextChanged(QString)),this, SLOT(pluginNameChanged(QString)));

    if (selection >= 0)
    {
        QModelIndex index = d_tableModel->index(selection, 0);
        if (index.isValid())
            d_table->setCurrentIndex(index);
    }
}

DCRenameCellDialog::~DCRenameCellDialog()
{
    if (d_tableModel)
        d_tableModel->deleteLater();
}

bool DCRenameCellDialog::checkInput()
{
    bool valid = true;

    d_message->setText("");
    d_statusText->setText("");
    if (d_selectedPagePath.length() == 0)
    {
        d_message->setText(tr("Select a page to place the cell"));
        valid = false;
    }
    else if (d_textField->text().length() > 0)
    {
        QString name;
        if (d_container->getIsPluginType(d_cell->getType()))
        {
            if (d_pluginLibraryNameComboBox->currentText().length() == 0)
            {
                d_message->setText(tr("Input plugin name"));
                valid = false;
            }
            else
            {
                name = d_container->createPluginCellName(d_textField->text(), d_cell->getType(),d_pluginLibraryNameComboBox->currentText());
            }
        }
        else
        {
            name = d_textField->text();
        }

        std::string path = getFQNString(d_selectedPagePath.toLocal8Bit().data(), name.toLocal8Bit().data());
        TKCell *cell = d_container->getCell(path);
        if (cell)
        {
            if (cell != d_cell)
            {
                d_message->setText(tr("The cell name exists"));
                valid = false;
            }
        }
        else
        {
            d_statusText->setText(QString::fromStdString(path));
        }
    }
    else
    {
        d_message->setText(tr("Input cell name"));
        valid = false;
    }

    if (valid)
    {
        d_okButton->setEnabled(true);
        d_okButton->setDefault(true);
    }
    else
    {
        d_okButton->setEnabled(false);
        d_cancelButton->setDefault(true);
    }

    return valid;
}

void DCRenameCellDialog::textInputChanged(const QString &text)
{
    (void)text;

    checkInput();
}

void DCRenameCellDialog::okButtonClicked()
{
    if (checkInput())
    {
        QString newName;
        if (d_container->getIsPluginType(d_cell->getType()))
        {
            newName = d_container->createPluginCellName(d_textField->text(), d_cell->getType(), d_pluginLibraryNameComboBox->currentText());
        }
        else
        {
            newName = d_textField->text();
        }

        d_creator->doCommandRenameCell(this, d_cell, d_selectedPagePath, newName);
    }
}

void DCRenameCellDialog::cancelButtonClicked()
{
    done(false);
}


void DCRenameCellDialog::listSelectionChanged(const QItemSelection &selected, const QItemSelection &deselected)
{
    (void)deselected;

    if (selected.count()>0)
    {
        d_selectedPagePath = d_tableModel->data(selected.at(0).indexes().at(0), Qt::DisplayRole).toString();
    }
    else
    {
        d_selectedPagePath = "";
    }
    checkInput();
}

void DCRenameCellDialog::pluginNameChanged(const QString &text)
{
    (void)text;
    checkInput();
}

void DCRenameCellDialog::commandExecuted(const QUndoCommand *executedCommand)
{
    const DCCommand *command = dynamic_cast<const DCCommand*>(executedCommand);
    if (command && command->getRequester() == this)
    {
        done(true);
    }
}
