//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Nov-3, 2012.
//
#include "dccellscriptseditorpagewidget.h"

#include <QVBoxLayout>
#include <QLabel>

#include "dccontainer.h"
#include "dccodeeditor.h"
#include "dccell.h"
#include "dccellcode.h"
#include "mainwindow.h"
#include "dialog/dccodeeditorexternaleditorsettingdialog.h"

static const char* S_EXTERNAL_EDITOR_BUTTON_LABEL1 = "Use external editor...";
static const char* S_EXTERNAL_EDITOR_BUTTON_LABEL2 = "Editor script";

DCCellScriptsEditorPageWidget::DCCellScriptsEditorPageWidget(DCCell *targetCell, QWidget *parent) :
    QWidget(parent), d_cell(targetCell), d_modified(false),
    d_customScriptExternalMode(false), d_cellCodeScriptExternalMode(false)
{
    if (d_cell->getIsCellCodeAssgined())
    {
        d_cellCodeOrg = d_cell->getCellCode();
    }
    else
    {
        d_cellCodeOrg = NULL;
    }

    d_splitter = new QSplitter(Qt::Vertical, this);

    d_customScriptEditor = new DCCodeEditor(this);
    d_cellCodeScriptEditor = new DCCodeEditor(this);

    QVBoxLayout *customScriptEditorBaseLayout = new QVBoxLayout;
    QHBoxLayout *customScriptEditorTopLayout = new QHBoxLayout;
    customScriptEditorTopLayout->addWidget(new QLabel(tr("custom script")));
    d_customScriptEditExternalButton = new QPushButton(tr(S_EXTERNAL_EDITOR_BUTTON_LABEL1));
    customScriptEditorTopLayout->addStretch();
    customScriptEditorTopLayout->addWidget(d_customScriptEditExternalButton);
    customScriptEditorBaseLayout->addLayout(customScriptEditorTopLayout);
    customScriptEditorBaseLayout->addWidget(d_customScriptEditor);
    QWidget *customScriptWidget = new QWidget;
    customScriptWidget->setLayout(customScriptEditorBaseLayout);

    QVBoxLayout *cellCodeScriptEditorBaseLayout = new QVBoxLayout;
    QHBoxLayout *cellCodeScriptEditorTopLayout = new QHBoxLayout;
    cellCodeScriptEditorTopLayout->addWidget(new QLabel(tr("cell code script")));
    d_cellCodeScriptEditExternalButton = new QPushButton(tr(S_EXTERNAL_EDITOR_BUTTON_LABEL1));
    cellCodeScriptEditorTopLayout->addStretch();
    cellCodeScriptEditorTopLayout->addWidget(d_cellCodeScriptEditExternalButton);
    cellCodeScriptEditorBaseLayout->addLayout(cellCodeScriptEditorTopLayout);
    cellCodeScriptEditorBaseLayout->addWidget(d_cellCodeScriptEditor);
    QWidget *cellCodeScriptWidget = new QWidget;
    cellCodeScriptWidget->setLayout(cellCodeScriptEditorBaseLayout);

    d_splitter->addWidget(customScriptWidget);
    d_splitter->addWidget(cellCodeScriptWidget);

    QVBoxLayout *rootLayout = new QVBoxLayout;
    rootLayout->setMargin(0);
    rootLayout->addWidget(d_splitter);

    setLayout(rootLayout);

    d_customScriptFolder.attach(d_cell);
    d_cellCodeScriptFolder.attach(d_cellCodeOrg);

    connect(targetCell, SIGNAL(destroyed()), this, SLOT(cellDestroyed()));
    connect(targetCell, SIGNAL(cellCodeChanged()), this, SLOT(assignedCellCodeChanged()));
    connect(&d_customScriptFolder, SIGNAL(scriptChanged()), this, SLOT(folderCustomScriptChanged()));
    connect(&d_cellCodeScriptFolder, SIGNAL(scriptChanged()), this, SLOT(folderCellCodeScriptChanged()));
    connect(&d_customScriptFolder, SIGNAL(modificationStatusChanged(bool)), this, SLOT(folderCustomScriptModifiedStatusChanged(bool)));
    connect(&d_cellCodeScriptFolder, SIGNAL(modificationStatusChanged(bool)), this, SLOT(folderCellCodeScriptModifiedStatusChanged(bool)));
    connect(&d_customScriptFolder, SIGNAL(fileModifiedByExternalEditor(qint64)), this, SLOT(folderCustomScriptFileModifiedByExternalEditor(qint64)));
    connect(&d_cellCodeScriptFolder, SIGNAL(fileModifiedByExternalEditor(qint64)), this, SLOT(folderCellCodeScriptFileModifiedByExternalEditor(qint64)));
    connect(d_customScriptEditExternalButton, SIGNAL(clicked()), this, SLOT(useExternalEditorForCustomScriptPressed()));
    connect(d_cellCodeScriptEditExternalButton, SIGNAL(clicked()), this, SLOT(useExternalEditorForCellCodeScriptPressed()));

    loadScripts();
}

bool DCCellScriptsEditorPageWidget::loadScripts()
{
    d_customScriptEditor->disconnect(this);
    d_cellCodeScriptEditor->disconnect(this);

    d_customScriptEditor->setPlainText(d_customScriptFolder.getCurrentScript());
    d_customScriptEditor->setReadOnly(d_customScriptExternalMode);
    if (d_cell->getIsCellCodeAssgined())
    {
        d_cellCodeScriptEditor->setPlainText(d_cellCodeScriptFolder.getCurrentScript());
        d_cellCodeScriptEditor->setReadOnly(d_cellCodeScriptExternalMode);
        d_cellCodeScriptEditExternalButton->setEnabled(true);
        d_cellCodeScriptEditor->document()->setModified(false);
        d_splitter->widget(1)->show();
    }
    else
    {
        d_cellCodeScriptEditor->setPlainText("");
        d_cellCodeScriptEditor->setReadOnly(true);
        d_cellCodeScriptEditExternalButton->setEnabled(false);
        d_splitter->widget(1)->hide();
    }

    updateModifiedStatus();

    connect(d_customScriptEditor, SIGNAL(textChanged()), this, SLOT(editorCustomScriptChanged()));
    connect(d_cellCodeScriptEditor, SIGNAL(textChanged()), this, SLOT(editorCellCodeScriptChanged()));

    return true;
}

DCCellScriptsEditorPageWidget::~DCCellScriptsEditorPageWidget()
{
    if (d_cell)
    {
        disconnect(d_cell, 0, this, 0);
    }
    this->disconnect();
}

//slot
void DCCellScriptsEditorPageWidget::cellDestroyed()
{
    DCCell *cell = d_cell;
    d_cell = NULL;
    emit editingCellDestroyed(cell);
}

//slot
void DCCellScriptsEditorPageWidget::assignedCellCodeChanged()
{
    d_cellCodeScriptEditor->disconnect(this);
    d_cellCodeScriptFolder.deattach();

    if (d_cell->getIsCellCodeAssgined())
    {
        DCCellCode *cellCode = d_cell->getCellCode();
        d_cellCodeScriptFolder.attach(cellCode);
        d_cellCodeScriptEditor->setPlainText(d_cellCodeScriptFolder.getCurrentScript());
        d_cellCodeScriptEditor->setReadOnly(false);
        d_cellCodeScriptEditExternalButton->setEnabled(true);
        d_splitter->widget(1)->show();
        d_splitter->setStretchFactor(0,1);
        d_splitter->setStretchFactor(1,4);
        d_cellCodeScriptEditor->setFocus();
    }
    else
    {
        d_cellCodeScriptEditor->setPlainText("");
        d_cellCodeScriptEditor->setReadOnly(true);
        d_cellCodeScriptEditExternalButton->setEnabled(false);
        d_splitter->widget(1)->hide();
    }

    updateModifiedStatus();
    connect(d_cellCodeScriptEditor, SIGNAL(textChanged()), this, SLOT(editorCellCodeScriptChanged()));
}

//slot
void DCCellScriptsEditorPageWidget::editorCustomScriptChanged()
{
    d_customScriptFolder.setScript(d_customScriptEditor->toPlainText());
}

//slot
void DCCellScriptsEditorPageWidget::editorCellCodeScriptChanged()
{
    d_cellCodeScriptFolder.setScript(d_cellCodeScriptEditor->toPlainText());
}

//slot
void DCCellScriptsEditorPageWidget::folderCustomScriptChanged()
{
    d_customScriptEditor->disconnect(this);
    d_customScriptEditor->setPlainText(d_customScriptFolder.getCurrentScript());
    connect(d_customScriptEditor, SIGNAL(textChanged()), this, SLOT(editorCustomScriptChanged()));
}

//slot
void DCCellScriptsEditorPageWidget::folderCellCodeScriptChanged()
{
    d_cellCodeScriptEditor->disconnect(this);
    d_cellCodeScriptEditor->setPlainText(d_cellCodeScriptFolder.getCurrentScript());
    connect(d_cellCodeScriptEditor, SIGNAL(textChanged()), this, SLOT(editorCellCodeScriptChanged()));
}

//slot
void DCCellScriptsEditorPageWidget::folderCustomScriptModifiedStatusChanged(bool modified)
{
    (void)modified;
    updateModifiedStatus();
}

//slot
void DCCellScriptsEditorPageWidget::folderCellCodeScriptModifiedStatusChanged(bool modified)
{
    (void)modified;
    updateModifiedStatus();
}

//slot
void DCCellScriptsEditorPageWidget::folderCustomScriptFileModifiedByExternalEditor(qint64 notifiedTime)
{
    emit customScriptModifiedByExternalEditor(this,notifiedTime);
}

//slot
void DCCellScriptsEditorPageWidget::folderCellCodeScriptFileModifiedByExternalEditor(qint64 notifiedTime)
{
    emit cellCodeScriptModifiedByExternalEditor(this,notifiedTime);
}

//slot
void DCCellScriptsEditorPageWidget::useExternalEditorForCustomScriptPressed()
{
    if (d_customScriptExternalMode)
    {
        d_customScriptExternalMode = false;
        d_customScriptEditor->setReadOnly(false);
        d_customScriptEditExternalButton->setText(tr(S_EXTERNAL_EDITOR_BUTTON_LABEL1));
    }
    else
    {
        if (MainWindow::openExternalEditorFor(d_cell->getWorkFilePathForCustomScript()))
        {
            saveCustomScriptOnlyToFile(true);
            d_customScriptExternalMode = true;
            d_customScriptEditor->setReadOnly(true);
            d_customScriptEditExternalButton->setText(tr(S_EXTERNAL_EDITOR_BUTTON_LABEL2));
        }
    }

}

//slot
void DCCellScriptsEditorPageWidget::useExternalEditorForCellCodeScriptPressed()
{
    if (!d_cell->getIsCellCodeAssgined())
        return;

    if (d_cellCodeScriptExternalMode)
    {
        d_cellCodeScriptExternalMode = false;
        d_cellCodeScriptEditor->setReadOnly(false);
        d_cellCodeScriptEditExternalButton->setText(tr(S_EXTERNAL_EDITOR_BUTTON_LABEL1));
    }
    else
    {
        DCCellCode *cellCode = d_cell->getCellCode();
        if (MainWindow::openExternalEditorFor(cellCode->getWorkFilePathForCellCodeScript()))
        {
            saveCellCodeScriptOnlyToFile(true);
            d_cellCodeScriptExternalMode = true;
            d_cellCodeScriptEditor->setReadOnly(true);
            d_cellCodeScriptEditExternalButton->setText(tr(S_EXTERNAL_EDITOR_BUTTON_LABEL2));
        }
    }

}

void DCCellScriptsEditorPageWidget::updateModifiedStatus()
{
    bool modified = d_customScriptFolder.getIsModified();
    if (!modified)
        modified = d_cellCodeScriptFolder.getIsModified();

    if (modified != d_modified)
    {
        d_modified = modified;
        emit cellScriptsModifiedStatusChanged(this, modified);
    }
}

bool DCCellScriptsEditorPageWidget::getIsModified() const
{
    return d_modified;
}


void DCCellScriptsEditorPageWidget::setReadOnly(bool readOnly)
{
    d_customScriptEditor->setReadOnly(readOnly);
    d_cellCodeScriptEditor->setReadOnly(readOnly);
    d_customScriptEditExternalButton->setEnabled(!readOnly);
    d_cellCodeScriptEditExternalButton->setEnabled(!readOnly);
}


QSet<DCVCPage*> DCCellScriptsEditorPageWidget::saveScriptsToFile(bool saveModifiedOnly)
{
    bool save = false;
    bool result = true;

    QSet<DCVCPage*> modifiedPages;

    if (!saveModifiedOnly || d_customScriptFolder.getIsModified())
    {
        save = true;
        result = d_customScriptFolder.saveScript();
        if (result)
        {
            modifiedPages.insert(d_cell->getPageBelonging());
        }
    }

    if (result)
    {
        if (d_cell->getIsCellCodeAssgined())
        {
            if (!saveModifiedOnly || d_cellCodeScriptFolder.getIsModified())
            {
                result = d_cellCodeScriptFolder.saveScript();
                DCCellCode *cellCode = d_cell->getCellCode();
                if (result)
                {
                    modifiedPages.insert(cellCode->getPageBelonging());
                }
                d_cellCodeOrg = cellCode;
            }
        }
        else
        {
            d_cellCodeOrg = NULL;
        }
    }

    if (result)
    {
        d_modified = false;
        emit cellScriptsSaved(this);
    }
    else
    {
        QMessageBox::warning(this, tr("Failed to save"), tr("The change couldn't be saved"));
    }
    return modifiedPages;
}

bool DCCellScriptsEditorPageWidget::saveCustomScriptOnlyToFile(bool saveModifiedOnly)
{
    bool result = true;
    bool saved = false;
    if (!saveModifiedOnly || d_customScriptFolder.getIsModified())
    {
        saved = true;
        result = d_customScriptFolder.saveScript();
    }

    if (result)
    {
        if (saved)
        {
            d_modified = getIsModified();
            emit cellScriptsSaved(this);
        }
    }
    else
    {
        QMessageBox::warning(this, tr("Failed to save"), tr("The change couldn't be saved"));
    }
    return result;
}

bool DCCellScriptsEditorPageWidget::saveCellCodeScriptOnlyToFile(bool saveModifiedOnly)
{
    bool result = true;
    bool saved = false;
    DCCellCode *cellCode = NULL;
    if (d_cell->getIsCellCodeAssgined())
        cellCode = d_cell->getCellCode();

    if (cellCode && (!saveModifiedOnly || d_cellCodeScriptFolder.getIsModified()))
    {
        saved = true;
        result = d_cellCodeScriptFolder.saveScript();
    }

    if (result)
    {
        if (saved)
        {
            d_modified = getIsModified();
            emit cellScriptsSaved(this);
        }
    }
    else
    {
        QMessageBox::warning(this, tr("Failed to save"), tr("The change couldn't be saved"));
    }
    return result;
}

bool DCCellScriptsEditorPageWidget::getIsCustomScriptModifiedByExternalEditor() const
{
    return d_customScriptFolder.getIsFileModifiedByExternalEditor();
}

bool DCCellScriptsEditorPageWidget::getIsCellCodeModifiedByExternalEditor() const
{
    return d_cellCodeScriptFolder.getIsFileModifiedByExternalEditor();
}

qint64 DCCellScriptsEditorPageWidget::getCustomScriptLoadedTime() const
{
    return d_customScriptFolder.getLoadedTime();
}

qint64 DCCellScriptsEditorPageWidget::getCellCodeScriptLoadedTime() const
{
    return d_cellCodeScriptFolder.getLoadedTime();
}

void DCCellScriptsEditorPageWidget::focusCustomScript()
{
    d_splitter->setStretchFactor(0,1);
    d_splitter->setStretchFactor(1,4);
    d_customScriptEditor->setFocus();
}

void DCCellScriptsEditorPageWidget::focusCellCodeScript()
{
    d_splitter->setStretchFactor(0,1);
    d_splitter->setStretchFactor(1,4);
    d_cellCodeScriptEditor->setFocus();

}

void DCCellScriptsEditorPageWidget::reloadCustomScript()
{
    d_customScriptEditor->setPlainText(d_customScriptFolder.getCurrentScript(true));
    d_customScriptEditor->setReadOnly(d_customScriptExternalMode);
    updateModifiedStatus();
}

void DCCellScriptsEditorPageWidget::reloadCellCodeScript()
{
    if (d_cell->getIsCellCodeAssgined())
    {
        d_cellCodeScriptEditor->setPlainText(d_cellCodeScriptFolder.getCurrentScript(true));
        d_cellCodeScriptEditor->setReadOnly(d_cellCodeScriptExternalMode);
        d_cellCodeScriptEditor->document()->setModified(false);
        d_cellCodeScriptEditExternalButton->setEnabled(true);
        d_splitter->widget(1)->show();
    }
    else
    {
        d_cellCodeScriptEditor->setPlainText("");
        d_cellCodeScriptEditor->setReadOnly(true);
        d_cellCodeScriptEditExternalButton->setEnabled(false);
        d_splitter->widget(1)->hide();
    }

    updateModifiedStatus();
}

