//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Apr/15/2012.
//
#include "qtdnserialportimpl.h"

#include "ui_portinfodialog.h"
#include "DNServerSerialPort.h"
#include "DNThread.h"
#include "mainwindow.h"
#include "TKLog.h"
#include "DNUtils.h"
#include "DNSettings.h"

#include <QThread>
#include <QtCore/QVariant>

//
#include "DNEngine.h"

class SleeperThread : public QThread
{
public:
    static void msleep(unsigned long msecs)
    {
        QThread::msleep(msecs);
    }
};

DNSerialPortImpl * DNSerialPortImpl::create(DNServerSerialPort *server)
{
    return new QtDNSerialPortImpl(server);
}


QtDNSerialPortImpl::QtDNSerialPortImpl(DNServerSerialPort *server) :
    DNSerialPortImpl(server), mSerialCommunicationThread(NULL), mHandlerThread(NULL), mStopping(false),
    mResponseString(""),
    mPortName(""),
    mBaudRate(SerialPort::Rate9600),
    mDataBits(SerialPort::Data8),
    mParity(SerialPort::NoParity),
    mStopBits(SerialPort::OneStop),
    mFlowControl(SerialPort::NoFlowControl)
{
}

QtDNSerialPortImpl::~QtDNSerialPortImpl()
{
    stop();
    if (mSerialCommunicationThread)
    {
        delete mSerialCommunicationThread;
        mSerialCommunicationThread = NULL;
    }
    if (mHandlerThread)
    {
        delete mHandlerThread;
        mHandlerThread = NULL;
    }
}

bool QtDNSerialPortImpl::setup()
{
    if (!MainWindow::instance)
        return false;

    QtDNSerialPortSettingDialog dialog(MainWindow::instance);
    dialog.exec();

    if (dialog.isOK())
    {
        mPortName = dialog.getSelectedPortName();
        mBaudRate = (SerialPort::Rate) dialog.getSelectedRate();
        mDataBits = dialog.getSelectedDataBits();
        mParity = dialog.getSelectedParity();
        mStopBits = dialog.getSelectedStopBits();
        mFlowControl = dialog.getSelectedFlowControl();

        return true;
    }
    return false;

}

bool QtDNSerialPortImpl::isRunning()
{
    DNLocker lock(&mStartStopLock);
    return !mStopping;
}

void QtDNSerialPortImpl::start()
{
    stop();
    mStartStopLock.lock();
    if (mStopping)
    {
        mStopping = false;
        mSerialCommunicationThread = DNThread::createThread(QtDNSerialPortImpl::serialCommunicationThreadBody, this);
        mHandlerThread = DNThread::createThread(QtDNSerialPortImpl::requestHandelrThreadBody, this);

        mSerialCommunicationThread->start();
        mHandlerThread->start();
    }
    mStartStopLock.unlock();
}

void QtDNSerialPortImpl::stop()
{
    mStartStopLock.lock();
    if (!mStopping)
    {
        mStopping = true;
        if (mSerialCommunicationThread)
        {
            mSerialCommunicationThread->waitForExit(1000);
            delete mSerialCommunicationThread;
            mSerialCommunicationThread = NULL;
        }
        if (mHandlerThread)
        {
            mHandlerThread->waitForExit(1000);
            delete mHandlerThread;
            mHandlerThread = NULL;
        }
        mRequestQueue.clear();
    }
    mStartStopLock.unlock();
}

void QtDNSerialPortImpl::sendMessage(const char* message)
{
    mResponseStringLock.lock();
    mResponseString.append(message);
    mResponseStringLock.unlock();
    mSerialWriteSemaphore.release(1);
}

//static
void QtDNSerialPortImpl::serialCommunicationThreadBody(void *_impl)
{
    if (!_impl)
        return;

    QtDNSerialPortImpl *impl = (QtDNSerialPortImpl*)_impl;

    if (impl->mPortName.length()==0 || impl->mBaudRate == 0)
    {
        impl->mStartStopLock.lock();
        impl->mStopping = true;
        impl->mStartStopLock.unlock();
        return;
    }

    SerialPort *port = NULL;
    bool r = true;
    int retry = 6;
    do
    {
        port = new SerialPort(impl->mPortName);
        r = port != NULL;

        if (r)
        {
            r = port->open(QIODevice::ReadWrite | QIODevice::Unbuffered);
        }
        if (r)
        {
            if (!port->setRate(impl->mBaudRate))
            {
                r = false;
            }
        }
        if (r)
        {
            if (!port->setDataBits(impl->mDataBits))
            {
                r = false;
            }
        }
        if (r)
        {
            if (!port->setParity(impl->mParity))
            {
                r = false;
            }
        }
        if (r)
        {
            if (!port->setStopBits(impl->mStopBits))
            {
                r = false;
            }
        }
        if (r)
        {
            if (!port->setFlowControl(impl->mFlowControl))
            {
                r = false;
            }
        }

        if (!r)
        {
            SleeperThread::msleep(500);
            if (port)
            {
                port->close();
                delete port;
                port = NULL;
            }
            retry--;
            dnNotifyWarning("SerialPort initialization", "Failed to open the serial port. retrying...");
        }
    } while (!r && retry >= 0);


    if (!r)
    {
        if (port)
            delete port;

        impl->mStartStopLock.lock();
        impl->mStopping = true;
        impl->mStartStopLock.unlock();
        dnNotifyWarning("SerialPort initialization", "Failed to open the serial port. giving up!");
        return;
    }

    int waitTime = 1;

    char buffer[1024];
    int p = 0;
    while(!impl->mStopping)
    {
        if (port->bytesAvailable() > 0)
        {
            int len = port->read( (buffer + p), sizeof(buffer) - p);
            if (len != -1)
            {
                int i = p;
                while (i < p + len)
                {
                    if (buffer[i] == '\r' || buffer[i] == '\n' || i == sizeof(buffer)-1)
                    {
                        if (i>0)
                        {
                            buffer[i] = 0;
                            impl->mRequestQueueLock.lock();
                            impl->mRequestQueue.push_back(QString::fromLocal8Bit(buffer));
                            impl->mRequestQueueLock.unlock();
                            impl->mQueueSemaphore.release(1);
                        }
                        int j = i + 1;
                        while(j < p + len)
                        {
                            if (buffer[j]!='\r' && buffer[j]!='\n')
                            {
                                break;
                            }
                            j++;
                        }
                        int d = 0;
                        while (j < p + len)
                        {
                            buffer[d] = buffer[j];
                            d++;
                            j++;
                        }
                        p = 0;
                        i = 0;
                        len = d;
                        continue;
                    }
                    i++;
                }
                p = p + len;
                waitTime = 1;
            }
        }
        else
        {
            if (!impl->mSerialWriteSemaphore.tryAcquire(1,waitTime))
            {
                waitTime++;
                if (waitTime>100)
                {
                    waitTime = 100;
                    if (!port->isOpen())
                    {
                        //TODO
                    }
                }
            }
        }

        impl->mResponseStringLock.lock();
        if (impl->mResponseString.length()>0 && port->isOpen())
        {
            QString response = impl->mResponseString;
            impl->mResponseString.clear();
            impl->mResponseStringLock.unlock();
            port->write(response.toLocal8Bit());
        }
        else
        {
            impl->mResponseStringLock.unlock();
        }

    }
    port->close();
    delete port;
}

//static
void QtDNSerialPortImpl::requestHandelrThreadBody(void *_impl)
{
    if (!_impl)
        return;

    QtDNSerialPortImpl *impl = (QtDNSerialPortImpl*)_impl;
    while(!impl->mStopping)
    {
        impl->mQueueSemaphore.tryAcquire(1,100);
        impl->mRequestQueueLock.lock();
        if (!impl->mRequestQueue.isEmpty())
        {
            QString line = impl->mRequestQueue.dequeue();
            impl->mRequestQueueLock.unlock();

            if (line.length()>3)
            {
                QString seq = line.left(2);
                QString cmd = line.mid(2,1);
                QString par = line.mid(3);
                if (cmd == "M")
                {
                    //message
                    TKLog::printf(TKLog::NORMAL, "(SERIALl):%s", par.toLocal8Bit().constData());
                }
                else if (cmd == "S")
                {
                    //set data
                    int idx = par.indexOf(",");
                    if (idx > 0 && par.length() > idx + 1)
                    {
                        QString path = par.left(idx);
                        float val = par.mid(idx+1).toFloat();
                        impl->mServer->getEngine()->doClientSetRequest(path.toLocal8Bit().constData(),val);
                    }
                }
                else if (cmd == "G")
                {
                    //get data
                    float val = impl->mServer->getEngine()->doClientGetRequest(par.toLocal8Bit().constData());
                    QString rep = seq;
                    rep.append(QString::number(val));
                    rep.append("\n");
                    impl->sendMessage(rep.toLocal8Bit().constData());
                }
            }
        }
        else
        {
            impl->mRequestQueueLock.unlock();
        }

    }
}


Q_DECLARE_METATYPE(SerialPortInfo)

QtDNSerialPortSettingDialog::QtDNSerialPortSettingDialog(QWidget *parent)
    : QDialog(parent)
    , ui(new Ui::QtDNSerialPortSettingDialog)
    , mIsOK(false)
{
    ui->setupUi(this);
    procUpdateAvailablePorts();
    procItemPortChanged(0);

    connect(ui->updateButton, SIGNAL(clicked()), this, SLOT(procUpdateAvailablePorts()));
    connect(ui->portsComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(procItemPortChanged(int)));
    connect(ui->okPushButton, SIGNAL(clicked()), this, SLOT(procOKButtonClick()));
    connect(ui->cancelPushButton, SIGNAL(clicked()), this, SLOT(procCancelButtonClick()));

    //default setting
    QString defaultPortName = QString::fromStdString(DNSettings::getValue(DNSettings::SERIAL_PORTNAME , ""));
    for (int i = 0; i < ui->portsComboBox->count(); i++)
    {
        if (ui->portsComboBox->itemText(i) == defaultPortName)
        {
            ui->portsComboBox->setCurrentIndex(i);
            break;
        }
    }

    QString defaultBaudRate = QString::fromStdString(DNSettings::getValue(DNSettings::SERIAL_RATE, "9600"));
    for (int i = 0; i < ui->ratesComboBox->count(); i++)
    {
        if (ui->ratesComboBox->itemText(i) == defaultBaudRate)
        {
            ui->ratesComboBox->setCurrentIndex(i);
            break;
        }
    }

    QString defaultDataBits = QString::fromStdString(DNSettings::getValue(DNSettings::SERIAL_DATABITS, "8"));
    for (int i = 0; i < ui->dataBitsComboBox->count(); i++)
    {
        if (ui->dataBitsComboBox->itemText(i) == defaultDataBits)
        {
            ui->dataBitsComboBox->setCurrentIndex(i);
            break;
        }
    }

    QString defaultParity = QString::fromStdString(DNSettings::getValue(DNSettings::SERIAL_PARITY, "No parity"));
    for (int i = 0; i < ui->parityComboBox->count(); i++)
    {
        if (ui->parityComboBox->itemText(i) == defaultParity)
        {
            ui->parityComboBox->setCurrentIndex(i);
            break;
        }
    }

    QString defaultStopBits = QString::fromStdString(DNSettings::getValue(DNSettings::SERIAL_STOPBITS, "one stop"));
    for (int i = 0; i < ui->stopBitsComboBox->count(); i++)
    {
        if (ui->stopBitsComboBox->itemText(i) == defaultStopBits)
        {
            ui->stopBitsComboBox->setCurrentIndex(i);
            break;
        }
    }

    QString defaultFlowControl = QString::fromStdString(DNSettings::getValue(DNSettings::SERIAL_FLOWCONTROL, "No flow control"));
    for (int i = 0; i < ui->flowControlComboBox->count(); i++)
    {
        if (ui->flowControlComboBox->itemText(i) == defaultFlowControl)
        {
            ui->flowControlComboBox->setCurrentIndex(i);
            break;
        }
    }

}

QtDNSerialPortSettingDialog::~QtDNSerialPortSettingDialog()
{
    delete ui;
}

QString QtDNSerialPortSettingDialog::getSelectedPortName()
{
    return ui->portsComboBox->currentText();
}

int QtDNSerialPortSettingDialog::getSelectedRate()
{
    return ui->ratesComboBox->currentText().toInt();
}

SerialPort::DataBits QtDNSerialPortSettingDialog::getSelectedDataBits()
{
    SerialPort::DataBits r = SerialPort::UnknownDataBits;

    switch(ui->dataBitsComboBox->currentIndex())
    {
    case 0:
        r = SerialPort::Data5;
        break;
    case 1:
        r = SerialPort::Data6;
        break;
    case 2:
        r = SerialPort::Data7;
        break;
    case 3:
        r = SerialPort::Data8;
        break;
    }

    return r;
}

SerialPort::Parity   QtDNSerialPortSettingDialog::getSelectedParity()
{
    SerialPort::Parity r = SerialPort::UnknownParity;

    switch(ui->parityComboBox->currentIndex())
    {
    case 0:
        r = SerialPort::NoParity;
        break;
    case 1:
        r = SerialPort::EvenParity;
        break;
    case 2:
        r = SerialPort::OddParity;
        break;
    case 3:
        r = SerialPort::SpaceParity;
        break;
    case 4:
        r = SerialPort::MarkParity;
        break;
    }

    return r;
}

SerialPort::StopBits QtDNSerialPortSettingDialog::getSelectedStopBits()
{
    SerialPort::StopBits r = SerialPort::UnknownStopBits;

    switch(ui->stopBitsComboBox->currentIndex())
    {
    case 0:
        r = SerialPort::OneStop;
        break;
    case 1:
        r = SerialPort::OneAndHalfStop;
        break;
    case 2:
        r = SerialPort::TwoStop;
        break;
    }

    return r;
}

SerialPort::FlowControl QtDNSerialPortSettingDialog::getSelectedFlowControl()
{
    SerialPort::FlowControl r = SerialPort::UnknownFlowControl;

    switch(ui->flowControlComboBox->currentIndex())
    {
    case 0:
        r = SerialPort::NoFlowControl;
        break;
    case 1:
        r = SerialPort::HardwareControl;
        break;
    case 2:
        r = SerialPort::SoftwareControl;
        break;
    }

    return r;
}


void QtDNSerialPortSettingDialog::procUpdateAvailablePorts()
{
    ui->portsComboBox->clear();
    foreach (SerialPortInfo info, SerialPortInfo::availablePorts()) {
        QVariant v;
        v.setValue(info);
        ui->portsComboBox->addItem(info.portName(), v);
    }

    ui->okPushButton->setEnabled(ui->portsComboBox->count() > 0);
}

void QtDNSerialPortSettingDialog::procItemPortChanged(int idx)
{
    QVariant v = ui->portsComboBox->itemData(idx);
    if (v.isValid()) {
        SerialPortInfo info = v.value<SerialPortInfo>();

        ui->locationValueLabel->setText(info.systemLocation());
        ui->descriptionValueLabel->setText(info.description());
        ui->manufacturerValueLabel->setText(info.manufacturer());
    }
}

void QtDNSerialPortSettingDialog::procOKButtonClick()
{
    mIsOK = true;
    done(1);
}

void QtDNSerialPortSettingDialog::procCancelButtonClick()
{
    mIsOK = false;
    done(0);
}

bool QtDNSerialPortSettingDialog::isOK()
{
    // save selections
    DNSettings::setValue(DNSettings::SERIAL_PORTNAME, ui->portsComboBox->currentText().toStdString());
    DNSettings::setValue(DNSettings::SERIAL_RATE, ui->ratesComboBox->currentText().toStdString());
    DNSettings::setValue(DNSettings::SERIAL_DATABITS, ui->dataBitsComboBox->currentText().toStdString());
    DNSettings::setValue(DNSettings::SERIAL_PARITY, ui->parityComboBox->currentText().toStdString());
    DNSettings::setValue(DNSettings::SERIAL_STOPBITS, ui->stopBitsComboBox->currentText().toStdString());
    DNSettings::setValue(DNSettings::SERIAL_FLOWCONTROL, ui->flowControlComboBox->currentText().toStdString());

    return mIsOK;
}
