//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 2/22/2012.
//
#include "dnqscontainer.h"

#include "TKCell.h"
#include "TKCellCode.h"
#include "TKAxon.h"
#include "TKReceptor.h"
#include "TKAxonTerminal.h"
#include "TKContainer.h"
#include "TKLog.h"
#include "DNUtils.h"
#include "versioninfo.h"
#include "dnqsbasiccell.h"
#include "dnqsinputcell.h"
#include "dnqsoutputcell.h"
#include "dnqsbasicstoragecell.h"
#include "dnplugininputcell.h"
#include "dnpluginoutputcell.h"
#include "dnqscellcode.h"
#include "DNGlobal.h"

static QScriptValue scriptPrint(QScriptContext *context, QScriptEngine *engine)
{
    if (context->argumentCount() >= 1)
    {
        QString value = context->argument(0).toString();
        TKLog::printf(TKLog::NORMAL, "%s", value.toLocal8Bit().data());
    }
    return engine->nullValue();
}

static QScriptValue scriptFQN(QScriptContext *context, QScriptEngine *engine)
{
    if (context->argumentCount() >= 2)
    {
        const char *location = context->argument(0).toString().toUtf8().constData();
        const char *name = context->argument(1).toString().toUtf8().constData();

        std::string fqnString = getFQNString(location, name);
        return QScriptValue(fqnString.c_str());
    }
    else
    {
        return engine->nullValue();
    }
}

//static
TKContainer* TKContainer::createContainer()
{
    TKContainer *container = new DNQSContainer();
    container->init();
    return container;
}

DNQSContainer::DNQSContainer()
{
    mQSEngine = new QScriptEngine();
    mQSGlobalObject = mQSEngine->globalObject();

    mQSGlobalObject.setProperty("print",mQSEngine->newFunction(scriptPrint, 1));
    mQSGlobalObject.setProperty("FQN",mQSEngine->newFunction(scriptFQN, 2));

    QScriptValue global = mQSEngine->newObject();
    mQSGlobalObject.setProperty("global", global,QScriptValue::Undeletable);
    global.setProperty("container_version", QScriptValue(CONTAINER_VERSION),QScriptValue::ReadOnly|QScriptValue::Undeletable);

    mQSCellContainer = mQSEngine->newObject();
    mQSGlobalObject.setProperty("__private_DNCellContainer__",mQSCellContainer);
}

DNQSContainer::~DNQSContainer()
{
    if (mQSEngine)
        delete mQSEngine;
}

void DNQSContainer::setValue(std::string key, float value)
{
    mQSGlobalObject.setProperty(QString::fromStdString(key), QScriptValue(value));
}

float DNQSContainer::getValue(std::string key) const
{
    QScriptValue v = mQSGlobalObject.property(QString::fromStdString(key));
    if (v.isNumber())
    {
        return v.toNumber();
    }
    else
    {
        return 0;
    }
}

TKCell* DNQSContainer::cellFactory(std::string location, std::string name, std::string type, bool canInterfaceIn, bool canInterfaceOut)
{
    TKCell *cell = NULL;

    if (type == CELLTYPE_JSBASIC || type.length() == 0)
    {
        cell = new DNQSBasicCell(this, location, name, canInterfaceIn, canInterfaceOut);
    }
    else if (type == CELLTYPE_IN)
    {
        cell = new DNQSInputCell(this, location, name);
    }
    else if (type == CELLTYPE_OUT)
    {
        cell = new DNQSOutputCell(this, location, name);
    }
    else if (type == CELLTYPE_BASICSTORAGE)
    {
        cell = new DNQSBasicStorageCell(this, location, name, canInterfaceIn, canInterfaceOut);
    }

    if (cell)
        cell->init();

    return cell;
}

TKCell* DNQSContainer::pluginCellFactory(std::string location, std::string fullName, std::string type, std::string pluginName, std::string pluginValue, bool canInterfaceIn, bool canInterfaceOut)
{
    Q_ASSERT(canInterfaceIn != canInterfaceOut);

    (void)canInterfaceOut;

    TKCell *cell = NULL;

    if (canInterfaceIn)
    {
        cell = new DNPluginInputCell(this, location, fullName, pluginName, pluginValue);
    }
    else
    {
        cell = new DNPluginOutputCell(this, location, fullName, pluginName, pluginValue);
    }

    if (cell)
        cell->init();

    return cell;
}

TKAxon* DNQSContainer::axonFactory(TKCell *theOwner)
{
    return new TKAxon(theOwner);
}

TKReceptor* DNQSContainer::receptorFactory(TKCell *theOwner)
{
    return new TKReceptor(theOwner);
}

TKAxonTerminal* DNQSContainer::axonTerminalFactory(TKAxon *theOwner)
{
    return new TKAxonTerminal(theOwner);
}

TKCellCode* DNQSContainer::cellCodeFactory(std::string name, std::string cellapi, std::string code)
{
    return new DNQSCellCode(name, cellapi, this, code);
}
