﻿(*----------------------------------------------------------------------------
-- Copyright 2020 Levashev Ivan Aleksandrovich                              --
--                                                                          --
-- Licensed under the Apache License, Version 2.0 (the "License");          --
-- you may not use this file except in compliance with the License.         --
-- You may obtain a copy of the License at                                  --
--                                                                          --
--     http://www.apache.org/licenses/LICENSE-2.0                           --
--                                                                          --
-- Unless required by applicable law or agreed to in writing, software      --
-- distributed under the License is distributed on an "AS IS" BASIS,        --
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. --
-- See the License for the specific language governing permissions and      --
-- limitations under the License.                                           --
----------------------------------------------------------------------------*)

(***************************************************************************\
* Отладочный менеджер памяти                                                *
*****************************************************************************
* - Выделяет под объекты память страницами.                                 *
* - При освобождении страницы освобождаются логически, но так, что не может *
*   тот же виртуальный адрес быть повторно использован. Обращения к         *
*   уничтоженному объекту приводят к A/V сразу на месте без шанса походить  *
*   по висячим указателям и попортить структуры данных.                     *
* - В силу специфики менеджеров памяти не добавляется в проект. Вместо      *
*   этого в список путей поиска добавляется директория SafeMM3, а в проекте *
*   перед всеми другими модулями указывается SafeMM3.Install.               *
* - Принцип действия предполагает постоянный расход виртуальной памяти, так *
*   что имеет смысл отдавать предпочтение 64разрядным сборкам.              *
* - Даже на Win64 при полной загрузке АРМ исчерпываются ресурсы. В связи с  *
*   этим применён вероятностный подход, и вероятность задаётся аргументами  *
*   InstallDebugMemoryManager в модуле Install. На слабых рабочих станциях  *
*   придётся уменьшать.                                                     *
* - TThreadPool.TAbstractWorkerData.FreeInstance не вызывает inherited, а   *
*   напрямую FreeMem, что не вполне корректно и не было совместимо с        *
*   SafeMM3. Для обхода этой проблемы указатели, полученные VirtualAlloc и  *
*   vOldManager.GetMem, стало можно смешивать.                              *
*****************************************************************************
* Иван Левашев, май 2020                                                    *
****************************************************************************)

unit SafeMM3;

interface

var
  vRandomVirtualChance: Integer = 333;

function DebugGetMem(Size: NativeInt): Pointer;
function DebugFreeMem(P: Pointer): Integer;
function DebugReallocMem(P: Pointer; Size: NativeInt): Pointer;
function DebugAllocMem(ASize: NativeInt): Pointer;

function DebugMaybeVirtualGetMem(Size: NativeInt): PByte;
function DebugVirtualGetMem(Size: NativeInt): PByte;
function DebugNonVirtualGetMem(Size: NativeInt): PByte;

const
  DebugMemoryManager: TMemoryManagerEx = (
    GetMem: DebugGetMem;
    FreeMem: DebugFreeMem;
    ReallocMem: DebugReallocMem;
    AllocMem: DebugAllocMem;
    RegisterExpectedMemoryLeak: nil;
    UnregisterExpectedMemoryLeak: nil;
  );

/// <param name="RandomVirtualChance">
///    Шанс в промилле, что объект будет выделен VirtualAlloc
/// </param>
/// <param name="UseGuardPages">
///    Использовать защитные страницы после области выделенной памяти на случай,
///    если с объектом пытаются работать как с экземпляром, чьего класса он не
///    является
/// </param>
/// <param name="UseDeveloperHintPages">
///    Использовать ли страницы, подсказывающие разработчику, что по некоторому
///    адресу был выделен объект. Они расположены перед объектом, и надо
///    прокрутить сильно назад, чтоб их увидеть. И это только для уничтоженных
///    объектов. И отладчик незакоммиченные страницы может не отображать как ??,
///    может отображать как 00, при том, что в CPU подсказка для того же адреса
///    может быть [...] = ???? Размер страницы в шестнадцатеричном коде $1000,
///    так что насколько далеко прокручивать, должно быть легко посчитать.
///    Крутим вверх обзор памяти, пока не стали три последние цифры нулями, а
///    выше должна начаться страница, вся заполненная одним и тем же, а именно
///    TClass уничтоженного объекта. Можно в Evaluate/Modify запросить
///    TClass($....).ClassName
/// </param>
procedure InstallDebugMemoryManager
  (RandomVirtualChance: Integer = 333;
   UseGuardPages: Boolean = False;
   UseDeveloperHintPages: Boolean = False);
procedure UninstallDebugMemoryManager;

implementation

var
  vOldManager: TMemoryManagerEx;
  vNewInstanceAddressBegin,
  vNewInstanceAddressEnd,
  vFreeInstanceAddressBegin,
  vFreeInstanceAddressEnd: PByte;
  vPageSize, vAllocationGranularity: UInt32;
  vGuardSize, vDeveloperHintSize: UInt32; // vPageSize или 0
  vUseGuardPages: Boolean = False;
  vUseDeveloperHintPages: Boolean = False;

type
  TSafeMM3Preamble = record
    FullSize: NativeInt; // отрицательные значения = обычный менеджер памяти
    MagicWord: UInt64; // должно быть последним полем
  end;
  PSafeMM3Preamble = ^TSafeMM3Preamble;

const
  cSafeMM3MagicWord = $DEFA15EEDFBE01C0;

type
  TSystemInfo = record
    case Integer of
      0: (
        dwOemId: UInt32);
      1: (
        wProcessorArchitecture: UInt16;
        wReserved: UInt16;
        dwPageSize: UInt32;
        lpMinimumApplicationAddress: Pointer;
        lpMaximumApplicationAddress: Pointer;
        dwActiveProcessorMask: ^UInt32;
        dwNumberOfProcessors: UInt32;
        dwProcessorType: UInt32;
        dwAllocationGranularity: UInt32;
        wProcessorLevel: UInt16;
        wProcessorRevision: UInt16);
  end;
procedure GetSystemInfo(var SystemInfo: TSystemInfo); stdcall;
  external 'kernel32.dll' name 'GetSystemInfo';

procedure MeasureSystem;
type
  TNewInstanceHandler = function: TObject of object;
  TFreeInstanceHandler = procedure of object;
var
  newInstanceMethod: TNewInstanceHandler;
  fakeObject: TClass; // отсюда читается VMT
  freeInstanceMethod: TFreeInstanceHandler;
  systemInfo: TSystemInfo;
begin
  newInstanceMethod := TObject.NewInstance;
  vNewInstanceAddressBegin := TMethod(newInstanceMethod).Code;
  vNewInstanceAddressEnd := vNewInstanceAddressBegin + $50; // реальный размер $22
  fakeObject := TObject;
  freeInstanceMethod := TObject(@fakeObject).FreeInstance;
  vFreeInstanceAddressBegin := TMethod(freeInstanceMethod).Code;
  vFreeInstanceAddressEnd := vFreeInstanceAddressBegin + $50; // реально $29 (x64)
  GetSystemInfo(systemInfo);
  vPageSize := systemInfo.dwPageSize;
  vAllocationGranularity := systemInfo.dwAllocationGranularity;
end;

procedure InstallDebugMemoryManager
  (RandomVirtualChance: Integer = 333;
   UseGuardPages: Boolean = False;
   UseDeveloperHintPages: Boolean = False);
begin
  Assert(GetHeapStatus.TotalAllocated = 0);
  GetMemoryManager(vOldManager);
  vRandomVirtualChance := RandomVirtualChance;
  vUseGuardPages := UseGuardPages;
  vUseDeveloperHintPages := UseDeveloperHintPages;
  SetMemoryManager(DebugMemoryManager);
  MeasureSystem;
  if UseGuardPages then begin
    vGuardSize := vPageSize;
  end
  else begin
    vGuardSize := 0;
  end {if};
  if UseDeveloperHintPages then begin
    vDeveloperHintSize := vPageSize;
  end
  else begin
    vDeveloperHintSize := 0;
  end {if};
  Randomize;
end;

procedure UninstallDebugMemoryManager;
begin
  SetMemoryManager(vOldManager);
end;

{$IFDEF WIN64}
function GetReturnAddress2: PByte;
// Обычный ReturnAddress возвращает адрес в System._GetMem, а нам это не
// интересно, нам бы надо на ещё один уровень глубже

// В Win32 процедуры сначала копируют esp в ebp, потом уменьшают esp
// В Win64 процедуры сначала уменьшают rsp, потом копируют rsp в rbp
// Как результат, на Win32 довольно просто пробегать по стеку, а на Win64
// по-хорошему нужно активно обращаться к WinAPI вроде RtlLookupFunctionEntry.
// Пример есть в ECallStack.TEurekaBaseStackList.Caller

// Здесь сделана попытка всё же по-простому узнавать недостающий адрес, опираясь
// на информацию об устройстве стека DebugGetMem и System._GetMem и т.п.
// функций.

// Но из-за этого при обновлении Delphi может ломаться. Данный модуль не для
// релизных сборок, а для разработчиков, так что пока и так сойдёт.
asm
  // mov rax, [rbp+$30]
  // mov rax, [rax+$8]
  // // System._FreeMem не делает push rbp, вместо этого rbx
  // // короче, проще со стека по смещению от rsp достать
  mov rax, [rsp+$70]
end;
{$ENDIF WIN64}
{$IFDEF WIN32}
function GetReturnAddress2: PByte;
asm
  mov eax, [esp+$18]
end;
{$ENDIF}

function DebugGetMem(Size: NativeInt): Pointer;
var
  returnAddress2: PByte;
begin
  returnAddress2 := GetReturnAddress2;
  if (returnAddress2 >= vNewInstanceAddressBegin) and
    (returnAddress2 <= vNewInstanceAddressEnd)
  then begin
    Result := DebugMaybeVirtualGetMem(Size);
  end
  else begin
    Result := DebugNonVirtualGetMem(Size);
  end {if};
end;

function DebugReallocMem(P: Pointer; Size: NativeInt): Pointer;
begin
  // Методы TObject не вызывают ReallocMem
  Dec(PByte(P), SizeOf(TSafeMM3Preamble));
  Result := vOldManager.ReallocMem(P, SizeOf(TSafeMM3Preamble) + Size);
  if not Assigned(Result) then begin
    Exit;
  end {if};
  PSafeMM3Preamble(Result).FullSize := -(SizeOf(TSafeMM3Preamble) + Size);
  Inc(PByte(Result), SizeOf(TSafeMM3Preamble));
end;

function DebugAllocMem(ASize: NativeInt): Pointer;
begin
  // Методы TObject не вызывают AllocMem
  Result := DebugNonVirtualGetMem(ASize);
  if Assigned(Result) then begin
    FillChar(Result^, ASize, 0);
  end {if};
end;

function VirtualAlloc
  (lpvAddress: Pointer; dwSize: NativeUInt;
   flAllocationType, flProtect: UInt32): Pointer; stdcall;
  external 'kernel32.dll' name 'VirtualAlloc';
function VirtualFree
  (lpAddress: Pointer; dwSize: NativeUInt;
   dwFreeType: UInt32): LongBool; stdcall;
  external 'kernel32.dll' name 'VirtualFree';
function VirtualProtect
  (lpAddress: Pointer; dwSize: NativeUInt;
   flNewProtect: UInt32; var OldProtect: UInt32): LongBool; stdcall;
  external 'kernel32.dll' name 'VirtualProtect';

const
  MEM_COMMIT   = $00001000;
  MEM_RESERVE  = $00002000;
  MEM_DECOMMIT = $00004000;
  MEM_RELEASE  = $00008000;
  PAGE_NOACCESS  = $01;
  PAGE_READONLY  = $02;
  PAGE_READWRITE = $04;

function DebugMaybeVirtualGetMem(Size: NativeInt): PByte;
begin
  if Random(1000) >= vRandomVirtualChance then begin
    // выделение обычным менеджером памяти
    Exit(DebugNonVirtualGetMem(Size));
  end
  else begin
    Exit(DebugVirtualGetMem(Size));
  end {if};
end;

threadvar
  tvReservedMemory: PByte;
  /// <remarks>
  ///    Округление до dwPageSize
  /// </remarks>
  tvReservedAmount: NativeInt;

/// <summary>
///    Данная функция выделяет память обычным VirtualAlloc и сохраняет в
///    threadvar, если (а это, скорее всего, так) в блоке
///    dwAllocationGranularity ещё осталось пространство. Последующие вызовы по
///    возможности отъедают место от уже выделенного блока
/// </summary>
/// <remarks>
///    Для памяти не сделан MEM_COMMIT
/// </remarks>
function CompactVirtualAlloc(Size: NativeInt): PByte;
// VirtualAlloc из WinAPI выделяет память блоками по dwAllocationGranularity
// (64кб), а хотелось бы dwPageSize (4кб), ведь это разница на целый порядок
// Если поддерживать честный Free, то нужно было бы дополнительные структуры
// в памяти устраивать, чтоб распиливать dwAllocationGranularity на dwPageSize
// Но менеджер памяти только для отладки, и можно кое-что упростить. FreeMem
// реализуется через MEM_DECOMMIT, а для MEM_DECOMMIT принадлежность блоку
// MEM_RESERVE не важна. Только для MEM_RELEASE нужно адрес указывать на начало
// блока, полученного от VirtualAlloc, и он высвобождается сразу весь. Нет
// MEM_RELEASE, нет проблем.

var
  // округление до dwPageSize
  sizeRoundedUpToPageSize, newFreeSpace: NativeInt;

  // округление до dwAllocationGranularity
  newAllocBlockSize: NativeInt;
begin
  sizeRoundedUpToPageSize := Size + vPageSize - 1;
  sizeRoundedUpToPageSize := sizeRoundedUpToPageSize -
    (sizeRoundedUpToPageSize mod vPageSize);

  if tvReservedAmount < sizeRoundedUpToPageSize then begin
    newAllocBlockSize := sizeRoundedUpToPageSize + vAllocationGranularity - 1;
    newAllocBlockSize := newAllocBlockSize -
      (newAllocBlockSize mod vAllocationGranularity);

    Result := VirtualAlloc
      ((* lpvAddress       => *) nil,
       (* dwSize           => *) newAllocBlockSize,
       (* flAllocationType => *) MEM_RESERVE,
       (* flProtect        => *) PAGE_NOACCESS);

    if not Assigned(Result) then begin
      Exit;
    end {if};

    // В новом блоке памяти свободного места может быть больше, чем старом

    newFreeSpace := newAllocBlockSize - sizeRoundedUpToPageSize;
    if newFreeSpace < tvReservedAmount then begin
      Exit;
    end {if};

    // Тогда лучше заменить на новый

    tvReservedMemory := Result + sizeRoundedUpToPageSize;
    tvReservedAmount := newFreeSpace;
  end
  else if tvReservedAmount > sizeRoundedUpToPageSize then begin
    Result := tvReservedMemory;
    Inc(tvReservedMemory, sizeRoundedUpToPageSize);
    Dec(tvReservedAmount, sizeRoundedUpToPageSize);
  end
  else begin
    Result := tvReservedMemory;
    tvReservedMemory := nil;
    tvReservedAmount := 0;
  end {if};
end;

function DebugVirtualGetMem(Size: NativeInt): PByte;
var
  fullSize: NativeInt;
begin
  fullSize :=
    vDeveloperHintSize +       // подсказка разработчику

                               // здесь неявно присутствует выравнивание по
                               // границе между страницами

    SizeOf(TSafeMM3Preamble) + // преамбула
    ((Size + 7) and not 7) +   // память объекта округлена по границе 8 байт
    vGuardSize;                // защитная страница

  Result := CompactVirtualAlloc(fullSize);

  if not Assigned(Result) then begin
    Exit;
  end {if};

  Inc(Result, vDeveloperHintSize); // у живых объектов подсказка не заполнена

  // Выделен цельный диапазон виртуальной памяти без логической.
  // Теперь в этом цельном диапазоне нужно выделить страницы кроме отладочных

  if not Assigned(VirtualAlloc
    ((* lpvAddress       => *) Result,
     (* dwSize           => *) fullSize - vGuardSize - vDeveloperHintSize,
     (* flAllocationType => *) MEM_COMMIT,
     (* flProtect        => *) PAGE_READWRITE))
  then begin
    Exit(nil);
  end {if};

  // Теперь сдвигаем Result так, чтоб прижать вплотную к защитной странице

  Inc(Result, (vPageSize - fullSize mod vPageSize) mod vPageSize);

  // Заполняем преамбулу

  PSafeMM3Preamble(Result).FullSize := fullSize;
  PSafeMM3Preamble(Result).MagicWord := cSafeMM3MagicWord;

  // Готовим окончательный результат

  Inc(Result, SizeOf(TSafeMM3Preamble));
end;

procedure SetupDeveloperHintPage(HintPage: PByte; WasClass: TClass);
type
  PClass = ^TClass;
var
  index: Integer;
  oldProtect: UInt32;
begin
  // Сначала нужно эту страницу закоммитить

  if not Assigned(VirtualAlloc
    ((* lpvAddress       => *) HintPage,
     (* dwSize           => *) vDeveloperHintSize,
     (* flAllocationType => *) MEM_COMMIT,
     (* flProtect        => *) PAGE_READWRITE))
  then begin
    Exit; // видимо, не будет у разработчика подсказки
  end {if};

  for index := 0 to vPageSize div SizeOf(TClass) - 1 do begin
    PClass(HintPage + SizeOf(TClass) * index)^ := WasClass;
  end {loop};

  // Для тяжёлых случаев можно ещё в страницу вставить дамп с вершины стека,
  // чтоб потом, при A/V, вычислять, кто уничтожил объект

  // Закроем доступ на запись

  VirtualProtect
    ((* lpAddress    => *) HintPage,
     (* dwSize       => *) vDeveloperHintSize,
     (* flNewProtect => *) PAGE_READONLY,
     (* OldProtect   => *) oldProtect);
end;

function DebugFreeMem(P: Pointer): Integer;
var
  fullSize: NativeInt;
  wasClass: TClass;
begin
  // возвращаемся к преамбуле
  Dec(PByte(P), SizeOf(TSafeMM3Preamble));

  if PSafeMM3Preamble(P).MagicWord <> cSafeMM3MagicWord then begin
    Exit(1);
  end {if};

  fullSize := PSafeMM3Preamble(P).FullSize;

  if fullSize <= 0 then begin
    // память была выделена не через VirtualAlloc

    // напоследок испортим байты так, чтобы это не было похоже на хороший
    // указатель

    FillChar(P^, -fullSize, Byte($FE));

    Exit(0); // и если уж портить, то чтоб без переиспользования
    // Exit(vOldManager.FreeMem(P));
  end {if};

  if vUseDeveloperHintPages then begin
    // Класс надо успеть сохранить сейчас
    wasClass := TObject(PByte(P) + SizeOf(TSafeMM3Preamble)).ClassType;
  end {if};

  // Возвращаемся к самому началу страницы

  Dec(PByte(P), (vPageSize - fullSize mod vPageSize) mod vPageSize);

  // Оставляем виртуальные адреса навечно в резерве

  if not VirtualFree
    ((* lpAddress  => *) P,
     (* dwSize     => *) fullSize - vGuardSize - vDeveloperHintSize,
     (* dwFreeType => *) MEM_DECOMMIT)
  then begin
    Exit(1);
  end {if};

  if vUseDeveloperHintPages then begin
    SetupDeveloperHintPage(PByte(P) - vDeveloperHintSize, wasClass);
  end {if};

  Exit(0);
end;

function DebugNonVirtualGetMem(Size: NativeInt): PByte;
begin
  Result := vOldManager.GetMem(SizeOf(TSafeMM3Preamble) + Size);

  if not Assigned(Result) then begin
    Exit;
  end {if};

  // Заполняем преамбулу

  PSafeMM3Preamble(Result).FullSize := -(SizeOf(TSafeMM3Preamble) + Size);
  PSafeMM3Preamble(Result).MagicWord := cSafeMM3MagicWord;

  // Готовим окончательный результат

  Inc(Result, SizeOf(TSafeMM3Preamble));
end;

end.

