/* File: do-spell.c */

/* Purpose: Do everything for each spell */

#include "angband.h"
#include "cave.h"
#include "command.h"
#include "creature_const.h"
#include "creature_melee.h"
#include "creature_throwing.h"
#include "floors.h"
#include "mutation.h"
#include "object.h"
#include "spell.h"


/*
 * Generate dice info string such as "foo 2d10"
 */
static cptr info_string_dice(cptr str, int dice, int sides, int base)
{
	if(!dice) return format("%s%d", str, base); /* Fix value */
	else if(!base) return format("%s%dd%d", str, dice, sides); /* Dice only */
	else return format("%s%dd%d%+d", str, dice, sides, base); /* Dice plus base value */
}


/*
 * Generate damage-dice info string such as "dam 2d10"
 */
static cptr info_damage(int dice, int sides, int base)
{
	return info_string_dice(get_keyword("KW_DAM"), dice, sides, base);
}


/*
 * Generate duration info string such as "dur 20+1d20"
 */
static cptr info_duration(int base, int sides)
{
	return format("%s:%d+1d%d", get_keyword("KW_DURING"), base, sides);
}


/*
 * Generate range info string such as "range 5"
 */
static cptr info_range(COODINATES range)
{
	return format("%s:%d", get_keyword("KW_RANGE"), range);
}


/*
 * Generate heal info string such as "heal 2d8"
 */
static cptr info_heal(int dice, int sides, int base)
{
	return info_string_dice(get_keyword("KW_HEAL"), dice, sides, base);
}


/*
 * Generate delay info string such as "delay 15+1d15"
 */
static cptr info_delay(int base, int sides)
{
	return format("%s:%d+1d%d", get_keyword("KW_DELAY"), base, sides);
}


/*
 * Generate multiple-damage info string such as "dam 25 each"
 */
static cptr info_multi_damage(POWER dam)
{
	return format("%s e%d", get_keyword("KW_DAM"), dam);
}


/*
 * Generate multiple-damage-dice info string such as "dam 5d2 each"
 */
static cptr info_multi_damage_dice(int dice, int sides)
{
	return format("%s e%dd%d", get_keyword("KW_DAM"), dice, sides);
}


/*
 * Generate power info string such as "power 100"
 */
static cptr info_power(POWER power)
{
	return format("%s %d", get_keyword("KW_POW"), power);
}


/*
 * Generate power info string such as "power 1d100"
 */
static cptr info_power_dice(int dice, int sides)
{
	return format("%s %dd%d", get_keyword("KW_POW"), dice, sides);
}


/*
 * Generate radius info string such as "rad 100"
 */
static cptr info_radius(COODINATES rad)
{
	return format("%s %d", get_keyword("KW_RAD"), rad);
}


/*
 * Generate weight info string such as "max wgt 15"
 */
static cptr info_weight(int weight)
{
	char buf[30];
	format_weight(buf, weight);
	return format("%s:%s", MES_SYS_MAX_WEIGHT, buf);
}

// Prepare standard probability to become beam for cast_bolt()
int beam_chance(creature_type *creature_ptr)
{
	if(has_trait(creature_ptr, TRAIT_MAGIC_SPECIALIST)) return creature_ptr->lev;
	return creature_ptr->lev / 2;
}

// Handle summoning and failure of trump spells
static bool trump_summoning(creature_type *creature_ptr, int num, bool pet, COODINATES y, COODINATES x, FLOOR_LEV lev, int type, FLAGS_32 mode)
{
	int who;
	int i;
	bool success = FALSE;

	if(!lev) lev = creature_ptr->lev * 2 / 3 + (FLOOR_LEV)randint1(creature_ptr->lev / 2); // TODO Default level

	if(pet)
	{
		mode |= PC_FORCE_PET;
		if(mode & PC_ALLOW_UNIQUE)
			if(randint1(50 + creature_ptr->lev) >= creature_ptr->lev / 10) mode &= ~PC_ALLOW_UNIQUE;
		who = -1;
	}
	else
	{
		mode |= PC_NO_PET;
		who = 0;
	}
	for (i = 0; i < num; i++) if(summoning(creature_ptr, y, x, lev, type, mode)) success = TRUE;
	if(!success) msg_print(MES_TRAIT_NO_TRUMP_SUMMON);

	return success;
}


/*
 * This spell should become more useful (more controlled) as the
 * player gains experience levels.  Thus, add 1/5 of the player's
 * level to the die roll.  This eliminates the worst effects later on,
 * while keeping the results quite random.  It also allows some potent
 * effects only at high level.
 */
void cast_wonder(creature_type *caster_ptr, TRAIT_ID trait_id)
{
	//TODO target_select
	COODINATES y = 0, x = 0;
	COODINATES lev_bonus = caster_ptr->lev;
	int die = randint1(100) + lev_bonus / 5;
	// TODO: add Karma of Fortune feature.

	if(die > 100) msg_print(MES_TRAIT_CAST_WONDER_MIGHTY);

	if(die < 8) cast_bolt(caster_ptr, DO_EFFECT_CLONE, MAX_RANGE_SUB, 0, trait_id);
	else if(die < 14) cast_bolt(caster_ptr, DO_EFFECT_HASTE, MAX_RANGE_SUB, caster_ptr->lev, trait_id);
	else if(die < 26) cast_bolt(caster_ptr, DO_EFFECT_HEAL_HP, MAX_RANGE_SUB, diceroll(4, 6), trait_id);
	else if(die < 31) cast_bolt(caster_ptr, DO_EFFECT_POLY, MAX_RANGE_SUB, lev_bonus, trait_id);
	else if(die < 36) cast_bolt(caster_ptr, DO_EFFECT_MISSILE, MAX_RANGE_SUB, diceroll(3 + ((lev_bonus - 1) / 5), 4), trait_id);
	else if(die < 41) cast_bolt(caster_ptr, DO_EFFECT_CONFUSION, MAX_RANGE_SUB, lev_bonus, trait_id);
	else if(die < 46) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_POIS, 20 + (lev_bonus / 2), 3, trait_id, 0);
	else if(die < 51) (void)cast_beam(caster_ptr, DO_EFFECT_LITE_WEAK, MAX_RANGE_SUB, diceroll(6, 8), trait_id);
	else if(die < 56) cast_bolt(caster_ptr, DO_EFFECT_ELEC, MAX_RANGE_SUB, diceroll(3 + ((lev_bonus - 5) / 4), 8), trait_id);
	else if(die < 61) cast_bolt(caster_ptr, DO_EFFECT_COLD, MAX_RANGE_SUB, diceroll(5 + ((lev_bonus - 5) / 4), 8), trait_id);
	else if(die < 66) cast_bolt(caster_ptr, DO_EFFECT_ACID, MAX_RANGE_SUB, diceroll(6 + ((lev_bonus - 5) / 4), 8), trait_id);
	else if(die < 71) cast_bolt(caster_ptr, DO_EFFECT_FIRE, MAX_RANGE_SUB, diceroll(8 + ((lev_bonus - 5) / 4), 8), trait_id);
	else if(die < 76) cast_bolt(caster_ptr, DO_EFFECT_DRAIN, MAX_RANGE_SUB, 75, trait_id);
	else if(die < 81) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_ELEC, 30 + lev_bonus / 2, 2, trait_id, 0);
	else if(die < 86) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_ACID, 40 + lev_bonus, 2, trait_id, 0);
	else if(die < 91) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_ICE, 70 + lev_bonus, 3, trait_id, 0);
	else if(die < 96) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_FIRE, 80 + lev_bonus, 3, trait_id, 0);
	else if(die < 101) cast_bolt(caster_ptr, DO_EFFECT_DRAIN, MAX_RANGE_SUB, 100 + lev_bonus, trait_id);
	else if(die < 104) earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 12);
	else if(die < 106) (void)destroy_area(caster_ptr, caster_ptr->fy, caster_ptr->fx, 13 + (COODINATES)randint0(5), FALSE);
	else if(die < 108) symbol_genocide(caster_ptr, lev_bonus + 50, TRUE);
	else if(die < 110) project_all_vision(caster_ptr, DO_EFFECT_DISP_ALL, 120);
	else
	{
		project_all_vision(caster_ptr, DO_EFFECT_DISP_ALL, 150);
		project_all_vision(caster_ptr, DO_EFFECT_SLOW, caster_ptr->lev);
		project_all_vision(caster_ptr, DO_EFFECT_SLEEP, caster_ptr->lev);
		heal_creature(caster_ptr, 300);
	}
}

void cast_invoke_spirits(creature_type *caster_ptr, TRAIT_ID trait_id)
{
	//TODO: target
	COODINATES y = 0, x = 0;
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);
	int lev_bonus = caster_ptr->lev;
	int die = randint1(100) + lev_bonus / 5;
	// TODO: Add Karma of Fortune feature.

	msg_print(INVOKE_SPIRIT_START);
	if(die > 100) msg_print(INVOKE_SPIRIT_HIGH_POWER);

	if(die < 8)
	{
		msg_print(INVOKE_SPIRIT_LOW1);
		(void)summoning(0, caster_ptr->fy, caster_ptr->fx, floor_ptr->depth, TRAIT_S_UNDEAD, (PC_ALLOW_GROUP | PC_ALLOW_UNIQUE | PC_NO_PET));
	}
	else if(die < 14)
	{
		msg_print(INVOKE_SPIRIT_LOW2);
		add_timed_trait(caster_ptr, TRAIT_AFRAID, randint1(4) + 4, TRUE);
	}
	else if(die < 26)
	{
		msg_print(INVOKE_SPIRIT_LOW3);
		add_timed_trait(caster_ptr, TRAIT_CONFUSED, randint1(4) + 4, TRUE);
	}
	else if(die < 31) cast_bolt(caster_ptr, DO_EFFECT_POLY, MAX_RANGE_SUB, lev_bonus, trait_id);
	else if(die < 36) cast_bolt(caster_ptr, DO_EFFECT_MISSILE, MAX_RANGE_SUB, diceroll(3 + ((lev_bonus - 1) / 5), 4), trait_id);
	else if(die < 41) cast_bolt(caster_ptr, DO_EFFECT_CONFUSION, MAX_RANGE_SUB, lev_bonus, trait_id);
	else if(die < 46) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_POIS, 20 + (lev_bonus / 2), 3, trait_id, 0);
	else if(die < 51) (void)cast_beam(caster_ptr, DO_EFFECT_LITE_WEAK, MAX_RANGE_SUB, diceroll(6, 8), trait_id);
	else if(die < 56) cast_bolt(caster_ptr, DO_EFFECT_ELEC, MAX_RANGE_SUB, diceroll(3+((lev_bonus-5)/4),8), trait_id);
	else if(die < 61) cast_bolt(caster_ptr, DO_EFFECT_COLD, MAX_RANGE_SUB, diceroll(5+((lev_bonus-5)/4),8), trait_id);
	else if(die < 66) cast_bolt(caster_ptr, DO_EFFECT_ACID, MAX_RANGE_SUB, diceroll(6+((lev_bonus-5)/4),8), trait_id);
	else if(die < 71) cast_bolt(caster_ptr, DO_EFFECT_FIRE, MAX_RANGE_SUB, diceroll(8+((lev_bonus-5)/4),8), trait_id);
	else if(die < 76) cast_bolt(caster_ptr, DO_EFFECT_DRAIN, MAX_RANGE_SUB, 75, trait_id);
	else if(die < 81) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_ELEC, 30 + lev_bonus / 2, 2, trait_id, 0);
	else if(die < 86) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_ACID, 40 + lev_bonus, 2, trait_id, 0);
	else if(die < 91) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_ICE, 70 + lev_bonus, 3, trait_id, 0);
	else if(die < 96) cast_ball_aux(caster_ptr, y, x, DO_EFFECT_FIRE, 80 + lev_bonus, 3, trait_id, 0);
	else if(die < 101) cast_bolt(caster_ptr, DO_EFFECT_DRAIN, MAX_RANGE_SUB, 100 + lev_bonus, trait_id);
	else if(die < 104) earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 12);
	else if(die < 106) (void)destroy_area(caster_ptr, caster_ptr->fy, caster_ptr->fx, 13 + (DIRECTION)randint0(5), FALSE);
	else if(die < 108) symbol_genocide(caster_ptr, lev_bonus + 50, TRUE);
	else if(die < 110) project_all_vision(caster_ptr, DO_EFFECT_DISP_ALL, 120);
	else
	{
		project_all_vision(caster_ptr, DO_EFFECT_DISP_ALL, 150);
		project_all_vision(caster_ptr, DO_EFFECT_SLOW, caster_ptr->lev);
		project_all_vision(caster_ptr, DO_EFFECT_SLEEP, caster_ptr->lev);
		heal_creature(caster_ptr, 300);
	}

	if(die < 31) msg_print(INVOKE_SPIRIT_FUNBLE_AFTER);
}


void wild_magic(creature_type *caster_ptr, int spell)
{
	int counter = 0;
	int type = TRAIT_S_MOLD + randint0(6);
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);

	if(type < TRAIT_S_MOLD) type = TRAIT_S_MOLD;
	else if(type > TRAIT_S_MIMIC) type = TRAIT_S_MIMIC;

	switch (randint1(spell) + randint1(8) + 1)
	{
	case 1:
	case 2:
	case 3:
		teleport_creature(caster_ptr, 10, TELEPORT_PASSIVE); break;
	case 4:
	case 5:
	case 6:
		teleport_creature(caster_ptr, 100, TELEPORT_PASSIVE); break;
	case 7:
	case 8:
		teleport_creature(caster_ptr, 200, TELEPORT_PASSIVE); break;
	case 9:
	case 10:
	case 11:
		unlite_area(caster_ptr, 10, 3); break;
	case 12:
	case 13:
	case 14:
		lite_area(caster_ptr, diceroll(2, 3), 2);
		break;
	case 15:
		project(caster_ptr, 0, 1, caster_ptr->fy, caster_ptr->fx, 0, DO_EFFECT_KILL_DOOR, PROJECT_GRID | PROJECT_ITEM | PROJECT_HIDE, -1);
		break;
	case 16: case 17:
		wall_breaker(caster_ptr);
	case 18:
		project(caster_ptr, 0, 1, caster_ptr->fy, caster_ptr->fx, caster_ptr->lev, DO_EFFECT_SLEEP, PROJECT_KILL | PROJECT_HIDE, -1);
		break;
	case 19:
	case 20:
		project(caster_ptr, 0, 1, caster_ptr->fy, caster_ptr->fx, 0, DO_EFFECT_MAKE_TRAP, PROJECT_GRID | PROJECT_ITEM | PROJECT_HIDE, -1);
		break;
	case 21:
	case 22:
		project(caster_ptr, 0, 1, caster_ptr->fy, caster_ptr->fx, 0, DO_EFFECT_MAKE_DOOR, PROJECT_GRID | PROJECT_ITEM | PROJECT_HIDE, -1);
		break;
	case 23:
	case 24:
	case 25:
		aggravate_creatures(caster_ptr);
		break;
	case 26:
		earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 5);
		break;
	case 27:
	case 28:
		(void)get_mutative_trait(caster_ptr, 0, TRUE);
		break;
	case 29:
	case 30: apply_disenchant(caster_ptr, TRUE); break;
	case 31: lose_all_info(caster_ptr); break;
	case 32: cast_ball(caster_ptr, DO_EFFECT_CHAOS, MAX_RANGE_SUB, spell + 5, 1 + (spell / 10)); break;
	case 33: wall_stone(caster_ptr); break;
	case 34:
	case 35:
		while (counter++ < 8) (void)summoning(0, caster_ptr->fy, caster_ptr->fx, (floor_ptr->depth * 3) / 2, type, (PC_ALLOW_GROUP | PC_NO_PET));
		break;
	case 36:
	case 37: summoning(NULL, caster_ptr->fy, caster_ptr->fx, floor_ptr->depth, TRAIT_S_MONSTER, (PC_ALLOW_GROUP | PC_ALLOW_UNIQUE | PC_NO_PET)); break;
	case 38: summoning(NULL, caster_ptr->fy, caster_ptr->fx, 150, TRAIT_S_CYBER, (PC_ALLOW_GROUP)); break;
	default:
		{
			int count = 0;
			(void)activate_ty_curse(caster_ptr, FALSE, &count);
			break;
		}
	}

	return;
}


void cast_shuffle(creature_type *caster_ptr)
{
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);
	int lev_bonus = caster_ptr->lev;
	int die;
	// TODO: Add Karma of Fortune feature.
	int i;

	// Card sharks and high mages get a level bonus
	if((caster_ptr->class_idx == CLASS_ROGUE) || (caster_ptr->class_idx == CLASS_HIGH_MAGE) || (caster_ptr->class_idx == CLASS_SORCERER))
		die = (randint1(110)) + lev_bonus / 5;
	else
		die = randint1(120);

	msg_print(SHUFFLE_START);

	if(die < 7)
	{
		msg_print(SHUFFLE_DEATH);
		for (i = 0; i < randint1(3); i++)
		{
			summoning(NULL, caster_ptr->fy, caster_ptr->fx, floor_ptr->depth, TRAIT_S_MONSTER, (PC_ALLOW_GROUP | PC_ALLOW_UNIQUE | PC_NO_PET));
		}
	}
	else if(die < 14)
	{
		msg_print(SHUFFLE_DEVIL);
		summoning(0, caster_ptr->fy, caster_ptr->fx, floor_ptr->depth, TRAIT_S_DEMON, (PC_ALLOW_GROUP | PC_ALLOW_UNIQUE | PC_NO_PET));
	}
	else if(die < 18)
	{
		int count = 0;
		msg_print(SHUFFLE_HANGEDMAN);
		activate_ty_curse(caster_ptr, FALSE, &count);
	}
	else if(die < 22)
	{
		msg_print(SHUFFLE_DISCODE);
		aggravate_creatures(caster_ptr);
	}
	else if(die < 26)
	{
		msg_print(SHUFFLE_FOOL);
		do_dec_stat(caster_ptr, STAT_INT);
		do_dec_stat(caster_ptr, STAT_WIS);
	}
	else if(die < 30)
	{
		msg_print(SHUFFLE_STRANGE);
		trump_summoning(caster_ptr, 1, FALSE, caster_ptr->fy, caster_ptr->fx, (floor_ptr->depth * 3 / 2), (32 + randint1(6)), PC_ALLOW_GROUP | PC_ALLOW_UNIQUE);
	}
	else if(die < 33)
	{
		msg_print(SHUFFLE_MOON);
		unlite_area(caster_ptr, 10, 3);
	}
	else if(die < 38)
	{
		msg_print(SHUFFLE_WHEEL);
		wild_magic(caster_ptr, randint0(32));
	}
	else if(die < 40)
	{
		msg_print(SHUFFLE_TELEPORT);
		teleport_creature(caster_ptr, 10, TELEPORT_PASSIVE);
	}
	else if(die < 42)
	{
		msg_print(SHUFFLE_JUSTICE);
		do_active_trait(caster_ptr, TRAIT_BLESSING_SELF, FALSE, 100, 0L);
	}
	else if(die < 47)
	{
		msg_print(SHUFFLE_TELEPORT);
		teleport_creature(caster_ptr, 100, TELEPORT_PASSIVE);
	}
	else if(die < 52)
	{
		msg_print(SHUFFLE_TELEPORT);
		teleport_creature(caster_ptr, 200, TELEPORT_PASSIVE);
	}
	else if(die < 60)
	{
		msg_print(SHUFFLE_TOWER);
		wall_breaker(caster_ptr);
	}
	else if(die < 72)
	{
		msg_print(SHUFFLE_TEMPERANCE);
		project(caster_ptr, 0, 1, caster_ptr->fy, caster_ptr->fx, caster_ptr->lev, DO_EFFECT_SLEEP, PROJECT_KILL | PROJECT_HIDE, -1);
	}
	else if(die < 80)
	{
		msg_print(SHUFFLE_TOWER);
		earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 5);
	}
	else if(die < 82)
	{
		msg_print(SHUFFLE_FRIEND);
		trump_summoning(caster_ptr, 1, TRUE, caster_ptr->fy, caster_ptr->fx, (floor_ptr->depth * 3 / 2), TRAIT_S_MOLD, 0L);
	}
	else if(die < 84)
	{
		msg_print(SHUFFLE_FRIEND);
		trump_summoning(caster_ptr, 1, TRUE, caster_ptr->fy, caster_ptr->fx, (floor_ptr->depth * 3 / 2), TRAIT_S_BAT, 0L);
	}
	else if(die < 86)
	{
		msg_print(SHUFFLE_FRIEND);
		trump_summoning(caster_ptr, 1, TRUE, caster_ptr->fy, caster_ptr->fx, (floor_ptr->depth * 3 / 2), TRAIT_S_VORTEX, 0L);
	}
	else if(die < 88)
	{
		msg_print(SHUFFLE_FRIEND);
		trump_summoning(caster_ptr, 1, TRUE, caster_ptr->fy, caster_ptr->fx, (floor_ptr->depth * 3 / 2), TRAIT_S_CREEPING_COIN, 0L);
	}
	else if(die < 96)
	{
		msg_print(SHUFFLE_LOVER);
		cast_ball(caster_ptr, DO_EFFECT_CHARM, MAX_RANGE_SUB, MIN(caster_ptr->lev, 20), 0);
	}
	else if(die < 101)
	{
		msg_print(SHUFFLE_HERMIT);
		wall_stone(caster_ptr);
	}
	else if(die < 111)
	{
		msg_print(SHUFFLE_JUDGEMENT);
		remove_all_mutative_traits(caster_ptr);
		do_cmd_rerate(caster_ptr, FALSE);
	}
	else if(die < 120)
	{
		msg_print(SHUFFLE_SUN);
		wiz_lite(floor_ptr, caster_ptr, FALSE);
	}
	else
	{
		msg_print(SHUFFLE_WORLD);
		if(caster_ptr->exp < CREATURE_MAX_EXP)
		{
			s32b ee = (caster_ptr->exp / 25) + 1;
			if(ee > 5000) ee = 5000;
			gain_exp(caster_ptr, ee, 0, TRUE);
		}
	}
}


// Drop 10+1d10 meteor ball at random places near the player
void cast_meteor(creature_type *caster_ptr, POWER dam, COODINATES rad)
{
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);
	int i, d;
	int b = 10 + randint1(10);
	COODINATES y = 0, x = 0;
	COODINATES dy, dx;
	int count;

	for (i = 0; i < b; i++)
	{
		for (count = 0; count <= 20; count++)
		{

			x = caster_ptr->fx - 8 + (COODINATES)randint0(17);
			y = caster_ptr->fy - 8 + (COODINATES)randint0(17);

			dx = (caster_ptr->fx > x) ? (caster_ptr->fx - x) : (x - caster_ptr->fx);
			dy = (caster_ptr->fy > y) ? (caster_ptr->fy - y) : (y - caster_ptr->fy);

			/* Approximate distance */
			d = (dy > dx) ? (dy + (dx >> 1)) : (dx + (dy >> 1));

			if(d >= 9) continue;

			if(!IN_BOUNDS(floor_ptr, y, x) || !projectable(floor_ptr, MAX_RANGE, caster_ptr->fy, caster_ptr->fx, y, x)
			    || !CAVE_HAVE_FLAG_BOLD(floor_ptr, y, x, FF_PROJECT)) continue;

			/* Valid position */
			break;
		}

		if(count > 20) continue;

		project(caster_ptr, 0, rad, y, x, dam, DO_EFFECT_METEOR, PROJECT_KILL | PROJECT_JUMP | PROJECT_ITEM, -1);
	}
}


// Drop 10+1d10 disintegration ball at random places near the target
bool cast_wrath_of_the_god(creature_type *creature_ptr, POWER dam, COODINATES rad)
{
	floor_type *floor_ptr = GET_FLOOR_PTR(creature_ptr);
	COODINATES x, y, tx, ty, nx, ny;
	DIRECTION dir;
	int i;
	int b = 10 + randint1(10);

	if(!get_aim_dir(creature_ptr, MAX_RANGE_SUB, &dir)) return FALSE;

	/* Use the given direction */
	tx = creature_ptr->fx + 99 * ddx[dir];
	ty = creature_ptr->fy + 99 * ddy[dir];

	/* Hack -- Use an actual "target" */
	if((dir == 5) && target_okay(creature_ptr))
	{
		tx = target_col;
		ty = target_row;
	}

	x = creature_ptr->fx;
	y = creature_ptr->fy;

	while(TRUE)
	{
		/* Hack -- Stop at the target */
		if((y == ty) && (x == tx)) break;

		ny = y;
		nx = x;
		mmove2(&ny, &nx, creature_ptr->fy, creature_ptr->fx, ty, tx);

		/* Stop at maximum range */
		if(MAX_RANGE <= distance(creature_ptr->fy, creature_ptr->fx, ny, nx)) break;

		/* Stopped by walls/doors */
		if(!CAVE_HAVE_FLAG_BOLD(floor_ptr, ny, nx, FF_PROJECT)) break;

		/* Stopped by creatures */
		if((dir != 5) && floor_ptr->cave[ny][nx].creature_idx != 0) break;

		/* Save the new location */
		x = nx;
		y = ny;
	}
	tx = x;
	ty = y;

	for (i = 0; i < b; i++)
	{
		int count = 20, d = 0;

		while (count--)
		{
			COODINATES dx, dy;

			x = tx - 5 + (COODINATES)randint0(11);
			y = ty - 5 + (COODINATES)randint0(11);

			dx = (tx > x) ? (tx - x) : (x - tx);
			dy = (ty > y) ? (ty - y) : (y - ty);

			/* Approximate distance */
			d = (dy > dx) ? (dy + (dx >> 1)) : (dx + (dy >> 1));
			/* Within the radius */
			if(d < 5) break;
		}

		if(count < 0) continue;

		/* Cannot penetrate perm walls */
		if(!IN_BOUNDS(floor_ptr, y, x) ||
		    cave_stop_disintegration(floor_ptr, y, x) ||
		    !in_disintegration_range(floor_ptr, ty, tx, y, x)) continue;
		project(creature_ptr, 0, rad, y, x, dam, DO_EFFECT_DISINTEGRATE, PROJECT_JUMP | PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL, -1);
	}

	return TRUE;
}

// An "item_tester_hook" for offer
static bool item_tester_offer(creature_type *creature_ptr, object_type *object_ptr)
{
	if(creature_ptr && !is_valid_creature(creature_ptr)) return FALSE;
	/* Flasks of oil are okay */
	if(object_ptr->tval != TV_CORPSE) return FALSE;
	if(object_ptr->sval != SV_CORPSE) return FALSE;
	if(my_strchr("pht", species_info[object_ptr->pval].d_char)) return TRUE;

	return FALSE; /* Assume not okay */
}

/*
 * Daemon spell Summon Greater Demon
 */
bool cast_summon_greater_demon(creature_type *creature_ptr)
{
	int lev_bonus = creature_ptr->lev;
	OBJECT_ID item;
	FLOOR_LEV summon_lev;
	object_type *object_ptr;

	if(!get_item(creature_ptr, &item, MES_OBJECT_WHICH_CORPSE, MES_OBJECT_NO_CORPSE, (USE_INVEN | USE_FLOOR), item_tester_offer, 0)) return FALSE;
	object_ptr = GET_ITEM(creature_ptr, item);

	summon_lev = lev_bonus * 2 / 3 + species_info[object_ptr->pval].level;

	if(summoning(NULL, creature_ptr->fy, creature_ptr->fx, summon_lev, TRAIT_S_HI_DEMON, (PC_ALLOW_GROUP | PC_FORCE_PET)))
	{
		msg_print(MES_TRAP_S_H_DEMON);
		msg_print(MES_SUMMON_SERVANT);
		increase_item(creature_ptr, item, -1, TRUE);
	}
	else msg_print(MES_SUMMON_NOTHING);
	return TRUE;
}

// Start singing if the player is a Bard 
static void start_singing(creature_type *creature_ptr, int spell, int song)
{
	creature_ptr->now_singing = song;
	creature_ptr->singing_turn = spell;

	set_action(creature_ptr, ACTION_SING);
	prepare_update(creature_ptr, CRU_BONUS);
	prepare_redraw(PR_STATUS);
}


/*
 * Stop singing if the player is a Bard 
 */
void stop_singing(creature_type *creature_ptr)
{
	if(creature_ptr->class_idx != CLASS_BARD) return;

 	/* Are there interupted song? */
	if(creature_ptr->pre_singing) /* Forget interupted song */
	{
		creature_ptr->pre_singing = 0;
		return;
	}

	/* The player is singing? */
	if(!creature_ptr->now_singing) return;

	/* Hack -- if called from set_action(creature_ptr, ), avoid recursive loop */
	if(creature_ptr->action == ACTION_SING) set_action(creature_ptr, ACTION_NONE);

	/* Message text of each song or etc. */
	do_spell(creature_ptr, REALM_MUSIC, creature_ptr->singing_turn, SPELL_STOP);

	creature_ptr->now_singing = MUSIC_NONE;
	creature_ptr->singing_turn = 0;

	prepare_update(creature_ptr, CRU_BONUS);

	prepare_redraw(PR_STATUS);
}


static cptr do_life_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	int lev_bonus = caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "y̎";
		if(desc) return "Ƒ̗͂񕜂B";
#else
		if(name) return "Cure Light Wounds";
		if(desc) return "Heals cut and HP a little.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(2, 10), 0L);
		break;

	case 1:
#ifdef JP
		if(name) return "j";
		if(desc) return "莞ԁAACɃ{[iX𓾂B";
#else
		if(name) return "Bless";
		if(desc) return "Gives bonus to hit and AC for a few turns.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_BLESSING_SELF, TRUE, 100, 0L);
		break;

	case 2:
#ifdef JP
		if(name) return "y";
		if(desc) return "1̂̃N[`[ɏ_[W^BRƖB";
#else
		if(name) return "Cause Light Wounds";
		if(desc) return "Wounds a creature a little unless resisted.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_CAUSE_1, TRUE, 100, 0L);
		break;

	case 3:
#ifdef JP
		if(name) return "̏";
		if(desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if(name) return "Call Light";
		if(desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_LIGHT_AREA, TRUE, 100, 0L);
		break;

	case 4:
#ifdef JP
		if(name) return " & Bm";
		if(desc) return "߂̑SĂ㩂ƔƊKimB";
#else
		if(name) return "Detect Doors & Traps";
		if(desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_DOOR_TRAP, TRUE, 100, 0L);
		break;

	case 5:
#ifdef JP
		if(name) return "d̎";
		if(desc) return "Ƒ̗͂𒆒x񕜂B";
#else
		if(name) return "Cure Medium Wounds";
		if(desc) return "Heals cut and HP more.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(4, 10), 0L);
		break;

	case 6:
#ifdef JP
		if(name) return "";
		if(desc) return "̓̓ł菜B";
#else
		if(name) return "Cure Poison";
		if(desc) return "Cure poison status.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_POISON, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "󕠏[";
		if(desc) return "ɂB";
#else
		if(name) return "Satisfy Hunger";
		if(desc) return "Satisfies hunger.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SATIATE, TRUE);
		break;

	case 8:
#ifdef JP
		if(name) return "";
		if(desc) return "ACeɂア􂢂B";
#else
		if(name) return "Remove Curse";
		if(desc) return "Removes normal curses from equipped items.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_CURSE_1, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "d";
		if(desc) return "1̂̃N[`[ɒ_[W^BRƖB";
#else
		if(name) return "Cause Medium Wounds";
		if(desc) return "Wounds a creature unless resisted.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_CAUSE_2, TRUE, 100, 0L);
		break;

	case 10:
#ifdef JP
		if(name) return "v̎";
		if(desc) return "̗͂啝ɉ񕜂AƞNOԂSB";
#else
		if(name) return "Cure Critical Wounds";
		if(desc) return "Heals cut, stun and HP greatly.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(8, 10), 0L);
		break;

	case 11:
#ifdef JP
		if(name) return "ϔMϊ";
		if(desc) return "莞ԁAΉƗCɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Heat and Cold";
		if(desc) return "Gives resistance to fire and cold. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_RES_COLD, TRUE, 20, 0L);
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE, 20, 0L);
		break;

	case 12:
#ifdef JP
		if(name) return "ӊm";
		if(desc) return "ӂ̒n`mB";
#else
		if(name) return "Sense Surroundings";
		if(desc) return "Maps nearby area.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_MAP, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "pjbNEAfbh";
		if(desc) return "ẼAfbh|BRƖB";
#else
		if(name) return "Turn Undead";
		if(desc) return "Attempts to scare undead creatures in sight.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TURN_UNDEAD, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "̗͉";
		if(desc) return "ɂ߂ċ͂ȉ񕜎ŁAƞNOԂSB";
#else
		if(name) return "Healing";
		if(desc) return "Much powerful healing magic, and heals cut and stun completely.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, 300, 0L);
		break;

	case 15:
#ifdef JP
		if(name) return "E̖";
		if(desc) return "̂鏰̏ɁAN[`[ʂ蔲菢ꂽ肷邱ƂłȂȂ郋[`B";
#else
		if(name) return "Glyph of Warding";
		if(desc) return "Sets a glyph on the floor beneath you. Creatures cannot attack you if you are on a glyph, but can try to break glyph.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_PROTECT_RUNE, TRUE, 300, 0L);
		break;

	case 16:
#ifdef JP
		if(name) return "**";
		if(desc) return "ACeɂ͂Ȏ􂢂B";
#else
		if(name) return "Dispel Curse";
		if(desc) return "Removes normal and heavy curse from equipped items.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_CURSE_2, TRUE);
		break;

	case 17:
#ifdef JP
		if(name) return "ӎ";
		if(desc) return "ACeʂB";
#else
		if(name) return "Perception";
		if(desc) return "Identifies an item.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_IDENTIFY, TRUE, 100, 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "AfbhގU";
		if(desc) return "ȆSẴAfbhɃ_[W^B";
#else
		if(name) return "Dispel Undead";
		if(desc) return "Damages all undead creatures in sight.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DISPEL_UNDEAD, TRUE, lev_bonus * 5, 0L);
		break;

	case 19:
#ifdef JP
		if(name) return "̍";
		if(desc) return "ȆSẴN[`[𖣗BRƖB";
#else
		if(name) return "Day of the Dove";
		if(desc) return "Attempts to charm all creatures in sight.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DOMINATE_LIVES, TRUE, 100, 0L);
		break;

	case 20:
#ifdef JP
		if(name) return "v";
		if(desc) return "1̂̃N[`[ɑ_[W^BRƖB";
#else
		if(name) return "Cause Critical Wounds";
		if(desc) return "Wounds a creature critically unless resisted.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_CAUSE_3, TRUE, 100, 0L);
		break;

	case 21:
#ifdef JP
		if(name) return "A҂̏";
		if(desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if(name) return "Word of Recall";
		if(desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_RECALL, TRUE, 100, 0L);
		break;

	case 22:
#ifdef JP
		if(name) return "^̍Ւd";
		if(desc) return "݂̊Kč\B";
#else
		if(name) return "Alter Reality";
		if(desc) return "Recreates current dungeon level.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_SHADOW_SHIFT, TRUE, 100, 0L);
		break;

	case 23:
#ifdef JP
		if(name) return "^EE";
		if(desc) return "̂鏰Ǝ8}X̏̏ɁAN[`[ʂ蔲菢ꂽ肷邱ƂłȂȂ郋[`B";
#else
		if(name) return "Warding True";
		if(desc) return "Creates glyphs in all adjacent squares and under you.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_PROTECT_RUNE, TRUE, 2, 0L);
		break;

	case 24:
#ifdef JP
		if(name) return "sщ";
		if(desc) return "̊K̑BN[`[BłȂȂB";
#else
		if(name) return "Sterilization";
		if(desc) return "Prevents any breeders on current level from breeding.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STERILITY, TRUE, 100, 0L);
		break;

	case 25:
#ifdef JP
		if(name) return "Sm";
		if(desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if(name) return "Detection";
		if(desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_ALL, TRUE, 100, 0L);
		break;

	case 26:
#ifdef JP
		if(name) return "Afbh";
		if(desc) return "̎͂ɂAfbh݂̊KBRƖB";
#else
		if(name) return "Annihilate Undead";
		if(desc) return "Eliminates all nearby undead creatures, exhausting you.  Powerful or unique creatures may be able to resist.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_GENOCIDE_UNDEAD, TRUE, lev_bonus + 50, 0L);
		break;

	case 27:
#ifdef JP
		if(name) return "痢";
		if(desc) return "̊KŜivɏƂ炵A_WׂẴACemB";
#else
		if(name) return "Clairvoyance";
		if(desc) return "Maps and lights whole dungeon level. Knows all objects location. And gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_ENLIGHTENMENT, TRUE, 100, 0L);
		break;

	case 28:
#ifdef JP
		if(name) return "S";
		if(desc) return "ׂẴXe[^Xƌol񕜂B";
#else
		if(name) return "Restoration";
		if(desc) return "Restores all stats and experience.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_RESTORE_ALL, TRUE, 100, 0L);
		break;

	case 29:
#ifdef JP
		if(name) return "*̗͉*";
		if(desc) return "ŋ̖̎@ŁAƞNOԂSB";
#else
		if(name) return "Healing True";
		if(desc) return "The greatest healing magic. Heals all HP, cut and stun.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, 2000, 0L);
		break;

	case 30:
#ifdef JP
		if(name) return "ȂrW";
		if(desc) return "ACe̎\͂SɒmB";
#else
		if(name) return "Holy Vision";
		if(desc) return "*Identifies* an item.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_IDENTIFY_TRUE, TRUE, 100, 0L);
		break;

	case 31:
#ifdef JP
		if(name) return "ɂ̑ϐ";
		if(desc) return "莞ԁAϐtAACƖ@h\͂㏸B";
#else
		if(name) return "Ultimate Resistance";
		if(desc) return "Gives ultimate resistance, bonus to AC and speed.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_GET_ULTRA_RESISTANCE, TRUE, lev_bonus / 2, 0L);
		break;

	}

	return "";
}


static cptr do_sorcery_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	POWER lev_bonus = (POWER)caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "N[`[m";
		if(desc) return "߂̑SĂ̌N[`[mB";
#else
		if(name) return "Detect Creatures";
		if(desc) return "Detects all creatures in your vicinity unless invisible.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SMELL_MON, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "V[gEe|[g";
		if(desc) return "ߋ̃e|[gB";
#else
		if(name) return "Phase Door";
		if(desc) return "Teleport short distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BLINK, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "㩂Ɣm";
		if(desc) return "߂̑SĂ̔㩂mB";
#else
		if(name) return "Detect Doors and Traps";
		if(desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_DOOR_TRAP, TRUE, 100, 0L);
		break;

	case 3:
#ifdef JP
		if(name) return "CgEGA";
		if(desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if(name) return "Light Area";
		if(desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_LIGHT_AREA, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "pjbNEN[`[";
		if(desc) return "N[`[1̂BRƖB";
#else
		if(name) return "Confuse Creature";
		if(desc) return "Attempts to confuse a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CONF, TRUE);
		break;

	case 5:
#ifdef JP
		if(name) return "e|[g";
		if(desc) return "̃e|[gB";
#else
		if(name) return "Teleport";
		if(desc) return "Teleport long distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ACTIVE_TELEPORT, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "X[vEN[`[";
		if(desc) return "N[`[1̂𖰂点BRƖB";
#else
		if(name) return "Sleep Creature";
		if(desc) return "Attempts to sleep a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SLEPT, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "͏[U";
		if(desc) return "/@_̏[U񐔂𑝂₷A[Ũbh̏[UԂ炷B";
#else
		if(name) return "Recharging";
		if(desc) return "Recharges staffs, wands or rods.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_CHARGE_2, TRUE, lev_bonus * 4, 0L);
		break;

	case 8:
#ifdef JP
		if(name) return "@̒n}";
		if(desc) return "ӂ̒n`mB";
#else
		if(name) return "Magic Mapping";
		if(desc) return "Maps nearby area.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_MAP, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "Ӓ";
		if(desc) return "ACeʂB";
#else
		if(name) return "Identify";
		if(desc) return "Identifies an item.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "XEEN[`[";
		if(desc) return "N[`[1̂BRƖB";
#else
		if(name) return "Slow Creature";
		if(desc) return "Attempts to slow a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SLOW, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "ӃX[v";
		if(desc) return "ȆSẴN[`[𖰂点BRƖB";
#else
		if(name) return "Mass Sleep";
		if(desc) return "Attempts to sleep all creatures in sight.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MASS_SLEEP, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "e|[gEN[`[";
		if(desc) return "N[`[e|[gr[BRƖB";
#else
		if(name) return "Teleport Away";
		if(desc) return "Teleports all creatures on the line away unless resisted.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_AWAY, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "Xs[h";
		if(desc) return "莞ԁAB";
#else
		if(name) return "Haste Self";
		if(desc) return "Hastes you for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HASTE, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "^Em";
		if(desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if(name) return "Detection True";
		if(desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_ALL, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "^EӒ";
		if(desc) return "ACe̎\͂SɒmB";
#else
		if(name) return "Identify True";
		if(desc) return "*Identifies* an item.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY_TRUE, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "̂ƍ󊴒m";
		if(desc) return "߂̑SẴACeƍmB";
#else
		if(name) return "Detect items and Treasure";
		if(desc) return "Detects all treasures and items in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_ITEM_GOLD, TRUE, 100, 0L);
		break;

	case 17:
#ifdef JP
		if(name) return "`[EN[`[";
		if(desc) return "N[`[1̂𖣗BRƖB";
#else
		if(name) return "Charm Creature";
		if(desc) return "Attempts to charm a creature.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DOMINATE_LIVE, TRUE, 100, 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "_m";
		if(desc) return "莞ԁAepV[\͂𓾂B";
#else
		if(name) return "Sense Minds";
		if(desc) return "Gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_GET_ESP, TRUE, 50, 0L);
		break;

	case 19:
#ifdef JP
		if(name) return "Xړ";
		if(desc) return "XֈړBnɂƂgȂB";
#else
		if(name) return "Teleport to town";
		if(desc) return "Teleport to a town which you choose in a moment. Can only be used outdoors.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_TELEPORT_TOWN, TRUE, 50, 0L);
		break;

	case 20:
#ifdef JP
		if(name) return "ȕ";
		if(desc) return "݂̎̏ԂSɒmB";
#else
		if(name) return "Self Knowledge";
		if(desc) return "Gives you useful info regarding your current resistances, the powers of your weapon and maximum limits of your stats.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SELF_KNOWLEDGE, TRUE);
		break;

	case 21:
#ifdef JP
		if(name) return "e|[gEx";
		if(desc) return "uɏォ̊KɃe|[gB";
#else
		if(name) return "Teleport Level";
		if(desc) return "Teleport to up or down stairs in a moment.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_LEVEL, TRUE);
		break;

	case 22:
#ifdef JP
		if(name) return "A҂̎";
		if(desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if(name) return "Word of Recall";
		if(desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RECALL, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "̔";
		if(desc) return "Z̎w肵ꏊɃe|[gB";
#else
		if(name) return "Dimension Door";
		if(desc) return "Teleport to given location.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DIMENSION_DOOR, TRUE, lev_bonus / 2 + 10, 0L);
		break;

	case 24:
#ifdef JP
		if(name) return "";
		if(desc) return "N[`[̑Ać̗Aő̗́AXs[hÂmB";
#else
		if(name) return "Probing";
		if(desc) return "Proves all creatures' alignment, HP, speed and their true character.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SCAN_CREATURE, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "̃[";
		if(desc) return "̂鏰̏ɁAN[`[ʂƔă_[W^郋[`B";
#else
		if(name) return "Explosive Rune";
		if(desc) return "Sets a glyph under you. The glyph will explode when a creature moves on it.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_EXPLOSIVE_RUNE, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "O";
		if(desc) return "ACȇֈړB";
#else
		if(name) return "Telekinesis";
		if(desc) return "Pulls a distant item close to you.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_TELEKINES, TRUE, lev_bonus * 15, 0L);
		break;

	case 27:
#ifdef JP
		if(name) return "痢";
		if(desc) return "̊KŜivɏƂ炵A_WׂẴACemBɁA莞ԃepV[\͂𓾂B";
#else
		if(name) return "Clairvoyance";
		if(desc) return "Maps and lights whole dungeon level. Knows all objects location. And gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_ENLIGHTENMENT, TRUE, 100, 0L);
		break;

	case 28:
#ifdef JP
		if(name) return "̎";
		if(desc) return "ȆSẴN[`[𖣗BRƖB";
#else
		if(name) return "Charm creatures";
		if(desc) return "Attempts to charm all creatures in sight.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DOMINATE_LIVES, TRUE, 100, 0L);
		break;

	case 29:
#ifdef JP
		if(name) return "Bp";
		if(desc) return "ACe1ɕςB";
#else
		if(name) return "Alchemy";
		if(desc) return "Turns an item into 1/3 of its value in gold.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MIDAS_TCH, TRUE, 100, 0L);
		break;

	case 30:
#ifdef JP
		if(name) return "Ǖ";
		if(desc) return "ȆSẴN[`[e|[gBRƖB";
#else
		if(name) return "Banishment";
		if(desc) return "Teleports all creatures in sight away unless resisted.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MASS_TELE_AWAY, TRUE, 100, 0L);
		break;

	case 31:
#ifdef JP
		if(name) return "̋";
		if(desc) return "莞ԁA_[W󂯂ȂȂoA𒣂B؂ꂽuԂɏ^[̂ŒӁB";
#else
		if(name) return "Globe of Invulnerability";
		if(desc) return "Generates barrier which completely protect you from almost all damages. Takes a few your turns when the barrier breaks or duration time is exceeded.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_INVULNER, TRUE, 100, 0L);
	}

	return "";
}


static cptr do_nature_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	static const char * s_dam = "KW_DAM";
	static const char * s_rng = "KW_RAN";

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "N[`[m";
		if(desc) return "߂̑SĂ̌N[`[mB";
#else
		if(name) return "Detect Creatures";
		if(desc) return "Detects all creatures in your vicinity unless invisible.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SMELL_MON, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "";
		if(desc) return "d̒Zr[B";
#else
		if(name) return "Lightning";
		if(desc) return "Fires a short beam of lightning.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_LIGHTNING_BEAM, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "㩂Ɣm";
		if(desc) return "߂̑SĂ㩂ƔmB";
#else
		if(name) return "Detect Doors and Traps";
		if(desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_DOOR_TRAP, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "HƐ";
		if(desc) return "HoB";
#else
		if(name) return "Produce Food";
		if(desc) return "Produces a Ration of Food.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CREATE_FOOD, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "̌";
		if(desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if(name) return "Daylight";
		if(desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_LIGHT_AREA, TRUE, 100, 0L);
		break;

	case 5:
#ifdef JP
		if(name) return "K";
		if(desc) return "1̂𖣗BRƖB";
#else
		if(name) return "Animal Taming";
		if(desc) return "Attempts to charm an animal.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_CHARM_ANIMAL, TRUE, 100, 0L);
		break;

	case 6:
#ifdef JP
		if(name) return "ւ̑ϐ";
		if(desc) return "莞ԁACAAdɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Environment";
		if(desc) return "Gives resistance to fire, cold and electricity for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_RES_COLD, TRUE, 20, 0L);
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE, 20, 0L);
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_RES_ELEC, TRUE, 20, 0L);
		break;

	case 7:
#ifdef JP
		if(name) return "ƓŎ";
		if(desc) return "SAł̂犮SɎ菜A̗͂񕜂B";
#else
		if(name) return "Cure Wounds & Poison";
		if(desc) return "Heals all cut and poison status. Heals HP a little.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, 20, 0L);
		break;

	case 8:
#ifdef JP
		if(name) return "Ηn";
		if(desc) return "ǂnďɂB";
#else
		if(name) return "Stone to Mud";
		if(desc) return "Turns one rock square to mud.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STONE_TO_MUD, TRUE, 100, 0L);
		break;

	case 9:
#ifdef JP
		if(name) return "ACXE{g";
		if(desc) return "C̃{g̓r[B";
#else
		if(name) return "Frost Bolt";
		if(desc) return "Fires a bolt or beam of cold.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_COLD, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "R̊o";
		if(desc) return "ӂ̒n`mA߂㩁AAKiASẴN[`[mB";
#else
		if(name) return "Nature Awareness";
		if(desc) return "Maps nearby area. Detects all creatures, traps, doors and stairs.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_NATURE_AWARENESS, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "t@CAE{g";
		if(desc) return "Ή̃{g̓r[B";
#else
		if(name) return "Fire Bolt";
		if(desc) return "Fires a bolt or beam of fire.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_FIRE, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "z";
		if(desc) return "BN[`[ɌʂB";
#else
		if(name) return "Ray of Sunlight";
		if(desc) return "Fires a beam of light which damages to light-sensitive creatures.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SUNLIGHT, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "";
		if(desc) return "ȆSẴN[`[BRƖB";
#else
		if(name) return "Entangle";
		if(desc) return "Attempts to slow all creatures in sight.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MASS_SLOW, TRUE, 100, 0L);
		break;

	case 14:
#ifdef JP
		if(name) return "";
		if(desc) return "1̏B";
#else
		if(name) return "Summon Animal";
		if(desc) return "Summons an animal.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_S_ANIMAL, TRUE, 100, 0L);
		break;

	case 15:
#ifdef JP
		if(name) return "򑐎";
		if(desc) return "̗͂啝ɉ񕜂AANOԁAłSB";
#else
		if(name) return "Herbal Healing";
		if(desc) return "Heals HP greatly. And heals cut, stun and poison completely.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, 500, 0L);
		break;

	case 16:
#ifdef JP
		if(name) return "Ki";
		if(desc) return "̂ʒuɊKiB";
#else
		if(name) return "Stair Building";
		if(desc) return "Creates a stair which goes down or up.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STAIR_BUILDING, TRUE, 0, 0L);
		break;

	case 17:
#ifdef JP
		if(name) return "Ή";
		if(desc) return "莞ԁAAC㏸B";
#else
		if(name) return "Stone Skin";
		if(desc) return "Gives bonus to AC for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STONE_SKIN, TRUE, randint1(20) + 30, 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "^Eϐ";
		if(desc) return "莞ԁA_AdAACAłɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resistance True";
		if(desc) return "Gives resistance to fire, cold, electricity, acid and poison for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_ELEMENT, TRUE);
		break;

	case 19:
#ifdef JP
		if(name) return "Xёn";
		if(desc) return "͂ɖ؂oB";
#else
		if(name) return "Forest Creation";
		if(desc) return "Creates trees in all adjacent squares.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CREATE_FOREST, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "Fa";
		if(desc) return "ȆSĂ̓𖣗BRƖB";
#else
		if(name) return "Animal Friendship";
		if(desc) return "Attempts to charm all animals in sight.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CHARM_ANIMALS, TRUE);
		break;

	case 21:
#ifdef JP
		if(name) return "";
		if(desc) return "ACe̎\͂SɒmB";
#else
		if(name) return "Stone Tell";
		if(desc) return "*Identifies* an item.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY_TRUE, TRUE);
		break;

	case 22:
#ifdef JP
		if(name) return "΂̕";
		if(desc) return "̎͂ɉԛ̕ǂB";
#else
		if(name) return "Wall of Stone";
		if(desc) return "Creates granite walls in all adjacent squares.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_STONE_WALL, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "Hh~";
		if(desc) return "ACe_ŏȂ悤HB";
#else
		if(name) return "Protect from Corrosion";
		if(desc) return "Makes an equipment acid-proof.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RUSTPROOF, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "nk";
		if(desc) return "͂̃_Wh炵AǂƏ_ɓςB";
#else
		if(name) return "Earthquake";
		if(desc) return "Shakes dungeon structure, and results in random swapping of floors and walls.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_EARTHQUAKE, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "J}C^`";
		if(desc) return "SɌčUB";
#else
		if(name) return "Cyclone";
		if(desc) return "Attacks all adjacent creatures.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MASSACRE, TRUE, 100, 0L);
		break;

	case 26:
#ifdef JP
		if(name) return "uU[h";
		if(desc) return "ȗC̋B";
#else
		if(name) return "Blizzard";
		if(desc) return "Fires a huge ball of cold.";
#endif
    
		break;

	case 27:
#ifdef JP
		if(name) return "ȗ";
		if(desc) return "ȓd̋B";
#else
		if(name) return "Lightning Storm";
		if(desc) return "Fires a huge electric ball.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_LIGHTNING_STORM, TRUE);
		break;

	case 28:
#ifdef JP
		if(name) return "Q";
		if(desc) return "Ȑ̋B";
#else
		if(name) return "Whirlpool";
		if(desc) return "Fires a huge ball of water.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_WATER_STORM, TRUE);
		break;

	case 29:
#ifdef JP
		if(name) return "z";
		if(desc) return "𒆐SƂ̋𔭐BɁÅKŜivɏƂ炵A_WׂẴACemB";
#else
		if(name) return "Call Sunlight";
		if(desc) return "Generates ball of light centered on you. Maps and lights whole dungeon level. Knows all objects location.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CALL_SUNLIGHT, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "̐n";
		if(desc) return "ɉC̑B";
#else
		if(name) return "Elemental Branding";
		if(desc) return "Makes current weapon fire or frost branded.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_ELEMENTAL_BRAND, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "R̋";
		if(desc) return "߂̑SẴN[`[Ƀ_[W^AnkNA𒆐SƂ̋𔭐B";
#else
		if(name) return "Nature's Wrath";
		if(desc) return "Damages all creatures in sight. Makes quake. Generates disintegration ball centered on you.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_NATURE_WRATH, TRUE);
		break;
	}

	return "";
}


static cptr do_chaos_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	static const char s_dam[] = "KW_DAM";
	static const char s_random[] = "KW_RANDOM";

	COODINATES lev_bonus = caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "}WbNE~TC";
		if(desc) return "ア@̖B";
#else
		if(name) return "Magic Missile";
		if(desc) return "Fires a weak bolt of magic.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_MISSILE, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "gbv/hAj";
		if(desc) return "אڂ㩂Ɣj󂷂B";
#else
		if(name) return "Trap / Door Destruction";
		if(desc) return "Destroys all traps in adjacent squares.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DESTROY_DOOR_TRAP, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "M";
		if(desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if(name) return "Flash of Light";
		if(desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_LIGHT_AREA, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "̎";
		if(desc) return "Uł悤ɂB";
#else
		if(name) return "Touch of Confusion";
		if(desc) return "Attempts to confuse the next creature that you hit.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_CONFUSING_MELEE, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "y";
		if(desc) return "@̋B";
#else
		if(name) return "Mana Burst";
		if(desc) return "Fires a ball of magic.";
#endif
		break;

	case 5:
#ifdef JP
		if(name) return "t@CAE{g";
		if(desc) return "̃{g̓r[B";
#else
		if(name) return "Fire Bolt";
		if(desc) return "Fires a bolt or beam of fire.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_FIRE, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "͂̌";
		if(desc) return "ȕ̋B";
#else
		if(name) return "Fist of Force";
		if(desc) return "Fires a tiny ball of disintegration.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_FORCE_FIST, TRUE);

	case 7:
#ifdef JP
		if(name) return "e|[g";
		if(desc) return "̃e|[gB";
#else
		if(name) return "Teleport Self";
		if(desc) return "Teleport long distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ACTIVE_TELEPORT, TRUE);
		break;

	case 8:
#ifdef JP
		if(name) return "_[";
		if(desc) return "N[`[Ƀ_Ȍʂ^B";
#else
		if(name) return "Wonder";
		if(desc) return "Fires something with random effects.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_WANDER, TRUE, 100, 0L);
		break;

	case 9:
#ifdef JP
		if(name) return "JIXE{g";
		if(desc) return "JIX̃{g̓r[B";
#else
		if(name) return "Chaos Bolt";
		if(desc) return "Fires a bolt or ball of chaos.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_BO_CHAO, TRUE, 100, 0L);

		break;

	case 10:
#ifdef JP
		if(name) return "\jbNEu[";
		if(desc) return "𒆐SƂ̋𔭐B";
#else
		if(name) return "Sonic Boom";
		if(desc) return "Generates a ball of sound centered on you.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SONIC_BOOM, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "jł̖";
		if(desc) return "Ȗ͂̃r[B";
#else
		if(name) return "Doom Bolt";
		if(desc) return "Fires a beam of pure mana.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_MANA, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "t@CAE{[";
		if(desc) return "̋B";
#else
		if(name) return "Fire Ball";
		if(desc) return "Fires a ball of fire.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_FIRE, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "e|[gEAEFC";
		if(desc) return "N[`[e|[gr[BRƖB";
#else
		if(name) return "Teleport Other";
		if(desc) return "Teleports all creatures on the line away unless resisted.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_AWAY, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "ǰt";
		if(desc) return "ӂ̃ACeAN[`[An`j󂷂B";
#else
		if(name) return "Word of Destruction";
		if(desc) return "Destroy everything in nearby area.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_STAR_DESTROY, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "OX";
		if(desc) return "ȃJIX̋B";
#else
		if(name) return "Invoke Logrus";
		if(desc) return "Fires a huge ball of chaos.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_STORM_CHAOS, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "ҕϗe";
		if(desc) return "N[`[1̂ϐgBRƖB";
#else
		if(name) return "Polymorph Other";
		if(desc) return "Attempts to polymorph a creature.";
#endif
		if(info) return info_power(lev_bonus);
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_POLYMORPH_OTHER, TRUE);
		break;

	case 17:
#ifdef JP
		if(name) return "A";
		if(desc) return "Sɑ΂ēd̃r[B";
#else
		if(name) return "Chain Lightning";
		if(desc) return "Fires lightning beams in all directions.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_CHAIN_LIGHTNING, TRUE, 90, 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "͕";
		if(desc) return "/@_̏[U񐔂𑝂₷A[Ũbh̏[UԂ炷B";
#else
		if(name) return "Arcane Binding";
		if(desc) return "Recharges staffs, wands or rods.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_CHARGE_2, TRUE, 90, 0L);
		break;

	case 19:
#ifdef JP
		if(name) return "q";
		if(desc) return "ȕ̋B";
#else
		if(name) return "Disintegrate";
		if(desc) return "Fires a huge ball of disintegration.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_DISI, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "ϗe";
		if(desc) return "݂̊Kč\B";
#else
		if(name) return "Alter Reality";
		if(desc) return "Recreates current dungeon level.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SHADOW_SHIFT, TRUE);
		break;

	case 21:
#ifdef JP
		if(name) return "}WbNEPbg";
		if(desc) return "Pbg𔭎˂B";
#else
		if(name) return "Magic Rocket";
		if(desc) return "Fires a magic rocket.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ROCKET, TRUE);
		break;

	case 22:
#ifdef JP
		if(name) return "ׂ̐n";
		if(desc) return "ɃJIX̑B";
#else
		if(name) return "Chaos Branding";
		if(desc) return "Makes current weapon a Chaotic weapon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_CHAOS_BRAND, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "";
		if(desc) return "1̏B";
#else
		if(name) return "Summon Demon";
		if(desc) return "Summons a demon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_DEMON, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "d͌";
		if(desc) return "d͂̃r[B";
#else
		if(name) return "Beam of Gravity";
		if(desc) return "Fires a beam of gravity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BE_GRAV, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "Q";
		if(desc) return "̎ӂ覐΂𗎂ƂB";
#else
		if(name) return "Meteor Swarm";
		if(desc) return "Makes meteor balls fall down to nearby random locations.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_METEOR_SWARM, TRUE, lev_bonus * 2, 0L);
		break;

	case 26:
#ifdef JP
		if(name) return "̈ꌂ";
		if(desc) return "𒆐SƂȉ̋𔭐B";
#else
		if(name) return "Flame Strike";
		if(desc) return "Generate a huge ball of fire centered on you.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STORM_FIRE, TRUE, lev_bonus * 2, 0L);
		break;

	case 27:
#ifdef JP
		if(name) return "׏";
		if(desc) return "_ȑ̋r[𔭐B";
#else
		if(name) return "Call Chaos";
		if(desc) return "Generate random kind of balls or beams.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_CALL_CHAOS, TRUE, lev_bonus * 2, 0L);
		break;

	case 28:
#ifdef JP
		if(name) return "ȕϗe";
		if(desc) return "ϐg悤ƂB";
#else
		if(name) return "Polymorph Self";
		if(desc) return "Polymorphs yourself.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_POLYMORPH, TRUE, lev_bonus * 2, 0L);
		break;

	case 29:
#ifdef JP
		if(name) return "̗͂";
		if(desc) return "ɋ͂ŋȏȖ͂̋B";
#else
		if(name) return "Mana Storm";
		if(desc) return "Fires an extremely powerful huge ball of pure mana.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_BA_MANA, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "OX̃uX";
		if(desc) return "ɋ͂ȃJIX̋B";
#else
		if(name) return "Breathe Logrus";
		if(desc) return "Fires an extremely powerful ball of chaos.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_BA_CHAO, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "";
		if(desc) return "Ɏ͂ɌāAPbgAȖ͂̋Aːp̋BAǂɗאڂĎgpƍL͈͂j󂷂B";
#else
		if(name) return "Call the Void";
		if(desc) return "Fires rockets, mana balls and nuclear waste balls in all directions each unless you are not adjacent to any walls. Otherwise *destroys* huge area.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_CALL_THE_VOID, TRUE);
		break;
	}

	return "";
}


static cptr do_death_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	static const char s_dam[] = "KW_DAM";
	static const char s_random[] = "KW_RANDOM";

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "m";
		if(desc) return "߂̐̂ȂN[`[mB";
#else
		if(name) return "Detect Unlife";
		if(desc) return "Detects all nonliving creatures in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_NONLIVING, TRUE, 100, 0L);
		break;

	case 1:
#ifdef JP
		if(name) return "Ee";
		if(desc) return "Ȏ׈ȗ͂{[BPǂȃN[`[ɂ͑傫ȃ_[W^B";
#else
		if(name) return "Malediction";
		if(desc) return "Fires a tiny ball of evil power which hurts good creatures greatly.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MALEDICTION, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "׈m";
		if(desc) return "߂̎׈ȃN[`[mB";
#else
		if(name) return "Detect Evil";
		if(desc) return "Detects all evil creatures in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_EVIL, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "L_";
		if(desc) return "ł̋B";
#else
		if(name) return "Stinking Cloud";
		if(desc) return "Fires a ball of poison.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_POIS, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "";
		if(desc) return "1̂̃N[`[𖰂点BRƖB";
#else
		if(name) return "Black Sleep";
		if(desc) return "Attempts to sleep a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SLEPT, TRUE);
		break;

	case 5:
#ifdef JP
		if(name) return "ϓ";
		if(desc) return "莞ԁAłւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Poison";
		if(desc) return "Gives resistance to poison. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_POIS, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "Q";
		if(desc) return "N[`[1̂|ANOBRƖB";
#else
		if(name) return "Horrify";
		if(desc) return "Attempts to scare and stun a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SCARE, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "Afbh]";
		if(desc) return "Afbh1̂𖣗BRƖB";
#else
		if(name) return "Enslave Undead";
		if(desc) return "Attempts to charm an undead creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CHARM_UNDEAD, TRUE);
		break;

	case 8:
#ifdef JP
		if(name) return "Ggs[̋";
		if(desc) return "̂҂Ɍʂ̂鋅B";
#else
		if(name) return "Orb of Entropy";
		if(desc) return "Fires a ball which damages living creatures.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_DRAI, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "n̖";
		if(desc) return "ñ{g̓r[B";
#else
		if(name) return "Nether Bolt";
		if(desc) return "Fires a bolt or beam of nether.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_NETH, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "EC_";
		if(desc) return "𒆐SƂł̋𔭐B";
#else
		if(name) return "Cloud kill";
		if(desc) return "Generate a ball of poison centered on you.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_POIS, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "N[`[";
		if(desc) return "N[`[1̂BolACe͎ɓȂBRƖB";
#else
		if(name) return "Genocide One";
		if(desc) return "Attempts to vanish a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GENOCIDE_ONE, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "ł̐n";
		if(desc) return "ɓł̑B";
#else
		if(name) return "Poison Branding";
		if(desc) return "Makes current weapon poison branded.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_POISON_BRAND, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "zhC";
		if(desc) return "N[`[1̂琶͂zƂBzƂ͂ɂĖxオB";
#else
		if(name) return "Vampiric Drain";
		if(desc) return "Absorbs some HP from a creature and gives them to you. You will also gain nutritional sustenance from this.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_VAMPIRIC_DRAIN_1, TRUE, 100, 0L);
		break;

	case 14:
#ifdef JP
		if(name) return "̏p";
		if(desc) return "͂̎̂⍜𐶂ԂB";
#else
		if(name) return "Animate dead";
		if(desc) return "Resurrects nearby corpse and skeletons. And makes these your pets.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ANIM_DEAD, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "E";
		if(desc) return "w肵̃N[`[݂̊KBRƖB";
#else
		if(name) return "Genocide";
		if(desc) return "Eliminates an entire class of creature, exhausting you.  Powerful or unique creatures may resist.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SYMBOL_GENOCIDE, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "m";
		if(desc) return "mA|B";
#else
		if(name) return "Berserk";
		if(desc) return "Gives bonus to hit and HP, immunity to fear for a while. But decreases AC.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BERSERK, TRUE);
		break;

	case 17:
#ifdef JP
		if(name) return "쏢";
		if(desc) return "_ŗlXȌʂNB";
#else
		if(name) return "Invoke Spirits";
		if(desc) return "Causes random effects.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_INVOKE_SPIRITS, TRUE);
		break;

	case 18:
#ifdef JP
		if(name) return "Í̖";
		if(desc) return "Í̃{g̓r[B";
#else
		if(name) return "Dark Bolt";
		if(desc) return "Fires a bolt or beam of darkness.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_DARK, TRUE);
		break;

	case 19:
#ifdef JP
		if(name) return "m";
		if(desc) return "mA|AB";
#else
		if(name) return "Battle Frenzy";
		if(desc) return "Gives another bonus to hit and HP, immunity to fear for a while. Hastes you. But decreases AC.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BATTLE_FRENZY, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "z̐n";
		if(desc) return "ɋz̑B";
#else
		if(name) return "Vampiric Branding";
		if(desc) return "Makes current weapon Vampiric.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_VAMPIRIC_BRAND, TRUE);
		break;

	case 21:
#ifdef JP
		if(name) return "^Ez";
		if(desc) return "N[`[1̂琶͂zƂBzƂ͂ɂđ̗͂񕜂B";
#else
		if(name) return "Vampirism True";
		if(desc) return "Fires 3 bolts. Each of the bolts absorbs some HP from a creature and gives them to you.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_VAMPIRIC_DRAIN_1, TRUE, 300, 0L);
		break;

	case 22:
#ifdef JP
		if(name) return "̌";
		if(desc) return "E̐̂N[`[Ƀ_[W^B";
#else
		if(name) return "Nether Wave";
		if(desc) return "Damages all living creatures in sight.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_DISPEL_LIVES, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "Í̗";
		if(desc) return "ȈÍ̋B";
#else
		if(name) return "Darkness Storm";
		if(desc) return "Fires a huge ball of darkness.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_BA_DARK, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "̌";
		if(desc) return "̌B";
#else
		if(name) return "Death Ray";
		if(desc) return "Fires a beam of death.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_DEATH_RAY, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "ҏ";
		if(desc) return "1̂̃AfbhB";
#else
		if(name) return "Raise the Dead";
		if(desc) return "Summons an undead creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_UNDEAD, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "҂̔`";
		if(desc) return "ACe1ʂBxƃACe̔\͂Sɒm邱ƂłB";
#else
		if(name) return "Esoteria";
		if(desc) return "Identifies an item. Or *identifies* an item at higher level.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY_TRUE, TRUE);
		break;

	case 27:
#ifdef JP
		if(name) return "zSω";
		if(desc) return "莞ԁAzSɕωBωĂԂ͖{̎푰̔\͂AɋzSƂĂ̔\͂𓾂B";
#else
		if(name) return "Polymorph Vampire";
		if(desc) return "Mimic a vampire for a while. Loses abilities of original race and gets abilities as a vampire.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_POLYMORPH_VAMPIRE, TRUE);

	case 28:
#ifdef JP
		if(name) return "͕";
		if(desc) return "ol񕜂B";
#else
		if(name) return "Restore Life";
		if(desc) return "Restore lost experience.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RESTORE_LIFE, TRUE);
		break;

	case 29:
#ifdef JP
		if(name) return "ӖE";
		if(desc) return "̎͂ɂN[`[݂̊KBRƖB";
#else
		if(name) return "Mass Genocide";
		if(desc) return "Eliminates all nearby creatures, exhausting you.  Powerful or unique creatures may be able to resist.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MASS_GENOCIDE, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "n̍";
		if(desc) return "׈ȗ͂BPǂȃN[`[ɂ͑傫ȃ_[W^B";
#else
		if(name) return "Hellfire";
		if(desc) return "Fires a powerful ball of evil power. Hurts good creatures greatly.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HELLFIRE, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "H̉";
		if(desc) return "莞ԁAǂʂ蔲邱Ƃł󂯂_[WyĤ̏ԂɕϐgB";
#else
		if(name) return "Wraithform";
		if(desc) return "Becomes wraith form which gives ability to pass walls and makes all damages half.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_WRAITH_FORM, TRUE);
		break;
	}

	return "";
}


static cptr do_trump_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	bool fail = (mode == SPELL_FAIL) ? TRUE : FALSE;

	static const char s_random[] = "KW_RANDOM";

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "V[gEe|[g";
		if(desc) return "ߋ̃e|[gB";
#else
		if(name) return "Phase Door";
		if(desc) return "Teleport short distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BLINK, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "w偂̃J[h";
		if(desc) return "w偂B";
#else
		if(name) return "Trump Spiders";
		if(desc) return "Summons spiders.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_SPIDER, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "Vbt";
		if(desc) return "J[h̐肢B";
#else
		if(name) return "Shuffle";
		if(desc) return "Causes random effects.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SHUFFLE, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "tAEZbg";
		if(desc) return "Ő[KύXB";
#else
		if(name) return "Reset Recall";
		if(desc) return "Resets the 'deepest' level for recall spell.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RESET_RECALL, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "e|[g";
		if(desc) return "̃e|[gB";
#else
		if(name) return "Teleport";
		if(desc) return "Teleport long distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ACTIVE_TELEPORT, TRUE);
		break;

	case 5:
#ifdef JP
		if(name) return "m̃J[h";
		if(desc) return "莞ԁAepV[\͂𓾂B";
#else
		if(name) return "Trump Spying";
		if(desc) return "Gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_GET_ESP, TRUE, 50, 0L);
		break;

	case 6:
#ifdef JP
		if(name) return "e|[gEN[`[";
		if(desc) return "N[`[e|[gr[BRƖB";
#else
		if(name) return "Teleport Away";
		if(desc) return "Teleports all creatures on the line away unless resisted.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_AWAY, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "̃J[h";
		if(desc) return "1̂̓B";
#else
		if(name) return "Trump Animals";
		if(desc) return "Summons an animal.";
#endif
    
		{
			if(cast || fail)
			{
				int type = (!fail ? TRAIT_S_ANIMAL_RANGER : TRAIT_S_ANIMAL);
				msg_print(MES_SUMMON_TRUMP_ANIMAL);
				if(trump_summoning(caster_ptr, 1, !fail, caster_ptr->fy, caster_ptr->fx, 0, type, 0L))
				{
					if(fail) msg_print(MES_SUMMON_FUMBLE_CREATURE);
				}
			}
		}
		break;

	case 8:
#ifdef JP
		if(name) return "ړ̃J[h";
		if(desc) return "ACȇֈړB";
#else
		if(name) return "Trump Reach";
		if(desc) return "Pulls a distant item close to you.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELEKINES, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "J~J[̃J[h";
		if(desc) return "̔N[`[B";
#else
		if(name) return "Trump Kamikaze";
		if(desc) return "Summons creatures which explode by itself.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_KAMIKAZE, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "쏢";
		if(desc) return "1̗̂HB";
#else
		if(name) return "Phantasmal Servant";
		if(desc) return "Summons a ghost.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_PHANTOM, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "Xs[hEN[`[";
		if(desc) return "N[`[1̂B";
#else
		if(name) return "Haste Creature";
		if(desc) return "Hastes a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HASTE_OTHER, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "e|[gEx";
		if(desc) return "uɏォ̊KɃe|[gB";
#else
		if(name) return "Teleport Level";
		if(desc) return "Teleport to up or down stairs in a moment.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_LEVEL, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "̔";
		if(desc) return "Z̎w肵ꏊɃe|[gB";
#else
		if(name) return "Dimension Door";
		if(desc) return "Teleport to given location.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DIMENSION_DOOR, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "A҂̎";
		if(desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if(name) return "Word of Recall";
		if(desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RECALL, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "Ǖ";
		if(desc) return "ȆSẴN[`[e|[gBRƖB";
#else
		if(name) return "Banish";
		if(desc) return "Teleports all creatures in sight away unless resisted.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MASS_TELE_AWAY, TRUE, 100, 0L);
		break;

	case 16:
#ifdef JP
		if(name) return "ʒũJ[h";
		if(desc) return "1̂̃N[`[ƈʒuB";
#else
		if(name) return "Swap Position";
		if(desc) return "Swap positions of you and a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SWAP_POS, TRUE);
		break;

	case 17:
#ifdef JP
		if(name) return "Afbh̃J[h";
		if(desc) return "1̂̃AfbhB";
#else
		if(name) return "Trump Undead";
		if(desc) return "Summons an undead creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_UNDEAD, TRUE);
		break;

	case 18:
#ifdef JP
		if(name) return "঒ނ̃J[h";
		if(desc) return "1̂̃qhB";
#else
		if(name) return "Trump Reptiles";
		if(desc) return "Summons a hydra.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_HYDRA, TRUE);
		break;

	case 19:
#ifdef JP
		if(name) return "N[`[̃J[h";
		if(desc) return "̃N[`[B";
#else
		if(name) return "Trump Creatures";
		if(desc) return "Summons some creatures.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, 0, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "nEh̃J[h";
		if(desc) return "1O[ṽnEhB";
#else
		if(name) return "Trump Hounds";
		if(desc) return "Summons a group of hounds.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_HOUND, TRUE);
		break;

	case 21:
#ifdef JP
		if(name) return "gv̐n";
		if(desc) return "Ƀgv̑B";
#else
		if(name) return "Trump Branding";
		if(desc) return "Makes current weapon a Trump weapon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_TRUMP_BRAND, TRUE);
		break;

	case 22:
#ifdef JP
		if(name) return "lԃgv";
		if(desc) return "_Ƀe|[gˑRψقÄӎvŃe|[gˑRψقgɂB";
#else
		if(name) return "Living Trump";
		if(desc) return "Gives mutation which makes you teleport randomly or makes you able to teleport at will.";
#endif
    
		break;

	case 23:
#ifdef JP
		if(name) return "TCo[f[̃J[h";
		if(desc) return "1̂̃TCo[f[B";
#else
		if(name) return "Trump Cyberdemon";
		if(desc) return "Summons a cyber demon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_CYBER, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "\̃J[h";
		if(desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if(name) return "Trump Divination";
		if(desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_ALL, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "m̃J[h";
		if(desc) return "ACe̎\͂SɒmB";
#else
		if(name) return "Trump Lore";
		if(desc) return "*Identifies* an item.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY_TRUE, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "񕜃N[`[";
		if(desc) return "N[`[1̗̂̑͂񕜂B";
#else
		if(name) return "Heal Creature";
		if(desc) return "Heal a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HEAL_OTHER, TRUE);

	case 27:
#ifdef JP
		if(name) return "hS̃J[h";
		if(desc) return "1̂̃hSB";
#else
		if(name) return "Trump Dragon";
		if(desc) return "Summons a dragon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_DRAGON, TRUE);
		break;

	case 28:
#ifdef JP
		if(name) return "覐΂̃J[h";
		if(desc) return "̎ӂ覐΂𗎂ƂB";
#else
		if(name) return "Trump Meteor";
		if(desc) return "Makes meteor balls fall down to nearby random locations.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_METEOR_SWARM, TRUE);
		break;

	case 29:
#ifdef JP
		if(name) return "f[̃J[h";
		if(desc) return "1̂̈B";
#else
		if(name) return "Trump Demon";
		if(desc) return "Summons a demon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_DEMON, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "ñJ[h";
		if(desc) return "1̂̏㋉AfbhB";
#else
		if(name) return "Trump Greater Undead";
		if(desc) return "Summons a greater undead.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_HI_UNDEAD, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "ÑhS̃J[h";
		if(desc) return "1̂̌ÑhSB";
#else
		if(name) return "Trump Ancient Dragon";
		if(desc) return "Summons an ancient dragon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_HI_DRAGON, TRUE);
		break;
	}

	return "";
}


static cptr do_arcane_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "d";
		if(desc) return "d̃{g̓r[B";
#else
		if(name) return "Zap";
		if(desc) return "Fires a bolt or beam of lightning.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_ELEC, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "@̎{";
		if(desc) return "ɌB";
#else
		if(name) return "Wizard Lock";
		if(desc) return "Locks a door.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_JAM, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "̊m";
		if(desc) return "߂̓ȃN[`[mB";
#else
		if(name) return "Detect Invisibility";
		if(desc) return "Detects all invisible creatures in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_INVISIBLE, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "N[`[m";
		if(desc) return "߂̑SĂ̌N[`[mB";
#else
		if(name) return "Detect Creatures";
		if(desc) return "Detects all creatures in your vicinity unless invisible.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SMELL_MON, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "V[gEe|[g";
		if(desc) return "ߋ̃e|[gB";
#else
		if(name) return "Blink";
		if(desc) return "Teleport short distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BLINK, TRUE);
		break;

	case 5:
#ifdef JP
		if(name) return "CgEGA";
		if(desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if(name) return "Light Area";
		if(desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_LIGHT_AREA, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "㩂Ɣ j";
		if(desc) return "꒼̑SĂ㩂Ɣj󂷂B";
#else
		if(name) return "Trap & Door Destruction";
		if(desc) return "Fires a beam which destroy traps and doors.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DESTROY_DOOR_TRAP_BEAM, TRUE); 
		break;

	case 7:
#ifdef JP
		if(name) return "y̎";
		if(desc) return "Ƒ̗͂񕜂B";
#else
		if(name) return "Cure Light Wounds";
		if(desc) return "Heals cut and HP a little.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(2, 10), 0L);
		break;

	case 8:
#ifdef JP
		if(name) return "㩂Ɣ m";
		if(desc) return "߂̑SĂ㩂ƔƊKimB";
#else
		if(name) return "Detect Doors & Traps";
		if(desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_DOOR_TRAP, TRUE, 100, 0L);
		break;

	case 9:
#ifdef JP
		if(name) return "Rf";
		if(desc) return "ɔR⋋B";
#else
		if(name) return "Phlogiston";
		if(desc) return "Adds more turns of light to a lantern or torch.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_PHLOGISTON, TRUE, 100, 0L);
		break;

	case 10:
#ifdef JP
		if(name) return "󊴒m";
		if(desc) return "߂̍mB";
#else
		if(name) return "Detect Treasure";
		if(desc) return "Detects all treasures in your vicinity.";
#endif
		do_active_trait(caster_ptr, TRAIT_SMELL_MET, TRUE, 100, 0L);
		break;

	case 11:
#ifdef JP
		if(name) return "@m";
		if(desc) return "߂̖@ACemB";
#else
		if(name) return "Detect Enchantment";
		if(desc) return "Detects all magical items in your vicinity.";
#endif
		do_active_trait(caster_ptr, TRAIT_DETECT_OBJECT_MAGIC, TRUE, 100, 0L);
		break;

	case 12:
#ifdef JP
		if(name) return "ACem";
		if(desc) return "߂̑SẴACemB";
#else
		if(name) return "Detect Objects";
		if(desc) return "Detects all items in your vicinity.";
#endif
		do_active_trait(caster_ptr, TRAIT_DETECT_OBJECT, TRUE, 100, 0L);
		break;

	case 13:
#ifdef JP
		if(name) return "";
		if(desc) return "ł̓犮SɎ菜B";
#else
		if(name) return "Cure Poison";
		if(desc) return "Cures poison status.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_POISON, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "ϗ";
		if(desc) return "莞ԁACւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Cold";
		if(desc) return "Gives resistance to cold. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_COLD, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "ω";
		if(desc) return "莞ԁAւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Fire";
		if(desc) return "Gives resistance to fire. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "ϓd";
		if(desc) return "莞ԁAdւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Lightning";
		if(desc) return "Gives resistance to electricity. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE);
		break;

	case 17:
#ifdef JP
		if(name) return "ώ_";
		if(desc) return "莞ԁA_ւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Acid";
		if(desc) return "Gives resistance to acid. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_ACID, TRUE);
		break;

	case 18:
#ifdef JP
		if(name) return "d̎";
		if(desc) return "Ƒ̗͂𒆒x񕜂B";
#else
		if(name) return "Cure Medium Wounds";
		if(desc) return "Heals cut and HP more.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(4, 10), 0L);
		break;

	case 19:
#ifdef JP
		if(name) return "e|[g";
		if(desc) return "̃e|[gB";
#else
		if(name) return "Teleport";
		if(desc) return "Teleport long distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ACTIVE_TELEPORT, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "Ӓ";
		if(desc) return "ACeʂB";
#else
		if(name) return "Identify";
		if(desc) return "Identifies an item.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY, TRUE);
		break;

	case 21:
#ifdef JP
		if(name) return "Ηn";
		if(desc) return "ǂnďɂB";
#else
		if(name) return "Stone to Mud";
		if(desc) return "Turns one rock square to mud.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STONE_TO_MUD, TRUE, 100, 0L);
		break;

	case 22:
#ifdef JP
		if(name) return "M";
		if(desc) return "BN[`[ɌʂB";
#else
		if(name) return "Ray of Light";
		if(desc) return "Fires a beam of light which damages to light-sensitive creatures.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SUNLIGHT, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "󕠏[";
		if(desc) return "ɂB";
#else
		if(name) return "Satisfy Hunger";
		if(desc) return "Satisfies hunger.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SATIATE, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "F";
		if(desc) return "莞ԁAȂ̂悤ɂȂB";
#else
		if(name) return "See Invisible";
		if(desc) return "Gives see invisible for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_SEE_INVISIBLE, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "G^";
		if(desc) return "1̂̃G^B";
#else
		if(name) return "Conjure Elemental";
		if(desc) return "Summons an elemental.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_ELEMENTAL, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "e|[gEx";
		if(desc) return "uɏォ̊KɃe|[gB";
#else
		if(name) return "Teleport Level";
		if(desc) return "Teleport to up or down stairs in a moment.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_LEVEL, TRUE);
		break;

	case 27:
#ifdef JP
		if(name) return "e|[gEN[`[";
		if(desc) return "N[`[e|[gr[BRƖB";
#else
		if(name) return "Teleport Away";
		if(desc) return "Teleports all creatures on the line away unless resisted.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TELE_AWAY, TRUE);
		break;

	case 28:
#ifdef JP
		if(name) return "f̋";
		if(desc) return "AdACA_̂ǂꂩ̋B";
#else
		if(name) return "Elemental Ball";
		if(desc) return "Fires a ball of some elements.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ELEMENTAL_BALL, TRUE);
		break;


	case 29:
#ifdef JP
		if(name) return "Sm";
		if(desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if(name) return "Detection";
		if(desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_ALL, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "A҂̎";
		if(desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if(name) return "Word of Recall";
		if(desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RECALL, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "痢";
		if(desc) return "̊KŜivɏƂ炵A_WׂẴACemBɁA莞ԃepV[\͂𓾂B";
#else
		if(name) return "Clairvoyance";
		if(desc) return "Maps and lights whole dungeon level. Knows all objects location. And gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_ENLIGHTENMENT, TRUE, 100, 0L);
		break;
	}

	return "";
}


static cptr do_craft_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "ԊO";
		if(desc) return "莞ԁAԊO͂B";
#else
		if(name) return "Infravision";
		if(desc) return "Gives infravision for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SEE_INFRA, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "񕜗͋";
		if(desc) return "莞ԁA񕜗͂B";
#else
		if(name) return "Regeneration";
		if(desc) return "Gives regeneration ability for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_REGENERATE, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "󕠏[";
		if(desc) return "ɂȂB";
#else
		if(name) return "Satisfy Hunger";
		if(desc) return "Satisfies hunger.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SATIATE, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "ϗC";
		if(desc) return "莞ԁACւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Cold";
		if(desc) return "Gives resistance to cold. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "ωΉ";
		if(desc) return "莞ԁAւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Fire";
		if(desc) return "Gives resistance to fire. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE);
		break;

	case 5:
#ifdef JP
		if(name) return "mCg";
		if(desc) return "莞ԁAq[[CɂȂB";
#else
		if(name) return "Heroism";
		if(desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BECOME_HERO, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "ϓd";
		if(desc) return "莞ԁAdւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Lightning";
		if(desc) return "Gives resistance to electricity. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_ELEC, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "ώ_";
		if(desc) return "莞ԁA_ւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Acid";
		if(desc) return "Gives resistance to acid. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_ACID, TRUE);
		break;

	case 8:
#ifdef JP
		if(name) return "F";
		if(desc) return "莞ԁAȂ̂悤ɂȂB";
#else
		if(name) return "See Invisibility";
		if(desc) return "Gives see invisible for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_SEE_INVISIBLE, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "";
		if(desc) return "ACeɂア􂢂B";
#else
		if(name) return "Remove Curse";
		if(desc) return "Removes normal curses from equipped items.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_CURSE_1, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "ϓ";
		if(desc) return "莞ԁAłւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Poison";
		if(desc) return "Gives resistance to poison. This resistance can be added to which from equipment for more powerful resistance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_POIS, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "m";
		if(desc) return "mA|B";
#else
		if(name) return "Berserk";
		if(desc) return "Gives bonus to hit and HP, immunity to fear for a while. But decreases AC.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BERSERK, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "ȕ";
		if(desc) return "݂̎̏ԂSɒmB";
#else
		if(name) return "Self Knowledge";
		if(desc) return "Gives you useful info regarding your current resistances, the powers of your weapon and maximum limits of your stats.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SELF_KNOWLEDGE, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "Ύ׈E";
		if(desc) return "׈ȃN[`[̍UhoA𒣂B";
#else
		if(name) return "Protection from Evil";
		if(desc) return "Gives aura which protect you from evil creature's physical attack.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_PROT_EVIL, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "";
		if(desc) return "ŁANOԁASAo𒼂B";
#else
		if(name) return "Cure";
		if(desc) return "Heals poison, stun, cut and hallucination completely.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CURING, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "@";
		if(desc) return "莞ԁAɗCAAdA_Ał̂ꂩ̑BȂƎgȂB";
#else
		if(name) return "Mana Branding";
		if(desc) return "Makes current weapon some elemental branded. You must wield weapons.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_ELEMENT_BRAND, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "epV[";
		if(desc) return "莞ԁAepV[\͂𓾂B";
#else
		if(name) return "Telepathy";
		if(desc) return "Gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_GET_ESP, TRUE, 50, 0L);
		break;

	case 17:
#ifdef JP
		if(name) return "Ή";
		if(desc) return "莞ԁAAC㏸B";
#else
		if(name) return "Stone Skin";
		if(desc) return "Gives bonus to AC for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_STONE_SKIN, TRUE, 50, 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "Sϐ";
		if(desc) return "莞ԁA_AdAACAłɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resistance";
		if(desc) return "Gives resistance to fire, cold, electricity, acid and poison for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_ELEMENT, TRUE);
		break;

	case 19:
#ifdef JP
		if(name) return "Xs[h";
		if(desc) return "莞ԁAB";
#else
		if(name) return "Haste Self";
		if(desc) return "Hastes you for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HASTE, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "ǔ";
		if(desc) return "莞ԁAǂʂ蔲悤ɂȂB";
#else
		if(name) return "Walk through Wall";
		if(desc) return "Gives ability to pass walls for a while.";
#endif
    

	case 21:
#ifdef JP
		if(name) return "";
		if(desc) return "ɔ˂̑B";
#else
		if(name) return "Polish Shield";
		if(desc) return "Makes a shield a shield of reflection.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_PURISH_SHIELD, TRUE);
		break;

	case 22:
#ifdef JP
		if(name) return "S[";
		if(desc) return "1̂̃S[𐻑B";
#else
		if(name) return "Create Golem";
		if(desc) return "Creates a golem.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_GOLEM, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "@̊Z";
		if(desc) return "莞ԁA@h͂ACオAƖӖڂ̑ϐA˔\́Aგm炸AV𓾂B";
#else
		if(name) return "Magical armor";
		if(desc) return "Gives resistance to magic, bonus to AC, resistance to confusion, blindness, reflection, free action and levitation for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_MAGIC_DEF, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "͉";
		if(desc) return "Ehɂꂽ閂͂SɉB";
#else
		if(name) return "Remove Enchantment";
		if(desc) return "Removes all magics completely from any weapon or armor.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_REMOVE, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "􂢕";
		if(desc) return "ACeɂ͂Ȏ􂢂B";
#else
		if(name) return "Remove All Curse";
		if(desc) return "Removes normal and heavy curse from equipped items.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_CURSE_2, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "SȂm";
		if(desc) return "ACe̎\͂SɒmB";
#else
		if(name) return "Knowledge True";
		if(desc) return "*Identifies* an item.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_IDENTIFY_TRUE, TRUE);
		break;

	case 27:
#ifdef JP
		if(name) return "틭";
		if(desc) return "̖Cƃ_[WCB";
#else
		if(name) return "Enchant Weapon";
		if(desc) return "Attempts to increase +to-hit, +to-dam of a weapon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_WEAPON_BOOST, TRUE);
		break;

	case 28:
#ifdef JP
		if(name) return "h";
		if(desc) return "Z̖hCB";
#else
		if(name) return "Enchant Armor";
		if(desc) return "Attempts to increase +AC of an armor.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_ARMOR_BOOST, TRUE);
		break;

	case 29:
#ifdef JP
		if(name) return "푮t^";
		if(desc) return "Ƀ_ɑB";
#else
		if(name) return "Brand Weapon";
		if(desc) return "Makes current weapon a random ego weapon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_RANDOM_BRAND, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "lԃgv";
		if(desc) return "_Ƀe|[gˑRψقÄӎvŃe|[gˑRψقgɂB";
#else
		if(name) return "Living Trump";
		if(desc) return "Gives mutation which makes you teleport randomly or makes you able to teleport at will.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_LIVING_TRUMP, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "ւ̖Ɖu";
		if(desc) return "莞ԁACAAdA_̂ꂩɑ΂Ɖu𓾂B";
#else
		if(name) return "Immunity";
		if(desc) return "Gives an immunity to fire, cold, electricity or acid for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_IMMUNE, TRUE);
		break;
	}

	return "";
}


static cptr do_daemon_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	static const char s_dam[] = "KW_DAM";

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "}WbNE~TC";
		if(desc) return "ア@̖B";
#else
		if(name) return "Magic Missile";
		if(desc) return "Fires a weak bolt of magic.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_MISSILE, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "m";
		if(desc) return "߂̐̂ȂN[`[mB";
#else
		if(name) return "Detect Unlife";
		if(desc) return "Detects all nonliving creatures in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_NONLIVING, TRUE, 100, 0L);
		break;

	case 2:
#ifdef JP
		if(name) return "ׂȂj";
		if(desc) return "莞ԁAACɃ{[iX𓾂B";
#else
		if(name) return "Evil Bless";
		if(desc) return "Gives bonus to hit and AC for a few turns.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_BLESSING_SELF, TRUE, 100, 0L);
		break;

	case 3:
#ifdef JP
		if(name) return "ωΉ";
		if(desc) return "莞ԁAւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Resist Fire";
		if(desc) return "Gives resistance to fire, cold and electricity for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_FIRE, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "Q";
		if(desc) return "N[`[1̂|ANOBRƖB";
#else
		if(name) return "Horrify";
		if(desc) return "Attempts to scare and stun a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TERROR, TRUE);
		break;

	case 5:
#ifdef JP
		if(name) return "n̖";
		if(desc) return "ñ{g̓r[B";
#else
		if(name) return "Nether Bolt";
		if(desc) return "Fires a bolt or beam of nether.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_NETH, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "Ñ̎쏢";
		if(desc) return "Ñ̎B";
#else
		if(name) return "Summon Manes";
		if(desc) return "Summons a manes.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_MANES, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "n̉";
		if(desc) return "׈ȗ͂{[BPǂȃN[`[ɂ͑傫ȃ_[W^B";
#else
		if(name) return "Hellish Flame";
		if(desc) return "Fires a ball of evil power. Hurts good creatures greatly.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_HELLFIRE, TRUE);

	case 8:
#ifdef JP
		if(name) return "f[xz";
		if(desc) return "1̂𖣗BRƖ";
#else
		if(name) return "Dominate Demon";
		if(desc) return "Attempts to charm a demon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DOMINATE_DEMON, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "rW";
		if(desc) return "ӂ̒n`mB";
#else
		if(name) return "Vision";
		if(desc) return "Maps nearby area.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_MAP, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "ϒn";
		if(desc) return "莞ԁAnւ̑ϐ𓾂B";
#else
		if(name) return "Resist Nether";
		if(desc) return "Gives resistance to nether for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_RES_NETH, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "vY}E{g";
		if(desc) return "vY}̃{g̓r[B";
#else
		if(name) return "Plasma bolt";
		if(desc) return "Fires a bolt or beam of plasma.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_PLAS, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "t@CAE{[";
		if(desc) return "̋B";
#else
		if(name) return "Fire Ball";
		if(desc) return "Fires a ball of fire.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_FIRE, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "̐n";
		if(desc) return "ɉ̑B";
#else
		if(name) return "Fire Branding";
		if(desc) return "Makes current weapon fire branded.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ENCHANT_FIRE_BRAND, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "n";
		if(desc) return "傫Ȓn̋B";
#else
		if(name) return "Nether Ball";
		if(desc) return "Fires a huge ball of nether.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_NETH, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "f[";
		if(desc) return "1̂B";
#else
		if(name) return "Summon Demon";
		if(desc) return "Summons a demon.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_DEMON, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "̖";
		if(desc) return "莞ԁAepV[\͂𓾂B";
#else
		if(name) return "Devilish Eye";
		if(desc) return "Gives telepathy for a while.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_GET_ESP, TRUE, 50, 0L);
		break;

	case 17:
#ifdef JP
		if(name) return "̃N[N";
		if(desc) return "|菜A莞ԁAƗC̑ϐÃI[𓾂Bϐ͑ɂϐɗݐςB";
#else
		if(name) return "Devil Cloak";
		if(desc) return "Removes fear. Gives resistance to fire and cold, and aura of fire. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DEVIL_CLOAK, TRUE, 50, 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "n◬";
		if(desc) return "𒆐SƂ̋oAnɕςB";
#else
		if(name) return "The Flow of Lava";
		if(desc) return "Generates a ball of fire centered on you which transforms floors to magma.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_FLOW_LAVA, TRUE, 50, 0L);
		break;

	case 19:
#ifdef JP
		if(name) return "vY}";
		if(desc) return "vY}̋B";
#else
		if(name) return "Plasma Ball";
		if(desc) return "Fires a ball of plasma.";
#endif    
		if(cast) do_active_trait(caster_ptr, TRAIT_BO_PLAS, TRUE, 50, 0L);
		break;

	case 20:
#ifdef JP
		if(name) return "ω";
		if(desc) return "莞ԁAɕωBωĂԂ͖{̎푰̔\͂AɈƂĂ̔\͂𓾂B";
#else
		if(name) return "Polymorph Demon";
		if(desc) return "Mimic a demon for a while. Loses abilities of original race and gets abilities as a demon.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_POLYMORPH_DEMON, TRUE, 50, 0L);
		break;

	case 21:
#ifdef JP
		if(name) return "n̔g";
		if(desc) return "ȆSẴN[`[Ƀ_[W^BPǂȃN[`[ɓɑ傫ȃ_[W^B";
#else
		if(name) return "Nather Wave";
		if(desc) return "Damages all creatures in sight. Hurts good creatures greatly.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_WAVE_NETHER, TRUE, 50, 0L);
		break;

	case 22:
#ifdef JP
		if(name) return "TLoX̐ڕ";
		if(desc) return "ʍ̋B";
#else
		if(name) return "Kiss of Succubus";
		if(desc) return "Fires a ball of nexus.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SUCCUBUS_KISS, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "jł̎";
		if(desc) return "jł̎BHN[`[͂̂ƂHP̔Õ_[W󂯂B";
#else
		if(name) return "Doom Hand";
		if(desc) return "Attempts to make a creature's HP almost half.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HAND_DOOM, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "mCg";
		if(desc) return "莞ԁAq[[CɂȂB";
#else
		if(name) return "Raise the Morale";
		if(desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BECOME_HERO, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "sł̓";
		if(desc) return "莞ԁAԋt]ւ̑ϐ𓾂B";
#else
		if(name) return "Immortal Body";
		if(desc) return "Gives resistance to time for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_RES_TIME, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "C̉~";
		if(desc) return "𒆐SƂJIX̋A̋𔭐A߂̃N[`[𖣗B";
#else
		if(name) return "Insanity Circle";
		if(desc) return "Generate balls of chaos, confusion and charm centered on you.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_INSANITY_CIRCLE, TRUE);
		break;

	case 27:
#ifdef JP
		if(name) return "ybgj";
		if(desc) return "SẴybgIɔjB";
#else
		if(name) return "Explode Pets";
		if(desc) return "Makes all pets explode.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DISCHARGE_MINION, TRUE);
		break;

	case 28:
#ifdef JP
		if(name) return "O[^[f[";
		if(desc) return "㋉f[Bɂ͐l('p','h','t'ŕ\N[`[)̎̂Ȃ΂ȂȂB";
#else
		if(name) return "Summon Greater Demon";
		if(desc) return "Summons greater demon. It need to sacrifice a corpse of human ('p','h' or 't').";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_GREATER_DEMON, TRUE);
		break;

	case 29:
#ifdef JP
		if(name) return "n";
		if(desc) return "Ȓn̋B";
#else
		if(name) return "Nether Storm";
		if(desc) return "Generate a huge ball of nether.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_STORM_NETHER, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "̎";
		if(desc) return "_[W󂯂邱ƂɂđΏۂɎ􂢂A_[W^lXȌʂNB";
#else
		if(name) return "Bloody Curse";
		if(desc) return "Puts blood curse which damages and causes various effects on a creature. You also take damage.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CAUSE_BLOODY_CURSE, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "ω";
		if(desc) return "̉ɕωBωĂԂ͖{̎푰̔\͂AɈ̉ƂĂ̔\͂𓾁Aǂj󂵂ȂB";
#else
		if(name) return "Polymorph Demonlord";
		if(desc) return "Mimic a demon lord for a while. Loses abilities of original race and gets great abilities as a demon lord. Even hard walls can't stop your walking.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_POLYMORPH_DEMONLORD, TRUE);
		break;

	}

	return "";
}


static cptr do_crusade_spell(creature_type *caster_ptr, int spell, int mode)
{

	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	COODINATES lev_bonus = caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "";
		if(desc) return "d̃{g̓r[B";
#else
		if(name) return "Punishment";
		if(desc) return "Fires a bolt or beam of lightning.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_PUNISHMENT, TRUE);
		break;

	case 1:
#ifdef JP
		if(name) return "׈݊m";
		if(desc) return "߂̎׈ȃN[`[mB";
#else
		if(name) return "Detect Evil";
		if(desc) return "Detects all evil creatures in your vicinity.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DETECT_EVIL, TRUE);
		break;

	case 2:
#ifdef JP
		if(name) return "|";
		if(desc) return "|菜B";
#else
		if(name) return "Remove Fear";
		if(desc) return "Removes fear.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_FEAR, TRUE);
		break;

	case 3:
#ifdef JP
		if(name) return "Ј";
		if(desc) return "N[`[1̂|BRƖB";
#else
		if(name) return "Scare Creature";
		if(desc) return "Attempts to scare a creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_TERROR, TRUE);
		break;

	case 4:
#ifdef JP
		if(name) return "";
		if(desc) return "אڂSẴN[`[𖰂点BRƖB";
#else
		if(name) return "Sanctuary";
		if(desc) return "Attempts to sleep creatures in the adjacent squares.";
#endif
		break;

	case 5:
#ifdef JP
		if(name) return "";
		if(desc) return "̃e|[gB";
#else
		if(name) return "Portal";
		if(desc) return "Teleport medium distance.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ACTIVE_TELEPORT, TRUE);
		break;

	case 6:
#ifdef JP
		if(name) return "X^[_Xg";
		if(desc) return "^[Qbgt߂ɑM̃{gA˂B";
#else
		if(name) return "Star Dust";
		if(desc) return "Fires many bolts of light near the target.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_STAR_DUST, TRUE);
		break;

	case 7:
#ifdef JP
		if(name) return "g̏";
		if(desc) return "AŁANOSB";
#else
		if(name) return "Purify";
		if(desc) return "Heals all cut, stun and poison status.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CURING, TRUE);
		break;

	case 8:
#ifdef JP
		if(name) return "׈΂";
		if(desc) return "׈ȃN[`[1̂e|[gBRƖB";
#else
		if(name) return "Scatter Evil";
		if(desc) return "Attempts to teleport an evil creature away.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SCATTER_EVIL, TRUE);
		break;

	case 9:
#ifdef JP
		if(name) return "Ȃ";
		if(desc) return "Ȃ͂B׈ȃN[`[ɑ΂đ傫ȃ_[W^邪APǂȃN[`[ɂ͌ʂȂB";
#else
		if(name) return "Holy Orb";
		if(desc) return "Fires a ball with holy power. Hurts evil creatures greatly, but don't effect good creatures.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_HOLYFIRE, TRUE);
		break;

	case 10:
#ifdef JP
		if(name) return "";
		if(desc) return "ȆSẴAfbhyшɃ_[W^A׈ȃN[`[|B";
#else
		if(name) return "Exorcism";
		if(desc) return "Damages all undead and demons in sight, and scares all evil creatures in sight.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_EXORCISM, TRUE);
		break;

	case 11:
#ifdef JP
		if(name) return "";
		if(desc) return "ACeɂア􂢂B";
#else
		if(name) return "Remove Curse";
		if(desc) return "Removes normal curses from equipped items.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_CURSE_1, TRUE);
		break;

	case 12:
#ifdef JP
		if(name) return "F";
		if(desc) return "莞ԁAȂ̂悤ɂȂB";
#else
		if(name) return "Sense Unseen";
		if(desc) return "Gives see invisible for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_SEE_INVISIBLE, TRUE);
		break;

	case 13:
#ifdef JP
		if(name) return "Ύ׈E";
		if(desc) return "׈ȃN[`[̍UhoA𒣂B";
#else
		if(name) return "Protection from Evil";
		if(desc) return "Gives aura which protect you from evil creature's physical attack.";
#endif
		do_active_trait_tmp(caster_ptr, TRAIT_PROT_EVIL, TRUE);
		break;

	case 14:
#ifdef JP
		if(name) return "ق̗";
		if(desc) return "͂ȓd̃{gB";
#else
		if(name) return "Judgment Thunder";
		if(desc) return "Fires a powerful bolt of lightning.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BO_ELEC, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "Ȃ䌾t";
		if(desc) return "E̎׈ȑ݂ɑ傫ȃ_[W^A̗͂񕜂AŁA|ANOԁASB";
#else
		if(name) return "Holy Word";
		if(desc) return "Damages all evil creatures in sight, heals HP somewhat, and completely heals poison, fear, stun and cut status.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_HOLY_WORD, TRUE);
		break;

	case 16:
#ifdef JP
		if(name) return "Jꂽ";
		if(desc) return "꒼̑SĂ㩂Ɣj󂷂B";
#else
		if(name) return "Unbarring Ways";
		if(desc) return "Fires a beam which destroy traps and doors.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DESTROY_DOOR_TRAP_BEAM, TRUE);
		break;

	case 17:
#ifdef JP
		if(name) return "";
		if(desc) return "׈ȃN[`[̓~߂B";
#else
		if(name) return "Arrest";
		if(desc) return "Attempts to paralyze an evil creature.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ARREST_EVIL, TRUE);
		break;

	case 18:
#ifdef JP
		if(name) return "ȂI[";
		if(desc) return "莞ԁA׈ȃN[`[鐹ȂI[𓾂B";
#else
		if(name) return "Holy Aura";
		if(desc) return "Gives aura of holy power which injures evil creatures which attacked you for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_HOLY_AURA, TRUE);
		break;

	case 19:
#ifdef JP
		if(name) return "Afbh&ގU";
		if(desc) return "ȆSẴAfbhyшɃ_[W^B";
#else
		if(name) return "Dispel Undead & Demons";
		if(desc) return "Damages all undead and demons in sight.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DISPEL_UNDEAD_DEMON, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "׈ގU";
		if(desc) return "ȆSĂ̎׈ȃN[`[Ƀ_[W^B";
#else
		if(name) return "Dispel Evil";
		if(desc) return "Damages all evil creatures in sight.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DISPEL_EVIL_1, TRUE, lev_bonus * 4, 0L);
		break;

	case 21:
#ifdef JP
		if(name) return "Ȃn";
		if(desc) return "ʏ̕ɖŎׂ̑B";
#else
		if(name) return "Holy Blade";
		if(desc) return "Makes current weapon especially deadly against evil creatures.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_ENCHANT_HOLY_BRAND, TRUE, lev_bonus * 4, 0L);
		break;

	case 22:
#ifdef JP
		if(name) return "X^[o[Xg";
		if(desc) return "ȑM̋B";
#else
		if(name) return "Star Burst";
		if(desc) return "Fires a huge ball of powerful light.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BA_LITE, TRUE);
		break;

	case 23:
#ifdef JP
		if(name) return "Vg";
		if(desc) return "Vg1̏B";
#else
		if(name) return "Summon Angel";
		if(desc) return "Summons an angel.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_S_ANGEL, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "mCg";
		if(desc) return "莞ԁAq[[CɂȂB";
#else
		if(name) return "Heroism";
		if(desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BECOME_HERO, TRUE);
		break;

	case 25:
#ifdef JP
		if(name) return "􂢑ގU";
		if(desc) return "ACeɂ͂Ȏ􂢂B";
#else
		if(name) return "Dispel Curse";
		if(desc) return "Removes normal and heavy curse from equipped items.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_REMOVE_CURSE_2, TRUE);
		break;

	case 26:
#ifdef JP
		if(name) return "׈Ǖ";
		if(desc) return "ȆSĂ̎׈ȃN[`[e|[gBRƖB";
#else
		if(name) return "Banish Evil";
		if(desc) return "Teleports all evil creatures in sight away unless resisted.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_BANISH_EVIL, TRUE);
		break;

	case 27:
#ifdef JP
		if(name) return "n}Qh";
		if(desc) return "ӂ̃ACeAN[`[An`j󂷂B";
#else
		if(name) return "Armageddon";
		if(desc) return "Destroy everything in nearby area.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_STAR_DESTROY, TRUE);
		break;

	case 28:
#ifdef JP
		if(name) return "ڂɂ͖ڂ";
		if(desc) return "莞ԁA_[W󂯂ƂɍUsN[`[ɑ΂ē̃_[W^B";
#else
		if(name) return "An Eye for an Eye";
		if(desc) return "Gives special aura for a while. When you are attacked by a creature, the creature are injured with same amount of damage as you take.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_GET_EYE_FOR_EYE, TRUE);
		break;

	case 29:
#ifdef JP
		if(name) return "_̓{";
		if(desc) return "^[Qbg̎͂ɕ̋𑽐ƂB";
#else
		if(name) return "Wrath of the God";
		if(desc) return "Drops many balls of disintegration near the target.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_WRATH_OF_GOD, TRUE);
		break;

	case 30:
#ifdef JP
		if(name) return "_";
		if(desc) return "אڂN[`[ɐȂ_[W^AẼN[`[Ƀ_[WAANOAA|A^Bɑ̗͂񕜂B";
#else
		if(name) return "Divine Intervention";
		if(desc) return "Damages all adjacent creatures with holy power. Damages and attempt to slow, stun, confuse, scare and freeze all creatures in sight. And heals HP.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_DIVINE_INTERVENTION, TRUE);
		break;

	case 31:
#ifdef JP
		if(name) return "";
		if(desc) return "ȆPǂȃN[`[ybgɂ悤ƂAȂȂꍇyёPǂłȂN[`[|Bɑ̉ꂽRmAq[[AjAAΎ׈E𓾂B";
#else
		if(name) return "Crusade";
		if(desc) return "Attempts to charm all good creatures in sight, and scare all non-charmed creatures, and summons great number of knights, and gives heroism, bless, speed and protection from evil.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_CRUSADE, TRUE);
		break;
	}

	return "";
}


static cptr do_music_spell(creature_type *caster_ptr, int spell, int mode)
{
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	bool fail = (mode == SPELL_FAIL) ? TRUE : FALSE;
	bool cont = (mode == SPELL_CONT) ? TRUE : FALSE;
	bool stop = (mode == SPELL_STOP) ? TRUE : FALSE;

	static const char s_dam[] = "KW_DAM";

	DIRECTION dir;
	COODINATES lev_bonus = caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "x݂̉";
		if(desc) return "ȆSẴN[`[BRƖB";
#else
		if(name) return "Song of Holding";
		if(desc) return "Attempts to slow all creatures in sight.";
#endif
		if(cast || fail) stop_singing(caster_ptr);
		if(cast) do_active_trait(caster_ptr, TRAIT_MASS_SLOW, TRUE, 100, 0L);
		break;

	case 1:
#ifdef JP
		if(name) return "j̉";
		if(desc) return "AC̃{[iX𓾂B";
#else
		if(name) return "Song of Blessing";
		if(desc) return "Gives bonus to hit and AC for a few turns.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("ȃfBtŎn߂DDD");
#else
			msg_print("The holy power of the Music of the Ainur enters you...");
#endif
			start_singing(caster_ptr, spell, MUSIC_BLESS);
		}
		break;

	case 2:
#ifdef JP
		if(name) return "̉F";
		if(desc) return "̃{gB";
#else
		if(name) return "Wrecking Note";
		if(desc) return "Fires a bolt of sound.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);
		if(cast) do_active_trait(caster_ptr, TRAIT_BO_SOUN, TRUE, 100, 0L);
		break;

	case 3:
#ifdef JP
		if(name) return "NO̐";
		if(desc) return "ȆSẴN[`[NOBRƖB";
#else
		if(name) return "Stun Pattern";
		if(desc) return "Attempts to stun all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("Ῐf郁fBtŎn߂DDD");
#else
			msg_print("You weave a pattern of sounds to bewilder and daze...");
#endif
			start_singing(caster_ptr, spell, MUSIC_STUN);
		}

		{
			int dice = lev_bonus / 10;
			int sides = 2;

			if(info) return info_power_dice(dice, sides);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_STUN, diceroll(dice, sides));
			}
		}

		break;

	case 4:
#ifdef JP
		if(name) return "̗";
		if(desc) return "̗͂񕜂B";
#else
		if(name) return "Flow of Life";
		if(desc) return "Heals HP a little.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("̂ʂđ̂ɊC߂ĂDDD");
#else
			msg_print("Life flows through you as you sing a song of healing...");
#endif
			start_singing(caster_ptr, spell, MUSIC_L_LIFE);
		}

		{
			int dice = 2;
			int sides = 6;

			if(info) return info_heal(dice, sides, 0);
			if(cont) heal_creature(caster_ptr, diceroll(dice, sides));
		}

		break;

	case 5:
#ifdef JP
		if(name) return "z̉";
		if(desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if(name) return "Song of the Sun";
		if(desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
		if(cast || cont) do_active_trait(caster_ptr, TRAIT_LIGHT_AREA, TRUE, 100, 0L);
		break;

	case 6:
#ifdef JP
		if(name) return "|̉";
		if(desc) return "ȆSẴN[`[|BRƖB";
#else
		if(name) return "Song of Fear";
		if(desc) return "Attempts to scare all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("ǂ남ǂ낵fBtŎn߂DDD");
#else
			msg_print("You start weaving a fearful pattern...");
#endif
			start_singing(caster_ptr, spell, MUSIC_FEAR);			
		}

		{
			POWER power = lev_bonus;

			if(info) return info_power(power);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_TURN_ALL, power);
			}
		}

		break;

	case 7:
#ifdef JP
		if(name) return "킢̉";
		if(desc) return "q[[CɂȂB";
#else
		if(name) return "Heroic Ballad";
		if(desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif

		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("킢̉̂̂DDD");
#else
			msg_print("You start singing a song of intense fighting...");
#endif

			(void)heal_creature(caster_ptr, 10);
			(void)set_timed_trait(caster_ptr, TRAIT_AFRAID, 0, TRUE);
			prepare_update(caster_ptr, CRU_HP);

			start_singing(caster_ptr, spell, MUSIC_HERO);
		}

			if(stop) prepare_update(caster_ptr, CRU_HP);

		break;

	case 8:
#ifdef JP
		if(name) return "Imo";
		if(desc) return "߂//KimBx15őSẴN[`[A20ōƃACemł悤ɂȂBx25Ŏӂ̒n`mA40ł̊KŜivɏƂ炵A_ŴׂẴACemB̌ʂ͉̂邱ƂŏɋNB";
#else
		if(name) return "Clairaudience";
		if(desc) return "Detects traps, doors and stairs in your vicinity. And detects all creatures at level 15, treasures and items at level 20. Maps nearby area at level 25. Lights and know the whole level at level 40. These effects occurs by turns while this song continues.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("Âȉyo܂DDD");
#else
			msg_print("Your quiet music sharpens your sense of hearing...");
#endif

			/* Hack -- Initialize the turn count */
			caster_ptr->singing_turn = 0;

			start_singing(caster_ptr, spell, MUSIC_DETECT);
		}

		{
			COODINATES rad = DETECT_RAD_DEFAULT;

			if(info) return info_radius(rad);

			if(cont)
			{
				GAME_TURN count = caster_ptr->singing_turn;

				if(count >= 19) wiz_lite(floor_ptr, caster_ptr, FALSE);
				if(count >= 11)
				{
					map_area(caster_ptr, rad);
					if(lev_bonus > 39 && count < 19) caster_ptr->singing_turn = count + 1;
				}
				if(count >= 6)
				{
					/* There are too many hidden treasure.  So... */
					/* detect_treasure(rad); */
					detect_objects_gold(caster_ptr, rad);
					detect_objects_normal(caster_ptr, rad);

					if(lev_bonus > 24 && count < 11) caster_ptr->singing_turn = count + 1;
				}
				if(count >= 3)
				{
					detect_creatures_invis(caster_ptr, rad);
					detect_creatures_normal(caster_ptr, rad);
					if(lev_bonus > 19 && count < 6) caster_ptr->singing_turn = count + 1;
				}
				detect_traps(caster_ptr, rad, TRUE);
				detect_doors(caster_ptr, rad);
				detect_stairs(caster_ptr, rad);

				if(lev_bonus > 14 && count < 3) caster_ptr->singing_turn = count + 1;
			}
		}

		break;

	case 9:
#ifdef JP
		if(name) return "̉";
		if(desc) return "ȆSẴN[`[ɑ΂Đ_UsB";
#else
		if(name) return "Soul Shriek";
		if(desc) return "Damages all creatures in sight with PSI damages.";
#endif

		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("_PȂ̂̂DDD");
#else
			msg_print("You start singing a song of soul in pain...");
#endif
			start_singing(caster_ptr, spell, MUSIC_PSI);
		}

		{
			int dice = 1;
			int sides = lev_bonus * 3 / 2;

			if(info) return info_damage(dice, sides, 0);
			if(cont) project_all_vision(caster_ptr, DO_EFFECT_PSI, diceroll(dice, sides));
		}

		break;

	case 10:
#ifdef JP
		if(name) return "m̉";
		if(desc) return "̂}XƗׂ̃}XɗĂACeӒ肷B";
#else
		if(name) return "Song of Lore";
		if(desc) return "Identifies all items which are in the adjacent squares.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("̐E̒mꍞłDDD");
#else
			msg_print("You recall the rich lore of the world...");
#endif
			start_singing(caster_ptr, spell, MUSIC_ID);
		}

		{
			COODINATES rad = 1;

			if(info) return info_radius(rad);

			/*
			 * ̂̊JnɂʔF
			 * MPsŊӒ肪Oɉ̂fĂ܂̂h~B
			 */
			if(cont || cast)
			{
				project(caster_ptr, 0, rad, caster_ptr->fy, caster_ptr->fx, 0, DO_EFFECT_IDENTIFY, PROJECT_ITEM, -1);
			}
		}

		break;

	case 11:
#ifdef JP
		if(name) return "Bق̉";
		if(desc) return "Bs\͂㏸B";
#else
		if(name) return "Hiding Tune";
		if(desc) return "Gives improved stealth.";
#endif
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);
		if(cast) start_singing(caster_ptr, spell, MUSIC_STEALTH);

		break;

	case 12:
#ifdef JP
		if(name) return "e̐";
		if(desc) return "ȆSẴN[`[BRƖB";
#else
		if(name) return "Illusion Pattern";
		if(desc) return "Attempts to confuse all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("ӂʂɌeꂽDDD");
#else
			msg_print("You weave a pattern of sounds to beguile and confuse...");
#endif
			start_singing(caster_ptr, spell, MUSIC_CONF);
		}

		{
			POWER power = lev_bonus * 2;

			if(info) return info_power(power);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_CONFUSION, power);
			}
		}

		break;

	case 13:
#ifdef JP
		if(name) return "jł̋";
		if(desc) return "ȆSẴN[`[ɑ΂čUsB";
#else
		if(name) return "Doomcall";
		if(desc) return "Damages all creatures in sight with booming sound.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("DDD");
#else
			msg_print("The fury of the Downfall of Numenor lashes out...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SOUND);
		}

		{
			int dice = 10 + lev_bonus / 5;
			int sides = 7;

			if(info) return info_damage(dice, sides, 0);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_SOUND, diceroll(dice, sides));
			}
		}

		break;

	case 14:
#ifdef JP
		if(name) return "tBG̉";
		if(desc) return "͂̎̂⍜𐶂ԂB";
#else
		if(name) return "Firiel's Song";
		if(desc) return "Resurrects nearby corpse and skeletons. And makes these your pets.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ANIM_DEAD, TRUE);
		break;

	case 15:
#ifdef JP
		if(name) return "̒";
		if(desc) return "ȆSẴN[`[𖣗BRƖB";
#else
		if(name) return "Fellowship Chant";
		if(desc) return "Attempts to charm all creatures in sight.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DOMINATE_LIVES, TRUE, 100, 0L);
		if(cont) do_active_trait(caster_ptr, TRAIT_DOMINATE_LIVES, TRUE, 100, 0L);
		break;

	case 16:
#ifdef JP
		if(name) return "g";
		if(desc) return "ǂ@iށB̑̃ACe͏B";
#else
		if(name) return "Sound of disintegration";
		if(desc) return "Makes you be able to burrow into walls. Objects under your feet evaporate.";
#endif

		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("ӂ郁fBtŎn߂DDD");
#else
			msg_print("You weave a violent pattern of sounds to break wall.");
#endif
			start_singing(caster_ptr, spell, MUSIC_WALL);
		}

		{
			/*
			 * ̂̊JnɂʔF
			 * MPsŌʂOɉ̂fĂ܂̂h~B
			 */
			if(cont || cast)
			{
				project(caster_ptr, 0, 0, caster_ptr->fy, caster_ptr->fx,
					0, DO_EFFECT_DISINTEGRATE, PROJECT_KILL | PROJECT_ITEM | PROJECT_HIDE, -1);
			}
		}
		break;

	case 17:
#ifdef JP
		if(name) return "fϐ";
		if(desc) return "_AdAACAłɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if(name) return "Finrod's Resistance";
		if(desc) return "Gives resistance to fire, cold, electricity, acid and poison. These resistances can be added to which from equipment for more powerful resistances.";
#endif
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_MAGIC_RES_ELEMENT, TRUE);
		break;

	case 18:
#ifdef JP
		if(name) return "zrbg̃fB";
		if(desc) return "B";
#else
		if(name) return "Hobbit Melodies";
		if(desc) return "Hastes you.";
#endif

		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("yȉ̂ݎn߂DDD");
#else
			msg_print("You start singing joyful pop song...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SPEED);
		}

		break;

	case 19:
#ifdef JP
		if(name) return "c񂾐E";
		if(desc) return "߂̃N[`[e|[gBRƖB";
#else
		if(name) return "World Contortion";
		if(desc) return "Teleports all nearby creatures away unless resisted.";
#endif
    
		{
			COODINATES rad = (COODINATES)lev_bonus / 15 + 1;
			POWER power = lev_bonus * 3 + 1;

			if(info) return info_radius(rad);

			/* Stop singing before start another */
			if(cast || fail) stop_singing(caster_ptr);

			if(cast)
			{
#ifdef JP
				msg_print("̂Ԃc߂DDD");
#else
				msg_print("Reality whirls wildly as you sing a dizzying melody...");
#endif

				project(caster_ptr, 0, rad, caster_ptr->fy, caster_ptr->fx, power, DO_EFFECT_AWAY_ALL, PROJECT_KILL, -1);
			}
		}
		break;

	case 20:
#ifdef JP
		if(name) return "ގỦ";
		if(desc) return "ȆSẴN[`[Ƀ_[W^B׈ȃN[`[ɓɑ傫ȃ_[W^B";
#else
		if(name) return "Dispelling chant";
		if(desc) return "Damages all creatures in sight. Hurts evil creatures greatly.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("ςȂsaGӂߗĂDDD");
#else
			msg_print("You cry out in an ear-wracking voice...");
#endif
			start_singing(caster_ptr, spell, MUSIC_DISPEL);
		}

		{
			int m_sides = lev_bonus * 3;
			int e_sides = lev_bonus * 3;

			if(info) return format("%s1d%d+1d%d", s_dam, m_sides, e_sides);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_DISP_ALL, randint1(m_sides));
				project_all_vision(caster_ptr, DO_EFFECT_DISP_EVIL, randint1(e_sides));
			}
		}
		break;

	case 21:
#ifdef JP
		if(name) return "T}̊Ì";
		if(desc) return "ȆSẴN[`[A点悤ƂBRƖB";
#else
		if(name) return "The Voice of Saruman";
		if(desc) return "Attempts to slow and sleep all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("DA͓Iȉ̂ݎn߂DDD");
#else
			msg_print("You start humming a gentle and attractive song...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SARUMAN);
		}

		{
			POWER power = lev_bonus;

			if(info) return info_power(power);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_SLOW, power);
				project_all_vision(caster_ptr, DO_EFFECT_SLEEP, power);
			}
		}

		break;

	case 22:
#ifdef JP
		if(name) return "̉F";
		if(desc) return "̃r[B";
#else
		if(name) return "Song of the Tempest";
		if(desc) return "Fires a beam of sound.";
#endif
    
		{
			int dice = 15 + (lev_bonus - 1) / 2;
			int sides = 10;

			if(info) return info_damage(dice, sides, 0);

			/* Stop singing before start another */
			if(cast || fail) stop_singing(caster_ptr);
			if(cast) cast_beam(caster_ptr, DO_EFFECT_SOUND, MAX_RANGE_SUB, diceroll(dice, sides), 0);
		}
		break;

	case 23:
#ifdef JP
		if(name) return "̐E";
		if(desc) return "݂̊Kč\B";
#else
		if(name) return "Ambarkanta";
		if(desc) return "Recreates current dungeon level.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_SHADOW_SHIFT, TRUE);
		break;

	case 24:
#ifdef JP
		if(name) return "j̐";
		if(desc) return "͂̃_Wh炵AǂƏ_ɓςB";
#else
		if(name) return "Wrecking Pattern";
		if(desc) return "Shakes dungeon structure, and results in random swapping of floors and walls.";
#endif

		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("jIȉ̂킽DDD");
#else
			msg_print("You weave a pattern of sounds to contort and shatter...");
#endif
			start_singing(caster_ptr, spell, MUSIC_QUAKE);
		}

		{
			COODINATES rad = 10;

			if(info) return info_radius(rad);

			if(cont)
			{
				earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 10);
			}
		}

		break;


	case 25:
#ifdef JP
		if(name) return "؂̉";
		if(desc) return "ȆSẴN[`[Ⴢ悤ƂBRƖB";
#else
		if(name) return "Stationary Shriek";
		if(desc) return "Attempts to freeze all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("ƂfBtŎn߂DDD");
#else
			msg_print("You weave a very slow pattern which is almost likely to stop...");
#endif
			start_singing(caster_ptr, spell, MUSIC_STASIS);
		}

		{
			POWER power = lev_bonus * 4;

			if(info) return info_power(power);

			if(cont)
			{
				project_all_vision(caster_ptr, DO_EFFECT_HOLD, power);
			}
		}

		break;

	case 26:
#ifdef JP
		if(name) return "̉";
		if(desc) return "̂鏰̏ɁAN[`[ʂ蔲菢ꂽ肷邱ƂłȂȂ郋[`B";
#else
		if(name) return "Endurance";
		if(desc) return "Sets a glyph on the floor beneath you. Creatures cannot attack you if you are on a glyph, but can try to break glyph.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_PROTECT_RUNE, TRUE);
		break;

	case 27:
#ifdef JP
		if(name) return "pY̎";
		if(desc) return "Aq[[CɂȂAȆSẴN[`[Ƀ_[W^B";
#else
		if(name) return "The Hero's Poem";
		if(desc) return "Hastes you. Gives heroism. Damages all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("pỶ̂񂾁DDD");
#else
			msg_print("You chant a powerful, heroic call to arms...");
#endif
			(void)heal_creature(caster_ptr, 10);
			(void)set_timed_trait(caster_ptr, TRAIT_AFRAID, 0, TRUE);
			prepare_update(caster_ptr, CRU_HP);

			start_singing(caster_ptr, spell, MUSIC_SHERO);
		}

		if(stop) prepare_update(caster_ptr, CRU_HP);

		{
			int dice = 1;
			int sides = lev_bonus * 3;

			if(info) return info_damage(dice, sides, 0);

			if(cont) project_all_vision(caster_ptr, DO_EFFECT_DISP_ALL, diceroll(dice, sides));
		}
		break;

	case 28:
#ifdef JP
		if(name) return "@ȉ";
		if(desc) return "͂ȉ񕜂̉̂ŁAƞNOԂSB";
#else
		if(name) return "Relief of Yavanna";
		if(desc) return "Powerful healing song. Also heals cut and stun completely.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
			msg_print("̂ʂđ̂ɊC߂ĂDDD");
#else
			msg_print("Life flows through you as you sing the song...");
#endif
			start_singing(caster_ptr, spell, MUSIC_H_LIFE);
		}

		{
			int dice = 15;
			int sides = 10;

			if(info) return info_heal(dice, sides, 0);
			if(cont) heal_creature(caster_ptr, diceroll(dice, sides));
		}

		break;

	case 29:
#ifdef JP
		if(name) return "Đ̉";
		if(desc) return "ׂẴXe[^Xƌol񕜂B";
#else
		if(name) return "Goddess' rebirth";
		if(desc) return "Restores all stats and experience.";
#endif
    
		{
			/* Stop singing before start another */
			if(cast || fail) stop_singing(caster_ptr);

			if(cast) do_active_trait_tmp(caster_ptr, TRAIT_RESTORE_ALL, TRUE);
		}
		break;

	case 30:
#ifdef JP
		if(name) return "TE̖p";
		if(desc) return "ɋ͂ł̋B";
#else
		if(name) return "Wizardry of Sauron";
		if(desc) return "Fires an extremely powerful tiny ball of sound.";
#endif
    
		{
			int dice = 50 + lev_bonus;
			int sides = 10;
			COODINATES rad = 0;

			if(info) return info_damage(dice, sides, 0);

			/* Stop singing before start another */
			if(cast || fail) stop_singing(caster_ptr);

			if(cast)
			{
				if(!get_aim_dir(caster_ptr, MAX_RANGE_SUB, &dir)) return NULL;

				cast_ball(caster_ptr, DO_EFFECT_SOUND, MAX_RANGE_SUB, diceroll(dice, sides), rad);
			}
		}
		break;

	case 31:
#ifdef JP
		if(name) return "tBStB̒";
		if(desc) return "_[W󂯂ȂȂoA𒣂B";
#else
		if(name) return "Fingolfin's Challenge";
		if(desc) return "Generates barrier which completely protect you from almost all damages. Takes a few your turns when the barrier breaks.";
#endif
    
		/* Stop singing before start another */
		if(cast || fail) stop_singing(caster_ptr);

		if(cast)
		{
#ifdef JP
				msg_print("tBStB̖ւ̒̂DDD");
#else
				msg_print("You recall the valor of Fingolfin's challenge to the Dark Lord...");
#endif

				prepare_redraw(PR_MAP);
				prepare_update(caster_ptr, PU_CREATURES);
				prepare_window(PW_OVERHEAD | PW_DUNGEON);
				start_singing(caster_ptr, spell, MUSIC_INVULN);
		}

		if(stop)
		{
			if(!has_trait(caster_ptr, TRAIT_INVULNERABLE))
			{
#ifdef JP
				msg_print("Gł͂ȂȂB");
#else
				msg_print("The invulnerability wears off.");
#endif
				prepare_redraw(PR_MAP);
				prepare_update(caster_ptr, PU_CREATURES);
				prepare_window(PW_OVERHEAD | PW_DUNGEON);
			}
		}

		break;
	}

	return "";
}


static cptr do_hissatsu_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	DIRECTION dir;
	int lev_bonus = caster_ptr->lev;
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);

	switch (spell)
	{
	case 0:
#ifdef JP
		if(name) return "эj";
		if(desc) return "2}XꂽƂɂN[`[UB";
#else
		if(name) return "Tobi-Izuna";
		if(desc) return "Attacks a two squares distant creature.";
#endif
    
		if(cast)
		{
			if(!get_aim_dir(caster_ptr, 2, &dir)) return NULL;
			cast_bolt(caster_ptr, DO_EFFECT_ATTACK, HISSATSU_2, PROJECT_STOP | PROJECT_KILL, -1);
		}
		break;

	case 1:
#ifdef JP
		if(name) return "܌Ja";
		if(desc) return "3ɑ΂čUB";
#else
		if(name) return "3-Way Attack";
		if(desc) return "Attacks in 3 directions in one time.";
#endif
    
		if(cast)
		{
			int cdir;
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			for (cdir = 0;cdir < 8; cdir++)
			{
				if(cdd[cdir] == dir) break;
			}

			if(cdir == 8) return NULL;

			y = caster_ptr->fy + ddy_cdd[cdir];
			x = caster_ptr->fx + ddx_cdd[cdir];
			close_combat(caster_ptr, y, x, 0);
			y = caster_ptr->fy + ddy_cdd[(cdir + 7) % 8];
			x = caster_ptr->fx + ddx_cdd[(cdir + 7) % 8];
			close_combat(caster_ptr, y, x, 0);
			y = caster_ptr->fy + ddy_cdd[(cdir + 1) % 8];
			x = caster_ptr->fx + ddx_cdd[(cdir + 1) % 8];
			close_combat(caster_ptr, y, x, 0);
		}
		break;

	case 2:
#ifdef JP
		if(name) return "u[";
		if(desc) return "茳ɖ߂Ă悤ɓB߂ĂȂƂB";
#else
		if(name) return "Boomerang";
		if(desc) return "Throws current weapon. And it'll return to your hand unless failed.";
#endif
    
		if(cast)
		{
			if(!do_cmd_throw_aux(caster_ptr, 1, TRUE, 0)) return NULL;
		}
		break;

	case 3:
#ifdef JP
		if(name) return "";
		if(desc) return "Ήϐ̂ȂN[`[ɑ_[W^B";
#else
		if(name) return "Burning Strike";
		if(desc) return "Attacks a creature with more damage unless it has resistance to fire.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_FIRE);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 4:
#ifdef JP
		if(name) return "ECm";
		if(desc) return "߂̎vl邱ƂłN[`[mB";
#else
		if(name) return "Detect Ferocity";
		if(desc) return "Detects all creatures except mindless in your vicinity.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_DETECT_FELOCITY, TRUE, 100, 0L);
		break;

	case 5:
#ifdef JP
		if(name) return "݂ˑł";
		if(desc) return "Ƀ_[W^ȂANOƂB";
#else
		if(name) return "Strike to Stun";
		if(desc) return "Attempts to stun a creature in the adjacent.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_MINEUCHI);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 6:
#ifdef JP
		if(name) return "JE^[";
		if(desc) return "ɍUꂽƂɔB邽тMPB";
#else
		if(name) return "Counter";
		if(desc) return "Prepares to counterattack. When attack by a creature, strikes back using SP each time.";
#endif
    
		if(cast)
		{
			if(caster_ptr->riding)
			{
				msg_print(MES_PREVENT_BY_RIDING);
				return NULL;
			}
			msg_print(MES_MELEE_COUNTER_PREPARE);
			caster_ptr->counter = TRUE;
		}
		break;

	case 7:
#ifdef JP
		if(name) return "";
		if(desc) return "UAΑɔB";
#else
		if(name) return "Harainuke";
		if(desc) return "Attacks creature with your weapons normally, then move through counter side of the creature.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(caster_ptr->riding)
			{
				msg_print(MES_PREVENT_BY_RIDING);
				return NULL;
			}
	
			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
	
			if(dir == 5) return NULL;
			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];
	
			if(!floor_ptr->cave[y][x].creature_idx)
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
	
			close_combat(caster_ptr, y, x, 0);
	
			if(!creature_can_cross_terrain(caster_ptr, floor_ptr->cave[y][x].feat, 0) || is_trap(floor_ptr->cave[y][x].feat))
				break;
	
			y += ddy[dir];
			x += ddx[dir];
	
			if(creature_can_cross_terrain(caster_ptr, floor_ptr->cave[y][x].feat, 0) && !is_trap(floor_ptr->cave[y][x].feat) && !floor_ptr->cave[y][x].creature_idx)
			{
				msg_print(NULL);
	
				/* Move the player */
				(void)move_creature(caster_ptr, NULL, y, x, MCE_FORGET_FLOW | MCE_HANDLE_STUFF | MCE_DONT_PICKUP);
			}
		}
		break;

	case 8:
#ifdef JP
		if(name) return "T[yc^";
		if(desc) return "őϐ̂ȂN[`[ɑ_[W^B";
#else
		if(name) return "Serpent's Tongue";
		if(desc) return "Attacks a creature with more damage unless it has resistance to poison.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_POISON);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 9:
#ifdef JP
		if(name) return "ȃ";
		if(desc) return "̂Ȃ׈ȃN[`[ɑ_[W^邪ÃN[`[ɂ͑SʂȂB";
#else
		if(name) return "Zammaken";
		if(desc) return "Attacks an evil unliving creature with great damage. No effect to other  creatures.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_ZANMA);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 10:
#ifdef JP
		if(name) return "";
		if(desc) return "U֐΂B";
#else
		if(name) return "Wind Blast";
		if(desc) return "Attacks an adjacent creature, and blow it away.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, 0);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
			if(dungeon_info[floor_ptr->dungeon_id].flags1 & DF1_NO_MELEE)
			{
				return "";
			}
			if(floor_ptr->cave[y][x].creature_idx)
			{
				int i;
				COODINATES ty = y, tx = x;
				COODINATES oy = y, ox = x;
				CREATURE_ID creature_idx = floor_ptr->cave[y][x].creature_idx;
				creature_type *m_ptr = &creature_list[creature_idx];
				char m_name[MAX_NLEN];
	
				creature_desc(m_name, m_ptr, 0);
	
				for (i = 0; i < 5; i++)
				{
					y += ddy[dir];
					x += ddx[dir];
					if(cave_empty_bold(floor_ptr, y, x))
					{
						ty = y;
						tx = x;
					}
					else break;
				}
				if((ty != oy) || (tx != ox))
				{
					msg_format(MES_BLOE_AWAY, m_name);
					floor_ptr->cave[oy][ox].creature_idx = 0;
					floor_ptr->cave[ty][tx].creature_idx = creature_idx;
					m_ptr->fy = ty;
					m_ptr->fx = tx;
	
					update_creature_view(player_ptr, creature_idx, TRUE);
					lite_spot(floor_ptr, oy, ox);
					lite_spot(floor_ptr, ty, tx);
	
					if(is_lighting_creature(m_ptr) || is_darken_creature(m_ptr))
						prepare_update(caster_ptr, PU_SPECIES_LITE);
				}
			}
		}
		break;

	case 11:
#ifdef JP
		if(name) return "̖ڗ";
		if(desc) return "Eh1ʂBx45ȏŕEh̔\͂Sɒm邱ƂłB";
#else
		if(name) return "Judge";
		if(desc) return "Identifies a weapon or armor. Or *identifies* these at level 45.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_WEAPON_MASTER_JUDGE, TRUE, 100, 0L);
		break;

	case 12:
#ifdef JP
		if(name) return "ja";
		if(desc) return "󂵁AΌñN[`[ɑ_[W^B";
#else
		if(name) return "Rock Smash";
		if(desc) return "Breaks rock. Or greatly damage a creature made by rocks.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_HAGAN);
	
			if(!CAVE_HAVE_FLAG_BOLD(floor_ptr, y, x, FF_HURT_ROCK)) break;
	
			/* Destroy the feature */
			cave_alter_feat(floor_ptr, y, x, FF_HURT_ROCK);
	
			/* Update some things */
			prepare_update(caster_ptr, PU_FLOW);
		}
		break;

	case 13:
#ifdef JP
		if(name) return "ጎ";
		if(desc) return "U񐔂ACϐ̂ȂN[`[ɑ_[W^B";
#else
		if(name) return "Midare-Setsugekka";
		if(desc) return "Attacks a creature with increased number of attacks and more damage unless it has resistance to cold.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_COLD);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 14:
#ifdef JP
		if(name) return "}˂";
		if(desc) return "N[`[ꌂœ|UJoBs1__[W^ȂB";
#else
		if(name) return "Spot Aiming";
		if(desc) return "Attempts to kill a creature instantly. If failed cause only 1HP of damage.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_KYUSHO);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 15:
#ifdef JP
		if(name) return "_a";
		if(desc) return "S̈ꌂōUBU킳₷B";
#else
		if(name) return "Majingiri";
		if(desc) return "Attempts to attack with critical hit. But this attack is easy to evade for a creature.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_MAJIN);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 16:
#ifdef JP
		if(name) return "̂Đg";
		if(desc) return "͂ȍUJoB̃^[܂ł̊ԁAH炤_[WB";
#else
		if(name) return "Desperate Attack";
		if(desc) return "Attacks with all of your power. But all damages you take will be doubled for one turn.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_SUTEMI);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
			caster_ptr->sutemi = TRUE;
		}
		break;

	case 17:
#ifdef JP
		if(name) return "h܎a";
		if(desc) return "dϐ̂ȂN[`[ɔɑ傫_[W^B";
#else
		if(name) return "Lightning Eagle";
		if(desc) return "Attacks a creature with more damage unless it has resistance to electricity.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_ELEC);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 18:
#ifdef JP
		if(name) return "g";
		if(desc) return "fɋߊUB";
#else
		if(name) return "Rush Attack";
		if(desc) return "Steps close to a creature and attacks at a time.";
#endif
    
		if(cast)
		{
			if(!rush_attack(caster_ptr, NULL)) return NULL;
		}
		break;

	case 19:
#ifdef JP
		if(name) return "ԗQ";
		if(desc) return "gȀ[قǑ傫З͂őS̓GUłBĂȂN[`[ɂ͌ʂȂB";
#else
		if(name) return "Bloody Maelstrom";
		if(desc) return "Attacks all adjacent creatures with power corresponding to your cut status. Then increases your cut status. No effect to unliving creatures.";
#endif
    
		if(cast)
		{
			int y = 0, x = 0;

			cave_type       *c_ptr;
			creature_type    *m_ptr;
	
			if(caster_ptr->timed_trait[TRAIT_CUT] < 300)
				add_timed_trait(caster_ptr, TRAIT_CUT, 300, TRUE);
			else
				set_timed_trait(caster_ptr, TRAIT_CUT, caster_ptr->timed_trait[TRAIT_CUT] * 2, TRUE);
	
			for (dir = 0; dir < 8; dir++)
			{
				y = caster_ptr->fy + ddy_ddd[dir];
				x = caster_ptr->fx + ddx_ddd[dir];
				c_ptr = &floor_ptr->cave[y][x];
	
				/* Get the creature */
				m_ptr = &creature_list[c_ptr->creature_idx];
	
				/* Hack -- attack creatures */
				if(c_ptr->creature_idx && (m_ptr->see_others || CAVE_HAVE_FLAG_BOLD(floor_ptr, y, x, FF_PROJECT)))
				{
					if(!creature_living(m_ptr))
					{
						char m_name[MAX_NLEN];
	
						creature_desc(m_name, m_ptr, 0);
						msg_format(MES_IS_UNAFFECTED); // TODO
					}
					else close_combat(caster_ptr, y, x, HISSATSU_SEKIRYUKA);
				}
			}
		}
		break;

	case 20:
#ifdef JP
		if(name) return "k";
		if(desc) return "nkNB";
#else
		if(name) return "Earthquake Blow";
		if(desc) return "Shakes dungeon structure, and results in random swapping of floors and walls.";
#endif
    
		if(cast)
		{
			int y,x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx) close_combat(caster_ptr, y, x, HISSATSU_QUAKE);
			else earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 10);
		}
		break;

	case 21:
#ifdef JP
		if(name) return "n";
		if(desc) return "Ռg̃r[B";
#else
		if(name) return "Crack";
		if(desc) return "Fires a beam of shock wave.";
#endif
    
		if(cast)
		{
			int total_damage = 0, basedam, i;
			u32b flgs[MAX_TRAITS_FLAG];
			object_type *object_ptr;
			if(!get_aim_dir(caster_ptr, MAX_RANGE_SUB, &dir)) return NULL;
			msg_print(MES_TRAIT_CRACK_DONE);
			for (i = 0; i < 2; i++) // TODO
			{
				POWER damage;
	
				if(!get_equipped_slot_ptr(caster_ptr, SLOT_ID_HAND, i)) break;
				object_ptr = get_equipped_slot_ptr(caster_ptr, SLOT_ID_HAND, i);
				basedam = (object_ptr->dd * (object_ptr->ds + 1)) * 50;
				damage = object_ptr->to_damage * 100;
				object_flags(object_ptr, flgs);
				if(have_flag(flgs, TRAIT_SUPER_VORPAL))
				{
					basedam *= 5;
					basedam /= 3;
				}
				else if(have_flag(flgs, TRAIT_VORPAL))
				{
					basedam *= 11;
					basedam /= 9;
				}
				damage += basedam;
				total_damage += damage / 200;
				if(i) total_damage = total_damage*7/10;
			}
			cast_beam(caster_ptr, DO_EFFECT_FORCE, MAX_RANGE_SUB, total_damage, 0);
		}
		break;

	case 22:
#ifdef JP
		if(name) return "C̗Y";
		if(desc) return "ȆSN[`[ɑ΂č̍UsBɁA߂ɂN[`[{点B";
#else
		if(name) return "War Cry";
		if(desc) return "Damages all creatures in sight with sound. Aggravate nearby creatures.";
#endif
		if(cast)
		{
			msg_print(MES_TRAIT_WARCRY_DONE);
			project_all_vision(caster_ptr, DO_EFFECT_SOUND, randint1(lev_bonus * 3));
			aggravate_creatures(caster_ptr);
		}
		break;

	case 23:
#ifdef JP
		if(name) return "oOi";
		if(desc) return "͂3iUJoB";
#else
		if(name) return "Musou-Sandan";
		if(desc) return "Attacks with powerful 3 strikes.";
#endif
    
		if(cast)
		{
			int i;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			for (i = 0; i < 3; i++)
			{
				COODINATES y, x, ny, nx;
				CREATURE_ID creature_idx;
				cave_type *c_ptr;
				creature_type *m_ptr;
	
				y = caster_ptr->fy + ddy[dir];
				x = caster_ptr->fx + ddx[dir];
				c_ptr = &floor_ptr->cave[y][x];
	
				if(c_ptr->creature_idx)
					close_combat(caster_ptr, y, x, HISSATSU_3DAN);
				else
				{
					msg_print(MES_NO_DICRECTION_CREATURE);
					return NULL;
				}
	
				if(dungeon_info[floor_ptr->dungeon_id].flags1 & DF1_NO_MELEE)
				{
					return "";
				}
	
				/* Creature is dead? */
				if(!c_ptr->creature_idx) break;
	
				ny = y + ddy[dir];
				nx = x + ddx[dir];
				creature_idx = c_ptr->creature_idx;
				m_ptr = &creature_list[creature_idx];
	
				/* Creature cannot move back? */
				if(!creature_can_enter(ny, nx, m_ptr, 0))
				{
					/* -more- */
					if(i < 2) msg_print(NULL);
					continue;
				}
	
				c_ptr->creature_idx = 0;
				floor_ptr->cave[ny][nx].creature_idx = creature_idx;
				m_ptr->fy = ny;
				m_ptr->fx = nx;
				update_creature_view(player_ptr, creature_idx, TRUE);
				lite_spot(floor_ptr, y, x);
				lite_spot(floor_ptr, ny, nx);

				if(creature_can_cross_terrain(caster_ptr, c_ptr->feat, 0)) /* Player can move forward? */
				{
					if(!move_creature(caster_ptr, NULL, y, x, MCE_FORGET_FLOW | MCE_HANDLE_STUFF | MCE_DONT_PICKUP)) break;  /* Move the player */
				}
				else break;
				if(i < 2) msg_print(NULL); /* -more- */
			}
		}
		break;

	case 24:
#ifdef JP
		if(name) return "zS̉";
		if(desc) return "Ȗ̗͂zƂȂ̗͂񕜂BȂN[`[ɂ͒ʂȂB";
#else
		if(name) return "Vampire's Fang";
		if(desc) return "Attacks with vampiric strikes which absorbs HP from a creature and gives them to you. No effect to unliving creatures.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_DRAIN);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 25:
#ifdef JP
		if(name) return "f";
		if(desc) return "E̋NĂSN[`[ɞNOAA^悤ƂB";
#else
		if(name) return "Moon Dazzling";
		if(desc) return "Attempts to stun, confuse and sleep all waking creatures.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MOON_DAZZLING, TRUE, 100, 0L);
		break;

	case 26:
#ifdef JP
		if(name) return "Sla";
		if(desc) return "AēgŃN[`[UBU邽тMPBMPȂȂ邩AN[`[|ȂSla͏IB";
#else
		if(name) return "Hundred Slaughter";
		if(desc) return "Performs a series of rush attacks. The series continues while killing each creature in a time and SP remains.";
#endif
		if(cast)
		{
			const int mana_cost_per_creature = 8;
			bool new = TRUE;
			bool mdeath;
			do
			{
				if(!rush_attack(caster_ptr, &mdeath)) break;
				if(new) /* Reserve needed mana point */
				{
					dec_mana(caster_ptr, technic_info[REALM_HISSATSU - MIN_TECHNIC][26].smana); 
					new = FALSE;
				}
				else dec_mana(caster_ptr, mana_cost_per_creature);

				if(!mdeath) break;
				command_dir = 0;

				prepare_redraw(PR_MANA);
				handle_stuff(caster_ptr);
			}
			while (caster_ptr->csp > mana_cost_per_creature);

			if(new) return NULL;
			inc_mana(caster_ptr, technic_info[REALM_HISSATSU - MIN_TECHNIC][26].smana); /* Restore reserved mana */
		}
		break;

	case 27:
#ifdef JP
		if(name) return "VėM";
		if(desc) return "Ȅꏊw肵āȀꏊƎ̊ԂɂSN[`[UȀꏊɈړB";
#else
		if(name) return "Dragonic Flash";
		if(desc) return "Runs toward given location while attacking all creatures on the path.";
#endif
    
		if(cast)
		{
			COODINATES y, x;
			if(!tgt_pt(caster_ptr, &x, &y)) return NULL;
			if(!cave_player_teleportable_bold(caster_ptr, y, x, 0L) ||
			    (distance(y, x, caster_ptr->fy, caster_ptr->fx) > MAX_SIGHT / 2) ||
			    !projectable(floor_ptr, MAX_RANGE, caster_ptr->fy, caster_ptr->fx, y, x))
			{
				msg_print(MES_TRAIT_FAILED_MOVING_ATTACK);
				break;
			}
			if(has_trait(caster_ptr, TRAIT_PREVENT_TELEPORT))
			{
				msg_print(MES_TELEPORT_PREVENTED);
				break;
			}
			project(caster_ptr, 0, 0, y, x, HISSATSU_ISSEN, DO_EFFECT_ATTACK, PROJECT_BEAM | PROJECT_KILL, -1);
			teleport_creature_to(caster_ptr, y, x, 0L);
		}
		break;

	case 28:
#ifdef JP
		if(name) return "ď";
		if(desc) return "1^[2xUsB";
#else
		if(name) return "Twin Slash";
		if(desc) return "double attacks at a time.";
#endif
    
		if(cast)
		{
			int x, y;
	
			if(!get_rep_dir(caster_ptr, &dir, FALSE)) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
			{
				close_combat(caster_ptr, y, x, 0);
				if(floor_ptr->cave[y][x].creature_idx)
				{
					handle_stuff(caster_ptr);
					close_combat(caster_ptr, y, x, 0);
				}
			}
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
		}
		break;

	case 29:
#ifdef JP
		if(name) return "ՕⓁ";
		if(desc) return "͂ȍUsA߂̏ꏊɂʂyԁB";
#else
		if(name) return "Kofuku-Zettousei";
		if(desc) return "Performs a powerful attack which even effect nearby creatures.";
#endif
    
		if(cast)
		{
			int total_damage = 0, basedam, i;
			COODINATES y, x;
			u32b flgs[MAX_TRAITS_FLAG];
			object_type *object_ptr;
	
			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(is_melee_limitation_field(floor_ptr)) return "";

			msg_print(MES_TRAIT_KOFUKU_DONE);
			for (i = 0; i < 2; i++)
			{
				POWER damage;
				if(!get_equipped_slot_ptr(caster_ptr, SLOT_ID_HAND, i)) break;
				object_ptr = get_equipped_slot_ptr(caster_ptr, SLOT_ID_HAND, i);
				basedam = (object_ptr->dd * (object_ptr->ds + 1)) * 50;
				damage = object_ptr->to_damage * 100;
				object_flags(object_ptr, flgs);
				if(have_flag(flgs, TRAIT_SUPER_VORPAL))
				{
					/* vorpal blade */
					basedam *= 5;
					basedam /= 3;
				}
				else if(have_flag(flgs, TRAIT_VORPAL))
				{
					/* vorpal flag only */
					basedam *= 11;
					basedam /= 9;
				}
				damage += basedam;
				damage += caster_ptr->to_damage[i] * 100;
				total_damage += (damage / 100);
			}
			project(caster_ptr, 0, (CAVE_HAVE_FLAG_BOLD(floor_ptr, y, x, FF_PROJECT) ? 5 : 0), y, x, total_damage * 3 / 2, DO_EFFECT_METEOR, PROJECT_KILL | PROJECT_JUMP | PROJECT_ITEM, -1);
		}
		break;

	case 30:
#ifdef JP
		if(name) return "c_SE";
		if(desc) return "_[W炤Aɔɑ傫ȃ_[W^BAfbhɂ͓ɌʂB";
#else
		if(name) return "Keiun-Kininken";
		if(desc) return "Attacks a creature with extremely powerful damage. But you also takes some damages. Hurts a undead creature greatly.";
#endif
    
		if(cast)
		{
			COODINATES y, x;

			if(!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if(dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if(floor_ptr->cave[y][x].creature_idx)
				close_combat(caster_ptr, y, x, HISSATSU_UNDEAD);
			else
			{
				msg_print(MES_NO_DICRECTION_CREATURE);
				return NULL;
			}
#ifdef JP
			take_damage_to_creature(NULL, caster_ptr, DAMAGE_NOESCAPE, 100 + randint1(100), "c_SEgՌ", NULL, -1);
#else
			take_damage_to_creature(NULL, caster_ptr, DAMAGE_NOESCAPE, 100 + randint1(100), "exhaustion on using Keiun-Kininken", NULL, -1);
#endif
		}
		break;

	case 31:
#ifdef JP
		if(name) return "ؕ";
		if(desc) return "umƂ́AʂƂƌBv";
#else
		if(name) return "Harakiri";
		if(desc) return "'Busido is found in death'";
#endif

		if(cast)
		{
			int i;
			if(!get_check(MES_GET_CHECK_SUICIDE)) return NULL;
			prt(MES_SYS_HIT_VERIFY_KEY, 0, 0);
	
			flush();
			i = inkey();
			prt("", 0, 0);
			if(i != '@') return NULL;
			if(caster_ptr->total_winner)
			{
				take_damage_to_creature(NULL, caster_ptr, DAMAGE_FORCE, 9999, COD_SEPPUKU, NULL, -1);
				caster_ptr->total_winner = TRUE;
			}
			else
			{
				msg_print(MES_TRAIT_SEPPUKU_DONE);
				take_damage_to_creature(NULL, caster_ptr, DAMAGE_FORCE, 9999, COD_SEPPUKU, NULL, -1);
			}
		}
		break;
	}

	return "";
}

/* Hex */
static bool item_tester_hook_weapon_except_bow(creature_type *creature_ptr, object_type *object_ptr)
{
	if(!is_valid_creature(creature_ptr)) return FALSE;
	if(!is_valid_object(object_ptr)) return FALSE;
	switch (object_ptr->tval)
	{
		case TV_SWORD:
		case TV_HAFTED:
		case TV_POLEARM:
		case TV_DIGGING:
		{
			return TRUE;
		}
	}
	return FALSE;
}

static bool item_tester_hook_cursed(creature_type *creature_ptr, object_type *object_ptr)
{
	if(!is_valid_creature(creature_ptr)) return FALSE;
	if(!is_valid_object(object_ptr)) return FALSE;
	return (bool)(object_is_cursed(object_ptr));
}

static cptr do_hex_spell(creature_type *caster_ptr, int spell, int mode)
{
	floor_type *floor_ptr = GET_FLOOR_PTR(caster_ptr);

	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	bool cont = (mode == SPELL_CONT) ? TRUE : FALSE;
	bool stop = (mode == SPELL_STOP) ? TRUE : FALSE;

	bool add = TRUE;

	int lev_bonus = caster_ptr->lev;
	POWER power;

	switch (spell)
	{
	/*** 1st book (0-7) ***/
	case 0:
#ifdef JP
		if(name) return "ׂȂj";
		if(desc) return "jɂUxƖh͂オB";
#else
		if(name) return "Evily blessing";
		if(desc) return "Attempts to increase +to_hit of a weapon and AC";
#endif
		break;

	case 1:
#ifdef JP
		if(name) return "y̎";
		if(desc) return "HP⏝񕜂B";
#else
		if(name) return "Cure light wounds";
		if(desc) return "Heals cut and HP a little.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(1, 10), 0L);
		if(cont) do_active_trait(caster_ptr, TRAIT_HEAL, FALSE, diceroll(1, 10), 0L);
		break;

	case 2:
#ifdef JP
		if(name) return "̃I[";
		if(desc) return "̃I[gɂ܂ƂA񕜑xȂB";
#else
		if(name) return "Demonic aura";
		if(desc) return "Gives fire aura and regeneration.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("̂̃I[ŕꂽB");
#else
			msg_print("You have enveloped by fiery aura!");
#endif
		}
		if(stop)
		{
#ifdef JP
			msg_print("̃I[B");
#else
			msg_print("Fiery aura disappeared.");
#endif
		}
		break;

	case 3:
#ifdef JP
		if(name) return "L";
		if(desc) return "ẼN[`[ɔʂ̓ł̃_[W^B";
#else
		if(name) return "Stinking mist";
		if(desc) return "Deals few damages of poison to all creatures in your sight.";
#endif
		power = lev_bonus / 2 + 5;
		if(info) return info_damage(1, power, 0);
		if(cast || cont)
		{
			project_all_vision(caster_ptr, DO_EFFECT_POIS, randint1(power));
		}
		break;

	case 4:
#ifdef JP
		if(name) return "r͋";
		if(desc) return "p҂̘r͂㏸B";
#else
		if(name) return "Extra might";
		if(desc) return "Attempts to increase your strength.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("͂NėB");
#else
			msg_print("You feel you get stronger.");
#endif
		}
		break;

	case 5:
#ifdef JP
		if(name) return "";
		if(desc) return "Ă镐􂤁B";
#else
		if(name) return "Curse weapon";
		if(desc) return "Curses your weapon.";
#endif
		if(cast)
		{
			OBJECT_ID item;
			char *q, *s;
			char object_name[MAX_NLEN];
			object_type *object_ptr;
			u32b f[MAX_TRAITS_FLAG];

			q = MES_HEX_WHICH_CURSE;
			s = MES_HEX_WHICH_NO_WEAPON;

			if(!get_item(caster_ptr, &item, q, s, (USE_EQUIP), item_tester_hook_weapon_except_bow, 0)) return FALSE;

			object_ptr = &caster_ptr->inventory[item];
			object_desc(object_name, object_ptr, OD_NAME_ONLY);
			object_flags(object_ptr, f);
			if(!get_check(format(MES_HEX_CURSE_CHECK(object_ptr)))) return FALSE;

			if(!one_in_(3) && (object_is_artifact(object_ptr) || have_flag(f, TRAIT_BLESSED_BRAND)))
			{
				msg_format(MES_OBJECT_RESISTED_CURSE1(object_ptr));
				if(one_in_(3)) object_disenchant(caster_ptr, object_ptr, 0);
			}
			else
			{
				POWER power = 0;
				msg_format(MES_BLACK_AURA_TO_OBJECT, object_name);
				add_flag(object_ptr->curse_flags, TRAIT_CURSED);

				if(object_is_artifact(object_ptr) || object_is_ego(object_ptr))
				{

					if(one_in_(3)) add_flag(object_ptr->curse_flags, TRAIT_HEAVY_CURSE);
					if(one_in_(666))
					{
						add_flag(object_ptr->curse_flags, TRAIT_TY_CURSE);
						if(one_in_(666)) add_flag(object_ptr->curse_flags, TRAIT_DIVINE_CURSE);

						add_flag(object_ptr->trait_flags, TRAIT_ANTIPATHY);
						add_flag(object_ptr->trait_flags, TRAIT_VORPAL);
						add_flag(object_ptr->trait_flags, TRAIT_VAMPIRIC_BRAND);
						msg_print(MES_HEX_DIVINE_CURSE_DONE);
						power = 2;
					}
				}

				object_ptr->curse_flags[0] |= get_curse(power, object_ptr);
			}

			prepare_update(caster_ptr, CRU_BONUS);
			add = FALSE;
		}
		break;

	case 6:
#ifdef JP
		if(name) return "׈m";
		if(desc) return "͂̎׈ȃN[`[mB";
#else
		if(name) return "Evil detection";
		if(desc) return "Detects evil creatures.";
#endif
		if(info) return info_range(MAX_SIGHT);
		if(cast)
		{
#ifdef JP
			msg_print("׈Ȑ݂̑낤ƂB");
#else
			msg_print("You attend to the presence of evil creatures.");
#endif
		}
		break;

	case 7:
#ifdef JP
		if(name) return "䖝";
		if(desc) return "^[UςA󂯂_[Wn̋Ɖ΂ƂĎ͂ɕoB";
#else
		if(name) return "Patience";
		if(desc) return "Bursts hell fire strongly after patients any damage while few turns.";
#endif
		power = MIN(200, (caster_ptr->revenge_damage * 2));
		if(info) return info_damage(0, 0, power);
		if(cast)
		{
			int a = 3 - (caster_ptr->speed - 100) / 10;
			int r = 3 + randint1(3) + MAX(0, MIN(3, a));

			if(caster_ptr->revenge_turn > 0)
			{
				msg_print(MES_HEX_ALREADY_PATIENT);
				return NULL;
			}

			caster_ptr->revenge_type = 1;
			caster_ptr->revenge_turn = r;
			caster_ptr->revenge_damage = 0;
			msg_print(MES_HEX_ALREADY_PATIENT);
			add = FALSE;
		}
		if(cont)
		{
			COODINATES rad = 2 + (COODINATES)(power / 50);
			caster_ptr->revenge_turn--;
			if((caster_ptr->revenge_turn <= 0) || (power >= 200))
			{
				msg_print(MES_HEX_REVENGE_DONE);
				if(power) project(caster_ptr, 0, rad, caster_ptr->fy, caster_ptr->fx, power, DO_EFFECT_HELL_FIRE, (PROJECT_STOP | PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL), -1);
				if(wizard) msg_format(MES_HEX_WIZ_REVENGE(power));
				caster_ptr->revenge_type = 0;
				caster_ptr->revenge_turn = 0;
				caster_ptr->revenge_damage = 0;
			}
		}
		break;

	/*** 2nd book (8-15) ***/
	case 8:
#ifdef JP
		if(name) return "X̊Z";
		if(desc) return "X̃I[gɂ܂ƂAh͂㏸B";
#else
		if(name) return "Ice armor";
		if(desc) return "Gives fire aura and bonus to AC.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("̂X̊ZŕꂽB");
#else
			msg_print("You have enveloped by ice armor!");
#endif
		}
		if(stop)
		{
#ifdef JP
			msg_print("X̊ZB");
#else
			msg_print("Ice armor disappeared.");
#endif
		}
		break;

	case 9:
#ifdef JP
		if(name) return "d̎";
		if(desc) return "̗͂⏝𑽏񕜂B";
#else
		if(name) return "Cure serious wounds";
		if(desc) return "Heals cut and HP more.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(2, 10), 0L);
		if(cont) do_active_trait(caster_ptr, TRAIT_HEAL, FALSE, diceroll(2, 10), 0L);
		break;

	case 10:
#ifdef JP
		if(name) return "iz";
		if(desc) return "r𒆎~邱ƂȂǍʂ𓾂邱ƂłB";
#else
		if(name) return "Inhail potion";
		if(desc) return "Quaffs a potion without canceling of casting a spell.";
#endif
		if(cast)
		{
			caster_ptr->spelling_hex |= (1L << HEX_INHAIL);
			do_cmd_quaff_potion(caster_ptr);
			caster_ptr->spelling_hex &= ~(1L << HEX_INHAIL);
			add = FALSE;
		}
		break;

	case 11:
#ifdef JP
		if(name) return "z";
		if(desc) return "ẼN[`[ɔʂ̐͋z̃_[W^B^_[W̕A̗͂񕜂B";
#else
		if(name) return "Vampiric mist";
		if(desc) return "Deals few dameges of drain life to all creatures in your sight.";
#endif
		power = (lev_bonus / 2) + 5;
		if(info) return info_damage(1, power, 0);
		if(cast || cont)
		{
			project_all_vision(caster_ptr, DO_EFFECT_DRAIN, randint1(power));
		}
		break;

	case 12:
#ifdef JP
		if(name) return "";
		if(desc) return "̍U͂グB؂ꖡ𓾁A􂢂ɉė^_[W㏸APǂȃN[`[ɑ΂_[W2{ɂȂB";
#else
		if(name) return "Swords to runeswords";
		if(desc) return "Gives vorpal ability to your weapon. Increases damages by your weapon acccording to curse of your weapon.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("Ȃ̕킪PB");
#else
			if(!empty_hands(caster_ptr, FALSE)) msg_print("Your weapons glow bright black.");
			else msg_print("Your weapon glows bright black.");
#endif
		}
		if(stop)
		{
#ifdef JP
			msg_print("̋PB");
#else
			msg_format("Brightness of weapon%s disappeared.", (empty_hands(caster_ptr, FALSE)) ? "" : "s");
#endif
		}
		break;

	case 13:
#ifdef JP
		if(name) return "̎";
		if(desc) return "UۃN[`[B";
#else
		if(name) return "Touch of confusion";
		if(desc) return "Confuses a creature when you attack.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("Ȃ̎肪ԂPn߂B");
#else
			msg_print("Your hands glow bright red.");
#endif
		}
		if(stop)
		{
#ifdef JP
			msg_print("̋PȂȂB");
#else
			msg_print("Brightness on your hands disappeard.");
#endif
		}
		break;

	case 14:
#ifdef JP
		if(name) return "̋";
		if(desc) return "p҂̘ŕApAϋv͂㏸BU񐔂̏ 1 B";
#else
		if(name) return "Building up";
		if(desc) return "Attempts to increases your strength, dexterity and constitusion.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("ĝȂCB");
#else
			msg_print("You feel your body is developed more now.");
#endif
		}
		break;

	case 15:
#ifdef JP
		if(name) return "e|[gE";
		if(desc) return "ẼN[`[̃e|[gjQoA𒣂B";
#else
		if(name) return "Anti teleport barrier";
		if(desc) return "Obstructs all teleportations by creatures in your sight.";
#endif
		power = lev_bonus * 3 / 2;
		if(info) return info_power(power);
		if(cast)
		{
#ifdef JP
			msg_print("e|[gh􂢂B");
#else
			msg_print("You feel anyone can not teleport except you.");
#endif
		}
		break;

	/*** 3rd book (16-23) ***/
	case 16:
#ifdef JP
		if(name) return "Ռ̃N[N";
		if(desc) return "dC̃I[gɂ܂ƂAȂB";
#else
		if(name) return "Cloak of shock";
		if(desc) return "Gives lightning aura and a bonus to speed.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("̂Ȃ̃I[ŕꂽB");
#else
			msg_print("You have enveloped by electrical aura!");
#endif
		}
		if(stop)
		{
#ifdef JP
			msg_print("Ȃ̃I[B");
#else
			msg_print("Electrical aura disappeared.");
#endif
		}
		break;

	case 17:
#ifdef JP
		if(name) return "v̎";
		if(desc) return "̗͂⏝񕜂B";
#else
		if(name) return "Cure critical wounds";
		if(desc) return "Heals cut and HP greatry.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_HEAL, TRUE, diceroll(4, 10), 0L);
		if(cont) do_active_trait(caster_ptr, TRAIT_HEAL, FALSE, diceroll(4, 10), 0L);
		break;

	case 18:
#ifdef JP
		if(name) return "͕";
		if(desc) return "@̓ɖ͂ď[UB";
#else
		if(name) return "Recharging";
		if(desc) return "Recharges a magic device.";
#endif
		if(cast) do_active_trait(caster_ptr, TRAIT_MAGIC_CHARGE_2, TRUE, lev_bonus * 2, 0L);
		break;

	case 19:
#ifdef JP
		if(name) return "ҕ";
		if(desc) return "̂h点ăybgɂB";
#else
		if(name) return "Animate Dead";
		if(desc) return "Raises corpses and skeletons from dead.";
#endif
		if(cast) do_active_trait_tmp(caster_ptr, TRAIT_ANIM_DEAD, TRUE);
		break;

	case 20:
#ifdef JP
		if(name) return "h";
		if(desc) return "ĂhɎ􂢂B";
#else
		if(name) return "Curse armor";
		if(desc) return "Curse a piece of armour that you wielding.";
#endif
		if(cast)
		{
			OBJECT_ID item;
			char *q, *s;
			char object_name[MAX_NLEN];
			object_type *object_ptr;
			u32b f[MAX_TRAITS_FLAG];

			q = MES_HEX_WHICH_CURSE;
			s = MES_HEX_WHICH_NO_ARMOUR;

			if(!get_item(caster_ptr, &item, q, s, (USE_EQUIP), object_is_armour2, 0)) return FALSE;

			object_ptr = &caster_ptr->inventory[item];
			object_desc(object_name, object_ptr, OD_NAME_ONLY);
			object_flags(object_ptr, f);
			if(!get_check(format(MES_HEX_CURSE_CHECK(object_ptr)))) return FALSE;

			if(!one_in_(3) && (object_is_artifact(object_ptr) || have_flag(f, TRAIT_BLESSED_BRAND)))
			{
				msg_format(MES_OBJECT_RESISTED_CURSE1(object_ptr));
				if(one_in_(3)) object_disenchant(caster_ptr, object_ptr, 0);
			}
			else
			{
				POWER power = 0;
				msg_format(MES_BLACK_AURA_TO_OBJECT, object_name);
				add_flag(object_ptr->curse_flags, TRAIT_CURSED);

				if(object_is_artifact(object_ptr) || object_is_ego(object_ptr))
				{

					if(one_in_(3)) add_flag(object_ptr->curse_flags, TRAIT_HEAVY_CURSE);
					if(one_in_(666))
					{
						add_flag(object_ptr->curse_flags, TRAIT_TY_CURSE);
						if(one_in_(666)) add_flag(object_ptr->curse_flags, TRAIT_DIVINE_CURSE);

						add_flag(object_ptr->trait_flags, TRAIT_ANTIPATHY);
						add_flag(object_ptr->trait_flags, TRAIT_RES_POIS);
						add_flag(object_ptr->trait_flags, TRAIT_RES_DARK);
						add_flag(object_ptr->trait_flags, TRAIT_RES_NETH);
						msg_print(MES_HEX_DIVINE_CURSE_DONE);
						power = 2;
					}
				}

				object_ptr->curse_flags[0] |= get_curse(power, object_ptr);
			}

			prepare_update(caster_ptr, CRU_BONUS);
			add = FALSE;
		}
		break;

	case 21:
#ifdef JP
		if(name) return "ẽN[N";
		if(desc) return "ẽI[gɂ܂ƂAGɉẽ_[W^B";
#else
		if(name) return "Cloak of shadow";
		if(desc) return "Gives aura of shadow.";
#endif
		if(cast)
		{
			//TODO: GET outer equipment.
			object_type *object_ptr = &caster_ptr->inventory[0];

			if(!is_valid_object(object_ptr))
			{
#ifdef JP
				msg_print("N[NgɂĂȂI");
#else
				msg_print("You don't ware any cloak.");
#endif
				return NULL;
			}
			else if(!object_is_cursed(object_ptr))
			{
#ifdef JP
				msg_print("N[N͎ĂȂI");
#else
				msg_print("Your cloak is not cursed.");
#endif
				return NULL;
			}
			else
			{
#ifdef JP
				msg_print("ẽI[gɂ܂ƂB");
#else
				msg_print("You have enveloped by shadow aura!");
#endif
			}
		}
		if(cont)
		{
			//TODO: GET outer Equipment
			object_type *object_ptr = &caster_ptr->inventory[0];

			if((!object_ptr->k_idx) || (!object_is_cursed(object_ptr)))
			{
				do_spell(caster_ptr, REALM_HEX, spell, SPELL_STOP);
				caster_ptr->spelling_hex &= ~(1L << spell);
				caster_ptr->spelling_hex_num--;
				if(!caster_ptr->spelling_hex_num) set_action(caster_ptr, ACTION_NONE);
			}
		}
		if(stop)
		{
#ifdef JP
			msg_print("ẽI[B");
#else
			msg_print("Shadow aura disappeared.");
#endif
		}
		break;

	case 22:
#ifdef JP
		if(name) return "ɂ𖂗͂";
		if(desc) return "ẼN[`[ɐ__[W^A͂zB";
#else
		if(name) return "Pains to mana";
		if(desc) return "Deals psychic damages to all creatures in sight, and drains some mana.";
#endif
		power = lev_bonus * 3 / 2;
		if(info) return info_damage(1, power, 0);
		if(cast || cont)
		{
			project_all_vision(caster_ptr, DO_EFFECT_PSI_DRAIN, randint1(power));
		}
		break;

	case 23:
#ifdef JP
		if(name) return "ڂɂ͖ڂ";
		if(desc) return "Ō▂@Ŏ󂯂_[WAŨN[`[ɂ^B";
#else
		if(name) return "Eye for an eye";
		if(desc) return "Returns same damage which you got to the creature which damaged you.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("Q~]ɂꂽB");
#else
			msg_print("You wish strongly you want to revenge anything.");
#endif
		}
		break;

	/*** 4th book (24-31) ***/
	case 24:
#ifdef JP
		if(name) return "BE";
		if(desc) return "̊K̑BN[`[̑Bj~B";
#else
		if(name) return "Anti multiply barrier";
		if(desc) return "Obstructs all multiplying by creatures in entire floor.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("Bj~􂢂B");
#else
			msg_print("You feel anyone can not already multiply.");
#endif
		}
		break;

	case 25:
#ifdef JP
		if(name) return "͕";
		if(desc) return "olXɕA\͒l񕜂B";
#else
		if(name) return "Restore life";
		if(desc) return "Restores life energy and status.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("͂߂n߂B");
#else
			msg_print("You feel your life energy starting to return.");
#endif
		}
		if(cast || cont)
		{
			bool flag = FALSE;
			int d = (caster_ptr->max_exp - caster_ptr->exp);
			int r = (caster_ptr->exp / 20);
			int i;

			if(d > 0)
			{
				if(d < r)
					caster_ptr->exp = caster_ptr->max_exp;
				else
					caster_ptr->exp += r;

				/* Check the experience */
				check_experience(caster_ptr);

				flag = TRUE;
			}
			for (i = STAT_STR; i < STAT_MAX; i ++)
			{
				if(caster_ptr->stat_cur[i] < caster_ptr->stat_max[i])
				{
					caster_ptr->stat_cur[i] += 10;
					if(caster_ptr->stat_cur[i] > caster_ptr->stat_max[i]) caster_ptr->stat_cur[i] = caster_ptr->stat_max[i];
					prepare_update(caster_ptr, CRU_BONUS); // Recalculate bonuses

					flag = TRUE;
				}
			}

			if(!flag)
			{
#ifdef JP
				msg_format("%s̎̉r߂B", do_spell(caster_ptr, REALM_HEX, HEX_RESTORE, SPELL_NAME));
#else
				msg_format("Finish casting '%^s'.", do_spell(caster_ptr, REALM_HEX, HEX_RESTORE, SPELL_NAME));
#endif
				caster_ptr->spelling_hex &= ~(1L << HEX_RESTORE);
				if(cont) caster_ptr->spelling_hex_num--;
				if(!caster_ptr->spelling_hex_num) caster_ptr->action = ACTION_NONE;

				prepare_update(caster_ptr, CRU_BONUS | CRU_HP | CRU_MANA | CRU_SPELLS);
				prepare_redraw(PR_EXTRA);

				return "";
			}
		}
		break;

	case 26:
#ifdef JP
		if(name) return "͋z";
		if(desc) return "ꂽ̎􂢂zĖ͂񕜂B";
#else
		if(name) return "Drain curse power";
		if(desc) return "Drains curse on your weapon and heals SP a little.";
#endif
		if(cast)
		{
			OBJECT_ID item;
			char *s, *q;
			u32b f[MAX_TRAITS_FLAG];
			object_type *object_ptr;
#ifdef JP
			q = "ǂ̑iz܂H";
			s = "ꂽACe𑕔ĂȂB";
#else
			q = "Which cursed equipment do you drain mana from?";
			s = "You have no cursed equipment.";
#endif
			if(!get_item(caster_ptr, &item, q, s, (USE_EQUIP), item_tester_hook_cursed, 0)) return FALSE;

			object_ptr = &caster_ptr->inventory[item];
			object_flags(object_ptr, f);

			inc_mana(caster_ptr, (caster_ptr->lev / 5) + randint1(caster_ptr->lev / 5));
			if(has_trait(caster_ptr, TRAIT_TY_CURSE)) inc_mana(caster_ptr, randint1(5));

			if(have_flag(object_ptr->curse_flags, TRAIT_DIVINE_CURSE))
			{
				// Nothing
			}
			else if(have_flag(object_ptr->curse_flags, TRAIT_HEAVY_CURSE))
			{
				if(one_in_(7))
				{
					msg_print(MES_HEX_DRAIN_CURSE);
					object_ptr->curse_flags[0] = 0L;
				}
			}
			else if(have_flag(object_ptr->curse_flags, TRAIT_CURSED) && one_in_(3))
			{
				msg_print(MES_HEX_DRAIN_CURSE);
				object_ptr->curse_flags[0] = 0L;
			}
			add = FALSE;
		}
		break;

	case 27:
#ifdef JP
		if(name) return "z̐n";
		if(desc) return "zōUB";
#else
		if(name) return "Swords to vampires";
		if(desc) return "Gives vampiric ability to your weapon.";
#endif
		if(cast)
		{
#ifdef JP
			msg_print("Ȃ̕킪~ĂB");
#else
			if(!empty_hands(caster_ptr, FALSE))
				msg_print("Your weapons want more blood now.");
			else
				msg_print("Your weapon wants more blood now.");
#endif
		}
		if(stop)
		{
#ifdef JP
			msg_print("̊]B");
#else
			msg_format("Thirsty of weapon%s disappeared.", (empty_hands(caster_ptr, FALSE)) ? "" : "s");
#endif
		}
		break;

	case 28:
#ifdef JP
		if(name) return "NǑt";
		if(desc) return "ẼN[`[NOƂB";
#else
		if(name) return "Word of stun";
		if(desc) return "Stuns all creatures in your sight.";
#endif
		power = lev_bonus * 4;
		if(info) return info_power(power);
		if(cast || cont)
		{
			project_all_vision(caster_ptr, DO_EFFECT_STUN, power);
		}
		break;

	case 29:
#ifdef JP
		if(name) return "eړ";
		if(desc) return "N[`[ׂ̗̃}XɏuԈړB";
#else
		if(name) return "Moving into shadow";
		if(desc) return "Teleports you close to a creature.";
#endif
		if(cast)
		{
			int i;
			COODINATES y, x, dir;
			bool flag;

			for (i = 0; i < 3; i++)
			{
				if(!tgt_pt(caster_ptr, &x, &y)) return FALSE;

				flag = FALSE;

				for (dir = 0; dir < 8; dir++)
				{
					COODINATES dy = y + ddy_ddd[dir];
					COODINATES dx = x + ddx_ddd[dir];
					if(dir == 5) continue;
					if(floor_ptr->cave[dy][dx].creature_idx) flag = TRUE;
				}

				if(!cave_empty_bold(floor_ptr, y, x) || (floor_ptr->cave[y][x].info & CAVE_ICKY) || (distance(y, x, caster_ptr->fy, caster_ptr->fx) > lev_bonus + 2))
				{
					msg_print(MES_TELEPORT_CANNOT);
					continue;
				}
				break;
			}

			if(flag && randint0(lev_bonus * lev_bonus / 2)) teleport_creature_to(caster_ptr, y, x, 0L);
			else
			{
				msg_print(MES_TRAIT_SHADOW_MOVE_FAILED);
				teleport_creature(caster_ptr, 30, 0L);
			}

			add = FALSE;
		}
		break;

	case 30:
#ifdef JP
		if(name) return "@E";
		if(desc) return "ẼN[`[̖@jQoA𒣂B";
#else
		if(name) return "Anti magic barrier";
		if(desc) return "Obstructs all magic spell of creatures in your sight.";
#endif
		power = lev_bonus * 3 / 2;
		if(info) return info_power(power);
		if(cast)
		{
#ifdef JP
			msg_print("@h􂢂B");
#else
			msg_print("You feel anyone can not cast spells except you.");
#endif
		}
		break;

	case 31:
#ifdef JP
		if(name) return "Q̐鍐";
		if(desc) return "^[ɂ܂Ŏ󂯂_[WɉЗ͂̒n̍΂̒eB";
#else
		if(name) return "Revenge sentence";
		if(desc) return "Fires  a ball of hell fire to try revenging after few turns.";
#endif
		power = caster_ptr->revenge_damage;
		if(info) return info_damage(0, 0, power);
		if(cast)
		{
			int r;
			int a = 3 - (caster_ptr->speed - 100) / 10;
			r = 1 + randint1(2) + MAX(0, MIN(3, a));

			if(caster_ptr->revenge_turn > 0)
			{
#ifdef JP
				msg_print("łɕQ͐鍐ς݂B");
#else
				msg_print("You already pronounced your revenge.");
#endif
				return NULL;
			}

			caster_ptr->revenge_type = 2;
			caster_ptr->revenge_turn = r;
#ifdef JP
			msg_format("Ȃ͕Q鍐B %d ^[B", r);
#else
			msg_format("You pronounce your revenge. %d turns left.", r);
#endif
			add = FALSE;
		}
		if(cont)
		{
			caster_ptr->revenge_turn--;

			if(caster_ptr->revenge_turn <= 0)
			{
				DIRECTION dir;

				if(power)
				{
					command_dir = 0;

					do
					{
#ifdef JP
						msg_print("Q̎I");
#else
						msg_print("Time to revenge!");
#endif
					}
					while (!get_aim_dir(caster_ptr, MAX_RANGE_SUB, &dir));

					cast_ball(caster_ptr, DO_EFFECT_HELL_FIRE, dir, power, 1);

					if(wizard)
					{
#ifdef JP
						msg_format("%d_̃_[WԂB", power);
#else
						msg_format("You return %d damages.", power);
#endif
					}
				}
				else
				{
#ifdef JP
					msg_print("QCB");
#else
					msg_print("You are not a mood to revenge.");
#endif
				}
				caster_ptr->revenge_damage = 0;
			}
		}
		break;
	}

	/* start casting */
	if((cast) && (add))
	{
		/* add spell */
		caster_ptr->spelling_hex |= 1L << (spell);
		caster_ptr->spelling_hex_num++;

		if(caster_ptr->action != ACTION_SPELL) set_action(caster_ptr, ACTION_SPELL);
	}

	if(!info)
	{
		prepare_update(caster_ptr, CRU_BONUS | CRU_HP | CRU_MANA | CRU_SPELLS);
		prepare_redraw(PR_EXTRA | PR_HP | PR_MANA);
	}

	return "";
}


/*
 * Do everything for each spell
 */
cptr do_spell(creature_type *creature_ptr, REALM_ID realm, int spell, int mode)
{
	switch (realm)
	{
	case REALM_LIFE:     return do_life_spell(creature_ptr, spell, mode);
	case REALM_SORCERY:  return do_sorcery_spell(creature_ptr, spell, mode);
	case REALM_NATURE:   return do_nature_spell(creature_ptr, spell, mode);
	case REALM_CHAOS:    return do_chaos_spell(creature_ptr, spell, mode);
	case REALM_DEATH:    return do_death_spell(creature_ptr, spell, mode);
	case REALM_TRUMP:    return do_trump_spell(creature_ptr, spell, mode);
	case REALM_ARCANE:   return do_arcane_spell(creature_ptr, spell, mode);
	case REALM_CRAFT:    return do_craft_spell(creature_ptr, spell, mode);
	case REALM_DAEMON:   return do_daemon_spell(creature_ptr, spell, mode);
	case REALM_CRUSADE:  return do_crusade_spell(creature_ptr, spell, mode);
	case REALM_MUSIC:    return do_music_spell(creature_ptr, spell, mode);
	case REALM_HISSATSU: return do_hissatsu_spell(creature_ptr, spell, mode);
	case REALM_HEX:      return do_hex_spell(creature_ptr, spell, mode);
	}

	return NULL;
}

