/*
 * File: grid.c
 * Purpose: low-level dungeon creation primitives
 */

/*
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research,
 * and not for profit purposes provided that this copyright and statement
 * are included in all such copies.  Other copyrights may also apply.
 */

#include "angband.h"
#include "generate.h"
#include "grid.h"


/*
 * Returns random co-ordinates for player/creature/object
 */
bool new_creature_spot(floor_type *floor_ptr, creature_type *creature_ptr)
{
	COODINATES y = 0, x = 0;
	int max_attempts = MAX_TRIES;

	cave_type *c_ptr;
	feature_type *f_ptr;

	// Place the player
	while (max_attempts--)
	{
		// Pick a legal spot
		y = (COODINATES)rand_range(1, floor_ptr->height - 2);
		x = (COODINATES)rand_range(1, floor_ptr->width - 2);

		c_ptr = &floor_ptr->cave[y][x];

		// Must be a "naked" floor grid
		if(c_ptr->creature_idx) continue;
		if(floor_ptr->depth)
		{
			f_ptr = &feature_info[c_ptr->feat];

			if(max_attempts > 5000) /* Rule 1 */
			{
				if(!have_flag(f_ptr->flags, FF_FLOOR)) continue;
			}
			else // Rule 2
			{
				if(!have_flag(f_ptr->flags, FF_MOVE)) continue;
				if(have_flag(f_ptr->flags, FF_HIT_TRAP)) continue;
			}

			// Refuse to start on anti-teleport grids in dungeon
			if(!have_flag(f_ptr->flags, FF_TELEPORTABLE)) continue;
		}
		if(!creature_can_cross_terrain(creature_ptr, c_ptr->feat, 0)) continue;
		if(!IN_BOUNDS(floor_ptr, y, x)) continue;

		// Refuse to start on anti-teleport grids
		if(c_ptr->info & (CAVE_ICKY)) continue;

		break; // Done
	}

	if(max_attempts < 1) // (someone's comment) Should be -1, actually if we failed...
	{
		msg_format("Warning: new creature spot failed...");
		return FALSE;
	}

	// Save the new player grid
	move_creature(creature_ptr, floor_ptr, y, x, 0);
	return TRUE;
}


/*
 * Place an up/down staircase at given location
 */
void place_random_stairs(floor_type *floor_ptr, int y, int x)
{
	bool up_stairs = TRUE;
	bool down_stairs = TRUE;
	cave_type *c_ptr;


	c_ptr = &floor_ptr->cave[y][x];
	if(!is_floor_grid(c_ptr) || c_ptr->object_idx) return;

	/* Town */
	if(!floor_ptr->depth)
		up_stairs = FALSE;

	/* Ironman */
	if(ironman_downward)
		up_stairs = FALSE;

	/* Bottom */
	if(floor_ptr->depth >= dungeon_info[floor_ptr->dun_type].maxdepth)
		down_stairs = FALSE;

	/* Quest-level */
	if(quest_number(floor_ptr) && (floor_ptr->depth > 1))
		down_stairs = FALSE;

	/* We can't place both */
	if(down_stairs && up_stairs)
	{
		/* Choose a staircase randomly */
		if(PERCENT(50))
			up_stairs = FALSE;
		else
			down_stairs = FALSE;
	}

	/* Place the stairs */
	if(up_stairs) place_up_stairs(floor_ptr, y, x);
	else if(down_stairs) place_down_stairs(floor_ptr, y, x);
}


/*
 * Place a random type of door at the given location
 */
void place_random_door(floor_type *floor_ptr, int y, int x, bool room)
{
	int tmp, type;
	s16b feat = feat_none;
	cave_type *c_ptr = &floor_ptr->cave[y][x];

	/* Initialize mimic info */
	c_ptr->mimic = 0;

	if(dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_DOORS)
	{
		place_floor_bold(floor_ptr, y, x);
		return;
	}

	type = ((dungeon_info[floor_ptr->dun_type].flags1 & DF1_CURTAIN) &&
		one_in_((dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_CAVE) ? 16 : 256)) ? DOOR_CURTAIN :
		((dungeon_info[floor_ptr->dun_type].flags1 & DF1_GLASS_DOOR) ? DOOR_GLASS_DOOR : DOOR_DOOR);

	/* Choose an object */
	tmp = randint0(1000);

	/* Open doors (300/1000) */
	if(tmp < 300)
	{
		/* Create open door */
		feat = feat_door[type].open;
	}

	/* Broken doors (100/1000) */
	else if(tmp < 400)
	{
		/* Create broken door */
		feat = feat_door[type].broken;
	}

	/* Secret doors (200/1000) */
	else if(tmp < 600)
	{
		/* Create secret door */
		place_closed_door(floor_ptr, y, x, type);

		if(type != DOOR_CURTAIN)
		{
			/* Hide. If on the edge of room, use outer wall. */
			c_ptr->mimic = room ? feat_wall_outer : fill_type[randint0(100)];

			/* Floor type terrain cannot hide a door */
			if(FEAT_SUPPORTS_LOS(c_ptr->mimic) && !FEAT_SUPPORTS_LOS(c_ptr->feat))
			{
				if(have_flag(feature_info[c_ptr->mimic].flags, FF_MOVE) || have_flag(feature_info[c_ptr->mimic].flags, FF_CAN_FLY))
				{
					c_ptr->feat = one_in_(2) ? c_ptr->mimic : feat_floor_rand_table[randint0(100)];
				}
				c_ptr->mimic = 0;
			}
		}
	}

	/* Closed, locked, or stuck doors (400/1000) */
	else place_closed_door(floor_ptr, y, x, type);

	if(tmp < 400)
	{
		if(feat != feat_none)
		{
			set_cave_feat(floor_ptr, y, x, feat);
		}
		else
		{
			place_floor_bold(floor_ptr, y, x);
		}
	}

	delete_creature(floor_ptr, y, x);
}


/*
 * Place a random type of normal door at the given location.
 */
void place_closed_door(floor_type *floor_ptr, int y, int x, int type)
{
	int tmp;
	s16b feat = feat_none;

	if(dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_DOORS)
	{
		place_floor_bold(floor_ptr, y, x);
		return;
	}

	/* Choose an object */
	tmp = randint0(400);

	/* Closed doors (300/400) */
	if(tmp < 300)
	{
		/* Create closed door */
		feat = feat_door[type].closed;
	}

	/* Locked doors (99/400) */
	else if(tmp < 399)
	{
		/* Create locked door */
		feat = feat_locked_door_random(type);
	}

	/* Stuck doors (1/400) */
	else
	{
		/* Create jammed door */
		feat = feat_jammed_door_random(type);
	}

	if(feat != feat_none)
	{
		cave_set_feat(floor_ptr, y, x, feat);

		/* Now it is not floor */
		floor_ptr->cave[y][x].info &= ~(CAVE_MASK);
	}
	else place_floor_bold(floor_ptr, y, x);
}


/*
 * Make an empty square floor, for the middle of rooms
 */
void place_floor(floor_type *floor_ptr, int x1, int x2, int y1, int y2, bool light)
{
	int x, y;

	/* Place a full floor under the room */
	for (y = y1 - 1; y <= y2 + 1; y++)
	{
		for (x = x1 - 1; x <= x2 + 1; x++)
		{
			place_floor_bold(floor_ptr, y, x);
			add_cave_info(floor_ptr, y, x, CAVE_ROOM);
			if(light) add_cave_info(floor_ptr, y, x, CAVE_GLOW);
		}
	}
}


/*
 * Make an empty square room, only floor and wall grids (UNUSED?)
 */
void place_room(floor_type *floor_ptr, COODINATES x1, COODINATES x2, COODINATES y1, COODINATES y2, bool light)
{
	COODINATES y, x;

	place_floor(floor_ptr, x1, x2, y1, y2, light);

	/* Walls around the room */
	for (y = y1 - 1; y <= y2 + 1; y++)
	{
		place_outer_bold(floor_ptr, y, x1 - 1);
		place_outer_bold(floor_ptr, y, x2 + 1);
	}
	for (x = x1 - 1; x <= x2 + 1; x++)
	{
		place_outer_bold(floor_ptr, y1 - 1, x);
		place_outer_bold(floor_ptr, y2 + 1, x);
	}
}


/*
 * Create up to "num" objects near the given coordinates
 * Only really called by some of the "vault" routines.
 */
void vault_objects(floor_type *floor_ptr, COODINATES y, COODINATES x, int num)
{
	int dummy = 0;
	int i = 0;
	COODINATES j = y, k = x;
	cave_type *c_ptr;


	/* Attempt to place 'num' objects */
	for (; num > 0; --num)
	{
		/* Try up to 11 spots looking for empty space */
		for (i = 0; i < 11; ++i)
		{
			/* Pick a random location */
			while (dummy < SAFE_MAX_ATTEMPTS)
			{
				j = (COODINATES)rand_spread(y, 2);
				k = (COODINATES)rand_spread(x, 3);
				dummy++;
				if(!IN_BOUNDS(floor_ptr, j, k)) continue;
				break;
			}


			if(dummy >= SAFE_MAX_ATTEMPTS)
			{
				if(cheat_room)
				{
#ifdef JP
					msg_print("xIñACezuł܂I");
#else
					msg_print("Warning! Could not place vault object!");
#endif

				}
			}


			/* Require "clean" floor space */
			c_ptr = &floor_ptr->cave[j][k];
			if(!is_floor_grid(c_ptr) || c_ptr->object_idx) continue;

			/* Place an item */
			if(PERCENT(75))
			{
				place_object(floor_ptr, j, k, 0L, NULL);
			}

			/* Place gold */
			else
			{
				place_gold(floor_ptr, j, k);
			}

			/* Placement accomplished */
			break;
		}
	}
}


/*
 * Place a trap with a given displacement of point
 */
void vault_trap_aux(floor_type *floor_ptr, int y, int x, int yd, int xd)
{
	int count = 0, y1 = y, x1 = x;
	int dummy = 0;

	cave_type *c_ptr;

	/* Place traps */
	for (count = 0; count <= 5; count++)
	{
		/* Get a location */
		while (dummy < SAFE_MAX_ATTEMPTS)
		{
			y1 = rand_spread(y, yd);
			x1 = rand_spread(x, xd);
			dummy++;
			if(!IN_BOUNDS(floor_ptr, y1, x1)) continue;
			break;
		}

		if(dummy >= SAFE_MAX_ATTEMPTS)
		{
			if(cheat_room)
			{
#ifdef JP
msg_print("xIñgbvzuł܂I");
#else
				msg_print("Warning! Could not place vault trap!");
#endif

			}
		}

		/* Require "naked" floor grids */
		c_ptr = &floor_ptr->cave[y1][x1];
		if(!is_floor_grid(c_ptr) || c_ptr->object_idx || c_ptr->creature_idx) continue;

		/* Place the trap */
		place_trap(floor_ptr, y1, x1);

		break;
	}
}


/*
 * Place some traps with a given displacement of given location
 */
void vault_traps(floor_type *floor_ptr, int y, int x, int yd, int xd, int num)
{
	int i;

	for (i = 0; i < num; i++)
	{
		vault_trap_aux(floor_ptr, y, x, yd, xd);
	}
}


/*
 * Hack -- Place some sleeping creatures near the given location
 */
void vault_creatures(floor_type *floor_ptr, COODINATES y1, COODINATES x1, int num)
{
	int k, i;
	COODINATES y, x;
	cave_type *c_ptr;

	/* Try to summon "num" creatures "near" the given location */
	for (k = 0; k < num; k++)
	{
		/* Try nine locations */
		for (i = 0; i < 9; i++)
		{
			int d = 1;

			/* Pick a nearby location */
			scatter(floor_ptr, &y, &x, y1, x1, d, 0);

			/* Require "empty" floor grids */
			c_ptr = &floor_ptr->cave[y][x];
			if(!cave_empty_grid(c_ptr)) continue;

			/* Place the creature (allow groups) */
			floor_ptr->enemy_level = floor_ptr->depth + 2;
			(void)place_creature(NULL, floor_ptr, y, x, (PC_ALLOW_SLEEP | PC_ALLOW_GROUP));
			floor_ptr->enemy_level = floor_ptr->depth;
		}
	}
}


/*
 * Always picks a correct direction
 */
void correct_dir(COODINATES *rdir, COODINATES *cdir, COODINATES y1, COODINATES x1, COODINATES y2, COODINATES x2)
{
	/* Extract vertical and horizontal directions */
	*rdir = (y1 == y2) ? 0 : (y1 < y2) ? 1 : -1;
	*cdir = (x1 == x2) ? 0 : (x1 < x2) ? 1 : -1;

	/* Never move diagonally */
	if(*rdir && *cdir)
	{
		if(PERCENT(50))
			*rdir = 0;
		else
			*cdir = 0;
	}
}


/*
 * Pick a random direction
 */
void rand_dir(COODINATES *rdir, COODINATES *cdir)
{
	/* Pick a random direction */
	int i = randint0(4);

	/* Extract the dy/dx components */
	*rdir = ddy_ddd[i];
	*cdir = ddx_ddd[i];
}


// Function that sees if a square is a floor.  (Includes range checking.)
bool get_floor(floor_type *floor_ptr, int x, int y)
{
	if(!IN_BOUNDS(floor_ptr, y, x)) return FALSE; // Out of bounds
	if(is_floor_bold(floor_ptr, y, x)) return TRUE; // Do the real check
	return FALSE;
}


/* Set a square to be floor.  (Includes range checking.) */
void set_floor(floor_type *floor_ptr, int x, int y)
{
	if(!IN_BOUNDS(floor_ptr, y, x))
	{
		/* Out of bounds */
		return;
	}

	if(floor_ptr->cave[y][x].info & CAVE_ROOM)
	{
		/* A room border don't touch. */
		return;
	}

	/* Set to be floor if is a wall (don't touch lakes). */
	if(is_extra_bold(floor_ptr, y, x))
		place_floor_bold(floor_ptr, y, x);
}



/*
 * Constructs a tunnel between two points
 *
 * This function must be called BEFORE any streamers are created,
 * since we use the special "granite wall" sub-types to keep track
 * of legal places for corridors to pierce rooms.
 *
 * We use "door_flag" to prevent excessive construction of doors
 * along overlapping corridors.
 *
 * We queue the tunnel grids to prevent door creation along a corridor
 * which intersects itself.
 *
 * We queue the wall piercing grids to prevent a corridor from leaving
 * a room and then coming back in through the same entrance.
 *
 * We "pierce" grids which are "outer" walls of rooms, and when we
 * do so, we change all adjacent "outer" walls of rooms into "solid"
 * walls so that no two corridors may use adjacent grids for exits.
 *
 * The "solid" wall check prevents corridors from "chopping" the
 * corners of rooms off, as well as "silly" door placement, and
 * "excessively wide" room entrances.
 *
 * Kind of walls:
 *   extra -- walls
 *   inner -- inner room walls
 *   outer -- outer room walls
 *   solid -- solid room walls
 */
bool build_tunnel(floor_type *floor_ptr, COODINATES row1, COODINATES col1, COODINATES row2, COODINATES col2)
{
	COODINATES y, x, tmp_row, tmp_col, row_dir, col_dir, start_row, start_col;
	int main_loop_count = 0;

	bool door_flag = FALSE;

	cave_type *c_ptr;

	/* Save the starting location */
	start_row = row1;
	start_col = col1;

	/* Start out in the correct direction */
	correct_dir(&row_dir, &col_dir, row1, col1, row2, col2);

	/* Keep going until done (or bored) */
	while ((row1 != row2) || (col1 != col2))
	{
		/* Mega-Hack -- Paranoia -- prevent infinite loops */
		if(main_loop_count++ > 2000) return FALSE;

		/* Allow bends in the tunnel */
		if(randint0(100) < dun_tun_chg)
		{
			/* Acquire the correct direction */
			correct_dir(&row_dir, &col_dir, row1, col1, row2, col2);

			/* Random direction */
			if(randint0(100) < dun_tun_rnd)
			{
				rand_dir(&row_dir, &col_dir);
			}
		}

		/* Get the next location */
		tmp_row = row1 + row_dir;
		tmp_col = col1 + col_dir;


		/* Extremely Important -- do not leave the dungeon */
		while (!IN_BOUNDS(floor_ptr, tmp_row, tmp_col))
		{
			/* Acquire the correct direction */
			correct_dir(&row_dir, &col_dir, row1, col1, row2, col2);

			/* Random direction */
			if(randint0(100) < dun_tun_rnd) rand_dir(&row_dir, &col_dir);

			/* Get the next location */
			tmp_row = row1 + row_dir;
			tmp_col = col1 + col_dir;
		}


		/* Access the location */
		c_ptr = &floor_ptr->cave[tmp_row][tmp_col];

		/* Avoid "solid" walls */
		if(is_solid_grid(c_ptr)) continue;

		/* Pierce "outer" walls of rooms */
		if(is_outer_grid(c_ptr))
		{
			/* Acquire the "next" location */
			y = tmp_row + row_dir;
			x = tmp_col + col_dir;

			/* Hack -- Avoid outer/solid walls */
			if(is_outer_bold(floor_ptr, y, x)) continue;
			if(is_solid_bold(floor_ptr, y, x)) continue;

			/* Accept this location */
			row1 = tmp_row;
			col1 = tmp_col;

			/* Save the wall location */
			if(dungeon_ptr->wall_n < WALL_MAX)
			{
				dungeon_ptr->wall[dungeon_ptr->wall_n].y = row1;
				dungeon_ptr->wall[dungeon_ptr->wall_n].x = col1;
				dungeon_ptr->wall_n++;
			}
			else return FALSE;

			/* Forbid re-entry near this piercing */
			for (y = row1 - 1; y <= row1 + 1; y++)
			{
				for (x = col1 - 1; x <= col1 + 1; x++)
				{
					/* Convert adjacent "outer" walls as "solid" walls */
					if(is_outer_bold(floor_ptr, y, x))
					{
						/* Change the wall to a "solid" wall */
						place_solid_noperm_bold(floor_ptr, y, x);
					}
				}
			}
		}

		/* Travel quickly through rooms */
		else if(c_ptr->info & (CAVE_ROOM))
		{
			/* Accept the location */
			row1 = tmp_row;
			col1 = tmp_col;
		}

		/* Tunnel through all other walls */
		else if(is_extra_grid(c_ptr) || is_inner_grid(c_ptr) || is_solid_grid(c_ptr))
		{
			/* Accept this location */
			row1 = tmp_row;
			col1 = tmp_col;

			/* Save the tunnel location */
			if(dungeon_ptr->tunn_n < TUNN_MAX)
			{
				dungeon_ptr->tunn[dungeon_ptr->tunn_n].y = row1;
				dungeon_ptr->tunn[dungeon_ptr->tunn_n].x = col1;
				dungeon_ptr->tunn_n++;
			}
			else return FALSE;

			/* Allow door in next grid */
			door_flag = FALSE;
		}

		/* Handle corridor intersections or overlaps */
		else
		{
			/* Accept the location */
			row1 = tmp_row;
			col1 = tmp_col;

			/* Collect legal door locations */
			if(!door_flag)
			{
				/* Save the door location */
				if(dungeon_ptr->door_n < DOOR_MAX)
				{
					dungeon_ptr->door[dungeon_ptr->door_n].y = row1;
					dungeon_ptr->door[dungeon_ptr->door_n].x = col1;
					dungeon_ptr->door_n++;
				}
				else return FALSE;

				/* No door in next grid */
				door_flag = TRUE;
			}

			/* Hack -- allow pre-emptive tunnel termination */
			if(randint0(100) >= dun_tun_con)
			{
				/* Distance between row1 and start_row */
				tmp_row = row1 - start_row;
				if(tmp_row < 0) tmp_row = (-tmp_row);

				/* Distance between col1 and start_col */
				tmp_col = col1 - start_col;
				if(tmp_col < 0) tmp_col = (-tmp_col);

				/* Terminate the tunnel */
				if((tmp_row > 10) || (tmp_col > 10)) break;
			}
		}
	}

	return TRUE;
}


/*
 * This routine adds the square to the tunnel
 * It also checks for SOLID walls - and returns a nearby
 * non-SOLID square in (x,y) so that a simple avoiding
 * routine can be used. The returned boolean value reflects
 * whether or not this routine hit a SOLID wall.
 *
 * "affectwall" toggles whether or not this new square affects
 * the boundaries of rooms. - This is used by the catacomb
 * routine.
 */
static bool set_tunnel(floor_type *floor_ptr, COODINATES *x, COODINATES *y, bool affectwall)
{
	int i, j;
	COODINATES dx, dy;

	cave_type *c_ptr = &floor_ptr->cave[*y][*x];

	if(!IN_BOUNDS(floor_ptr, *y, *x)) return TRUE;

	if(is_inner_grid(c_ptr))
	{
		return TRUE;
	}

	if(is_extra_bold(floor_ptr, *y,*x))
	{
		/* Save the tunnel location */
		if(dungeon_ptr->tunn_n < TUNN_MAX)
		{
			dungeon_ptr->tunn[dungeon_ptr->tunn_n].y = *y;
			dungeon_ptr->tunn[dungeon_ptr->tunn_n].x = *x;
			dungeon_ptr->tunn_n++;

			return TRUE;
		}
		else return FALSE;
	}

	if(is_floor_bold(floor_ptr, *y, *x))
	{
		/* Don't do anything */
		return TRUE;
	}

	if(is_outer_grid(c_ptr) && affectwall)
	{
		/* Save the wall location */
		if(dungeon_ptr->wall_n < WALL_MAX)
		{
			dungeon_ptr->wall[dungeon_ptr->wall_n].y = *y;
			dungeon_ptr->wall[dungeon_ptr->wall_n].x = *x;
			dungeon_ptr->wall_n++;
		}
		else return FALSE;

		/* Forbid re-entry near this piercing */
		for (j = *y - 1; j <= *y + 1; j++)
		{
			for (i = *x - 1; i <= *x + 1; i++)
			{
				/* Convert adjacent "outer" walls as "solid" walls */
				if(is_outer_bold(floor_ptr, j, i))
				{
					/* Change the wall to a "solid" wall */
					place_solid_noperm_bold(floor_ptr, j, i);
				}
			}
		}

		/* Clear mimic type */
		floor_ptr->cave[*y][*x].mimic = 0;

		place_floor_bold(floor_ptr, *y, *x);

		return TRUE;
	}

	if(is_solid_grid(c_ptr) && affectwall)
	{
		/* cannot place tunnel here - use a square to the side */

		/* find usable square and return value in (x,y) */

		i = 50;

		dy = 0;
		dx = 0;
		while ((i > 0) && is_solid_bold(floor_ptr, *y + dy, *x + dx))
		{
			dy = (COODINATES)randint0(3) - 1;
			dx = (COODINATES)randint0(3) - 1;

			if(!IN_BOUNDS(floor_ptr, *y + dy, *x + dx))
			{
				dx = 0;
				dy = 0;
			}

			i--;
		}

		if(i == 0)
		{
			/* Failed for some reason: hack - ignore the solidness */
			place_outer_grid(c_ptr);
			dx = 0;
			dy = 0;
		}

		/* Give new, acceptable coordinate. */
		*x = *x + dx;
		*y = *y + dy;

		return FALSE;
	}

	return TRUE;
}


/*
 * This routine creates the catacomb-like tunnels by removing extra rock.
 * Note that this routine is only called on "even" squares - so it gives
 * a natural checkerboard pattern.
 */
static void create_cata_tunnel(floor_type *floor_ptr, COODINATES x, COODINATES y)
{
	COODINATES x1, y1;

	/* Build tunnel */
	x1 = x - 1;
	y1 = y;
	set_tunnel(floor_ptr, &x1, &y1, FALSE);

	x1 = x + 1;
	y1 = y;
	set_tunnel(floor_ptr, &x1, &y1, FALSE);

	x1 = x;
	y1 = y - 1;
	set_tunnel(floor_ptr, &x1, &y1, FALSE);

	x1 = x;
	y1 = y + 1;
	set_tunnel(floor_ptr, &x1, &y1, FALSE);
}


/*
 * This routine does the bulk of the work in creating the new types of tunnels.
 * It is designed to use very simple algorithms to go from (x1,y1) to (x2,y2)
 * It doesn't need to add any complexity - straight lines are fine.
 * The SOLID walls are avoided by a recursive algorithm which tries random ways
 * around the obstical until it works.  The number of itterations is counted, and it
 * this gets too large the routine exits. This should stop any crashes - but may leave
 * small gaps in the tunnel where there are too many SOLID walls.
 *
 * Type 1 tunnels are extremely simple - straight line from A to B.  This is only used
 * as a part of the dodge SOLID walls algorithm.
 *
 * Type 2 tunnels are made of two straight lines at right angles. When this is used with
 * short line segments it gives the "cavelike" tunnels seen deeper in the dungeon.
 *
 * Type 3 tunnels are made of two straight lines like type 2, but with extra rock removed.
 * This, when used with longer line segments gives the "catacomb-like" tunnels seen near
 * the surface.
 */
static void short_seg_hack(floor_type *floor_ptr, COODINATES x1, COODINATES y1, COODINATES x2, COODINATES y2, int type, int count, bool *fail)
{
	COODINATES i, x, y, length;

	/* Check for early exit */
	if(!(*fail)) return;

	length = (COODINATES)distance(x1, y1, x2, y2);

	count++;

	if((type == 1) && (length != 0))
	{

		for (i = 0; i <= length; i++)
		{
			x = x1 + i * (x2 - x1) / length;
			y = y1 + i * (y2 - y1) / length;
			if(!set_tunnel(floor_ptr, &x, &y, TRUE))
			{
				if(count > 50)
				{
					/* This isn't working - probably have an infinite loop */
					*fail = FALSE;
					return;
				}

				/* solid wall - so try to go around */
				short_seg_hack(floor_ptr, x, y, x1 + (i - 1) * (x2 - x1) / length, y1 + (i - 1) * (y2 - y1) / length, 1, count, fail);
				short_seg_hack(floor_ptr, x, y, x1 + (i + 1) * (x2 - x1) / length, y1 + (i + 1) * (y2 - y1) / length, 1, count, fail);
			}
		}
	}
	else if((type == 2) || (type == 3))
	{
		if(x1 < x2)
		{
			for (i = x1; i <= x2; i++)
			{
				x = i;
				y = y1;
				if(!set_tunnel(floor_ptr, &x, &y, TRUE))
				{
					/* solid wall - so try to go around */
					short_seg_hack(floor_ptr, x, y, i - 1, y1, 1, count, fail);
					short_seg_hack(floor_ptr, x, y, i + 1, y1, 1, count, fail);
				}
				if((type == 3) && ((x + y) % 2))
				{
					create_cata_tunnel(floor_ptr, i, y1);
				}
			}
		}
		else
		{
			for (i = x2; i <= x1; i++)
			{
				x = i;
				y = y1;
				if(!set_tunnel(floor_ptr, &x, &y, TRUE))
				{
					/* solid wall - so try to go around */
					short_seg_hack(floor_ptr, x, y, i - 1, y1, 1, count, fail);
					short_seg_hack(floor_ptr, x, y, i + 1, y1, 1, count, fail);
				}
				if((type == 3) && ((x + y) % 2))
				{
					create_cata_tunnel(floor_ptr, i, y1);
				}
			}

		}
		if(y1 < y2)
		{
			for (i = y1; i <= y2; i++)
			{
				x = x2;
				y = i;
				if(!set_tunnel(floor_ptr, &x, &y, TRUE))
				{
					/* solid wall - so try to go around */
					short_seg_hack(floor_ptr, x, y, x2, i - 1, 1, count, fail);
					short_seg_hack(floor_ptr, x, y, x2, i + 1, 1, count, fail);
				}
				if((type == 3) && ((x + y) % 2))
				{
					create_cata_tunnel(floor_ptr, x2, i);
				}
			}
		}
		else
		{
			for (i = y2; i <= y1; i++)
			{
				x = x2;
				y = i;
				if(!set_tunnel(floor_ptr, &x, &y, TRUE))
				{
					/* solid wall - so try to go around */
					short_seg_hack(floor_ptr, x, y, x2, i - 1, 1, count, fail);
					short_seg_hack(floor_ptr, x, y, x2, i + 1, 1, count, fail);
				}
				if((type == 3) && ((x + y) % 2))
				{
					create_cata_tunnel(floor_ptr, x2, i);
				}
			}
		}
	}
}


/*
 * This routine maps a path from (x1, y1) to (x2, y2) avoiding SOLID walls.
 * Permanent rock is ignored in this path finding- sometimes there is no
 * path around anyway -so there will be a crash if we try to find one.
 * This routine is much like the river creation routine in Zangband.
 * It works by dividing a line segment into two.  The segments are divided
 * until they are less than "cutoff" - when the corresponding routine from
 * "short_seg_hack" is called.
 * Note it is VERY important that the "stop if hit another passage" logic
 * stays as is.  Without this the dungeon turns into Swiss Cheese...
 */
bool build_tunnel2(floor_type *floor_ptr, COODINATES x1, COODINATES y1, COODINATES x2, COODINATES y2, int type, int cutoff)
{
	COODINATES x3, y3, dx, dy, length, changex, changey;
	int i;
	bool retval, firstsuccede;
	cave_type *c_ptr;

	length = distance(x1, y1, x2, y2);

	if(length > cutoff)
	{
		/*
		* Divide path in half and call routine twice.
		 */
		dx = (x2 - x1) / 2;
		dy = (y2 - y1) / 2;

		/* perturbation perpendicular to path */
		changex = (COODINATES)(randint0(abs(dy) + 2) * 2 - abs(dy) - 1) / 2;

		/* perturbation perpendicular to path */
		changey = (COODINATES)(randint0(abs(dx) + 2) * 2 - abs(dx) - 1) / 2;

		/* Work out "mid" ponit */
		x3 = x1 + dx + changex;
		y3 = y1 + dy + changey;

		/* See if in bounds - if not - do not perturb point */
		if(!IN_BOUNDS(floor_ptr, y3, x3))
		{
			x3 = (x1 + x2) / 2;
			y3 = (y1 + y2) / 2;
		}
		/* cache c_ptr */
		c_ptr = &floor_ptr->cave[y3][x3];
		if(is_solid_grid(c_ptr))
		{
			/* move midpoint a bit to avoid problem. */

			i = 50;

			dy = 0;
			dx = 0;
			while ((i > 0) && is_solid_bold(floor_ptr, y3 + dy, x3 + dx))
			{
				dy = (COODINATES)randint0(3) - 1;
				dx = (COODINATES)randint0(3) - 1;
				if(!IN_BOUNDS(floor_ptr, y3 + dy, x3 + dx))
				{
					dx = 0;
					dy = 0;
				}
				i--;
			}

			if(i == 0)
			{
				/* Failed for some reason: hack - ignore the solidness */
				place_outer_bold(floor_ptr, y3, x3);
				dx = 0;
				dy = 0;
			}
			y3 += dy;
			x3 += dx;
			c_ptr = &floor_ptr->cave[y3][x3];
		}

		if(is_floor_grid(c_ptr))
		{
			if(build_tunnel2(floor_ptr, x1, y1, x3, y3, type, cutoff))
			{
				if((floor_ptr->cave[y3][x3].info & CAVE_ROOM) || (randint1(100) > 95))
				{
					/* do second half only if works + if have hit a room */
					retval = build_tunnel2(floor_ptr, x3, y3, x2, y2, type, cutoff);
				}
				else
				{
					/* have hit another tunnel - make a set of doors here */
					retval = FALSE;

					/* Save the door location */
					if(dungeon_ptr->door_n < DOOR_MAX)
					{
						dungeon_ptr->door[dungeon_ptr->door_n].y = y3;
						dungeon_ptr->door[dungeon_ptr->door_n].x = x3;
						dungeon_ptr->door_n++;
					}
					else return FALSE;
				}
				firstsuccede = TRUE;
			}
			else
			{
				/* false- didn't work all the way */
				retval = FALSE;
				firstsuccede = FALSE;
			}
		}
		else
		{
			/* tunnel through walls */
			if(build_tunnel2(floor_ptr, x1, y1, x3, y3, type, cutoff))
			{
				retval = build_tunnel2(floor_ptr, x3, y3, x2, y2, type, cutoff);
				firstsuccede = TRUE;
			}
			else
			{
				/* false- didn't work all the way */
				retval = FALSE;
				firstsuccede = FALSE;
			}
		}
		if(firstsuccede)
		{
			/* only do this if the first half has worked */
			set_tunnel(floor_ptr, &x3, &y3, TRUE);
		}
		/* return value calculated above */
		return retval;
	}
	else
	{
		/* Do a short segment */
		retval = TRUE;
		short_seg_hack(floor_ptr, x1, y1, x2, y2, type, 0, &retval);

		/* Hack - ignore return value so avoid infinite loops */
		return TRUE;
	}
}

