package jp.dbdiffreport;

import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;

public class DdrCreateDiffReport {

	private static String beforeDir = "before";
	private static String afterDir = "after";

	private static int rowCount = 0;
	private static int colCount = 0;
	private static int headCount = 0;

	private static HSSFFont styleFont = null;
	private static HSSFCellStyle styleBox = null;
	private static HSSFCellStyle styleHead = null;
	private static HSSFCellStyle styleDelete = null;
	private static HSSFCellStyle styleUpdate = null;
	private static HSSFCellStyle styleUpdateBefore = null;
	private static HSSFCellStyle styleUpdateAfter = null;

	public static void main(String[] args) throws Exception {

		if (args.length > 0) beforeDir = args[0];
		if (args.length > 1) afterDir = args[1];

		HSSFWorkbook workbook = new HSSFWorkbook();
		HSSFSheet sheet = workbook.createSheet("diff");

		styleFont = workbook.createFont();
		styleFont.setFontName("ＭＳ ゴシック");
		styleFont.setFontHeightInPoints((short)9);

		styleBox = workbook.createCellStyle();
		styleBox.setBorderBottom(HSSFCellStyle.BORDER_THIN);
		styleBox.setBorderTop(HSSFCellStyle.BORDER_THIN);
		styleBox.setBorderLeft(HSSFCellStyle.BORDER_THIN);
		styleBox.setBorderRight(HSSFCellStyle.BORDER_THIN);
		styleBox.setFont(styleFont);

		styleDelete = createStyle(workbook, HSSFColor.GREY_25_PERCENT.index);
		styleUpdate = createStyle(workbook, HSSFColor.LIGHT_YELLOW.index);
		styleUpdateBefore = createStyle(workbook, HSSFColor.LIGHT_YELLOW.index);
		styleUpdateAfter = createStyle(workbook, HSSFColor.ROSE.index);

		styleHead = createStyle(workbook, HSSFColor.PALE_BLUE.index);
		styleHead.setAlignment(HSSFCellStyle.ALIGN_CENTER);

		diff(sheet, DdrUtil.getProperties());
		diff(sheet, DdrUtil.getProperties("other."));

		FileOutputStream fileOut = new FileOutputStream("diff.xls");
		workbook.write(fileOut);
		fileOut.close();
	}

	private static void diff(HSSFSheet sheet, DdrProperties p) throws Exception {

		if (p.getDriver() == null
		||	p.getUrl() == null
		||	p.getTableList() == null) return;

		Connection con = DdrUtil.getConnection(p);

		for (String table : p.getTableList()) {
			List<HashMap<String, Object>> columnList = DdrUtil.getColumns(con, p.getSchema(), table);
			headCount = columnList.size() + 1;
			addRow(sheet, null);
			addHeader(sheet, table, columnList);

			List<Integer> pkeyList = new ArrayList<Integer>();
			for (String pkey : DdrUtil.getPrimaryKeys(con, p.getSchema(), table)) {
				for (int i = 0; i < columnList.size(); i++) {
					if (pkey.equals(columnList.get(i).get("COLUMN_NAME"))) {
						pkeyList.add(new Integer(i + 1));
					}
				}
			}

			Runtime runtime = Runtime.getRuntime();
			String command = p.getDiff()
							+ " " + beforeDir + "/" + table + ".txt"
							+ " " + afterDir + "/" + table + ".txt"
							;
			Process process = runtime.exec(command);
			InputStream is = process.getInputStream();
			BufferedReader br = new BufferedReader(new InputStreamReader(is, "MS932"));

			boolean diff = false;
			String line= br.readLine();
			while (line != null) {
				if (find(line, 'a')) {
					/*
					 * add
					 */
					diff = true;
					line = br.readLine();
					do {
						addRow(sheet, line.replaceFirst(">", "追加"));
						line = br.readLine();
					} while(equal(line, '>'));
				} else if (find(line, 'd')) {
					/*
					 * delete
					 */
					diff = true;
					line = br.readLine();
					do {
						addRow(sheet, line.replaceFirst("<", "削除"), styleDelete);
						line = br.readLine();
					} while(equal(line, '<'));
				} else if (find(line, 'c')) {
					/*
					 * update
					 */
					diff = true;
					line = br.readLine();
					List<String> beforeList = new ArrayList<String>();
					do {
						beforeList.add(line);
						line = br.readLine();
					} while(equal(line, '<'));

					while(equal(line, '-')) {
						line = br.readLine();
					}

					List<String> afterList = new ArrayList<String>();
					do {
						afterList.add(line);
						line = br.readLine();
					} while(equal(line, '>'));

					if (pkeyList.size() == 0) {
						updateDiff(sheet, beforeList, afterList);
					} else {
						pkeyDiff(sheet, pkeyList, beforeList, afterList);
					}
				} else {
					System.out.println(line);
					throw new Exception();
				}
			}
			if (diff == false) {
				addRow(sheet, "一致");
			}
			br.close();
		}
	}

	private static void updateDiff(HSSFSheet sheet, List<String> beforeList, List<String> afterList) {

		if (beforeList.size() == afterList.size()) {
			for (int i = 0; i < beforeList.size(); i++) {
				String before = beforeList.get(i);
				String after = afterList.get(i);
				addRowBeforeAfter(sheet, before, after);
			}
		} else {
			for (String before : beforeList) {
				addRow(sheet, before.replaceFirst("<", "更新前/削除"), styleDelete);
			}
			for (String after : afterList) {
				addRow(sheet, after.replaceFirst(">", "更新後/追加"), styleUpdate);
			}
		}
	}

	private static void addRowBeforeAfter(HSSFSheet sheet, String before, String after) {

		HSSFRow rowBefore = addRow(sheet, before.replaceFirst("<", "更新前"));
		HSSFRow rowaAfter = addRow(sheet, after.replaceFirst(">", "更新後"), styleUpdate);
		String[] beforeCells = before.split("\t");
		String[] afterCells = after.split("\t");
		for (int c = 1; c < beforeCells.length; c++) {
			if (!beforeCells[c].equals(afterCells[c])) {
				rowBefore.getCell(c+1).setCellStyle(styleUpdateBefore);
				rowaAfter.getCell(c+1).setCellStyle(styleUpdateAfter);
			}
		}
	}

	private static void pkeyDiff(HSSFSheet sheet, List<Integer> pkeyList, List<String> beforeList, List<String> afterList) {

		for (String before : beforeList) {
			int afterIndex = findPkey(pkeyList, before, afterList);
			if (afterIndex == -1) {
				addRow(sheet, before.replaceFirst("<", "削除"), styleDelete);
			} else {
				for (int i = 0; i < afterIndex; i++) {
					addRow(sheet, afterList.get(i).replaceFirst(">", "追加"));
				}
				addRowBeforeAfter(sheet, before, afterList.get(afterIndex));

				List<String> newList = new ArrayList<String>();
				for (int i = afterIndex + 1; i < afterList.size(); i++) {
					newList.add(afterList.get(i));
				}
				afterList = newList;
			}
		}
		for (String after : afterList) {
			addRow(sheet, after.replaceFirst(">", "追加"));
		}
	}

	private static int findPkey(List<Integer> pkeyList, String before, List<String> afterList) {

		String[] beforeCells = before.split("\t");
		for (int i = 0; i < afterList.size(); i++) {
			String after = afterList.get(i);
			String[] afterCells = after.split("\t");
			if (equalPkey(pkeyList, beforeCells, afterCells)) {
				return i;
			}
		}
		return -1;
	}

	private static boolean equalPkey(List<Integer> pkeyList, String[] beforeCells, String[] afterCells) {

		for (int i : pkeyList) {
			if (!beforeCells[i].equals(afterCells[i])) {
				return false;
			}
		}
		return true;
	}

	private static boolean find(String line, char c) {
		return line.indexOf(c) != -1;
	}

	private static boolean equal(String line, char c) {
		if (line == null) return false;
		return line.charAt(0) == c;
	}

	private static HSSFCellStyle createStyle(HSSFWorkbook workbook, short color) {

		HSSFCellStyle style = workbook.createCellStyle();
		style.cloneStyleFrom(styleBox);
		style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND);
		style.setFillForegroundColor(color);
		return style;
	}

	private static HSSFRow addRow(HSSFSheet sheet, String line) {
		return addRow(sheet, line, styleBox);
	}

	private static HSSFRow addRow(HSSFSheet sheet, String line, HSSFCellStyle style) {

		HSSFRow row = sheet.createRow(rowCount++);
		colCount = 1;
		if (line != null) {
			addCells(row, line, style);
		}
		return row;
	}

	private static HSSFRow addHeader(HSSFSheet sheet, String table, List<HashMap<String, Object>> list) {

		HSSFRow row = sheet.createRow(rowCount++);
		colCount = 1;
		addCell(row, table, styleHead);
		for (HashMap<String, Object> map : list) {
			String s = (String)map.get("COLUMN_NAME");
			addCell(row, s, styleHead);
		}
		return row;
	}

	private static void addCells(HSSFRow row, String line, HSSFCellStyle style) {

		for (String s : line.split("\t", headCount)) {
			addCell(row, s, style);
		}
	}

	private static HSSFCell addCell(HSSFRow row, String s, HSSFCellStyle style) {

		HSSFCell cell = row.createCell(colCount++);
		cell.setCellType( HSSFCell.CELL_TYPE_STRING );
		cell.setCellValue(s);
		cell.setCellStyle(style);
		return cell;
	}
}
