﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using DataUnit.Core;
using System.IO;
using log4net;
using System.Reflection;

namespace DataUnit.Windows
{
    public partial class frmSql : Form
    {

        private bool _isWaiting = false;
        private bool _isDataExits = false;
        private Config _config = null;
        private DateTime _nextExecTime;
        private DateTime _StartTime;

        private static readonly ILog _logger
           = LogManager.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        private frmSql()
        {
            InitializeComponent();
        }

        public frmSql(Config config):this()
        {
            _config = config;
        }

        private void frmSql_Load(object sender, EventArgs e)
        {
            cmbUnit.SelectedIndex = 0;
            txtDirSql.Text = _config.SqlPath;
            txtDirResult.Text = _config.OutputPath;
            btnExpResult.Enabled = Directory.Exists(this.txtDirResult.Text);
        }

        private void btnWait_Click(object sender, EventArgs e)
        {
            ChangeControlStatus();
        }

        private void SetNextExecTime()
        {
            long nanoSecond = 0;
            switch (this.cmbUnit.SelectedIndex)
            {
                case 0:
                    nanoSecond = Convert.ToInt64(nudSapn.Value) * 60 * 1000 * 10000;
                    break;

                case 1:
                    nanoSecond = Convert.ToInt64(nudSapn.Value) * 60 * 60 * 1000 * 10000;
                    break;

                case 2:
                    nanoSecond = Convert.ToInt64(nudSapn.Value) * 24 * 60 * 60 * 1000 * 10000;
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }

            TimeSpan ts = new TimeSpan(nanoSecond);
            _nextExecTime = GetNowDateTime().Add(ts);
            lblNext.Text = String.Format("次回実行時刻：{0:yyyy/MM/dd HH:mm}", _nextExecTime);
            lblNext.Visible = true;
        }

        private void ChangeControlStatus()
        {
            _isWaiting = !_isWaiting;
            if (_isWaiting)
            {
                SetNextExecTime();
                btnWait.Text = "　タイマー解除";
                btnWait.ImageIndex = 1;
                rtxResult.Text = String.Format("{0:yyyy/MM/dd HH:mm:ss} 監視を開始します。\r\n", DateTime.Now);
                this.Text = "DataUnit";
                pgbExec.ColorBarBorder = Color.Lime;
                btnExpSql.Enabled = Directory.Exists(this.txtDirSql.Text);
                btnExpResult.Enabled = Directory.Exists(this.txtDirResult.Text);
                this.BackColor = System.Drawing.SystemColors.ControlLight;
                picLock.Visible = true;
            }
            else
            {
                lblNext.Visible = false;
                btnWait.Text = "　タイマー実行";
                btnWait.ImageIndex = 0;
                btnExpSql.Enabled = Directory.Exists(txtDirSql.Text);
                btnExpResult.Enabled = Directory.Exists(txtDirResult.Text);
                BackColor = SystemColors.Control;
                picLock.Visible = false;
            }

            pgbExec.Position = 0;
            pgbExec.Text = "";
            btnSQL.Enabled = !_isWaiting;
            btnClose.Enabled = !_isWaiting;
            timExec.Enabled = _isWaiting;
            cmbUnit.Enabled = !_isWaiting;
            nudSapn.Enabled = !_isWaiting;
        }

        private void timExec_Tick(object sender, EventArgs e)
        {
            
            if (_nextExecTime <= GetNowDateTime())
            {
                timExec.Enabled = false;
                SetNextExecTime();
                ExecSQL();
                timExec.Enabled = true;
            }
            
        }

        private void ExecSQL()
        {
            try
            {
                _StartTime = DateTime.Now;
                _isDataExits = false;
                Host host = new Host(_config);
                _logger.Info(String.Format("「{0}」フォルダのSQLを実行します。", host.Config.SqlPath));
                host.Status += host_Status;
                host.CheckData();
                pgbExec.ColorBarBorder = _isDataExits ? Color.Yellow : Color.Lime;
            }
            catch (Exception ex)
            {
                _logger.Error("エラーが発生しました。", ex);
                string msg = String.Format("\r\nエラーが発生しました。\r\n{0}\r\n{1}\r\n", ex.Message,ex.ToString());
                pgbExec.ColorBarBorder = Color.Red;
                rtxResult.Text += msg;
                ScrollRichText();
            }
            finally
            {
                this.Text += GetFinishedTitle();
            }
        }

        private void host_Status(object sender, StatusEventArgs e)
        {
            pgbExec.PositionMax = e.AllCount;
            pgbExec.Position = e.CurrentCount;

            if (e.Kind == StatusKind.ExistData)
            {
                _isDataExits = true;
            }

            this.Text = GetExecTitle(e.CurrentCount, e.AllCount);
            pgbExec.Text = String.Format("{0}/{1}", e.CurrentCount.ToString(), e.AllCount.ToString());
            string msg = String.Format("{0:yyyy/MM/dd HH:mm:ss} {1}\r\n", DateTime.Now,e.Message);
            if (e.Kind == StatusKind.Start)
            {
                msg += String.Format("{0:yyyy/MM/dd HH:mm:ss} 全{1}件のSQLを実行します。\r\n", DateTime.Now, e.AllCount.ToString());
            }
            rtxResult.Text += msg;
            ScrollRichText();
            Application.DoEvents();
        }

        private DateTime GetNowDateTime()
        {
            DateTime now = DateTime.Now;
            return new DateTime(now.Year, now.Month, now.Day, 
                                now.Hour, now.Minute, 0);
        }

        /// <summary>
        /// フォームを閉じる時のイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void frmTimer_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (!btnClose.Enabled)
            {
                const string MSG = "タイマー実行中です。\r\nタイマーを解除してから画面を閉じてください。\r\n";
                rtxResult.Text += MSG;
                ScrollRichText();
                e.Cancel = true;
                return;
            }

            //呼び出し元のフォームを再表示
            Owner.Visible = true;
        }

        private void ScrollRichText()
        {
            rtxResult.SelectionStart = rtxResult.Text.Length;
            rtxResult.Focus();
            rtxResult.ScrollToCaret();
        }

        private void btnClose_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void btnSQL_Click(object sender, EventArgs e)
        {
            Cursor = Cursors.WaitCursor;
            rtxResult.Text = "";
            btnClose.Enabled = false;
            btnWait.Enabled = false;
            btnSQL.Enabled = false;
            btnExpSql.Enabled = false;
            btnExpResult.Enabled = false;
            pgbExec.ColorBarBorder = Color.Lime;

            ExecSQL();

            btnClose.Enabled = true;
            btnWait.Enabled = true;
            btnSQL.Enabled = true;
            Cursor = Cursors.Default;
            btnExpSql.Enabled = Directory.Exists(txtDirSql.Text);
            btnExpResult.Enabled = Directory.Exists(txtDirResult.Text);

        }

        private string GetExecTitle(int count, int allCount)
        {
            const string title = "DataUnit ({0}/{1}) {2}";
            return String.Format(title,
                                count,
                                allCount,
                                _StartTime.ToString("HH:mm:ss"));
        }

        private string GetFinishedTitle()
        {
            DateTime endTime = DateTime.Now;
            TimeSpan ts = endTime - _StartTime;
            string span = "";
            if (ts.Hours > 0)
            {
                span = ts.Hours.ToString("d") + "時間";
            }
            if (ts.Minutes > 0)
            {
                span = span + ts.Minutes.ToString("d") + "分";
            }
            span = span + ts.Seconds.ToString("d") + "秒";
            return "→" + endTime.ToString("HH:mm:ss") + " [" + span + "]";

        }

        private void btnExpSql_Click(object sender, EventArgs e)
        {
            UICommon.ShowExplorer(txtDirSql.Text);
        }

        private void btnExpResult_Click(object sender, EventArgs e)
        {
            UICommon.ShowExplorer(txtDirResult.Text);
        }

        /// <summary>
        /// タイマー実行時に最小化すると、タスクバーにのみ表示
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void frmSql_Resize(object sender, EventArgs e)
        {
            if (WindowState == FormWindowState.Minimized && _isWaiting )
            {
                Visible = false;
                WindowState = FormWindowState.Normal;
                notifyIconTray.Visible = true;
            }
       
        }

        /// <summary>
        /// 通知アイコンをクリックした時のイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void notifyIconTray_Click(object sender, EventArgs e)
        {
            Visible = true;
            WindowState = FormWindowState.Normal;
            notifyIconTray.Visible = false;
            Activate();
            Application.DoEvents();
        }
    }
}
