﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Security;

namespace D2DBench
{
    class GDIBitmap : IBitmap
    {
        public IntPtr hbmp;
        public Graphics bmpGrp;
        public IntPtr bmpDC;
        public GDIBitmap(IntPtr hbmp, Graphics bmpGrp, IntPtr bmpDC,int width,int height)
        {
            this.hbmp = hbmp;
            this.bmpGrp = bmpGrp;
            this.bmpDC = bmpDC;
            this.Width = width;
            this.Height = height;
        }

        public int Width
        {
            get;
            private set;
        }

        public int Height
        {
            get;
            private set;
        }
    }
    class GDI : IDisposable,IRender
    {
        [StructLayout(LayoutKind.Sequential)]
        struct SIZE
        {
            public int cx;
            public int cy;

            public SIZE(int cx, int cy)
            {
                this.cx = cx;
                this.cy = cy;
            }
        }
        [StructLayout(LayoutKind.Sequential)]
        struct RECT
        {
            public int left;
            public int top;
            public int right;
            public int bottom;

            public RECT(int left, int top, int right, int bottom)
            {
                this.left = left;
                this.top = top;
                this.right = right;
                this.bottom = bottom;
            }
        }

        [Flags]
        enum ETOOptions : uint
        {
            ETO_NONE = 0,
            ETO_CLIPPED = 0x4,
            ETO_GLYPH_INDEX = 0x10,
            ETO_IGNORELANGUAGE = 0x1000,
            ETO_NUMERICSLATIN = 0x800,
            ETO_NUMERICSLOCAL = 0x400,
            ETO_OPAQUE = 0x2,
            ETO_PDY = 0x2000,
            ETO_RTLREADING = 0x800,
        }

        [Flags]
        enum PenStyle : int
        {
            PS_SOLID = 0,
            PS_DASH = 1,
            PS_DOT = 2,
            PS_DASHDOT = 3,
            PS_DASHDOTDOT = 4,
            PS_NULL = 5,
            PS_INSIDEFRAME = 6,
            PS_USERSTYLE = 7,
            PS_ALTERNATE = 8,
            PS_STYLE_MASK = 0x0000000F,
            PS_ENDCAP_ROUND = 0x00000000,
            PS_ENDCAP_SQUARE = 0x00000100,
            PS_ENDCAP_FLAT = 0x00000200,
            PS_ENDCAP_MASK = 0x00000F00,
            PS_JOIN_ROUND = 0x00000000,
            PS_JOIN_BEVEL = 0x00001000,
            PS_JOIN_MITER = 0x00002000,
            PS_JOIN_MASK = 0x0000F000,
            PS_COSMETIC = 0x00000000,
            PS_GEOMETRIC = 0x00010000,
            PS_TYPE_MASK = 0x000F0000
        };

        [Flags]
        enum FontWeight : int
        {
            FW_DONTCARE = 0,
            FW_THIN = 100,
            FW_EXTRALIGHT = 200,
            FW_LIGHT = 300,
            FW_NORMAL = 400,
            FW_MEDIUM = 500,
            FW_SEMIBOLD = 600,
            FW_BOLD = 700,
            FW_EXTRABOLD = 800,
            FW_HEAVY = 900,
        }
        [Flags]
        enum FontCharSet : uint
        {
            ANSI_CHARSET = 0,
            DEFAULT_CHARSET = 1,
            SYMBOL_CHARSET = 2,
            SHIFTJIS_CHARSET = 128,
            HANGEUL_CHARSET = 129,
            HANGUL_CHARSET = 129,
            GB2312_CHARSET = 134,
            CHINESEBIG5_CHARSET = 136,
            OEM_CHARSET = 255,
            JOHAB_CHARSET = 130,
            HEBREW_CHARSET = 177,
            ARABIC_CHARSET = 178,
            GREEK_CHARSET = 161,
            TURKISH_CHARSET = 162,
            VIETNAMESE_CHARSET = 163,
            THAI_CHARSET = 222,
            EASTEUROPE_CHARSET = 238,
            RUSSIAN_CHARSET = 204,
            MAC_CHARSET = 77,
            BALTIC_CHARSET = 186,
        }
        [Flags]
        enum FontPrecision : uint
        {
            OUT_DEFAULT_PRECIS = 0,
            OUT_STRING_PRECIS = 1,
            OUT_CHARACTER_PRECIS = 2,
            OUT_STROKE_PRECIS = 3,
            OUT_TT_PRECIS = 4,
            OUT_DEVICE_PRECIS = 5,
            OUT_RASTER_PRECIS = 6,
            OUT_TT_ONLY_PRECIS = 7,
            OUT_OUTLINE_PRECIS = 8,
            OUT_SCREEN_OUTLINE_PRECIS = 9,
            OUT_PS_ONLY_PRECIS = 10,
        }
        [Flags]
        enum FontClipPrecision : uint
        {
            CLIP_DEFAULT_PRECIS = 0,
            CLIP_CHARACTER_PRECIS = 1,
            CLIP_STROKE_PRECIS = 2,
            CLIP_MASK = 0xf,
            CLIP_LH_ANGLES = (1 << 4),
            CLIP_TT_ALWAYS = (2 << 4),
            CLIP_DFA_DISABLE = (4 << 4),
            CLIP_EMBEDDED = (8 << 4),
        }
        [Flags]
        enum FontQuality : uint
        {
            DEFAULT_QUALITY = 0,
            DRAFT_QUALITY = 1,
            PROOF_QUALITY = 2,
            NONANTIALIASED_QUALITY = 3,
            ANTIALIASED_QUALITY = 4,
            CLEARTYPE_QUALITY = 5,
            CLEARTYPE_NATURAL_QUALITY = 6,
        }
        [Flags]
        enum FontPitchAndFamily : uint
        {
            DEFAULT_PITCH = 0,
            FIXED_PITCH = 1,
            VARIABLE_PITCH = 2,
            FF_DONTCARE = (0 << 4),
            FF_ROMAN = (1 << 4),
            FF_SWISS = (2 << 4),
            FF_MODERN = (3 << 4),
            FF_SCRIPT = (4 << 4),
            FF_DECORATIVE = (5 << 4),
        }

        public enum StockObjects
        {
            WHITE_BRUSH = 0,
            LTGRAY_BRUSH = 1,
            GRAY_BRUSH = 2,
            DKGRAY_BRUSH = 3,
            BLACK_BRUSH = 4,
            NULL_BRUSH = 5,
            HOLLOW_BRUSH = NULL_BRUSH,
            DC_BRUSH = 18,
            DC_PEN = 19,
        }

        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern IntPtr CreateSolidBrush(uint crColor);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern uint SetTextColor(IntPtr hdc, int crColor);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern uint SetBkColor(IntPtr hdc, int crColor);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern IntPtr SelectObject(IntPtr hdc, IntPtr hgdiobj);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern bool DeleteObject(IntPtr hObject);
        [DllImport("gdi32.dll", EntryPoint = "GetTextExtentExPointW"), SuppressUnmanagedCodeSecurity]
        static extern bool GetTextExtentExPoint(IntPtr hdc, [MarshalAs(UnmanagedType.LPWStr)] string lpszStr, int cchString, int nMaxExtent, out int lpnFit, int[] alpDx, out SIZE lpSize);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern bool Rectangle(IntPtr hdc, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect);
        [DllImport("gdi32.dll", EntryPoint = "ExtTextOutW"), SuppressUnmanagedCodeSecurity]
        static extern bool ExtTextOut(IntPtr hdc, int X, int Y, uint fuOptions, [In] ref RECT lprc, [MarshalAs(UnmanagedType.LPWStr)] string lpString, uint cbCount, [In] int[] lpDx);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern IntPtr CreatePen(PenStyle fnPenStyle, int nWidth, uint crColor);
        [DllImport("gdi32.dll"), SuppressUnmanagedCodeSecurity]
        static extern IntPtr CreateFont(int nHeight, int nWidth, int nEscapement, int nOrientation, FontWeight fnWeight, uint fdwItalic, uint fdwUnderline, uint fdwStrikeOut, FontCharSet fdwCharSet, FontPrecision fdwOutputPrecision, FontClipPrecision fdwClipPrecision, FontQuality fdwQuality, FontPitchAndFamily fdwPitchAndFamily, string lpszFace);
        [System.Runtime.InteropServices.DllImportAttribute("gdi32.dll")]
        static extern int BitBlt(
          IntPtr hdcDest,     // handle to destination DC (device context)
          int nXDest,         // x-coord of destination upper-left corner
          int nYDest,         // y-coord of destination upper-left corner
          int nWidth,         // width of destination rectangle
          int nHeight,        // height of destination rectangle
          IntPtr hdcSrc,      // handle to source DC
          int nXSrc,          // x-coordinate of source upper-left corner
          int nYSrc,          // y-coordinate of source upper-left corner
          System.Int32 dwRop  // raster operation code
          );
        [DllImport("gdi32.dll", EntryPoint = "CreateCompatibleDC")]
        public static extern IntPtr CreateCompatibleDC(IntPtr hdc);
        [DllImport("gdi32.dll", EntryPoint = "DeleteDC")]
        public static extern IntPtr DeleteDC(IntPtr hdc);
        [DllImport("gdi32.dll", EntryPoint = "CreateCompatibleBitmap")]
        public static extern IntPtr CreateCompatibleBitmap(IntPtr hdc,int nWidth, int nHeight);
        [DllImport("gdi32.dll", ExactSpelling = true, PreserveSig = true, SetLastError = true)]
        static extern bool Ellipse(IntPtr hdc, int nLeftRect, int nTopRect,int nRightRect, int nBottomRect);
        [DllImport("gdi32.dll")]
        static extern bool BeginPath(IntPtr hdc);
        [DllImport("gdi32.dll")]
        static extern bool EndPath(IntPtr hdc);
        [DllImport("gdi32.dll")]
        static extern bool StrokeAndFillPath(IntPtr hdc);
        [DllImport("gdi32.dll")]
        static extern IntPtr GetStockObject(StockObjects fnObject);

        Graphics frontgrp;
        Rectangle ClientRect;
        IntPtr frontdc,backdc,backbmp;
        ColorTableGDI colors = new ColorTableGDI();
        Font font;

        public GDI(Control ctrl)
        {
            ClientRect = ctrl.ClientRectangle;
            this.font = ctrl.Font;
            this.frontgrp = ctrl.CreateGraphics();
            this.frontdc = this.frontgrp.GetHdc();
            this.backbmp = CreateCompatibleBitmap(this.frontdc, ClientRect.Width, ClientRect.Height);
        }

        public void Dispose()
        {
            DeleteObject(this.backbmp);
            this.frontgrp.ReleaseHdc(this.frontdc);
            this.frontgrp.Dispose();
        }

        public void BeginDraw()
        {
            this.backdc = CreateCompatibleDC(this.frontdc);
            SelectObject(this.backdc, this.backbmp);
        }

        public void EndDraw()
        {
            BitBlt(this.frontdc, 0, 0, ClientRect.Width, ClientRect.Height, this.backdc, 0, 0, 0xcc0020);
            DeleteDC(this.backdc);
        }

        public int ColorCount { get { return this.colors.Count; } }
        
        public IBitmap Load(Bitmap src)
        {
            IntPtr hbmp = src.GetHbitmap();
            Graphics bmpGrp = Graphics.FromImage(src);
            IntPtr hsrc = bmpGrp.GetHdc();
            return new GDIBitmap(hbmp, bmpGrp, hsrc,src.Width,src.Height);
        }

        public void UnLoad(IBitmap bmp)
        {
            GDIBitmap gbmp = (GDIBitmap)bmp;
            gbmp.bmpGrp.ReleaseHdc(gbmp.bmpDC);
            DeleteObject(gbmp.hbmp);
        }


        public void DrawBitmap(IBitmap bmp,int left,int top,int right,int bottom)
        {
            GDIBitmap gbmp = (GDIBitmap)bmp;
            IntPtr hbmpold = SelectObject(gbmp.bmpDC, gbmp.hbmp);

            BitBlt(backdc, left, top, right, bottom, gbmp.bmpDC, 0, 0, 0xcc0020);

            SelectObject(gbmp.bmpDC, hbmpold);
        }
        
        public void FillRectangle(int left, int top, int right, int bottom, int colorno)
        {
            int OldBackColor = (int)SetBkColor(backdc, ColorTranslator.ToWin32(this.colors[colorno]));
            RECT rect = new RECT(left, top, right + 1, bottom + 1);
            ExtTextOut(backdc, rect.left, rect.top, (uint)ETOOptions.ETO_OPAQUE, ref rect, string.Empty, (uint)0, null);
            SetTextColor(backdc, OldBackColor);
        }

        public void DrawEllipse(int x, int y, int radiusX, int radiusY, int colorno)
        {
            IntPtr hBrush = GetStockObject(StockObjects.NULL_BRUSH);
            IntPtr oldbrush = SelectObject(backdc,hBrush);

            PenStyle style = PenStyle.PS_SOLID;
            IntPtr pen = CreatePen(style, 1, (uint)ColorTranslator.ToWin32(this.colors[colorno]));

            IntPtr oldpen = SelectObject(backdc, pen);

            int left = x - radiusX;
            int top = y - radiusY;
            int right = x + radiusX;
            int bottom = y + radiusY;
            Ellipse(backdc, left, top, right, bottom);

            DeleteObject(pen);
            SelectObject(backdc, oldpen);

            DeleteObject(hBrush);
            SelectObject(backdc, oldbrush);
        }

        public void DrawEllipseByBrush(int x, int y, int radiusX, int radiusY, int colorno, int backcolorno)
        {
            IntPtr hBrush = CreateSolidBrush((uint)ColorTranslator.ToWin32(this.colors[backcolorno]));
            IntPtr oldbrush = SelectObject(backdc, hBrush);

            BeginPath(backdc);

            int left = x - radiusX;
            int top = y - radiusY;
            int right = x + radiusX;
            int bottom = y + radiusY;
            Ellipse(backdc, left, top, right, bottom);

            EndPath(backdc);

            PenStyle style = PenStyle.PS_SOLID;
            IntPtr pen = CreatePen(style, 1, (uint)ColorTranslator.ToWin32(this.colors[colorno]));
            IntPtr oldpen = SelectObject(backdc, pen);

            StrokeAndFillPath(backdc);

            DeleteObject(pen);
            SelectObject(backdc, oldpen);

            DeleteObject(hBrush);
            SelectObject(backdc, oldbrush);
        }

        public void DrawRectangleByBrush(int left, int top, int right, int bottom, int colorno, int backcolorno)
        {
            IntPtr hBrush = CreateSolidBrush((uint)ColorTranslator.ToWin32(this.colors[backcolorno]));
            IntPtr oldbrush = SelectObject(backdc, hBrush);

            BeginPath(backdc);
            
            Rectangle(backdc, left, top, right, bottom);
            
            EndPath(backdc);

            PenStyle style = PenStyle.PS_SOLID;
            IntPtr pen = CreatePen(style, 1, (uint)ColorTranslator.ToWin32(this.colors[colorno]));
            IntPtr oldpen = SelectObject(backdc, pen);

            StrokeAndFillPath(backdc);

            DeleteObject(pen);
            SelectObject(backdc, oldpen);

            DeleteObject(hBrush);
            SelectObject(backdc, oldbrush);
        }

        public void DrawRectangle(int left, int top, int right, int bottom, int colorno)
        {
            IntPtr hBrush = GetStockObject(StockObjects.NULL_BRUSH);
            IntPtr oldbrush = SelectObject(backdc, hBrush);

            PenStyle style = PenStyle.PS_SOLID;
            IntPtr pen = CreatePen(style, 1, (uint)ColorTranslator.ToWin32(this.colors[colorno]));

            IntPtr oldpen = SelectObject(backdc, pen);

            Rectangle(backdc, left, top, right, bottom);

            DeleteObject(pen);
            SelectObject(backdc, oldpen);

            DeleteObject(hBrush);
            SelectObject(backdc, oldbrush);
        }

        const string showStr = "D2DBenchMark";
        public void DrawString(int x, int y, int colorno)
        {
            int ForeColorCode = ColorTranslator.ToWin32(this.colors[colorno]);

            IntPtr hfont = GetHFont(this.font);
            IntPtr OldFont = SelectObject(backdc, hfont);
            int OldForeColor = (int)SetTextColor(backdc, ForeColorCode);
            int OldBackColor = (int)SetBkColor(backdc, 1);

            ETOOptions opt = ETOOptions.ETO_NONE;
            RECT rect = new RECT(x, y, x + 100, y + 100);
            ExtTextOut(backdc, x, y, (uint)opt, ref rect, showStr, (uint)showStr.Length, null);

            DeleteObject(hfont);
            SelectObject(backdc, OldFont);
            SetTextColor(backdc, OldForeColor);
            SetBkColor(backdc, OldBackColor);
        }

        IntPtr GetHFont(Font font)
        {
            int height = -(int)(font.Size * 96.0F / 72.0F);
            return CreateFont(height, 0, 0, 0, FontWeight.FW_NORMAL, 0, 0, 0, FontCharSet.DEFAULT_CHARSET, FontPrecision.OUT_DEFAULT_PRECIS, FontClipPrecision.CLIP_DEFAULT_PRECIS, FontQuality.DEFAULT_QUALITY, FontPitchAndFamily.DEFAULT_PITCH, font.Name);
        }
    }
}
