/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ksk.contacts.util;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.android.internal.telephony.ITelephony;

import android.content.Context;
import android.database.Cursor;
import android.os.RemoteException;
import android.provider.CallLog;
import android.telephony.TelephonyManager;

public class CallLogMgr {

	public static final Integer KIND_NONE = 0;

	public static final Integer KIND_SEND = 1;

	public static final Integer KIND_RECV = 2;

	//Context情報
	private Context mContext;

	//CallLog情報
	public class CallLogData {
		public int     contact_id;
		public String  name;
		public String  number;
		public Integer type;
		public Date    date;
		public Long    duration;
	}

	//コンテキスト
	public CallLogMgr(Context context){
		mContext = context;
	}

	public List<CallLogData> getCallLogData(Integer kind){

		Cursor cursor = null;

		List<CallLogData> list = new ArrayList<CallLogData>();

		ContactMgr mgr = new ContactMgr(mContext);

		//ContentProviderよりCallLogを取得(最大30件分)
		if (kind == KIND_SEND){
			//発信履歴
			cursor = mContext.getContentResolver().query( CallLog.Calls.CONTENT_URI,
														  null,
														  CallLog.Calls.TYPE + "=?",
														  new String[]{String.valueOf(CallLog.Calls.OUTGOING_TYPE)},
														  CallLog.Calls.DEFAULT_SORT_ORDER + " LIMIT 30" );

		}else if (kind == KIND_RECV){
			//着信履歴
			cursor = mContext.getContentResolver().query( CallLog.Calls.CONTENT_URI,
														  null, CallLog.Calls.TYPE + "=?",
														  new String[]{String.valueOf(CallLog.Calls.INCOMING_TYPE)},
														  CallLog.Calls.DEFAULT_SORT_ORDER + " LIMIT 30" );

		}else{
			//発信履歴＋着信履歴
			cursor = mContext.getContentResolver().query( CallLog.Calls.CONTENT_URI,
														  null,
														  null,
														  null,
														  CallLog.Calls.DEFAULT_SORT_ORDER + " LIMIT 30" );
		}

		//CallLogの存在有無をチェック
		if(cursor != null){
			if(cursor.moveToFirst()){

				do {
					//CallLog情報を設定
					CallLogData data = new CallLogData();
					//相手の名前
					//  →該当者が電話帳に未登録の場合には、NULLが設定される
					data.name     = cursor.getString(cursor.getColumnIndex(CallLog.Calls.CACHED_NAME));
					//相手の電話番号
					data.number   = cursor.getString(cursor.getColumnIndex(CallLog.Calls.NUMBER));
					if(data.number.equals("-2")){
						//Android仕様により、非通知時はNUMBERには"-2"(文字列)が格納されるため、文言置換
						data.number = "非通知";
					}
					//種別(発信 or 着信 or 不在)
					data.type     = cursor.getInt(cursor.getColumnIndex(CallLog.Calls.TYPE));
					//日付
					data.date     = new Date(cursor.getLong(cursor.getColumnIndex(CallLog.Calls.DATE)));
					//通話時間
					data.duration = cursor.getLong(cursor.getColumnIndex(CallLog.Calls.DURATION));

					//相手の名前が存在する場合(電話帳登録済み)は、電話番号からContactIDを取得する。
					if(data.name == null){
						//異常系(ありえないはず)
						data.contact_id = -1;

					}else{
						data.contact_id = mgr.getContactIDfromNumber(data.number);
					}

					//リストに追加
					list.add(data);

				} while(cursor.moveToNext());
			}
		}

		return list;
	}

	public List<CallLogData> getNewCallLogData(){

		Cursor cursor = null;

		List<CallLogData> list = new ArrayList<CallLogData>();

		ContactMgr mgr = new ContactMgr(mContext);

		//ContentProviderより未確認のCallLogを取得(最大30件分)
		//CallLog.Calls.NEWの値は、「確認済みの不在着信」のみが0になり、未確認の不在着信や発信、通話、着信は1になる
		cursor = mContext.getContentResolver().query( CallLog.Calls.CONTENT_URI,
													  null,
													  CallLog.Calls.TYPE + "=? AND " + CallLog.Calls.NEW + "=?",
													  new String[]{String.valueOf(CallLog.Calls.MISSED_TYPE),"1"},
													  CallLog.Calls.DEFAULT_SORT_ORDER + " LIMIT 30" );

		//CallLogの存在有無をチェック
		if(cursor != null){
			if(cursor.moveToFirst()){

				do {
					//CallLog情報を設定
					CallLogData data = new CallLogData();
					//相手の名前
					//  →該当者が電話帳に未登録の場合には、NULLが設定される
					data.name     = cursor.getString(cursor.getColumnIndex(CallLog.Calls.CACHED_NAME));
					//相手の電話番号
					data.number   = cursor.getString(cursor.getColumnIndex(CallLog.Calls.NUMBER));
					if(data.number.equals("-2")){
						//Android仕様により、非通知時はNUMBERには"-2"(文字列)が格納されるため、文言置換
						data.number = "非通知";
					}
					//種別(発信 or 着信 or 不在)
					data.type     = cursor.getInt(cursor.getColumnIndex(CallLog.Calls.TYPE));
					//日付
					data.date     = new Date(cursor.getLong(cursor.getColumnIndex(CallLog.Calls.DATE)));
					//通話時間
					data.duration = cursor.getLong(cursor.getColumnIndex(CallLog.Calls.DURATION));

					//相手の名前が存在する場合(電話帳登録済み)は、電話番号からContactIDを取得する。
					if(data.name == null){
						//異常系(ありえないはず)
						data.contact_id = -1;

					}else{
						data.contact_id = mgr.getContactIDfromNumber(data.number);
					}

					//リストに追加
					list.add(data);

				} while(cursor.moveToNext());
			}
		}

		return list;
	}

	public void cancelMissedCallNotification() {

		//TODO
		TelephonyManager telephony = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
		try {
			Class<?> c = Class.forName(telephony.getClass().getName());
			Method m = c.getDeclaredMethod("getITelephony");
			m.setAccessible(true);
			ITelephony telephonyService = (ITelephony) m.invoke(telephony);
			//呼び出し音を消す。
			telephonyService.cancelMissedCallsNotification();

		} catch (Exception e) {
			e.printStackTrace();
		}
	}

}
