/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ksk.contacts.tel.util;

import java.util.List;
import com.ksk.contacts.R;
import com.ksk.contacts.util.CommonData;
import com.ksk.contacts.util.SendAction;
import com.ksk.contacts.util.CommonData.MemberInfo;
import com.ksk.contacts.util.CommonData.MemberInfoHolder;
import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

public class MemberListAdapter extends BaseAdapter implements OnClickListener {

	//共通データクラス
	private CommonData     mCommonData = new CommonData();

	//メンバーリスト情報
	private List<MemberInfo> mListData;

	//Context情報
	private Context        mContext;

	//レイアウトID
	private int            mLayoutID = R.layout.memberlist_row;

	//コンストラクタ
    public MemberListAdapter(Context context) {
    	mContext  = context;
	}

	//コンストラクタ
    public MemberListAdapter(Context context, int layout) {
    	mContext  = context;
    	mLayoutID = layout;
	}

    /**
     * @param dataList
     */
    public void setDataList(List<MemberInfo> dataList) {
    	mListData = dataList;
	}

    public List<MemberInfo> getDataList() {
    	return mListData;
	}

	/* (非 Javadoc)
	 * @see android.widget.Adapter#getCount()
	 */
	@Override
    public int getCount() {
    	return mListData.size();
    }

    /* (非 Javadoc)
     * @see android.widget.Adapter#getItem(int)
     */
    @Override
    public Object getItem(int position) {
    	return mListData.get(position);
    }

    /* (非 Javadoc)
     * @see android.widget.Adapter#getItemId(int)
     */
    @Override
    public long getItemId(int position) {
    	return position;
    }

    /* (非 Javadoc)
     * @see android.widget.Adapter#getView(int, android.view.View, android.view.ViewGroup)
     */
    public View getView(int position, View convertView, ViewGroup parent) {
    	MemberInfoHolder holder = null;
		View   v = convertView;

		//選択位置を元にリスト情報を取得する。
		MemberInfo item = (MemberInfo) getItem(position);

		//viewが既に作成済みかをチェック
		if(v == null){
			//未作成の場合は、レイアウトを元にViewを新規作成する
			LayoutInflater inflater = (LayoutInflater)mContext.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
			v = inflater.inflate(mLayoutID, null, false);

			//Holderを作成し、各パラメータを設定する。
			holder = mCommonData.new MemberInfoHolder();
			holder.user   = (ImageView) v.findViewById(R.id.user);   //ユーザー画像
			holder.name   = (TextView)  v.findViewById(R.id.name);    //ユーザー名(フルネーム)
			holder.tel    = (ImageView) v.findViewById(R.id.btn_tel);     //音声発信ボタン
			holder.tel.setOnClickListener(this);                          //  →クリックリスナーの登録
			holder.email   = (ImageView) v.findViewById(R.id.btn_mail);   //Emailボタン
			holder.email.setOnClickListener(this);                        //  →クリックリスナーの登録
			holder.custom = (ImageView) v.findViewById(R.id.btn_custom);  //カスタムボタン
			holder.custom.setOnClickListener(this);                       //  →クリックリスナーの登録

			holder.tag1    = (TextView) v.findViewById(R.id.row_title);
			holder.tag2    = (LinearLayout) v.findViewById(R.id.row_userdata);
			//ViewにHolderを登録する。
			v.setTag(holder);

		}else{
			//View作成済みのため、Viewに登録されているTagからHolder情報を取得する。
			holder = (MemberInfoHolder) v.getTag();
		}

		//50音順の索引行を表示する
		if( (item.tag != null) && (!item.tag.equals("")) ){
			//索引インデックス行を表示する
			if(holder.tag1 != null){
				holder.tag1.setText(item.tag);
				holder.tag1.setVisibility(View.VISIBLE);
			}

			//索引インデックス行として表示する場合は、ユーザーデータ類は不要なため非表示にする。
			holder.tag2.setVisibility(View.GONE);
			return v;

		}else{
			//索引インデックス行ではないため、ユーザーデータ表示を可能にする。
			holder.tag2.setVisibility(View.VISIBLE);

			//索引インデックス表示用のTextViewは非表示に
			if(holder.tag1 != null){
				holder.tag1.setVisibility(View.GONE);
			}
		}

		//リスト情報がNULLではない、かつHolderがNULLでない場合は、各情報の設定を行う。
		if((item != null) && (holder != null) ){

			//HolderのDataに、ListData情報を登録する。
			//  →音声発信/Email/カスタムボタン押下時の各種情報取得用
			holder.data = item;
			holder.tel.setTag(holder);
			holder.email.setTag(holder);
			holder.custom.setTag(holder);

			//ユーザー画像の描画
			if(holder.user != null){
				if(item.photo != null){
					holder.user.setImageBitmap(item.photo);
				}else{
					holder.user.setImageResource(R.drawable.ic_launcher_android);
				}
			}

			//ユーザー名の設定
			holder.name.setText(item.name);

			//音声発信ボタンの表示
			if( (item.tel != null) && (!item.tel.equals("")) ){
				//音声発信ボタンに登録すべき電話番号がある場合
				holder.tel.setVisibility(View.VISIBLE);
			}else{
				//音声発信ボタンに登録すべき電話番号がない場合は非表示にする。
				holder.tel.setVisibility(View.INVISIBLE);
			}

			//Emailボタンの表示
			if( (item.email != null) && (!item.email.equals("")) ){
				//Emailボタンに登録すべきEmailがある場合
				holder.email.setVisibility(View.VISIBLE);
			}else{
				//Emailボタンに登録すべきEmailがない場合は非表示にする。
				holder.email.setVisibility(View.INVISIBLE);
			}

			//カスタムボタンの表示
			if(item.custom.kind != mCommonData.SEND_KIND_NONE){
				//カスタムボタンに登録すべきデータがある場合
				holder.custom.setVisibility(View.VISIBLE);
			}else{
				//カスタムボタンボタンに登録すべきデータがない場合は非表示にする。
				holder.custom.setVisibility(View.INVISIBLE);
			}
		}

		return v;
    }

	@Override
	public void onClick(View v) {

		SendAction action = new SendAction(mContext);

		//ViewのTagよりHolder情報を取得する。
		MemberInfoHolder holder = (MemberInfoHolder) v.getTag();

		//押下されたボタンに応じた処理を実施。
		if(holder.tel.equals(v)){
			action.sendCall(holder.data.tel);

		}else if(holder.email.equals(v)){
			action.sendMail( holder.data.email);

		}else if(holder.custom.equals(v)){
			if(holder.data.custom.kind == mCommonData.SEND_KIND_TEL){
				action.sendCall(holder.data.tel);

			}else if(holder.data.custom.kind == mCommonData.SEND_KIND_MAIL){
				action.sendMail(holder.data.email);
			}

		}
	}

	@Override
	public boolean isEnabled(int position) {

		if( (((MemberInfo)getItem(position)).tag != null) &&
		   (!((MemberInfo)getItem(position)).tag.equals("")) ){
			return false;
		}

		return super.isEnabled(position);
	}
}
