/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.account;

import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.access.Transaction;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.cayenne.om.account.AvzMPositionPosition;
import com.aimluck.eip.cayenne.om.account.EipMPosition;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALAbstractCheckList;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;

/**
 * 役職を複数削除するためのクラスです．
 * 
 */
public class AccountPositionMultiDelete extends ALAbstractCheckList {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AccountPositionMultiDelete.class
      .getName());

  /**
   * 一括削除処理
   * 
   * @param action
   *            アクションクラス
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param values
   *            役職IDリスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   */
  @Override
  // protected boolean action(RunData rundata, Context context,
  // List<String> values, List<String> msgList) {
  // try {
  // // オブジェクトモデル群を取得
  // List<EipMPosition> list = getEipMPositions(rundata, context, values);
  // if (list == null || list.size() == 0) {
  // return false;
  // }
  //
  // EipMPosition position = null;
  // List<Integer> ids = new ArrayList<Integer>();
  // int size = list.size();
  // for (int i = 0; i < size; i++) {
  // position = list.get(i);
  // ids.add(position.getPositionId());
  // }
  //
  // // 役職を削除
  // Expression exp =
  // ExpressionFactory.inDbExp(EipMPosition.POSITION_ID_PK_COLUMN, ids);
  // SelectQuery<EipMPosition> query = Database.query(EipMPosition.class, exp);
  // List<EipMPosition> postisions = query.fetchList();
  //
  // // 役職を削除
  // int psize = postisions.size();
  // for (int i = 0; i < psize; i++) {
  // Database.delete(list.get(i));
  // }
  //
  // // この役職に設定されているユーザーの役職IDを0とする
  // int idssize = ids.size();
  // for (int i = 0; i < idssize; i++) {
  // String sql =
  // "UPDATE turbine_user set POSITION_ID = 0 where POSITION_ID = "
  // + (ids.get(i)).intValue();
  // Database.sql(TurbineUser.class, sql).execute();
  // }
  // Database.commit();
  //
  // ALEipManager.getInstance().reloadPosition();
  //
  // } catch (Exception ex) {
  // Database.rollback();
  // logger.error("Exception", ex);
  // return false;
  // }
  // return true;
  // }
  protected boolean action(RunData rundata, Context context,
      List<String> values, List<String> msgList) {

    // 標準のトランザクション（オートコミット）
    Transaction baseTx = Transaction.getThreadTransaction();

    DataContext dataContext = DataContext.getThreadDataContext();

    // 自己管理トランザクション
    Transaction tx =
      Transaction.internalTransaction(DataContext
        .getThreadDataContext()
        .getParentDataDomain()
        .getTransactionDelegate());

    // 標準のトランザクションを自己管理トランザクションに置き換えます。
    Transaction.bindThreadTransaction(tx);

    try {
      // トランザクション開始
      tx.begin();
      // オブジェクトモデル群を取得
      List<EipMPosition> entities = getEipMPositions(rundata, context, values);
      if (entities == null || entities.size() == 0) {
        return false;
      }

      for (EipMPosition entity : entities) {
        // この役職に設定されているユーザーの役職IDをＮＵＬＬとする
        String sql =
          "UPDATE TURBINE_USER_GROUP_ROLE SET POSITION_ID = NULL where POSITION_ID = "
            + "#bind($positionId)";
        Database.sql(TurbineUserGroupRole.class, sql).param(
          "positionId",
          entity.getPositionId()).execute();

        Database.delete(entity.getAvzMPositionPositionArray());
        Database.delete(entity);
        dataContext.commitChanges();
      }

      // 他の役職の順番を変更する．
      SelectQuery<AvzMPositionPosition> p_query =
        Database.query(AvzMPositionPosition.class);
      p_query.orderAscending(AvzMPositionPosition.POSITION_PROPERTY);
      List<AvzMPositionPosition> entityPositions = p_query.fetchList();

      if (entityPositions != null && entityPositions.size() > 0) {

        int PositionSize = entityPositions.size();
        for (int i = 0; i < PositionSize; i++) {
          AvzMPositionPosition entityPosition = entityPositions.get(i);
          entityPosition.setPosition(Integer.valueOf(i + 1));
          dataContext.commitChanges();
        }
      }

      Database.commit();

      ALEipManager.getInstance().reloadPosition();

    } catch (Exception ex) {
      Database.rollback();
      String msg = "役職情報の一括削除に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    } finally {
      // 自己管理トランザクションを標準のトランザクションに戻します。
      Transaction.bindThreadTransaction(baseTx);
    }
    return true;
  }

  /**
   * 役職情報リスト取得
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @return 役職情報リスト
   * @throws Exception
   */
  private List<EipMPosition> getEipMPositions(RunData rundata, Context context,
      List<String> values) throws Exception {
    List<EipMPosition> list = null;

    try {
      if (values == null || values.size() == 0) {
        logger.debug("values are empty...");
        return null;
      }

      Expression exp =
        ExpressionFactory.inDbExp(EipMPosition.POSITION_ID_PK_COLUMN, values);
      SelectQuery<EipMPosition> query = Database.query(EipMPosition.class, exp);
      list = query.fetchList();
      if (list == null || list.size() == 0) {
        logger.debug("Not found ID...");
        return null;
      }
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      // list = null;
      throw ex;
      // change end
    }
    return list;
  }
}
