/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.account;

import java.util.Date;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.access.Transaction;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.account.util.AccountUtils;
import com.aimluck.eip.cayenne.om.account.AvzMPositionPosition;
import com.aimluck.eip.cayenne.om.account.EipMPosition;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 役職を管理するフォームデータを管理するクラスです。 <BR>
 * 
 */
public class AccountPositionFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService
      .getLogger(AccountPositionFormData.class.getName());

  /** 役職名 */
  private ALStringField position_name;

  /** 役職ID */
  private int position_id;

  /**
   * 初期化します。
   * 
   * @param action
   * @param rundata
   * @param context
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * 
   */
  public void initField() {

    // 役職名
    position_name = new ALStringField();
    position_name.setFieldName("役職名");
    position_name.setTrim(true);
  }

  /**
   * 各フィールドに対する制約条件を設定します。 <BR>
   * 
   * 
   */
  @Override
  protected void setValidator() {
    position_name.setNotNull(true);
    position_name.limitMaxLength(50);
  }

  /**
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    if (res) {
      try {
        if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
          position_id =
            Integer.parseInt(ALEipUtils.getTemp(
              rundata,
              context,
              ALEipConstants.ENTITY_ID));
        }
      } catch (Exception ex) {
        String msg = "役職情報の更新に失敗しました。";
        logger.error(msg, ex);
        msgList.add(msg);
        return false;
      }
    }
    return res;
  }

  /**
   * フォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return
   * 
   */
  @Override
  protected boolean validate(List<String> msgList) {
    position_name.validate(msgList);

    try {
      SelectQuery<EipMPosition> query = Database.query(EipMPosition.class);

      if (ALEipConstants.MODE_INSERT.equals(getMode())) {
        Expression exp =
          ExpressionFactory.matchExp(
            EipMPosition.POSITION_NAME_PROPERTY,
            position_name.getValue());
        query.setQualifier(exp);
      } else if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        Expression exp1 =
          ExpressionFactory.matchExp(
            EipMPosition.POSITION_NAME_PROPERTY,
            position_name.getValue());
        query.setQualifier(exp1);
        Expression exp2 =
          ExpressionFactory.noMatchDbExp(
            EipMPosition.POSITION_ID_PK_COLUMN,
            Integer.valueOf(position_id));
        query.andQualifier(exp2);
      }

      if (query.fetchList().size() != 0) {
        msgList.add("役職名『 <span class='em'>"
          + position_name
          + "</span> 』は既に登録されています。");
      }
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String mode = "登録";
      if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        mode = "更新";
      }
      String msg = "役職情報の" + mode + "に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      // change end
      return false;
    }

    return (msgList.size() == 0);
  }

  /**
   * 役職編集画面表示処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @throws ALDBErrorException
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    try {
      // オブジェクトモデルを取得
      EipMPosition record = AccountUtils.getEipMPosition(rundata, context);
      if (record == null) {
        // change start
        // return false;
        throw new Exception("指定された役職情報がデータベースにありません。");
        // change end
      }
      position_name.setValue(record.getPositionName());
    } catch (Exception ex) {
      String msg = "役職情報の編集画面の表示に失敗しました。";
      logger.error(msg, ex);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * 『役職』を追加します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      EipMPosition position = Database.create(EipMPosition.class);
      position.setPositionName(position_name.getValue());
      Date now = new Date();
      position.setCreateDate(now);
      position.setUpdateDate(now);

      // add by motegi start 役職順番に登録
      List<AvzMPositionPosition> posposlist =
        Database.query(AvzMPositionPosition.class).fetchList();
      int new_pos =
        (posposlist != null && posposlist.size() > 0)
          ? posposlist.size() + 1
          : 1;
      AvzMPositionPosition positionposition =
        Database.create(AvzMPositionPosition.class);
      positionposition.setToEipMPosition(position);
      positionposition.setPosition(Integer.valueOf(new_pos));
      // add end

      Database.commit();

      position_id = position.getPositionId().intValue();

      ALEipManager.getInstance().reloadPosition();
    } catch (Exception ex) {
      Database.rollback();
      String msg = "役職情報の登録に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    }
    return true;
  }

  /**
   * 『役職』を更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      EipMPosition record = AccountUtils.getEipMPosition(rundata, context);
      if (record == null) {
        // change start
        // return false;
        throw new Exception("指定された役職情報がデータベースにありません。");
        // change end
      }
      record.setPositionName(position_name.getValue());
      record.setUpdateDate(new Date());
      Database.commit();
      ALEipManager.getInstance().reloadPosition();
    } catch (Exception ex) {
      Database.rollback();
      String msg = "役職情報の更新に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    }
    return true;
  }

  /**
   * 『役職』を削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {

    // 標準のトランザクション（オートコミット）
    Transaction baseTx = Transaction.getThreadTransaction();

    DataContext dataContext = DataContext.getThreadDataContext();

    // 自己管理トランザクション
    Transaction tx =
      Transaction.internalTransaction(DataContext
        .getThreadDataContext()
        .getParentDataDomain()
        .getTransactionDelegate());

    // 標準のトランザクションを自己管理トランザクションに置き換えます。
    Transaction.bindThreadTransaction(tx);

    try {

      // トランザクション開始
      tx.begin();

      // オブジェクトモデルを取得
      EipMPosition entity = AccountUtils.getEipMPosition(rundata, context);
      if (entity == null) {
        return false;
      }

      // 役職IDを取得
      int positionId = entity.getPositionId();

      // 役職を削除
      Database.delete(entity.getAvzMPositionPositionArray());
      Database.delete(entity);
      dataContext.commitChanges();

      // この役職に設定されているユーザーの役職IDを0とする
      String sql =
        "UPDATE TURBINE_USER_GROUP_ROLE SET POSITION_ID = NULL where POSITION_ID = "
          + "#bind($positionId)";
      Database.sql(TurbineUserGroupRole.class, sql).param(
        "positionId",
        positionId).execute();
      dataContext.commitChanges();

      // 他の役職の順番を変更する．
      SelectQuery<AvzMPositionPosition> p_query =
        Database.query(AvzMPositionPosition.class);
      p_query.orderAscending(AvzMPositionPosition.POSITION_PROPERTY);
      List<AvzMPositionPosition> entityPositions = p_query.fetchList();

      if (entityPositions != null && entityPositions.size() > 0) {

        int size = entityPositions.size();
        for (int i = 0; i < size; i++) {
          AvzMPositionPosition entityPosition = entityPositions.get(i);
          entityPosition.setPosition(Integer.valueOf(i + 1));
        }
      }

      Database.commit();
      ALEipManager.getInstance().reloadPosition();

    } catch (Exception ex) {
      Database.rollback();
      String msg = "役職情報の削除に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    } finally {
      // 自己管理トランザクションを標準のトランザクションに戻します。
      Transaction.bindThreadTransaction(baseTx);
    }
    return true;
  }

  /**
   * 『役職名』を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getPositionName() {
    return position_name;
  }

  /**
   * 
   * @return
   */
  public int getPositionId() {
    return position_id;
  }
}
