/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2008 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "ConsoleScrollArea.h"
#include "wcwidth.h"

#include <QtGui/QFont>
#include <QtGui/QFontMetrics>
#include <QtGui/QPainter>
#include <QtGui/QResizeEvent>

const char * const ConsoleScrollArea::REPCHAR = 
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefgjijklmnopqrstuvwxyz0123456789./+@";


ConsoleScrollArea::ConsoleScrollArea(QWidget *parent)
: QAbstractScrollArea(parent), lock(false), cuX(0), cuY(0)
{
	QFont font;
	QFontMetrics fontMetrics(font);
	fontHeight = fontMetrics.height();
	fontLineSpacing = fontMetrics.lineSpacing();
	fontWidth = qRound(static_cast<double>(fontMetrics.width(REPCHAR))
			/ static_cast<double>(strlen(REPCHAR)));
	numX = width() / fontWidth;
	numY = height() / fontHeight;
}

void ConsoleScrollArea::keyPressEvent(QKeyEvent *e)
{
	QString str = e->text();
	switch (e->key()) {
	case Qt::Key_Backspace:
		if (cuX > 0) {
			rowList[rowList.size() - 1].chop(1);
			cuX--;
			viewport()->update();
		}
		return;
	case Qt::Key_Home:
		if (cuX != 0) {
			cuX = 0;
			viewport()->update();
		}
		return;
	case Qt::Key_End:
		{
			int x = rowList.at(cuY).size();
			if (cuX != x) {
				cuX = x;
				viewport()->update();
			}
			return;
		}
	case Qt::Key_Left:
		if (cuX > 0) {
			cuX--;
			viewport()->update();
		}
		return;
	case Qt::Key_Right:
		if (cuX < rowList.at(cuY).size()) {
			cuX++;
			viewport()->update();
		}
		return;
	}
	if (!str.isEmpty()) {
		if (str.endsWith('\r') || str.endsWith('\n')) {
			lock = true;
			if (!rowList.isEmpty()) {
				emit executed(rowList.at(rowList.size() - 1));
			}
		}
		append(str);
	}
}

void ConsoleScrollArea::paintEvent(QPaintEvent * /*event*/)
{
	QList<QChar> charList;
	int cursor = 0;
	for (int i = 0, j = rowList.size(); i < j; i++) {
		int l = rowList.at(i).size();
		int count = 0;
		for (int k = 0; k < l; k++) {
			QChar ch = rowList.at(i).at(k);
			charList.append(ch);
			count += wcwidth(ch.unicode()) - 1;
			if (i == cuY && k == cuX - 1) {
				cursor = charList.size();
			}
		}
		count += charList.size();
		for (int k = 0, m = numX - (count % numX); k < m; k++) {
			charList.append(' ');
		}
	}

	QPainter painter(viewport());
	int c = 0;
	int r = 0;
	for (int i = 0, j = charList.size(); i < j; i++) {
		if (c >= numX) {
			c = 0;
			r++;
		}
		QChar ch = charList[i];
		int w = wcwidth(ch.unicode());
		if (i == cursor) {
			painter.fillRect(c * fontWidth, r * fontHeight, 
					fontWidth * w, fontLineSpacing, QBrush(Qt::black));
			painter.setPen(Qt::white);
		}
		painter.drawText(c * fontWidth, r * fontHeight, 
				fontWidth * w, fontLineSpacing, Qt::AlignCenter, ch);
		if (i == cursor) {
			painter.setPen(Qt::black);
		}
		c += w;
	}
}

void ConsoleScrollArea::resizeEvent(QResizeEvent *event)
{
	QSize size = event->size();
	numX = size.width()  / fontWidth;
	numY = size.height()  / fontHeight;
}

void ConsoleScrollArea::append(const QString &text)
{
	QStringList list = text.split(QRegExp("(\n|\r)"));
	if (list.isEmpty()) {
		return;
	}
	if (!rowList.isEmpty()) {
		rowList[rowList.size() - 1] += list[0];
		list.takeFirst();
	}
	rowList << list;
	cuX = rowList[rowList.size() - 1].size();
	cuY = rowList.size() - 1;
	viewport()->update();
}

void ConsoleScrollArea::setLocked(bool status)
{
	lock = status;
}

bool ConsoleScrollArea::isLocked() const
{
	return lock;
}
