/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "Fit.h"

const double Fit::DEFAULT_FIT_LIMIT = 1e-5;

Fit::Fit() : functionName_("f(x)"),
	fitLimit_(DEFAULT_FIT_LIMIT), fitMaxIter_(DEFAULT_FIT_MAX_ITER)
{
}

void Fit::setRange(const FitRange &range)
{
	range_ = range;
}

const FitRange &Fit::range() const
{
	return range_;
}

void Fit::setFunctionName(const QString &name)
{
	functionName_ = name;
}

const QString &Fit::functionName() const
{
	return functionName_;
}

void Fit::setFunction(const QString &function)
{
	function_ = function;
}

const QString &Fit::function() const
{
	return function_;
}

void Fit::setDataFile(const QString &file)
{
	dataFile_ = file;
}

const QString &Fit::dataFile() const
{
	return dataFile_;
}

void Fit::setDataFileModifier(const FitModifier &mod)
{
	dataFileModifier_ = mod;
}

const FitModifier &Fit::dataFileModifier() const
{
	return dataFileModifier_;
}

void Fit::setParameterFile(bool isFile)
{
	isParameterFile_ = isFile;
}

bool Fit::isParameterFile() const
{
	return isParameterFile_;
}

void Fit::setParameterFile(const QString &file)
{
	parameterFile_ = file;
}

const QString &Fit::parameterFile() const
{
	return parameterFile_;
}

void Fit::setVariable(const QStringList &var)
{
	variable_ = var;
}

const QStringList &Fit::variable() const
{
	return variable_;
}

void Fit::setFitLimit(double limit)
{
	fitLimit_ = limit;
}

double Fit::fitLimit() const
{
	return fitLimit_;
}

void Fit::setFitMaxIter(int iter)
{
	fitMaxIter_ = iter;
}

int Fit::fitMaxIter() const
{
	return fitMaxIter_;
}

QString Fit::command() const
{
	QStringList commandList;
	if (fitLimit_ != DEFAULT_FIT_LIMIT) {
		commandList << "FIT_LIMIT = " + QString::number(fitLimit_);
	}
	if (fitMaxIter_ != DEFAULT_FIT_MAX_ITER) {
		commandList << "FIT_MAXITER = " + QString::number(fitMaxIter_);
	}
	commandList << functionName_ + "=" + function_;
	QString str = "fit " + range_.option() + " " + functionName_ + " '" 
		+ dataFile_ + "' " + dataFileModifier_.option() + " via ";
	if (isParameterFile_) {
		str += "'" + parameterFile_ + "'";
	} else {
		str += variable_.join(",");
	}
	commandList << str;
	return commandList.join("\n");
}
