﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using Utility;

namespace CTester.Setting
{
    public class Project : IDisposable
    {
        // ----------------------------------------
        // メンバークラス
        // ----------------------------------------
        #region Class Members

        public class SerializeData
        {
            public string SourcePath { get; set; }
            public EmuSetting EmuSetting { get; set; }
        }

        #endregion

        // ----------------------------------------
        // メンバー列挙型
        // ----------------------------------------
        #region Enum Members
        #endregion

        // ----------------------------------------
        // メンバー定数
        // ----------------------------------------
        #region Class Const Members
        #endregion

        // ----------------------------------------
        // フィールド
        // ----------------------------------------
        #region Fields

        private UnmanagedMethodBuilder _umb = null;
        private ITesterSource _source = null;
        private string _projectPath = string.Empty;
        private string _sourcePath = string.Empty;
        private EmuSetting _emuSetting = new EmuSetting();

        #endregion

        // ----------------------------------------
        // プロパティ
        // ----------------------------------------
        #region Properties

        public ITesterSource Source
        {
            get
            {
                return _source;
            }
        }

        public EmuSetting EmuSetting
        {
            get
            {
                return _emuSetting;
            }
            set
            {
                _emuSetting = value;
                OnChangeInfo(null);
            }
        }

        public bool CanOverwrite
        {
            get
            {
                return CanProjectWrite(_projectPath);
            }
        }

        public string ProjectName
        {
            get
            {
                if (_projectPath != string.Empty)
                {
                    return Path.GetFileNameWithoutExtension(_projectPath);
                }
                else if (_sourcePath != string.Empty)
                {
                    return Path.GetFileNameWithoutExtension(_sourcePath);
                }
                return string.Empty;
            }
        }

        #endregion

        // ----------------------------------------
        // イベント
        // ----------------------------------------
        #region Events

        /// <summary>
        /// データ変化イベント
        /// </summary>
        public event EventHandler ChangedInfo;

        /// <summary>
        /// データ変化イベント発生
        /// </summary>
        /// <param name="e">イベントデータ</param>
        private void OnChangeInfo(EventArgs e)
        {
            if (ChangedInfo != null)
            {
                ChangedInfo(this, e);
            }
        }

        #endregion

        // ----------------------------------------
        // コンストラクタ
        // ----------------------------------------
        #region Constructor

        public Project()
        {
        }

        public Project(string path)
        {
            _projectPath = path;
        }

        #endregion

        // ----------------------------------------
        // IDisposable インターフェース
        // ----------------------------------------
        #region IDisposable Interfaces

        /// <summary>
        /// メンバーの開放
        /// </summary>
        public void Dispose()
        {
            if (_umb != null)
            {
                _umb.Dispose();
                _umb = null;
            }
        }

        #endregion

        // ----------------------------------------
        // メソッド
        // ----------------------------------------
        #region Methods

        /// <summary>
        /// ソースロード
        /// </summary>
        /// <param name="sourcePath"></param>
        public bool LoadSource(string sourcePath)
        {
            if (LoadSourceSub(sourcePath))
            {
                _sourcePath = sourcePath;
                OnChangeInfo(null);
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// ソースロードサブ
        /// </summary>
        /// <param name="sourcePath"></param>
        /// <returns></returns>
        private bool LoadSourceSub(string sourcePath)
        {
            if (!System.IO.File.Exists(sourcePath))
            {
                return false;
            }
            _umb = new UnmanagedMethodBuilder(sourcePath);
            _source = _umb.Build(typeof(ITesterSource)) as ITesterSource;
            if(_source == null)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// プロジェクトロード
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public void LoadProject(string path)
        {
            if (LoadProjectSub(path))
            {
                OnChangeInfo(null);
            }
        }

        /// <summary>
        /// プロジェクトロードサブ
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private bool LoadProjectSub(string path)
        {
            if (!System.IO.File.Exists(path))
            {
                return false;
            }

            var data = Serializer.Deserialization(path, typeof(SerializeData)) as SerializeData;
            if (data != null)
            {
                _projectPath = path;
                LoadSourceSub(data.SourcePath);

                return true;
            }

            return false;
        }

        /// <summary>
        /// ロード
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public void Load(string path)
        {
            var ext = System.IO.Path.GetExtension(path);

            if (ext.CompareTo("." + App.SourceExtension) == 0)
            {
                LoadSource(path);
            }

            if (ext.CompareTo("." + App.ProjectExtension) == 0)
            {
                LoadProject(path);
            }
        }

        /// <summary>
        /// 新規プロジェクト
        /// </summary>
        /// <param name="folder"></param>
        /// <param name="name"></param>
        /// <param name="data"></param>
        public void NewProject(string folder, string name, SerializeData data)
        {
            if (!System.IO.Directory.Exists(folder))
            {
                System.IO.Directory.CreateDirectory(folder);
            }

            string path = System.IO.Path.Combine(folder, name + "." + App.ProjectExtension);
            Serializer.Serialization(path, data);

            _projectPath = path;
            if (!string.IsNullOrWhiteSpace(data.SourcePath) && System.IO.File.Exists(data.SourcePath))
            {
                LoadSource(data.SourcePath);
            }

            OnChangeInfo(null);
        }

        /// <summary>
        /// 上書きセーブ
        /// </summary>
        public void SaveOverwrite()
        {
            Save(_projectPath);
        }

        /// <summary>
        /// セーブ
        /// </summary>
        /// <param name="path"></param>
        public void Save(string path)
        {
            if (!CanProjectWrite(path))
            {
                return;
            }

            var data = new SerializeData();
            data.SourcePath = _umb.SourcePath;

            Serializer.Serialization(path, data);
        }

        /// <summary>
        /// プロジェクト書き込み可能チェック
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public bool CanProjectWrite(string path)
        {
            if (string.IsNullOrWhiteSpace(path))
            {
                return false;
            }
            if (System.IO.File.Exists(path))
            {
                return true;
            }
            return false;
        }

        #endregion
    }
}
