package jp.cssj.homare.css;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.cti2.helpers.URIHelper;
import jp.cssj.homare.css.property.ElementPropertySet;
import jp.cssj.homare.css.property.FontFacePropertySet;
import jp.cssj.homare.css.property.PagePropertySet;
import jp.cssj.homare.impl.css.property.CSSFontFamily;
import jp.cssj.homare.impl.css.property.CSSFontStyle;
import jp.cssj.homare.impl.css.property.FontWeight;
import jp.cssj.homare.impl.css.property.css3.CSSUnicodeRange;
import jp.cssj.homare.impl.css.property.css3.Src;
import jp.cssj.homare.message.MessageCodes;
import jp.cssj.homare.ua.DocumentContext;
import jp.cssj.homare.ua.UserAgent;
import jp.cssj.homare.xml.util.XMLUtils;
import jp.cssj.resolver.Source;
import jp.cssj.sakae.gc.font.FontFace;
import jp.cssj.sakae.gc.font.FontManager;
import jp.cssj.sakae.sac.parser.Parser;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.DocumentHandler;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.SACMediaList;
import org.w3c.css.sac.SelectorList;

/**
 * SACイベントからCSSStyleSheetオブジェクトを構築します。
 * 
 * @author <a href="mailto:tatsuhiko at miya.be">MIYABE Tatsuhiko </a>
 * @version $Id: StyleSheetBuilder.java,v 1.4 2007-05-06 04:24:06 miyabe Exp $
 */
public class CSSStyleSheetBuilder implements DocumentHandler {
	private static final Logger LOG = Logger
			.getLogger(CSSStyleSheetBuilder.class.getName());

	private static final boolean DEBUG = false;

	private static final int MAX_DEPTH = 10;

	private static final short NONE = 1;

	private static final short IN_SELECTOR = 2;

	private static final short IN_PAGE = 3;

	private static final short IN_FONT_FACE = 4;

	private final UserAgent ua;

	private final DeclarationBuilder declBuilder;

	/** スタイルシートのソースのスタック。 */
	private final List sourceStack = new ArrayList();

	/** スタイルシートのURIのスタック。 */
	private final List uriStack = new ArrayList();

	/** ブロックを無視するためのスタック。 */
	private final List mediaStack = new ArrayList();

	/** 状態。 */
	private short state = NONE;

	private CSSStyleSheet cssStyleSheet;

	public CSSStyleSheetBuilder(UserAgent ua) {
		this.ua = ua;
		this.declBuilder = new DeclarationBuilder(ua);
		this.declBuilder.setPropertySet(ElementPropertySet.getInstance());
	}

	public void setCSSStyleSheet(CSSStyleSheet cssStyleSheet) {
		this.cssStyleSheet = cssStyleSheet;
	}

	public CSSStyleSheet getCSSStyleSheet() {
		return this.cssStyleSheet;
	}

	public void comment(String text) throws CSSException {
		// ignore
	}

	public void ignorableAtRule(String atRule) throws CSSException {
		// ignore
	}

	public void namespaceDeclaration(String prefix, String uri)
			throws CSSException {
		// ignore
	}

	public void property(String name, LexicalUnit lu, boolean important)
			throws CSSException {
		if (this.state == NONE) {
			// 無視
			return;
		}
		this.declBuilder.property(name, lu, important);
	}

	public void startDocument(InputSource source) throws CSSException {
		URI uri = URI.create(source.getURI());
		this.sourceStack.add(source);
		this.uriStack.add(uri);
		this.declBuilder.setURI(uri);
		if (DEBUG) {
			System.out.println(uri);
		}
	}

	public void endDocument(InputSource source) throws CSSException {
		this.sourceStack.remove(this.sourceStack.size() - 1);
		this.uriStack.remove(this.uriStack.size() - 1);
		if (!this.uriStack.isEmpty()) {
			URI uri = (URI) this.uriStack.get(this.uriStack.size() - 1);
			this.declBuilder.setURI(uri);
			if (DEBUG) {
				System.out.println(uri);
			}
		}
	}

	protected InputSource getInputSource() {
		return (InputSource) this.sourceStack.get(this.sourceStack.size() - 1);
	}

	public void importStyle(String href, SACMediaList media,
			String defaultNamespaceURI) throws CSSException {
		if (DEBUG) {
			System.out.println("import:" + href);
		}
		StringBuffer buff = null;
		for (int i = 0; i < media.getLength(); ++i) {
			String medium = media.item(i);
			if (buff == null) {
				buff = new StringBuffer(medium);
			} else {
				buff.append(' ');
				buff.append(medium);
			}
		}
		// System.out.println(href+"/"+media);
		String mediaTypes = buff.toString();
		if (this.ua.is(mediaTypes)) {
			if (this.sourceStack.size() > MAX_DEPTH) {
				URI uri = (URI) this.uriStack.get(this.uriStack.size() - 1);
				this.ua.message(MessageCodes.WARN_DEEP_IMPORT, uri.toString(),
						String.valueOf(MAX_DEPTH));
				return;
			}
			URI baseURI = this.declBuilder.getURI();
			URI uri;
			try {
				uri = URIHelper.resolve(this.ua.getDocumentContext()
						.getEncoding(), baseURI, href);
			} catch (URISyntaxException e) {
				this.ua.message(MessageCodes.WARN_MISSING_CSS_STYLESHEET, href);
				return;
			}
			for (int i = 0; i < this.uriStack.size(); ++i) {
				URI stackURI = (URI) this.uriStack.get(i);
				if (stackURI.equals(uri)) {
					this.ua.message(MessageCodes.WARN_LOOP_IMPORT,
							baseURI.toString(), uri.toString());
					return;
				}
			}
			final Parser parser = new Parser();
			final boolean legacy = this.ua.getDocumentContext()
					.getCompatibleMode() >= DocumentContext.CM_MSIE;
			parser.setLegacy(legacy);
			parser.setDocumentHandler(this);
			try {
				Source source = this.ua.resolve(uri);
				try {
					InputSource inputSource = XMLUtils.toSACInputSource(source,
							this.getInputSource().getEncoding(), mediaTypes,
							null);
					parser.setDefaultCharset(this.ua.getDocumentContext()
							.getEncoding());
					parser.parseStyleSheet(inputSource);
				} finally {
					this.ua.release(source);
				}
			} catch (CSSException e) {
				this.ua.message(MessageCodes.WARN_BAD_CSS_SYNTAX,
						uri.toString(), e.getMessage());
				LOG.log(Level.FINE, "CSS文法エラー", e);
			} catch (IOException e) {
				this.ua.message(MessageCodes.WARN_MISSING_CSS_STYLESHEET,
						uri.toString());
				LOG.log(Level.FINE, "CSS読み込みエラー", e);
			}
		}
	}

	public void startMedia(SACMediaList media) throws CSSException {
		// System.out.println(media);
		for (int i = 0; i < media.getLength(); ++i) {
			String medium = media.item(i).toLowerCase();
			if (this.ua.is(medium)) {
				this.mediaStack.add(Boolean.TRUE);
				return;
			}
		}
		this.mediaStack.add(Boolean.FALSE);
	}

	public void endMedia(SACMediaList media) throws CSSException {
		// System.out.println("/"+media);
		this.mediaStack.remove(this.mediaStack.size() - 1);
	}

	protected boolean inProperMedia() {
		if (this.mediaStack.isEmpty()) {
			return true;
		}
		return ((Boolean) this.mediaStack.get(this.mediaStack.size() - 1))
				.booleanValue();
	}

	public void startPage(String name, String pseudoPage) throws CSSException {
		if (name != null) {
			URI uri = (URI) this.uriStack.get(this.uriStack.size() - 1);
			this.ua.message(MessageCodes.WARN_BAD_CSS_SYNTAX, uri.toString(),
					"名前つきページはサポートしていません");
			return;
		}
		if (this.inProperMedia()) {
			if (this.state != NONE) {
				URI uri = (URI) this.uriStack.get(this.uriStack.size() - 1);
				this.ua.message(MessageCodes.WARN_BAD_CSS_SYNTAX,
						uri.toString(), "@pageルールがネストされています。");
			}
			this.declBuilder.setDeclaration(null);
			this.declBuilder.setPropertySet(PagePropertySet.getInstance());
			this.state = IN_PAGE;
		}
	}

	public void endPage(String name, String pseudoPage) throws CSSException {
		if (name != null) {
			return;
		}
		if (this.inProperMedia()) {
			this.cssStyleSheet.addPage(pseudoPage,
					this.declBuilder.getDeclaration());
			this.state = NONE;
			this.declBuilder.setDeclaration(null);
			this.declBuilder.setPropertySet(ElementPropertySet.getInstance());
		}
	}

	public void startFontFace() throws CSSException {
		if (this.state != NONE) {
			URI uri = (URI) this.uriStack.get(this.uriStack.size() - 1);
			this.ua.message(MessageCodes.WARN_BAD_CSS_SYNTAX, uri.toString(),
					"@font-faceルールがネストされています。");
		}
		this.state = IN_FONT_FACE;
		this.declBuilder.setDeclaration(null);
		this.declBuilder.setPropertySet(FontFacePropertySet.getInstance());
	}

	public void endFontFace() throws CSSException {
		Declaration decl = this.declBuilder.getDeclaration();
		CSSStyle style = CSSStyle.getCSSStyle(this.ua, null, null);
		decl.applyProperties(style);
		URI[] uris = Src.get(style);
		if (uris != null) {
			boolean missing = true;
			for (int i = 0; i < uris.length; ++i) {
				URI uri = uris[i];
				try {
					Source src = null;
					try {
						FontFace face;
						src = this.ua.resolve(uri);
						if (!src.exists()) {
							continue;
						}
						face = new FontFace();
						face.src = src;

						face.fontFamily = CSSFontFamily.get(style);
						face.fontWeight = FontWeight.get(style);
						face.fontStyle = CSSFontStyle.get(style);
						face.unicodeRange = CSSUnicodeRange.get(style);
						FontManager fm = this.ua.getFontManager();
						fm.addFontFace(face);
						missing = false;
						break;
					} finally {
						if (src != null) {
							this.ua.release(src);
						}
					}
				} catch (Exception e) {
					// ignore
				}
			}
			if (missing) {
				this.ua.message(MessageCodes.WARN_MISSING_FONT_FILE,
						uris.toString());
			}
		}

		this.state = NONE;
		this.declBuilder.setDeclaration(null);
		this.declBuilder.setPropertySet(ElementPropertySet.getInstance());
	}

	public void startSelector(SelectorList selectors) throws CSSException {
		if (DEBUG) {
			System.out.println(selectors);
		}
		if (this.inProperMedia()) {
			// 宣言の構築を準備する
			if (this.state != NONE) {
				URI uri = (URI) this.uriStack.get(this.uriStack.size() - 1);
				this.ua.message(MessageCodes.WARN_BAD_CSS_SYNTAX,
						uri.toString(), "セレクタがネストされています。");
			}
			this.declBuilder.setDeclaration(null);
			this.state = IN_SELECTOR;
		}
	}

	public void endSelector(SelectorList selectors) throws CSSException {
		if (this.inProperMedia()) {
			this.cssStyleSheet.addRule(selectors,
					this.declBuilder.getDeclaration());
			// 宣言の構築を終了した
			this.state = NONE;
		}
	}
}
