package jp.cssj.cr.epub;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.List;

import jp.cssj.android.BookActivity;
import jp.cssj.cr.gc.AndroidUserAgent;
import jp.cssj.cr.gc.RecorderGC;
import jp.cssj.cr.gc.TranscoderHandler;
import jp.cssj.homare.ua.props.UAProps;
import jp.cssj.homare.xml.Parser;
import jp.cssj.homare.xml.XMLHandler;
import jp.cssj.homare.xml.XMLHandlerFilter;
import jp.cssj.homare.xml.XMLHandlerWrapper;
import jp.cssj.homare.xml.jaxp.JAXPXMLParser;
import jp.cssj.homare.xml.xhtml.XHTMLNSFilter;
import jp.cssj.homare.xml.xhtml.XHTMLPreprocessFilter;
import jp.cssj.print.epub.util.XHTMLHandler;
import jp.cssj.resolver.Source;
import jp.cssj.resolver.SourceResolver;
import jp.cssj.resolver.cache.CachedSourceResolver;
import jp.cssj.resolver.composite.CompositeSourceResolver;
import jp.cssj.resolver.helpers.MetaSourceImpl;
import jp.cssj.sakae.gc.GC;

public abstract class CopperThread extends Thread {
	public AndroidUserAgent ua;
	private final Cache cache;
	private final List<RecorderGC> pages;
	private final BookActivity parent;

	public CopperThread(BookActivity parent, Cache cache, List<RecorderGC> pages) {
		this.parent = parent;
		this.cache = cache;
		this.pages = pages;
	}

	protected abstract void newPage(int page);

	protected abstract void done();

	public void run() {
		try {
			this.ua = new AndroidUserAgent(this.parent.getFontManager(),
					this.cache, this.pages) {
				@Override
				public void closePage(GC gc) throws IOException {
					super.closePage(gc);
					newPage(this.pageCount);
				}
			};

			CompositeSourceResolver resolver = new CompositeSourceResolver();
			SourceResolver epub = new ArchiveFileSourceResolver(
					this.parent.archive);
			File tmpDir = new File(this.parent.getCacheDir(), "tmp");
			tmpDir.mkdirs();
			CachedSourceResolver cached = new CachedSourceResolver(tmpDir);
			resolver.setDefaultSourceResolver(epub);
			resolver.addSourceResolver("cached", cached);
			String path = this.parent.item.fullPath;
			URI uri = URI.create(path);

			this.sendFile(cached, "styles/main.css", "main.css");
			this.sendFile(cached, "styles/common.css", "common.css");
			this.sendFile(cached, "styles/columns-"
					+ this.parent.setting.columns + ".css", "columns.css");
			switch (parent.setting.direction) {
			case 0:
				this.sendFile(cached, "styles/direction.css", "direction.css");
				break;
			case 1:
				this.sendFile(cached, "styles/direction-h.css", "direction.css");
				break;
			case 2:
				this.sendFile(cached, "styles/direction-v.css", "direction.css");
				break;
			}
			if (parent.setting.reverse) {
				this.sendFile(cached, "styles/reverse.css", "reverse.css");
			}

			this.ua.setProperty(UAProps.INPUT_DEFAULT_STYLESHEET.name,
					"cached:main.css");
			this.ua.setProperty(UAProps.OUTPUT_TEXT_SIZE.name,
					String.valueOf((double) parent.setting.fontSize / 100.0));

			float paperWidth = this.parent.setting.width;
			float paperHeight = this.parent.setting.height;
			float width = paperWidth - this.parent.setting.margin / 100f * 2f;
			float height = paperHeight - this.parent.setting.margin / 100f * 2f;
			this.ua.setProperty(UAProps.OUTPUT_PAGE_WIDTH.name, width + "mm");
			this.ua.setProperty(UAProps.OUTPUT_PAGE_HEIGHT.name, height + "mm");
			this.ua.setSourceResolver(resolver);
			Source source = resolver.resolve(uri);
			this.ua.getDocumentContext().setBaseURI(source.getURI());

			XMLHandler handler = new TranscoderHandler(ua);
			Parser parser = new JAXPXMLParser();

			XMLHandlerFilter filter = new XHTMLPreprocessFilter(ua);
			filter.setXMLHandler(handler);

			XHTMLNSFilter nshandler = new XHTMLNSFilter();
			nshandler.setXMLHandler(filter);

			XHTMLHandler xhandler = new XHTMLHandler(nshandler,
					this.parent.item, parent.setting.direction == 2);

			// System.err.println("*** parse");
			parser.parse(this.ua, source, new XMLHandlerWrapper(xhandler, null));
			this.done();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			this.pages.clear();
		} finally {
			try {
				this.ua.finish();
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				this.pages.clear();
			}
		}
	}

	protected void sendFile(CachedSourceResolver cached, String asset,
			String uri) throws IOException {
		File file = cached.putFile(new MetaSourceImpl(URI.create("cached:"
				+ uri)));
		OutputStream out = new FileOutputStream(file);
		try {
			InputStream in = this.parent.getAssets().open(asset);
			try {
				byte[] buff = new byte[1024];
				for (int len = in.read(buff); len != -1; len = in.read(buff)) {
					out.write(buff, 0, len);
				}
			} finally {
				in.close();
			}
		} finally {
			out.close();
		}
	}
}
