/*

 Copyright 2001  The Apache Software Foundation 

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 */
package net.zamasoft.font.table;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

import net.zamasoft.font.util.RandomAccessFileInputStream;

/**
 * @version $Id: CmapTable.java 646 2011-08-28 12:55:08Z miyabe $
 * @author <a href="mailto:david@steadystate.co.uk">David Schweinsberg</a>
 */
public class CmapTable implements Table {

	private final int numTables;

	private final CmapIndexEntry[] entries;

	private final CmapFormat[] formats;

	protected CmapTable(DirectoryEntry de, RandomAccessFile raf)
			throws IOException {
		synchronized (raf) {
			raf.seek(de.getOffset());
			long fp = raf.getFilePointer();
			DataInputStream in = new DataInputStream(new BufferedInputStream(
					new RandomAccessFileInputStream(raf)));
			try {
				in.readUnsignedShort(); // version
				this.numTables = in.readUnsignedShort();
				this.entries = new CmapIndexEntry[this.numTables];
				this.formats = new CmapFormat[this.numTables];

				// Get each of the index entries
				for (int i = 0; i < this.numTables; i++) {
					this.entries[i] = new CmapIndexEntry(in);
				}
			} finally {
				in.close();
			}

			// Get each of the tables
			for (int i = 0; i < this.numTables; i++) {
				raf.seek(fp + this.entries[i].getOffset());
				in = new DataInputStream(new BufferedInputStream(
						new RandomAccessFileInputStream(raf)));
				try {
					int format = in.readUnsignedShort();
					this.formats[i] = CmapFormat.create(format, in);
				} finally {
					in.close();
				}
			}
		}
	}

	public CmapFormat getCmapFormat(short platformId, short encodingId) {
		// Find the requested format
		for (int i = 0; i < this.numTables; i++) {
			if (this.entries[i].getPlatformId() == platformId
					&& this.entries[i].getEncodingId() == encodingId) {
				return this.formats[i];
			}
		}
		return null;
	}

	public int getType() {
		return cmap;
	}

	public String toString() {
		StringBuffer sb = new StringBuffer().append("cmap\n");

		// Get each of the index entries
		for (int i = 0; i < this.numTables; i++) {
			sb.append("\t").append(this.entries[i]).append("\n");
		}

		// Get each of the tables
		for (int i = 0; i < this.numTables; i++) {
			sb.append("\t").append(this.formats[i]).append("\n");
		}
		return sb.toString();
	}
}
