package jp.cssj.print.epub;

import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipFile;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import jp.cssj.print.epub.Container.Rootfile;
import jp.cssj.print.epub.Contents.Item;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * EPUBファイルを読み込みます。
 * 
 * @author <a href="mailto:miyabe at gnn.co.jp">MIYABE Tatsuhiko</a>
 */
public class EPubFile {
	private static final Logger LOG = Logger
			.getLogger(EPubFile.class.getName());

	protected ArchiveFile archive;
	private SAXParserFactory pf = SAXParserFactory.newInstance();
	{
		this.pf.setValidating(false);
		setFeature(this.pf,
				"http://xml.org/sax/features/external-general-entities", false);
		setFeature(this.pf,
				"http://xml.org/sax/features/external-parameter-entities",
				false);
		setFeature(this.pf, "http://xml.org/sax/features/namespaces", true);
		setFeature(this.pf, "http://xml.org/sax/features/validation", false);
	}

	/**
	 * 与えられたEPUBファイルを読み込むためのインスタンスを作ります。
	 * 
	 * @param zip
	 *            EPUBファイル。
	 */
	public EPubFile(ArchiveFile archive) {
		this.archive = archive;
	}

	public EPubFile(ZipFile zip) {
		this(new ZipArchiveFile(zip));
	}

	private static void setFeature(SAXParserFactory pf, String key, boolean b) {
		try {
			if (pf.getFeature(key) == b) {
				return;
			}
			pf.setFeature(key, b);
		} catch (Exception e) {
			LOG.log(Level.FINE, "サポートされない機能です", e);
		}
	}

	/**
	 * META-INF/container.xmlを読み込みます。
	 * 
	 * @return　META-INF/container.xmlの情報。
	 * @throws FileNotFoundException
	 *             META-INF/container.xmlが存在しない場合。
	 * @throws IOException
	 *             ファイルの読み込みエラーがあった場合。
	 * @throws SAXException
	 *             ファイルの形式に問題があった場合。
	 */
	public Container readContainer() throws FileNotFoundException, IOException,
			SAXException {
		InputStream in = archive.getInputStream("META-INF/container.xml");
		try {
			SAXParser parser = this.pf.newSAXParser();
			ContainerHandler handler = new ContainerHandler();
			parser.parse(new InputSource(in), handler);
			return handler.getContainer();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		} finally {
			in.close();
		}
	}

	/**
	 * OPFを読み込みます。
	 * 
	 * @param root
	 *            mimeTypeが"application/oebps-package+xml"のルートファイル。
	 * @return 解析済みOPF。
	 * @throws IOException
	 *             ファイルの読み込みエラーがあった場合。
	 * @throws SAXException
	 *             ファイルの形式に問題があった場合。
	 */
	public Contents readContents(Rootfile root) throws IOException,
			SAXException {
		InputStream in = new BufferedInputStream(
				this.archive.getInputStream(root.fullPath));
		try {
			SAXParser parser = this.pf.newSAXParser();
			OPFHandler handler = new OPFHandler(root.fullPath);
			parser.parse(new InputSource(in), handler);
			// TODO readTitle が異常に遅い
			// for (Item item : handler.items) {
			// if (item.title == null) {
			// item.title = this.readTitle(item);
			// }
			// }
			return handler.getContents();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		} finally {
			in.close();
		}
	}

	/**
	 * NCX形式の目次を取得します。
	 * 
	 * @param contents
	 *            解析済みOPF。
	 * @return 解析済みNCX。
	 * @throws IOException
	 *             ファイルの読み込みエラーがあった場合。
	 * @throws SAXException
	 *             ファイルの形式に問題があった場合。
	 */
	public Toc readToc(Contents contents) throws IOException, SAXException {
		if (contents.toc == null) {
			// spine@tocがない場合、ncxを検索する。
			// 不正なOPFに対応するための仕様外の動作です
			for (int i = 0; i < contents.items.length; ++i) {
				if ("application/x-dtbncx+xml"
						.equals(contents.items[i].mediaType)) {
					contents.toc = contents.items[i];
					break;
				}
			}
		}
		final InputStream in = this.archive
				.getInputStream(contents.toc.fullPath);
		try {
			final SAXParser parser = this.pf.newSAXParser();
			final NCXHandler handler = new NCXHandler(contents);
			parser.parse(new InputSource(in), handler);
			return handler.getToc();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		} finally {
			in.close();
		}
	}

	/**
	 * XHTMLのタイトルを取得します。
	 * 
	 * @param item
	 *            XHTMLファイルのアイテム。
	 * @return ファイルにタイトルがある場合はタイトルを返す。タイトルがない場合、ファイル形式にエラーがある場合はnull。
	 * @throws IOException
	 *             ファイルの読み込みエラーがあった場合。
	 */
	public String readTitle(Item item) throws IOException {
		if (!item.mediaType.equals("application/xhtml+xml")) {
			return null;
		}
		if (!this.archive.exists(item.fullPath)) {
			return null;
		}
		XHTMLTitleHandler handler = new XHTMLTitleHandler();
		InputStream in = new BufferedInputStream(
				this.archive.getInputStream(item.fullPath));
		try {
			SAXParser parser = this.pf.newSAXParser();
			parser.parse(new InputSource(in), handler);
		} catch (Exception e) {
			// ignore
		} finally {
			in.close();
		}
		return handler.getTitle();
	}
}
