package jp.cssj.sakae.pdf.gc;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.sakae.font.BBox;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.GroupImageGC;
import jp.cssj.sakae.gc.font.FontManager;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.image.Image;
import jp.cssj.sakae.gc.paint.CMYKColor;
import jp.cssj.sakae.gc.paint.Color;
import jp.cssj.sakae.gc.paint.GrayColor;
import jp.cssj.sakae.gc.paint.LinearGradient;
import jp.cssj.sakae.gc.paint.Paint;
import jp.cssj.sakae.gc.paint.Pattern;
import jp.cssj.sakae.gc.paint.RadialGradient;
import jp.cssj.sakae.gc.text.Text;
import jp.cssj.sakae.pdf.PdfGraphicsOutput;
import jp.cssj.sakae.pdf.PdfGroupImage;
import jp.cssj.sakae.pdf.PdfNamedGraphicsOutput;
import jp.cssj.sakae.pdf.PdfNamedOutput;
import jp.cssj.sakae.pdf.PdfOutput;
import jp.cssj.sakae.pdf.PdfWriter;
import jp.cssj.sakae.pdf.font.FontMetricsImpl;
import jp.cssj.sakae.pdf.font.PdfFont;
import jp.cssj.sakae.pdf.font.PdfFontSource;
import jp.cssj.sakae.pdf.params.PdfParams;
import jp.cssj.sakae.util.ColorUtils;

/* PDF命令早見表
 * 
 * w	line width
 * J	line cap
 * j	line join
 * M	miter limit
 * d	line dash pattern
 * ri	rendering intents
 * i	flatness tolerance
 * gs	special graphics state
 * 
 * q	save graphics state
 * Q	restore	graphics state
 * cm	current transformation matrics
 * 
 * m	moveTo
 * l	lineTo
 * c	curveTo (1,2,3)
 * v	curveTo (2,3)
 * y	curveTo (1,3)
 * h	closePath
 * re	rectangle
 * 
 * S	stroke
 * s	[h S]
 * f	fill Bonzero Winding Number Rule
 * F	[f]
 * f*	fill Even-Odd Rule
 * B	[f S]
 * B*	[f* S]
 * b	[h B]
 * b*	[h B*]
 * n	nop
 * 
 * W	clip Bonzero Winding Number Rule
 * W*	clip Even-Odd Rule
 * 
 * BT
 * ET
 * 
 * Tc
 * Tw
 * Tz
 * TL
 * Tf
 * Tr
 * Ts
 * 
 * Td
 * TD
 * Tm
 * T*
 * 
 * Tj
 * TJ
 * '
 * "
 * 
 * d0
 * d1
 * 
 * CS	stroke color space
 * cs	nonstroke color space
 * SC	stroke color
 * SCN
 * sc
 * scn
 * G
 * g
 * RG
 * rg
 * K
 * k
 * 
 * sh
 * 
 * BI
 * ID
 * EI
 * 
 * Do	draw object
 * 
 * MP
 * DP
 * BMC
 * BDC
 * EMC
 * 
 * BX
 * EX
 * 
 * sh	shading pattern
 */

/**
 * PDFグラフィックコンテキストです。
 * 
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: PdfGC.java 759 2011-11-13 14:06:17Z miyabe $
 */
public class PdfGC implements GC {
	private static final Logger LOG = Logger.getLogger(PdfGC.class.getName());

	private static final boolean DEBUG = false;

	protected final PdfGraphicsOutput out;

	/**
	 * 設定されたグラフィック状態を保存するためのオブジェクト。
	 * 
	 * @author <a href="tatsuhiko at miya dot be">miyabe</a>
	 * @version $Id: PdfGC.java 759 2011-11-13 14:06:17Z miyabe $
	 */
	static class GraphicsState {
		public XGraphicsState gstate = null;

		public final double lineWidth;

		public final short lineCap;

		public final short lineJoin;

		public final double[] linePattern;

		public final Paint strokePaint;

		public final Paint fillPaint;

		public final float composite;

		public final AffineTransform actualTransform;

		public GraphicsState(PdfGC gc) {
			this.lineWidth = gc.lineWidth;
			this.lineCap = gc.lineCap;
			this.lineJoin = gc.lineJoin;
			this.linePattern = gc.linePattern;
			this.strokePaint = gc.strokePaint;
			this.fillPaint = gc.fillPaint;
			this.composite = gc.opacity;
			this.actualTransform = gc.actualTransform;
			if (this.actualTransform != null) {
				gc.actualTransform = new AffineTransform(this.actualTransform);
			}
		}

		public void restore(PdfGC gc) {
			gc.lineWidth = this.lineWidth;
			gc.lineCap = this.lineCap;
			gc.lineJoin = this.lineJoin;
			gc.linePattern = this.linePattern;
			gc.strokePaint = this.strokePaint;
			gc.fillPaint = this.fillPaint;
			gc.opacity = this.composite;
			gc.actualTransform = this.actualTransform;
		}
	}

	/**
	 * PDFのカレントのグラフィック状態を保存するためのオブジェクト。
	 * 
	 * @author <a href="tatsuhiko at miya dot be">miyabe</a>
	 * @version $Id: PdfGC.java 759 2011-11-13 14:06:17Z miyabe $
	 */
	static class XGraphicsState {
		public final double lineWidth;

		public final short lineCap;

		public final short lineJoin;

		public final double[] linePattern;

		public final Paint strokePaint;

		public final Paint fillPaint;

		public final float opacity;

		public final double letterSpacing;

		public XGraphicsState(PdfGC gc) {
			this.lineWidth = gc.xlineWidth;
			this.lineCap = gc.xlineCap;
			this.lineJoin = gc.xlineJoin;
			this.linePattern = gc.xlinePattern;
			this.strokePaint = gc.xstrokePaint;
			this.fillPaint = gc.xfillPaint;
			this.opacity = gc.xopacity;
			this.letterSpacing = gc.xletterSpacing;
		}

		public void restore(PdfGC gc) {
			gc.xlineWidth = this.lineWidth;
			gc.xlineCap = this.lineCap;
			gc.xlineJoin = this.lineJoin;
			gc.xlinePattern = this.linePattern;
			gc.xstrokePaint = this.strokePaint;
			gc.xfillPaint = this.fillPaint;
			gc.xopacity = this.opacity;
			gc.xletterSpacing = this.letterSpacing;
		}
	}

	private List<GraphicsState> stack = new ArrayList<GraphicsState>();

	private AffineTransform transform = null;

	private AffineTransform actualTransform = null;

	private Shape clip = null;

	/**
	 * 使用する線の末端の形状。
	 */
	private short lineCap = LINE_CAP_SQUARE;

	/**
	 * PDFのカレントの線の末端の形状。
	 */
	private short xlineCap = LINE_CAP_SQUARE;

	/**
	 * 使用する線の接続部分の形状。
	 */
	private short lineJoin = LINE_JOIN_MITER;

	/**
	 * PDFのカレントの線の接続部分の形状。
	 */
	private short xlineJoin = LINE_JOIN_MITER;

	/**
	 * 使用する線の幅。
	 */
	private double lineWidth = 1;

	/**
	 * PDFのカレントの線の幅。
	 */
	private double xlineWidth = 1;

	/**
	 * 使用する線のパターン。
	 */
	private double[] linePattern = STROKE_SOLID;

	/**
	 * PDFのカレントの線のパターン。
	 */
	private double[] xlinePattern = STROKE_SOLID;

	/**
	 * 使用する線の色。
	 */
	private Paint strokePaint = GrayColor.BLACK;

	/**
	 * PDFのカレントの線の色。
	 */
	private Paint xstrokePaint = GrayColor.BLACK;

	/**
	 * 使用する塗りつぶし色。
	 */
	private Paint fillPaint = GrayColor.BLACK;

	/**
	 * PDFのカレントの塗りつぶし色。
	 */
	private Paint xfillPaint = GrayColor.BLACK;

	private float opacity = 1;

	private float xopacity = 1;

	private double xletterSpacing = 0;

	private static class PatternKey {
		final double pageWidth;

		final double pageHeight;

		final Image image;

		final AffineTransform at;

		PatternKey(double pageWidth, double pageHeight, Image image,
				AffineTransform at) {
			this.pageWidth = pageWidth;
			this.pageHeight = pageHeight;
			this.image = image;
			this.at = at;
		}

		public boolean equals(Object o) {
			if (o instanceof PatternKey) {
				PatternKey key = (PatternKey) o;
				return key.pageWidth == this.pageWidth
						&& key.pageHeight == this.pageHeight
						&& key.image.equals(this.image)
						&& key.at.equals(this.at);
			}
			return false;
		}

		public int hashCode() {
			int hash = 1;
			long a = Double.doubleToLongBits(this.pageWidth);
			long b = Double.doubleToLongBits(this.pageHeight);
			hash = hash * 31 + (int) (a ^ (a >>> 32));
			hash = hash * 31 + (int) (b ^ (b >>> 32));
			hash = hash * 31 + this.image.hashCode();
			if (this.at != null) {
				hash = hash * 31 + this.at.hashCode();
			}
			return hash;
		}
	}

	private final Map<PatternKey, Object> patterns;

	private int qDepth = 0;

	private final int pdfVersion;

	private PdfGC(PdfGraphicsOutput out, Map<PatternKey, Object> patterns) {
		this.out = out;
		this.patterns = patterns;
		this.pdfVersion = this.out.getPdfWriter().getParams().getVersion();
	}

	public PdfGC(PdfGraphicsOutput out) {
		this(out, new HashMap<PatternKey, Object>());
	}

	public FontManager getFontManager() {
		return this.getPdfWriter().getFontManager();
	}

	public PdfGraphicsOutput getPDFGraphicsOutput() {
		return this.out;
	}

	public PdfWriter getPdfWriter() {
		return this.out.getPdfWriter();
	}

	public void begin() throws GraphicsException {
		if (DEBUG) {
			System.err.println("begin");
		}
		try {
			this.applyTransform();
			this.applyClip();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		this.stack.add(new GraphicsState(this));
	}

	public void end() throws GraphicsException {
		if (DEBUG) {
			System.err.println("end");
		}
		try {
			this.grestore();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		GraphicsState state = (GraphicsState) this.stack.remove(this.stack
				.size() - 1);
		state.restore(this);
		if (this.stack.isEmpty()) {
			this.transform = null;
		}
		this.clip = null;
	}

	public void resetState() throws GraphicsException {
		if (DEBUG) {
			System.err.println("reset");
		}
		try {
			this.grestore();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		GraphicsState state = (GraphicsState) this.stack
				.get(this.stack.size() - 1);
		state.restore(this);
		this.transform = null;
		this.clip = null;
	}

	public void setLineWidth(double lineWidth) {
		if (DEBUG) {
			System.err.println("setLineWidth: " + lineWidth);
		}
		this.lineWidth = lineWidth;
	}

	public double getLineWidth() {
		return this.lineWidth;
	}

	public void setLinePattern(double[] linePattern) {
		if (DEBUG) {
			System.err.println("setLinePattern: " + linePattern);
		}
		if (linePattern != null && linePattern.length > 0) {
			this.linePattern = linePattern;
		} else {
			this.linePattern = STROKE_SOLID;
		}
	}

	public double[] getLinePattern() {
		return this.linePattern;
	}

	public void setLineJoin(short lineJoin) {
		if (DEBUG) {
			System.err.println("setLineJoin: " + lineJoin);
		}
		this.lineJoin = lineJoin;
	}

	public short getLineJoin() {
		return this.lineJoin;
	}

	public void setLineCap(short lineCap) {
		if (DEBUG) {
			System.err.println("setLineCap: " + lineCap);
		}
		this.lineCap = lineCap;
	}

	public short getLineCap() {
		return this.lineCap;
	}

	public void setOpacity(float opacity) {
		if (DEBUG) {
			System.err.println("setOpacity: " + opacity);
		}
		this.opacity = opacity;
	}

	public float getOpacity() {
		return this.opacity;
	}

	public void setStrokePaint(Object paint) throws GraphicsException {
		if (DEBUG) {
			System.err.println("setStrokePaint: " + paint);
		}
		this.setPaint(paint, false);
	}

	public void setFillPaint(Object paint) throws GraphicsException {
		if (DEBUG) {
			System.err.println("setFillPaint: " + paint);
		}
		this.setPaint(paint, true);
	}

	protected void setPaint(Object paint, boolean fill)
			throws GraphicsException {
		if (fill) {
			this.fillPaint = (Paint) paint;
		} else {
			this.strokePaint = (Paint) paint;
		}
	}

	public void transform(AffineTransform at) throws GraphicsException {
		if (DEBUG) {
			System.err.println("transform: " + at);
		}
		if (at == null || at.isIdentity()) {
			return;
		}
		// assert at.getScaleX() != 0;
		// assert at.getScaleY() != 0;
		assert !Double.isNaN(at.getTranslateX());
		assert !Double.isNaN(at.getTranslateY());
		assert !Double.isNaN(at.getScaleX());
		assert !Double.isNaN(at.getScaleY());
		assert !Double.isNaN(at.getShearX());
		assert !Double.isNaN(at.getShearY());
		try {
			this.applyClip();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		if (this.transform == null) {
			this.transform = new AffineTransform(at);
		} else {
			this.transform.concatenate(at);
		}
		if (this.actualTransform == null) {
			this.actualTransform = new AffineTransform(at);
		} else {
			this.actualTransform.concatenate(at);
		}
	}

	public AffineTransform getTransform() {
		return this.actualTransform == null ? null : new AffineTransform(
				this.actualTransform);
	}

	public void clip(Shape clip) throws GraphicsException {
		if (DEBUG) {
			System.err.println("clip: "
					+ (clip == null ? clip : clip.getBounds2D()));
		}
		try {
			this.applyTransform();
			this.applyClip();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		this.clip = clip;
	}

	public Shape getClip() {
		return this.clip;
	}

	public void fill(Shape shape) throws GraphicsException {
		if (DEBUG) {
			System.err.println("fill: " + shape.getBounds2D());
		}
		try {
			this.applyStates();
			int winding;
			if (shape instanceof Rectangle2D) {
				Rectangle2D r = (Rectangle2D) shape;
				if (this.out.equals(r.getWidth(), 0.0)
						|| this.out.equals(r.getHeight(), 0.0)) {
					return;
				}
				winding = PathIterator.WIND_NON_ZERO;
				this.plotRect(r);
			} else {
				PathIterator i = shape.getPathIterator(null);
				winding = i.getWindingRule();
				this.plot(i);
			}

			switch (winding) {
			case PathIterator.WIND_NON_ZERO:
				this.out.writeOperator("f");
				break;
			case PathIterator.WIND_EVEN_ODD:
				this.out.writeOperator("f*");
				break;
			default:
				throw new IllegalStateException();
			}
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
	}

	public void draw(Shape shape) throws GraphicsException {
		if (DEBUG) {
			System.err.println("draw: " + shape.getBounds2D());
		}
		try {
			this.applyStates();
			boolean close;
			if (shape instanceof Rectangle2D) {
				Rectangle2D r = (Rectangle2D) shape;
				close = false;
				this.plotRect(r);
			} else {
				PathIterator i = shape.getPathIterator(null);
				close = this.plot(i);
			}

			this.out.writeOperator(close ? "s" : "S");
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
	}

	public void fillDraw(Shape shape) throws GraphicsException {
		if (DEBUG) {
			System.err.println("fillDraw: " + shape.getBounds2D());
		}
		try {
			this.applyStates();
			int winding;
			boolean close;
			if (shape instanceof Rectangle2D) {
				Rectangle2D r = (Rectangle2D) shape;
				winding = PathIterator.WIND_NON_ZERO;
				close = false;
				this.plotRect(r);
			} else {
				PathIterator i = shape.getPathIterator(null);
				winding = i.getWindingRule();
				close = this.plot(i);
			}

			switch (winding) {
			case PathIterator.WIND_NON_ZERO:
				this.out.writeOperator(close ? "b" : "B");
				break;
			case PathIterator.WIND_EVEN_ODD:
				this.out.writeOperator(close ? "b*" : "B*");
				break;
			default:
				throw new IllegalStateException();
			}
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
	}

	public void drawImage(Image image) throws GraphicsException {
		if (DEBUG) {
			System.err.println("drawImage: " + image);
		}
		try {
			this.applyStates();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		image.drawTo(this);
	}

	public void drawPDFImage(String name, double width, double height)
			throws GraphicsException {
		try {
			this.applyStates();
			this.begin();

			this.gsave();
			this.out.writeReal(width);
			this.out.writeReal(0);
			this.out.writeReal(0);
			this.out.writeReal(height);
			this.out.writePosition(0, height);
			this.out.writeOperator("cm");

			this.out.useResource("XObject", name);
			this.out.writeName(name);
			this.out.writeOperator("Do");

			this.end();
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
	}

	public void drawText(Text text, double x, double y)
			throws GraphicsException {
		if (DEBUG) {
			System.err.println("drawText: " + text);
		}
		if (text.getGLen() <= 0) {
			return;
		}
		assert text.getCLen() > 0;
		try {
			this.applyStates();

			FontMetricsImpl fm = (FontMetricsImpl) text.getFontMetrics();
			PdfFontSource source = (PdfFontSource) fm.getFontSource();
			if (this.pdfVersion == PdfParams.VERSION_PDFA1B) {
				byte type = source.getType();
				if (type != PdfFontSource.TYPE_EMBEDDED
						&& type != PdfFontSource.TYPE_MISSING) {
					throw new IllegalStateException(
							"PDF/A-1で埋め込みフォント以外は使用できません。");
				}
			}
			FontStyle fontStyle = text.getFontStyle();

			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("drawText: fontSource=" + source + " text=" + text);
			}

			boolean localContext = false;
			double size = fontStyle.getSize();
			double enlargement;
			short weight = fontStyle.getWeight();
			if (weight >= 500 && source.getWeight() < 500) {
				// 自前でBOLDを再現する
				switch (weight) {
				case 500:
					enlargement = size / 28.0;
					break;
				case 600:
					enlargement = size / 24.0;
					break;
				case 700:
					enlargement = size / 20.0;
					break;
				case 800:
					enlargement = size / 16.0;
					break;
				case 900:
					enlargement = size / 12.0;
					break;
				default:
					throw new IllegalStateException();
				}
				if (enlargement > 0) {
					// 自前でBOLDを再現する
					this.q();
					localContext = true;
					this.out.writeReal(enlargement);
					this.out.writeOperator("w");
					this.out.writeInt(2);
					this.out.writeOperator("Tr");
				}
			} else {
				enlargement = 0;
			}

			// 描画方向
			byte direction = fontStyle.getDirection();
			AffineTransform rotate = null;
			double center = 0;
			boolean verticalFont = false;
			switch (direction) {
			case FontStyle.DIRECTION_LTR:
			case FontStyle.DIRECTION_RTL:// TODO RTL
				// 横書き
				break;
			case FontStyle.DIRECTION_TB:
				// 縦書き
				if (source.getDirection() == direction) {
					// 縦組み
					verticalFont = true;
				} else {
					// ９０度回転横組み
					if (!localContext) {
						this.q();
						localContext = true;
					}
					rotate = AffineTransform.getRotateInstance(Math.PI / 2, x,
							y);
					this.out.writeTransform(rotate);
					this.out.writeOperator("cm");
					BBox bbox = source.getBBox();
					center = ((bbox.lly + bbox.ury) * size / FontSource.UNITS_PER_EM) / 2.0;
					y += center;
				}
				break;
			default:
				throw new IllegalStateException();
			}

			// テキスト開始
			this.out.writeOperator("BT");

			// イタリック
			short style = fontStyle.getStyle();
			if (style != FontStyle.FONT_STYLE_NORMAL && !source.isItalic()) {
				// 自前でイタリックを再現する
				if (verticalFont) {
					// 縦書きイタリック
					this.out.writeReal(1);
					this.out.writeReal(-0.25);
					this.out.writeReal(0);
					this.out.writeReal(1);
					this.out.writePosition(x, y);
					this.out.writeOperator("Tm");
				} else {
					// 横書きイタリック
					this.out.writeReal(1);
					this.out.writeReal(0);
					this.out.writeReal(0.25);
					this.out.writeReal(1);
					this.out.writePosition(x, y);
					this.out.writeOperator("Tm");
				}
			} else {
				this.out.writePosition(x, y);
				this.out.writeOperator("Td");
			}

			PdfFont font = (PdfFont) ((FontMetricsImpl) text.getFontMetrics())
					.getFont();

			// フォント名とサイズ
			String name = font.getName();
			this.out.useResource("Font", name);
			this.out.writeName(name);
			this.out.writeReal(size);
			this.out.writeOperator("Tf");

			// // 字間
			double letterSpacing = text.getLetterSpacing();
			// 縦書きでは負の値を使う(SPEC PDF1.3 8.7.1.1)
			if (verticalFont) {
				letterSpacing = -letterSpacing;
			}
			if (!this.out.equals(letterSpacing, this.xletterSpacing)) {
				this.out.writeReal(letterSpacing);
				this.out.writeOperator("Tc");
				if (!localContext) {
					this.xletterSpacing = letterSpacing;
				}
			}

			// 描画
			font.drawTo(this, text);

			// テキスト終了
			this.out.writeOperator("ET");

			if (enlargement > 0) {
				// Bold終了
				this.out.writeInt(0);
				this.out.writeOperator("Tr");
			}

			if (localContext) {
				this.Q();
			}
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
	}

	private static class PdfGroupImageGC extends PdfGC implements GroupImageGC {
		PdfGroupImageGC(PdfGroupImage image) {
			super(image);
		}

		public Image finish() throws GraphicsException {
			PdfGroupImage image = (PdfGroupImage) this.out;
			try {
				image.close();
			} catch (IOException e) {
				throw new GraphicsException(e);
			}
			return image;
		}
	}

	public GroupImageGC creatgeGroupImage(double width, double height) {
		try {
			PdfGroupImage image = this.getPdfWriter().createGroupImage(width,
					height);
			GroupImageGC gc = new PdfGroupImageGC(image);
			return gc;
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
	}

	protected void applyTransform() throws IOException {
		if (this.transform != null) {
			this.gsave();
			this.out.writeTransform(this.transform);
			this.out.writeOperator("cm");
			this.transform = null;
		}
	}

	protected void applyClip() throws IOException {
		if (this.clip != null) {
			this.gsave();
			int winding;
			if (this.clip instanceof Rectangle2D) {
				Rectangle2D r = (Rectangle2D) this.clip;
				winding = PathIterator.WIND_NON_ZERO;
				this.out.writeRect((double) r.getX(), (double) r.getY(),
						(double) r.getWidth(), (double) r.getHeight());
				this.out.writeOperator("re");
			} else {
				PathIterator i = this.clip.getPathIterator(null);
				winding = i.getWindingRule();
				this.plot(i);
			}

			switch (winding) {
			case PathIterator.WIND_NON_ZERO:
				this.out.writeOperator("W");
				break;
			case PathIterator.WIND_EVEN_ODD:
				this.out.writeOperator("W*");
				break;
			default:
				throw new IllegalStateException();
			}
			this.out.writeOperator("n");
			this.clip = null;
		}
	}

	private String getPaintName(Paint paint) throws GraphicsException {
		switch (paint.getPaintType()) {
		case Paint.PATTERN: {
			String name;
			Pattern pattern = (Pattern) paint;
			Image image = pattern.getImage();
			AffineTransform at = pattern.getTransform();

			PdfGraphicsOutput pout = (PdfGraphicsOutput) this.out;
			PatternKey key = new PatternKey(pout.getWidth(), pout.getHeight(),
					image, at);

			name = (String) this.patterns.get(key);
			if (name == null) {
				double width = image.getWidth();
				double height = image.getHeight();
				PdfNamedGraphicsOutput tout;
				try {
					tout = pout.getPdfWriter().createTilingPattern(width,
							height, pout.getHeight(), at);
					try {
						PdfGC pgc = new PdfGC(tout, null);
						image.drawTo(pgc);
					} finally {
						tout.close();
					}
					name = tout.getName();
				} catch (IOException e) {
					new GraphicsException(e);
				}
				this.patterns.put(key, name);
			}
			return name;
		}
		case Paint.LINEAR_GRADIENT: {
			// PDF Axial(Type 2) Shading
			if (this.out.getPdfWriter().getParams().getVersion() < PdfParams.VERSION_1_3) {
				return null;
			}
			LinearGradient gradient = (LinearGradient) paint;

			Color[] colors = gradient.getColors();
			double[] fractions = gradient.getFractions();

			AffineTransform at = this.getTransform();
			if (at == null) {
				at = gradient.getTransform();
			} else if (gradient.getTransform() != null) {
				at.concatenate(gradient.getTransform());
			}

			PdfGraphicsOutput pout = (PdfGraphicsOutput) this.out;
			try {
				PdfNamedOutput sout = pout.getPdfWriter().createShadingPattern(
						pout.getHeight(), at);
				try {
					sout.writeName("ShadingType");
					sout.writeInt(2);
					sout.lineBreak();

					sout.writeName("Coords");
					sout.startArray();
					sout.writeReal(gradient.getX1());
					sout.writeReal(gradient.getY1());
					sout.writeReal(gradient.getX2());
					sout.writeReal(gradient.getY2());
					sout.endArray();
					sout.lineBreak();
					this.shadingFunction(sout, colors, fractions);
				} finally {
					sout.close();
				}

				return sout.getName();
			} catch (IOException e) {
				throw new GraphicsException(e);
			}
		}
		case Paint.RADIAL_GRADIENT: {
			// PDF Radial(Type 3) Shading
			if (this.out.getPdfWriter().getParams().getVersion() < PdfParams.VERSION_1_3) {
				return null;
			}
			RadialGradient gp = (RadialGradient) paint;

			Color[] colors = gp.getColors();
			double[] fractions = gp.getFractions();
			double radius = gp.getRadius();

			AffineTransform at = this.getTransform();
			if (at == null) {
				at = gp.getTransform();
			} else if (gp.getTransform() != null) {
				at.concatenate(gp.getTransform());
			}

			double dx = gp.getFX() - gp.getCX();
			double dy = gp.getFY() - gp.getCY();
			double d = Math.sqrt(dx * dx + dy * dy);
			if (d > radius) {
				double scale = (radius * .9999) / d;
				dx = dx * scale;
				dy = dy * scale;
			}

			PdfGraphicsOutput pout = (PdfGraphicsOutput) this.out;
			try {
				PdfNamedOutput sout = pout.getPdfWriter().createShadingPattern(
						pout.getHeight(), at);
				try {
					sout.writeName("ShadingType");
					sout.writeInt(3);
					sout.lineBreak();

					sout.writeName("Coords");
					sout.startArray();
					sout.writeReal(gp.getCX() + dx);
					sout.writeReal(gp.getCY() + dy);
					sout.writeReal(0);
					sout.writeReal(gp.getCX());
					sout.writeReal(gp.getCY());
					sout.writeReal(radius);
					sout.endArray();
					sout.lineBreak();

					this.shadingFunction(sout, colors, fractions);
				} finally {
					sout.close();
				}
				return sout.getName();
			} catch (IOException e) {
				throw new GraphicsException(e);
			}
		}

		default:
			throw new IllegalStateException();
		}
	}

	protected void shadingFunction(PdfOutput sout, Color[] colors,
			double[] fractions) throws IOException {
		sout.writeName("ColorSpace");
		short colorType;
		if (this.getPdfWriter().getParams().getColorMode() == PdfParams.COLOR_MODE_GRAY) {
			colorType = Color.GRAY;
		} else {
			colorType = colors[0].getColorType();
			for (int i = 1; i < colors.length; ++i) {
				if (colorType != colors[i].getColorType()) {
					colorType = Color.RGB;
				}
			}
		}
		switch (colorType) {
		case Color.GRAY:
			sout.writeName("DeviceGray");
			break;
		case Color.RGB:
			sout.writeName("DeviceRGB");
			break;
		case Color.CMYK:
			sout.writeName("DeviceCMYK");
			break;
		default:
			throw new IllegalStateException();
		}
		sout.lineBreak();

		sout.writeName("Extend");
		sout.startArray();
		sout.writeBoolean(true);
		sout.writeBoolean(true);
		sout.endArray();
		sout.lineBreak();

		sout.writeName("Function");
		sout.startHash();
		if (colors.length <= 2
				&& (fractions == null || fractions.length == 0
						|| (fractions.length == 1 && fractions[0] == 0) || (fractions.length == 2
						&& fractions[0] == 0 && fractions[1] == 1))) {
			// 単純な場合
			Color c0 = colors[0];
			Color c1 = colors[1];

			sout.writeName("FunctionType");
			sout.writeInt(2);
			sout.lineBreak();

			sout.writeName("Domain");
			sout.startArray();
			sout.writeReal(0.0);
			sout.writeReal(1.0);
			sout.endArray();
			sout.lineBreak();

			sout.writeName("N");
			sout.writeReal(1.0);
			sout.lineBreak();

			sout.writeName("C0");
			sout.startArray();
			writeColor(sout, colorType, c0);
			sout.endArray();
			sout.lineBreak();

			sout.writeName("C1");
			sout.startArray();
			writeColor(sout, colorType, c1);
			sout.endArray();
			sout.lineBreak();
		} else {
			// 複雑な場合
			int segments = fractions.length - 1;
			if (fractions[0] != 0) {
				++segments;
			}
			if (fractions[fractions.length - 1] != 1) {
				++segments;
			}

			sout.writeName("FunctionType");
			sout.writeInt(3);
			sout.lineBreak();

			sout.writeName("Domain");
			sout.startArray();
			sout.writeReal(0.0);
			sout.writeReal(1.0);
			sout.endArray();
			sout.lineBreak();

			sout.writeName("Encode");
			sout.startArray();
			for (int i = 0; i < segments; ++i) {
				sout.writeReal(0.0);
				sout.writeReal(1.0);
			}
			sout.endArray();
			sout.lineBreak();

			sout.writeName("Bounds");
			sout.startArray();
			if (fractions[0] != 0) {
				sout.writeReal(fractions[0]);
			}
			for (int i = 1; i < fractions.length - 1; ++i) {
				sout.writeReal(fractions[i]);
			}
			if (fractions[fractions.length - 1] != 1) {
				sout.writeReal(fractions[fractions.length - 1]);
			}
			sout.endArray();
			sout.lineBreak();

			sout.writeName("Functions");
			sout.startArray();
			for (int i = -1; i < fractions.length; ++i) {
				Color c0, c1;
				if (i == -1) {
					if (fractions[0] != 0) {
						c0 = colors[0];
						c1 = colors[0];
					} else {
						continue;
					}
				} else if (i == fractions.length - 1) {
					if (fractions[i] != 1) {
						c0 = colors[i];
						c1 = colors[i];
					} else {
						break;
					}
				} else {
					c0 = colors[i];
					c1 = colors[i + 1];
				}

				sout.startHash();
				sout.writeName("FunctionType");
				sout.writeInt(2);
				sout.lineBreak();

				sout.writeName("Domain");
				sout.startArray();
				sout.writeReal(0.0);
				sout.writeReal(1.0);
				sout.endArray();
				sout.lineBreak();

				sout.writeName("N");
				sout.writeReal(1.0);
				sout.lineBreak();

				sout.writeName("C0");
				sout.startArray();
				writeColor(sout, colorType, c0);
				sout.endArray();
				sout.lineBreak();

				sout.writeName("C1");
				sout.startArray();
				writeColor(sout, colorType, c1);
				sout.endArray();
				sout.lineBreak();
				sout.endHash();
			}
			sout.endArray();
			sout.lineBreak();
		}
		sout.endHash();
		sout.lineBreak();
	}

	private static void writeColor(PdfOutput sout, short colorType, Color color)
			throws IOException {
		switch (colorType) {
		case Color.GRAY:
			if (color.getColorType() == Color.GRAY) {
				sout.writeReal(((GrayColor) color).getComponent(0));
				break;
			}
			sout.writeReal(ColorUtils.toGray(color.getRed(), color.getGreen(),
					color.getBlue()));
			break;
		case Color.RGB:
			sout.writeReal(color.getRed());
			sout.writeReal(color.getGreen());
			sout.writeReal(color.getBlue());
			break;
		case Color.CMYK:
			CMYKColor cmyk = (CMYKColor) color;
			sout.writeReal(cmyk.getComponent(CMYKColor.C));
			sout.writeReal(cmyk.getComponent(CMYKColor.M));
			sout.writeReal(cmyk.getComponent(CMYKColor.Y));
			sout.writeReal(cmyk.getComponent(CMYKColor.K));
			break;
		default:
			throw new IllegalStateException();
		}
	}

	protected void applyStates() throws IOException {
		// 変換
		this.applyTransform();
		this.applyClip();

		// ストローク
		if (this.lineWidth != this.xlineWidth) {
			this.xlineWidth = this.lineWidth;
			this.out.writeReal(this.lineWidth);
			this.out.writeOperator("w");
		}
		if (this.lineCap != this.xlineCap) {
			this.xlineCap = this.lineCap;
			this.out.writeInt(this.lineCap);
			this.out.writeOperator("J");
		}
		if (this.lineJoin != this.xlineJoin) {
			this.xlineCap = this.lineJoin;
			this.out.writeInt(this.lineJoin);
			this.out.writeOperator("j");
		}
		if (!Arrays.equals(this.linePattern, this.xlinePattern)) {
			this.xlinePattern = this.linePattern;
			this.out.startArray();
			if (this.linePattern != null) {
				for (int i = 0; i < this.linePattern.length; ++i) {
					assert this.linePattern[i] > 0;
					this.out.writeReal(this.linePattern[i]);
				}
			}
			this.out.endArray();
			this.out.writeInt(0);
			this.out.writeOperator("d");
		}

		// 塗り
		if (this.strokePaint != null
				&& !this.strokePaint.equals(this.xstrokePaint)) {
			switch (this.strokePaint.getPaintType()) {
			case Paint.COLOR:
				if (this.xstrokePaint != null
						&& this.xstrokePaint.getPaintType() != Paint.COLOR) {
					this.out.writeName("DeviceRGB");
					this.out.writeOperator("CS");
				}
				this.out.writeStrokeColor((Color) this.strokePaint);
				break;
			case Paint.PATTERN:
			case Paint.LINEAR_GRADIENT:
			case Paint.RADIAL_GRADIENT:
				String name = this.getPaintName(this.strokePaint);
				if (name != null) {
					this.out.writeName("Pattern");
					this.out.writeOperator("CS");
					this.out.useResource("Pattern", name);
					this.out.writeName(name);
					this.out.writeOperator("SCN");
				}
				break;
			default:
				throw new IllegalStateException();
			}
			this.xstrokePaint = this.strokePaint;
		}
		if (this.fillPaint != null && !this.fillPaint.equals(this.xfillPaint)) {
			switch (this.fillPaint.getPaintType()) {
			case Paint.COLOR:
				if (this.xfillPaint != null
						&& this.xfillPaint.getPaintType() != Paint.COLOR) {
					this.out.writeName("DeviceRGB");
					this.out.writeOperator("cs");
				}
				this.out.writeFillColor((Color) this.fillPaint);
				break;
			case Paint.PATTERN:
			case Paint.LINEAR_GRADIENT:
			case Paint.RADIAL_GRADIENT:
				String name = this.getPaintName(this.fillPaint);
				if (name != null) {
					this.out.writeName("Pattern");
					this.out.writeOperator("cs");
					this.out.useResource("Pattern", name);
					this.out.writeName(name);
					this.out.writeOperator("scn");
				}
				break;
			default:
				throw new IllegalStateException();
			}
			this.xfillPaint = this.fillPaint;
		}
		if (this.opacity != this.xopacity) {
			this.xopacity = this.opacity;
			this.applyComposite(this.opacity);
		}
	}

	private void applyComposite(float opacity) throws IOException {
		int pdfVersion = this.out.getPdfWriter().getParams().getVersion();
		if (pdfVersion < PdfParams.VERSION_1_4
				|| pdfVersion == PdfParams.VERSION_PDFA1B) {
			// 透明化処理がサポートされない場合。
			return;
		}
		Map<Float, String> alphaSgs = (Map<Float, String>) this.out
				.getPdfWriter().getAttribute("alphaSgs");
		if (alphaSgs == null) {
			alphaSgs = new HashMap<Float, String>();
			this.out.getPdfWriter().putAttribute("alphaSgs", alphaSgs);
		}
		String name = (String) alphaSgs.get(opacity);
		if (name == null) {
			PdfNamedOutput gsOut = this.out.getPdfWriter()
					.createSpecialGraphicsState();
			try {
				gsOut.writeName("CA");
				gsOut.writeReal(opacity);
				gsOut.writeName("ca");
				gsOut.writeReal(opacity);
			} finally {
				gsOut.close();
			}
			name = gsOut.getName();
			alphaSgs.put(opacity, name);
		}
		this.out.useResource("ExtGState", name);
		this.out.writeName(name);
		this.out.writeOperator("gs");
	}

	/**
	 * 現在のグラフィック状態が最初に適用された場合、 グラフィックコンテキスト開始命令(q)を出力し、現在のグラフィック状態を保存します。
	 * 
	 * @throws IOException
	 */
	private void gsave() throws IOException {
		if (this.stack.isEmpty()) {
			return;
		}
		GraphicsState state = (GraphicsState) this.stack
				.get(this.stack.size() - 1);
		if (state.gstate == null) {
			this.q();
			state.gstate = new XGraphicsState(this);
		}
	}

	/**
	 * 以前のグラフィック状態が保存されている場合、 グラフィックコンテキスト終了命令(Q)を出力し、現在のグラフィック状態を復帰します。
	 * 
	 * @throws IOException
	 */
	private void grestore() throws IOException {
		GraphicsState state = (GraphicsState) this.stack
				.get(this.stack.size() - 1);
		if (state.gstate != null) {
			this.Q();
			state.gstate.restore(this);
			state.gstate = null;
		}
	}

	private void q() throws IOException {
		++this.qDepth;
		if (this.pdfVersion == PdfParams.VERSION_PDFA1B) {
			if (this.qDepth > 28) {
				throw new IllegalStateException(
						"PDF/A-1ではグラフィックステートを28以上入れ子にできません。");
			}
		}
		this.out.writeOperator("q");
	}

	private void Q() throws IOException {
		--this.qDepth;
		this.out.writeOperator("Q");
	}

	protected void plotRect(Rectangle2D r) throws IOException {
		this.out.writeRect((double) r.getX(), (double) r.getY(),
				(double) r.getWidth(), (double) r.getHeight());
		this.out.writeOperator("re");
	}

	protected boolean plot(PathIterator i) throws IOException {
		double[] cord = new double[6];
		while (!i.isDone()) {
			int type = i.currentSegment(cord);
			double cx = 0f, cy = 0f;
			switch (type) {
			case PathIterator.SEG_MOVETO:
				this.out.writePosition(cx = cord[0], cy = cord[1]);
				this.out.writeOperator("m");
				break;
			case PathIterator.SEG_LINETO:
				this.out.writePosition(cx = cord[0], cy = cord[1]);
				this.out.writeOperator("l");
				break;
			case PathIterator.SEG_QUADTO:
				double x1 = cord[0];
				double y1 = -cord[1];
				double x2 = cord[2];
				double y2 = -cord[3];
				double xa = (cx + 2f * (x1 - cx) / 3f);
				double ya = (cy + 2f * (y1 - cy) / 3f);
				double xb = (xa + (x2 - cx) / 3f);
				double yb = (ya + (y2 - cy) / 3f);
				cx = x2;
				cy = y2;
				this.out.writePosition(xa, ya);
				this.out.writePosition(xb, yb);
				this.out.writePosition(cx, cy);
				this.out.writeOperator("c");
				break;
			case PathIterator.SEG_CUBICTO:
				this.out.writePosition(cord[0], cord[1]);
				this.out.writePosition(cord[2], cord[3]);
				this.out.writePosition(cx = cord[4], cy = cord[5]);
				this.out.writeOperator("c");
				break;
			case PathIterator.SEG_CLOSE:
				i.next();
				if (i.isDone()) {
					return true;
				}
				this.out.writeOperator("h");
				continue;
			}
			i.next();
		}
		return false;
	}
}