package jp.cssj.sakae.pdf.font.util;

import java.awt.Font;
import java.awt.font.FontRenderContext;
import java.awt.font.GlyphVector;
import java.awt.font.TextAttribute;
import java.awt.geom.AffineTransform;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.sakae.font.BBox;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.g2d.util.G2dUtils;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.font.FontMetrics;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.text.Text;
import jp.cssj.sakae.pdf.PdfGraphicsOutput;

/**
 * フォント関連のユーティリティです。
 * 
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: PdfFontUtils.java 764 2012-02-05 10:31:00Z miyabe $
 */
public class PdfFontUtils {
	private static final Logger LOG = Logger.getLogger(PdfFontUtils.class
			.getName());
	private static final FontRenderContext FRC = new FontRenderContext(null,
			true, true);

	private PdfFontUtils() {
		// unused
	}

	/**
	 * 代替のAWTフォントを返します。
	 * 
	 * @param metaFont
	 * @return
	 */
	public static Font toAwtFont(FontSource source) {
		Map atts = new HashMap();
		String fontName = source.getFontName();
		String awtName = null;
		if (!G2dUtils.isAvailable(fontName)) {
			String[] aliases = source.getAliases();
			for (int i = 0; i < aliases.length; ++i) {
				String alias = aliases[i];
				if (G2dUtils.isAvailable(alias)) {
					awtName = G2dUtils.toAwtFontName(alias);
					break;
				}
			}
			if (awtName == null) {
				awtName = fontName;
			}
		} else {
			awtName = G2dUtils.toAwtFontName(fontName);
		}
		atts.put(TextAttribute.FAMILY, awtName);
		return new Font(atts);
	}

	/**
	 * PDFグラフィックコンテキストにCIDテキストを描画します。
	 * 
	 * @param out
	 * @param text
	 * @throws IOException
	 */
	public static void drawCIDTo(PdfGraphicsOutput out, Text text,
			boolean verticalFont) throws IOException {
		int[] gids = text.getGIDs();
		int goff = text.getGOff();
		int glen = text.getGLen();
		double[] xadvances = text.getXAdvances(false);
		FontMetrics fm = text.getFontMetrics();
		out.startArray();
		int len = 0;
		int off = 0;
		double size = fm.getFontSize();
		for (int i = 0; i < glen; ++i) {
			double xadvance = xadvances == null ? 0 : xadvances[i];
			if (i > 0) {
				xadvance -= fm.getKerning(gids[i - 1], gids[i]);
			}
			if (xadvance != 0) {
				if (verticalFont) {
					xadvance = -xadvance;
				}
				if (len > 0) {
					out.writeBytes16(gids, off + goff, len);
					off += len;
					len = 0;
				}
				double kerning = -xadvance * FontSource.UNITS_PER_EM / size;
				out.writeReal(kerning);
			}
			++len;
		}
		if (len > 0) {
			out.writeBytes16(gids, off + goff, len);
		}
		out.endArray();
		out.writeOperator("TJ");
	}

	/**
	 * AWTフォントを描画します。
	 * 
	 * @param gc
	 * @param fontSource
	 * @param awtFont
	 * @param text
	 */
	public static void drawAwtFont(GC gc, FontSource fontSource, Font awtFont,
			Text text) throws GraphicsException {
		byte direction = text.getFontStyle().getDirection();
		double fontSize = text.getFontStyle().getSize();
		Map atts = new HashMap();
		G2dUtils.setFontAttributes(atts, text.getFontStyle());
		awtFont = awtFont.deriveFont(atts);
		int goff = text.getGOff();
		int glen = text.getGLen();
		int[] gids = text.getGIDs();
		byte[] clens = text.getCLens();
		char[] chars = text.getChars();
		double letterSpacing = text.getLetterSpacing();
		double[] xadvances = text.getXAdvances(false);
		FontMetrics fm = text.getFontMetrics();

		if (direction == FontStyle.DIRECTION_TB) {
			// 横倒し
			gc.transform(AffineTransform.getRotateInstance(Math.PI / 2.0));
			BBox bbox = fontSource.getBBox();
			gc.transform(AffineTransform
					.getTranslateInstance(
							0,
							((bbox.lly + bbox.ury) * fontSize / FontSource.UNITS_PER_EM) / 2f));
		}
		// 横書き
		int k = text.getCOff();
		int pgid = 0;
		for (int i = 0; i < glen; ++i) {
			int ii = goff + i;
			int gid = gids[ii];
			byte gclen = clens[ii];
			try {
				GlyphVector gv = awtFont.layoutGlyphVector(FRC, chars, k,
						gclen, Font.LAYOUT_LEFT_TO_RIGHT);
				gc.fill(gv.getOutline());
			} catch (Exception e) {
				LOG.log(Level.WARNING, new String(chars) + "/k=" + k
						+ "/gclen=" + gclen + "/clen=" + text.getCLen(), e);
			}
			double dx = fm.getAdvance(gid) + letterSpacing;
			if (i > 0) {
				dx -= fm.getKerning(pgid, gid);
			}
			if (xadvances != null) {
				dx += xadvances[i];
			}
			gc.transform(AffineTransform.getTranslateInstance(dx, 0));
			k += gclen;
			pgid = gid;
		}
	}
}
