package jp.cssj.sakae.pdf.font.type1;

import java.io.IOException;

import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.text.Text;
import jp.cssj.sakae.pdf.ObjectRef;
import jp.cssj.sakae.pdf.PdfFragmentOutput;
import jp.cssj.sakae.pdf.PdfGraphicsOutput;
import jp.cssj.sakae.pdf.XRef;
import jp.cssj.sakae.pdf.font.PdfFont;
import jp.cssj.sakae.pdf.font.util.PdfFontUtils;
import jp.cssj.sakae.pdf.gc.PdfGC;

/**
 * 標準Type1フォントです。
 * 
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: Type1Font.java 647 2011-08-28 13:05:54Z miyabe $
 */
class Type1Font implements PdfFont {
	private static final long serialVersionUID = 0L;

	private final AbstractType1FontSource source;

	private final String name, encoding;

	private final ObjectRef fontRef;

	Type1Font(AbstractType1FontSource source, String name, String encoding,
			ObjectRef fontRef) {
		this.source = source;
		this.name = name;
		this.encoding = encoding;
		this.fontRef = fontRef;
	}

	public FontSource getFontSource() {
		return this.source;
	}

	public int toGID(int c) {
		int gid = this.source.toGID(c);
		return gid;
	}

	public short getKerning(int scid, int cid) {
		return this.source.getKerning(scid, cid);
	}

	public int getLigature(int scid, int cid) {
		return this.source.getLigature(scid, cid);
	}

	public short getAdvance(int gid) {
		return this.source.getAdvance(gid);
	}

	public short getWidth(int gid) {
		return this.getAdvance(gid);
	}

	public String getName() {
		return this.name;
	}

	public void drawTo(GC gc, Text text) throws IOException, GraphicsException {
		if (gc instanceof PdfGC) {
			// PDF
			PdfGraphicsOutput out = ((PdfGC) gc).getPDFGraphicsOutput();
			int goff = text.getGOff();
			int glen = text.getGLen();
			int[] gids = text.getGIDs();
			double[] xadvances = text.getXAdvances(false);
			double size = text.getFontMetrics().getFontSize();
			out.startArray();
			int pgid = 0;
			StringBuffer buff = new StringBuffer();
			for (int j = 0; j < glen; ++j) {
				int jj = j + goff;
				int gid = gids[jj];
				short kerning = this.source.getKerning(gid, pgid);
				if (xadvances != null) {
					if (j == 0) {
						double xadvance = xadvances[j];
						if (xadvance != 0) {
							out.writeReal(-xadvance * FontSource.UNITS_PER_EM
									/ size);
						}
					} else {
						kerning += xadvances[j] * FontSource.UNITS_PER_EM
								/ size;
					}
				}
				if (kerning != 0) {
					out.writeString(buff.toString());
					buff.delete(0, buff.length());
					out.writeInt(-kerning);
				}
				buff.append((char) gid);
				pgid = gid;
			}
			out.writeString(buff.toString());
			out.endArray();
			out.writeOperator("TJ");
		} else {
			PdfFontUtils.drawAwtFont(gc, this.source, this.source.getAwtFont(),
					text);
		}
	}

	public void writeTo(PdfFragmentOutput out, XRef xref) throws IOException {
		out.startObject(this.fontRef);
		out.startHash();
		out.writeName("Type");
		out.writeName("Font");
		out.lineBreak();
		out.writeName("Subtype");
		out.writeName("Type1");
		out.lineBreak();
		if (this.encoding != null) {
			out.writeName("Encoding");
			out.writeName(this.encoding);
			out.lineBreak();
		}
		out.writeName("Name");
		out.writeName(this.name);
		out.lineBreak();
		out.writeName("BaseFont");
		out.writeName(this.source.getFontName());
		out.lineBreak();
		out.endHash();
		out.endObject();
	}

	public String toString() {
		return super.toString() + ":[PDFName=" + this.getName() + " source="
				+ this.getFontSource() + "]";
	}
}