package jp.cssj.sakae.pdf.font.cid;

import java.io.Serializable;
import java.lang.ref.SoftReference;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.resolver.Source;
import jp.cssj.sakae.util.IntList;

/**
 * CIDフォントのキャラクタマッピング情報です。
 * 
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: CIDTable.java 687 2011-09-17 06:42:53Z miyabe $
 */
public class CIDTable implements Serializable {
	private static final Logger LOG = Logger
			.getLogger(CIDTable.class.getName());

	private static final long serialVersionUID = 0;

	public static final char MISSING_CHAR = '?';

	protected final Source cmapSource;

	protected final String javaEncoding;

	transient protected SoftReference toCID = null;

	transient protected int missingCID = 0;

	transient protected Charset charset = null;

	public CIDTable(Source cmapSource, String javaEncoding) {
		this.cmapSource = cmapSource;
		this.javaEncoding = javaEncoding;
	}

	private int[] getToCid() {
		int[] toCid = null;
		if (this.toCID != null) {
			toCid = (int[]) this.toCID.get();
			if (toCid != null) {
				return toCid;
			}
		}
		if (LOG.isLoggable(Level.FINE)) {
			LOG.fine("load cid table:" + this.cmapSource);
		}

		CIDTableParser parser = new CIDTableParser();
		try {
			toCid = parser.parse(this.cmapSource);
			Charset charset = this.getCharset();
			// UTF-16BEの場合は、Javaの内部コードと一致する
			if (!charset.name().equalsIgnoreCase("UTF-16BE")) {
				IntList intList = new IntList();
				CharsetDecoder decoder = charset.newDecoder();
				ByteBuffer in = ByteBuffer.allocate(4);
				CharBuffer out = CharBuffer.allocate(1);
				for (int i = 0; i < toCid.length; ++i) {
					int cid = toCid[i];
					if (cid != 0) {
						in.clear();
						if (i < 0xFF) {
							in.put((byte) (i & 0xFF));
						} else if (i <= 0xFFFF) {
							in.put((byte) ((i >> 8) & 0xFF));
							in.put((byte) (i & 0xFF));
						} else if (i <= 0xFFFFFF) {
							in.put((byte) ((i >> 16) & 0xFF));
							in.put((byte) ((i >> 8) & 0xFF));
							in.put((byte) (i & 0xFF));
						} else {
							in.put((byte) ((i >> 24) & 0xFF));
							in.put((byte) ((i >> 16) & 0xFF));
							in.put((byte) ((i >> 8) & 0xFF));
							in.put((byte) (i & 0xFF));
						}
						in.flip();
						out.clear();
						decoder.reset();
						decoder.decode(in, out, true);
						decoder.flush(out);
						intList.set(out.get(0), cid);
					}
				}
				toCid = intList.toArray();
			}
			for (int i = 0; i < toCid.length; ++i) {
				if (toCid[i] == 0) {
					toCid[i] = -1;
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		this.toCID = new SoftReference(toCid);
		return toCid;
	}

	/**
	 * Javaエンコーディングを返します。
	 * 
	 * @return
	 */
	public Charset getCharset() {
		if (this.charset == null) {
			this.charset = Charset.forName(this.javaEncoding);
		}
		return this.charset;
	}

	/**
	 * 不明な文字のCIDを返します。
	 * 
	 * @param codes
	 * @return
	 */
	public int getMissingCID() {
		if (this.missingCID == 0) {
			this.missingCID = this.toCID(CIDTable.MISSING_CHAR);
		}
		return this.missingCID;
	}

	/**
	 * 文字をCIDに変換します。
	 * 
	 * @param c
	 *            文字
	 * @return cid
	 */
	public int toCID(int c) {
		int[] toCid = this.getToCid();
		if (c < 0 || c >= toCid.length) {
			return this.getMissingCID();
		}
		int cid = toCid[c];
		if (cid == -1) {
			return this.getMissingCID();
		}
		return cid;
	}

	/**
	 * 文字を含んでいればtrueを返します。
	 * 
	 * @param c
	 * @return
	 */
	public boolean containsChar(int c) {
		int[] toCid = this.getToCid();
		if (c < 0 || c >= toCid.length) {
			return false;
		}
		int cid = toCid[c];
		if (cid == -1) {
			return false;
		}
		return true;
	}

	public int getLength() {
		int[] toCid = this.getToCid();
		return toCid.length;
	}
}