package jp.cssj.sakae.pdf.font;

import java.awt.GraphicsEnvironment;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.resolver.Source;
import jp.cssj.resolver.SourceResolver;
import jp.cssj.resolver.composite.CompositeSourceResolver;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.font.FontSourceWrapper;
import jp.cssj.sakae.gc.font.FontFace;
import jp.cssj.sakae.gc.font.FontFamily;
import jp.cssj.sakae.gc.font.FontFamilyList;
import jp.cssj.sakae.gc.font.UnicodeRange;
import jp.cssj.sakae.pdf.ObjectRef;
import jp.cssj.sakae.pdf.font.cid.CMap;
import jp.cssj.sakae.pdf.font.type1.Encoding;
import jp.cssj.sakae.pdf.font.type1.GlyphMap;
import net.zamasoft.font.FontFile;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

class PdfFontSourceManagerConfigurationHandler extends DefaultHandler {
	private static final Logger LOG = Logger
			.getLogger(PdfFontSourceManagerConfigurationHandler.class.getName());

	private final URL base;

	private final SourceResolver resolver;

	private static final byte IN_FONTS = 0;

	private static final byte IN_ENCODINGS = 1;

	private static final byte IN_CORE_FONTS = 2;

	private static final byte IN_CMAPS = 3;

	private static final byte IN_CID_FONTS = 5;

	private static final byte IN_GENERIC_FONTS = 6;

	private byte state = IN_FONTS;

	private PDFFontSourceWrapper[] fontSources;

	private GlyphMap unicodeEncoding;

	private final Map<String, Encoding> nameToEncoding = new HashMap<String, Encoding>();

	private Encoding defaultEncoding;

	private final Map<String, CMap> nameToCMap = new HashMap<String, CMap>();

	final Map<String, FontSource> nameToFonts = new HashMap<String, FontSource>();

	final Map<String, FontFamilyList> genericToFamily = new HashMap<String, FontFamilyList>();

	PdfFontSourceManagerConfigurationHandler(URL base) throws IOException {
		this.base = base;
		this.resolver = CompositeSourceResolver
				.createGenericCompositeSourceResolver();
	}

	private File toFile(String src) throws SAXException {
		try {
			Source source = this.resolver.resolve(new URL(this.base, src).toURI());
			try {
				return source.getFile();
			} finally {
				this.resolver.release(source);
			}
		} catch (IOException e) {
			throw new SAXException(e);
		} catch (URISyntaxException e) {
			throw new SAXException(e);
		}
	}

	public void startElement(String uri, String lName, String qName,
			Attributes atts) throws SAXException {
		switch (this.state) {
		case IN_FONTS:
			if (qName.equals("encodings")) {
				// <encodings>

				this.state = IN_ENCODINGS;
			} else if (qName.equals("core-fonts")) {
				// <core-fonts>
				String encoding = atts.getValue("encoding");
				String unicodeSrc = atts.getValue("unicode-src");

				this.defaultEncoding = (Encoding) this.nameToEncoding
						.get(encoding);

				try {
					Source source = this.resolver.resolve(new URL(this.base, unicodeSrc).toURI());
					try {
						this.unicodeEncoding = GlyphMap.parse(source
								.getInputStream());
					} finally {
						this.resolver.release(source);
					}
				} catch (IOException e) {
					throw new SAXException(e);
				} catch (URISyntaxException e) {
					throw new SAXException(e);
				}
				this.state = IN_CORE_FONTS;
			} else if (qName.equals("cmaps")) {
				// <cmaps>

				this.state = IN_CMAPS;
			} else if (qName.equals("cid-fonts")) {
				// <cid-fonts>
				this.state = IN_CID_FONTS;
			} else if (qName.equals("generic-fonts")) {
				// <generic-fonts>
				this.state = IN_GENERIC_FONTS;
			}
			break;
		case IN_ENCODINGS:
			if (qName.equals("encoding")) {
				// <encoding>
				String src = atts.getValue("src");
				Source source = null;
				try {
						source = this.resolver.resolve(new URL(this.base, src).toURI());
					Encoding encoding = Encoding.parse(source.getInputStream());
					this.nameToEncoding.put(encoding.name, encoding);
				} catch (IOException e) {
					throw new SAXException(e);
				} catch (URISyntaxException e) {
					throw new SAXException(e);
				} finally {
					if (source != null) {
						this.resolver.release(source);
					}
				}
			}
			break;

		case IN_CORE_FONTS:
			if (qName.equals("letter-font")) {
				// letter-font
				String src = atts.getValue("src");
				Source source = null;
				try {
					source = this.resolver.resolve(new URL(this.base, src).toURI());

					Encoding pdfEncoding;
					String encoding = atts.getValue("encoding");
					if (encoding != null) {
						pdfEncoding = (Encoding) this.nameToEncoding
								.get(encoding);
					} else {
						pdfEncoding = this.defaultEncoding;
					}

					this.fontSources = new PDFFontSourceWrapper[] { new PDFFontSourceWrapper(
							FontLoader.readLetterType1Font(
									this.unicodeEncoding, pdfEncoding,
									source.getInputStream())) };
				} catch (Exception e) {
					LOG.log(Level.SEVERE, "AFMファイル'" + src + "'を読み込めません。", e);
					throw new SAXException(e);
				} finally {
					if (source != null) {
						this.resolver.release(source);
					}
				}

			} else if (qName.equals("symbol-font")) {
				// symbol-font
				String src = atts.getValue("src");
				Source source = null, encodingSource = null;
				try {
					source = this.resolver.resolve(new URL(this.base, src).toURI());

					String encodingSrc = atts.getValue("encoding-src");
					encodingSource = this.resolver.resolve(new URL(this.base, encodingSrc).toURI());

					this.fontSources = new PDFFontSourceWrapper[] { new PDFFontSourceWrapper(
							FontLoader.readSymbolType1Font(
									source.getInputStream(), encodingSource)) };
				} catch (Exception e) {
					LOG.log(Level.SEVERE, "AFMファイル'" + src + "'を読み込めません。", e);
					throw new SAXException(e);
				} finally {
					if (source != null) {
						this.resolver.release(source);
					}
					if (encodingSource != null) {
						this.resolver.release(encodingSource);
					}
				}
			}
			break;

		case IN_CMAPS:
			if (qName.equals("cmap")) {
				// <cmap>
				String src = atts.getValue("src");
				String javaEncoding = atts.getValue("java-encoding");
				try {
					Source source = this.resolver.resolve(new URL(this.base, src).toURI());
					CMap cmap = new CMap(source, javaEncoding);
					this.nameToCMap.put(cmap.getEncoding(), cmap);
				} catch (Exception e) {
					LOG.log(Level.SEVERE, "CMapファイル'" + src + "'を読み込めません", e);
					throw new SAXException(e);
				}
			}
			break;

		case IN_CID_FONTS:
			if (qName.equals("cid-keyed-font")) {
				String warraySrc = atts.getValue("warray");
				Source source = null;
				try {
					source = this.resolver.resolve(new URL(this.base, warraySrc).toURI());
					FontFace face = FontLoader.toFontFace(atts);
					PdfFontSource[] sources = FontLoader.readCIDKeyedFont(
							source, face, this.nameToCMap);
					this.fontSources = new PDFFontSourceWrapper[sources.length];
					for (int i = 0; i < sources.length; ++i) {
						this.fontSources[i] = new PDFFontSourceWrapper(
								sources[i]);
					}
				} catch (Exception e) {
					LOG.log(Level.SEVERE, "'" + source.getURI()
							+ "'の読み込みに失敗しました。", e);
					throw new SAXException(e);
				} finally {
					if (source != null) {
						this.resolver.release(source);
					}
				}
			} else if (qName.equals("font-file")) {
				String src = atts.getValue("src");
				String types = atts.getValue("types");

				try {
					File ttfFile = this.toFile(src);
					int index;
					try {
						index = Integer.parseInt(atts.getValue("index"));
					} catch (Exception e) {
						index = 0;
					}

					List<FontSource> list = new ArrayList<FontSource>();
					FontFace face = FontLoader.toFontFace(atts);

					if (types.indexOf("cid-keyed") != -1) {
						FontLoader.readTTF(list, face,
								FontLoader.TYPE_CID_KEYED, ttfFile, index,
								this.nameToCMap);
					}
					if (types.indexOf("cid-identity") != -1) {
						FontLoader.readTTF(list, face,
								FontLoader.TYPE_CID_IDENTITY, ttfFile, index,
								this.nameToCMap);
					}
					if (types.indexOf("embedded") != -1) {
						FontLoader.readTTF(list, face,
								FontLoader.TYPE_EMBEDDED, ttfFile, index,
								this.nameToCMap);
					}
					this.fontSources = new PDFFontSourceWrapper[list.size()];
					for (int i = 0; i < list.size(); ++i) {
						this.fontSources[i] = new PDFFontSourceWrapper(
								(PdfFontSource) list.get(i));
					}

				} catch (Exception e) {
					LOG.log(Level.WARNING, "'" + src + "'のフォント情報の取得に失敗しました。", e);
					this.fontSources = null;
				}
			} else if (qName.equals("font-dir")) {
				String dir = atts.getValue("dir");
				String types = atts.getValue("types");

				File dirFile = this.toFile(dir);
				if (LOG.isLoggable(Level.FINE)) {
					LOG.fine("scan: " + dirFile);
				}
				File[] files = dirFile.listFiles();
				if (files != null) {
					FontFace face = FontLoader.toFontFace(atts);
					for (int i = 0; i < files.length; ++i) {
						File ttfFile = files[i];
						if (ttfFile.isDirectory()) {
							continue;
						}
						String name = ttfFile.getName().toLowerCase();
						if (!name.endsWith(".ttf") && !name.endsWith(".ttc")
								&& !name.endsWith(".otf")) {
							continue;
						}
						try {
							FontFile ttFile = FontFile.getFontFile(ttfFile);

							List<FontSource> list = new ArrayList<FontSource>();

							for (int j = 0; j < ttFile.getNumFonts(); ++j) {
								if (types.indexOf("cid-identity") != -1) {
									FontLoader.readTTF(list, face,
											FontLoader.TYPE_CID_IDENTITY,
											ttfFile, j, this.nameToCMap);
								}
								if (types.indexOf("embedded") != -1) {
									FontLoader.readTTF(list, face,
											FontLoader.TYPE_EMBEDDED, ttfFile,
											j, this.nameToCMap);
								}
							}
							for (int j = 0; j < list.size(); ++j) {
								FontLoader.add(new PDFFontSourceWrapper(
										(PdfFontSource) list.get(j)),
										this.nameToFonts);
							}
						} catch (Exception e) {
							LOG.log(Level.WARNING, "'" + ttfFile
									+ "'のフォント情報の取得に失敗しました。", e);
						}
					}
				}
			} else if (qName.equals("system-font")) {
				String src = atts.getValue("src");
				String file = atts.getValue("file");
				String dir = atts.getValue("dir");
				String types = atts.getValue("types");

				List<FontSource> list = new ArrayList<FontSource>();
				FontFace face = FontLoader.toFontFace(atts);

				try {
					if (file != null) {
						File theFile = this.toFile(file);
						InputStream in = new FileInputStream(theFile);
						try {
							java.awt.Font font = java.awt.Font.createFont(
									java.awt.Font.TRUETYPE_FONT, in);
							FontLoader.readSystemFont(face, list, types, font,
									this.nameToCMap);
						} finally {
							in.close();
						}
					} else if (dir != null) {
						File theDir = this.toFile(dir);
						File[] files = theDir.listFiles();
						for (int i = 0; i < files.length; ++i) {
							File theFile = files[i];
							String name = theFile.getName().toLowerCase();
							if (name.endsWith(".ttf") || name.endsWith(".ttc")
									|| name.endsWith(".otf")) {
								InputStream in = new FileInputStream(theFile);
								try {
									java.awt.Font font = java.awt.Font
											.createFont(
													java.awt.Font.TRUETYPE_FONT,
													in);
									FontLoader.readSystemFont(face, list,
											types, font, this.nameToCMap);
								} finally {
									in.close();
								}
							}
						}
					} else {
						java.awt.Font font = java.awt.Font.decode(src);
						FontLoader.readSystemFont(face, list, types, font,
								this.nameToCMap);
					}
					this.fontSources = (PDFFontSourceWrapper[]) list
							.toArray(new PDFFontSourceWrapper[list.size()]);
				} catch (Exception e) {
					LOG.log(Level.WARNING, "'" + src + "'のフォント情報の取得に失敗しました。", e);
					this.fontSources = null;
				}
			} else if (qName.equals("all-system-fonts")) {
				String types = atts.getValue("types");
				String dir = atts.getValue("dir");

				java.awt.Font[] fonts;
				if (dir != null) {
					File dirFile = this.toFile(dir);
					File[] files = dirFile.listFiles();
					List<java.awt.Font> fontList = new ArrayList<java.awt.Font>();
					for (int i = 0; i < files.length; ++i) {
						try {
							InputStream in = new FileInputStream(files[i]);
							try {
								fontList.add(java.awt.Font.createFont(
										java.awt.Font.TRUETYPE_FONT, in));
							} finally {
								in.close();
							}
						} catch (Exception e) {
							LOG.log(Level.WARNING, "フォントファイルを読み込めません", e);
						}
					}
					fonts = (java.awt.Font[]) fontList
							.toArray(new java.awt.Font[fontList.size()]);
				} else {
					fonts = GraphicsEnvironment.getLocalGraphicsEnvironment()
							.getAllFonts();
				}
				FontFace face = FontLoader.toFontFace(atts);
				for (int i = 0; i < fonts.length; ++i) {
					java.awt.Font font = fonts[i];
					try {
						if (types.indexOf("cid-keyed") != -1) {
							PDFFontSourceWrapper fontSource = new PDFFontSourceWrapper(
									FontLoader.readSystemFont(face,
											FontLoader.TYPE_CID_KEYED, font,
											this.nameToCMap));
							FontLoader.add(fontSource, this.nameToFonts);
						}
						if (types.indexOf("cid-identity") != -1) {
							PDFFontSourceWrapper fontSource = new PDFFontSourceWrapper(
									FontLoader.readSystemFont(face,
											FontLoader.TYPE_CID_IDENTITY, font,
											this.nameToCMap));
							FontLoader.add(fontSource, this.nameToFonts);
						}
						if (types.indexOf("embedded") != -1) {
							PDFFontSourceWrapper fontSource = new PDFFontSourceWrapper(
									FontLoader.readSystemFont(face,
											FontLoader.TYPE_EMBEDDED, font,
											this.nameToCMap));
							FontLoader.add(fontSource, this.nameToFonts);
						}
					} catch (Exception e) {
						LOG.log(Level.WARNING, "'" + font.getFontName()
								+ "'のフォント情報の取得に失敗しました。", e);
					}
				}
			}
			break;

		case IN_GENERIC_FONTS:
			String genericFamily = lName;
			String fontFamily = atts.getValue("font-family");

			List<FontFamily> entries = new ArrayList<FontFamily>();
			for (StringTokenizer i = new StringTokenizer(fontFamily, ","); i
					.hasMoreTokens();) {
				FontFamily entry = new FontFamily(i.nextToken());
				entries.add(entry);
			}

			FontFamilyList family = new FontFamilyList(
					(FontFamily[]) entries.toArray(new FontFamily[entries
							.size()]));
			this.genericToFamily.put(genericFamily, family);
			break;
		}
		if (this.fontSources != null) {
			if (qName.equals("alias")) {
				// alias
				String name = atts.getValue("name");
				for (int i = 0; i < this.fontSources.length; ++i) {
					this.fontSources[i].addAliase(name);
				}
			} else if (qName.equals("include")) {
				String unicodeRange = atts.getValue("unicode-range");
				for (StringTokenizer st = new StringTokenizer(unicodeRange, ","); st
						.hasMoreTokens();) {
					UnicodeRange range = UnicodeRange
							.parseRange(st.nextToken());
					for (int i = 0; i < this.fontSources.length; ++i) {
						this.fontSources[i].addInclude(range);
					}
				}
			} else if (qName.equals("exclude")) {
				String unicodeRange = atts.getValue("unicode-range");
				for (StringTokenizer st = new StringTokenizer(unicodeRange, ","); st
						.hasMoreTokens();) {
					UnicodeRange range = UnicodeRange
							.parseRange(st.nextToken());
					for (int i = 0; i < this.fontSources.length; ++i) {
						this.fontSources[i].addExclude(range);
					}
				}
			}
		}
	}

	public void endElement(String uri, String lName, String qName)
			throws SAXException {
		if (qName.equals("letter-font") || qName.equals("symbol-font")
				|| qName.equals("cid-keyed-font") || qName.equals("font-file")
				|| qName.equals("system-font")) {
			if (this.fontSources == null) {
				throw new SAXException(qName);
			}
			for (int i = 0; i < this.fontSources.length; ++i) {
				FontLoader.add(this.fontSources[i], this.nameToFonts);
			}
			this.fontSources = null;
		} else if (qName.equals("encodings") || qName.equals("core-fonts")
				|| qName.equals("cmaps") || qName.equals("cid-fonts")
				|| qName.equals("generic-fonts")) {
			if (this.fontSources != null) {
				throw new SAXException(qName);
			}
			this.state = IN_FONTS;
		}
	}

	private static class PDFFontSourceWrapper extends FontSourceWrapper
			implements PdfFontSource {
		private static final long serialVersionUID = 1L;

		protected final List<String> aliasesList = new ArrayList<String>();

		protected final List<UnicodeRange> includes = new ArrayList<UnicodeRange>();

		protected final List<UnicodeRange> excludes = new ArrayList<UnicodeRange>();

		private transient String[] aliases = null;

		public PDFFontSourceWrapper(PdfFontSource source) {
			super(source);
		}

		public final synchronized void addAliase(String aliase) {
			this.aliasesList.add(aliase);
		}

		public final synchronized void addInclude(UnicodeRange range) {
			this.includes.add(range);
		}

		public final synchronized void addExclude(UnicodeRange range) {
			this.excludes.add(range);
		}

		public String[] getAliases() {
			String[] aliases = this.source.getAliases();
			int count = aliases.length + this.aliasesList.size();
			if (this.aliases == null || this.aliases.length != count) {
				this.aliases = new String[count];
				System.arraycopy(aliases, 0, this.aliases, 0, aliases.length);
				for (int i = 0; i < this.aliasesList.size(); ++i) {
					this.aliases[aliases.length + i] = (String) this.aliasesList
							.get(i);
				}
			}
			return this.aliases;
		}

		public boolean canDisplay(int c) {
			if (!this.excludes.isEmpty()) {
				for (int i = 0; i < this.excludes.size(); ++i) {
					UnicodeRange range = (UnicodeRange) this.excludes.get(i);
					if (range.contains(c)) {
						return false;
					}
				}
			}
			if (!this.includes.isEmpty()) {
				for (int i = 0; i < this.includes.size(); ++i) {
					UnicodeRange range = (UnicodeRange) this.includes.get(i);
					if (range.contains(c)) {
						return this.source.canDisplay(c);
					}
				}
				return false;
			}
			return this.source.canDisplay(c);
		}

		public PdfFont createFont(String name, ObjectRef fontRef) {
			return ((PdfFontSource) this.source).createFont(name, fontRef);
		}

		public byte getType() {
			return ((PdfFontSource) this.source).getType();
		}
	}
};
