package jp.cssj.sakae.pdf.font;

import java.io.IOException;

import jp.cssj.sakae.font.Font;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.font.FontStore;
import jp.cssj.sakae.gc.font.FontMetrics;
import jp.cssj.sakae.gc.font.FontStyle;

public class FontMetricsImpl implements FontMetrics {
	private static final long serialVersionUID = 1L;

	protected final FontStore fontStore;

	protected final FontSource source;

	protected final double size, xheight;

	protected final double ascent, descent;

	protected Font font = null;

	protected FontMetricsImpl(FontStore fontStore, FontSource fontSource,
			FontStyle fontStyle) {
		this.fontStore = fontStore;
		this.source = fontSource;
		this.size = fontStyle.getSize();
		this.xheight = this.size * this.source.getXHeight()
				/ FontSource.UNITS_PER_EM;

		double ascent, descent;
		byte direction = fontStyle.getDirection();
		switch (direction) {
		case FontStyle.DIRECTION_LTR:
		case FontStyle.DIRECTION_RTL:
			// 横書き
			ascent = this.size * this.source.getAscent()
					/ FontSource.UNITS_PER_EM;
			descent = this.size * this.source.getDescent()
					/ FontSource.UNITS_PER_EM;
			break;

		case FontStyle.DIRECTION_TB:
			// 縦書き
			ascent = descent = this.size / 2.0;
			break;

		default:
			throw new IllegalStateException();
		}
		double remainder = (this.size - ascent - descent);
		if (remainder != 0) {
			double afrac = ascent / (ascent + descent);
			double dfrac = descent / (ascent + descent);
			ascent = this.size * afrac;
			descent = this.size * dfrac;
		}
		this.ascent = ascent;
		this.descent = descent;
	}

	public Font getFont() {
		try {
			this.font = this.fontStore.useFont(this.source);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
		return this.font;
	}

	public FontSource getFontSource() {
		return this.source;
	}

	public double getAscent() {
		return this.ascent;
	}

	public double getDescent() {
		return this.descent;
	}

	public double getFontSize() {
		return this.size;
	}

	public double getXHeight() {
		return this.xheight;
	}

	public double getSpaceAdvance() {
		return this.size * this.source.getSpaceAdvance()
				/ FontSource.UNITS_PER_EM;
	}

	public double getAdvance(int gid) {
		return this.size * this.getFont().getAdvance(gid)
				/ FontSource.UNITS_PER_EM;
	}

	public double getWidth(int gid) {
		return this.size * this.getFont().getWidth(gid)
				/ FontSource.UNITS_PER_EM;
	}

	public double getKerning(int gid, int sgid) {
		return this.size * this.getFont().getKerning(gid, sgid)
				/ FontSource.UNITS_PER_EM;
	}

	public int getLigature(int gid, int sgid) {
		return this.getFont().getLigature(gid, sgid);
	}

	public boolean canDisplay(int c) {
		return this.getFontSource().canDisplay(c);
	}

	public String toString() {
		return super.toString() + ":[fontSource=" + this.source + "]";
	}
}
