package jp.cssj.sakae.pdf.font;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import jp.cssj.sakae.font.Font;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.font.FontSourceManager;
import jp.cssj.sakae.font.FontStore;
import jp.cssj.sakae.gc.font.FontFace;
import jp.cssj.sakae.gc.font.FontListMetrics;
import jp.cssj.sakae.gc.font.FontManager;
import jp.cssj.sakae.gc.font.FontMetrics;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.text.GlyphHandler;
import jp.cssj.sakae.gc.text.Glypher;
import jp.cssj.sakae.gc.text.Quad;
import jp.cssj.sakae.pdf.font.cid.missing.MissingCIDFontSource;

/**
 * Implementation of FontManager for PDF.
 * 
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: FontManagerImpl.java 669 2011-09-07 12:46:24Z miyabe $
 */
public class FontManagerImpl implements FontManager {
	private static final long serialVersionUID = 1L;

	protected final FontSourceManager globaldb;

	protected PdfFontSourceManager localdb = null;

	protected final FontStore fontStore;

	protected final Map<FontStyle, FontListMetrics> fontListMetricsCache = new HashMap<FontStyle, FontListMetrics>();

	public FontManagerImpl(FontSourceManager fontdb, FontStore fontStore) {
		this.globaldb = fontdb;
		this.fontStore = fontStore;
	}

	public FontManagerImpl(FontSourceManager fontdb) {
		this(fontdb, new DefaultFontStore());
	}

	public void addFontFace(FontFace face) throws IOException {
		if (this.localdb == null) {
			this.localdb = new PdfFontSourceManager();
		}
		this.localdb.addFontFace(face);
	}

	public FontListMetrics getFontListMetrics(FontStyle fontStyle) {
		FontListMetrics flm = (FontListMetrics) this.fontListMetricsCache
				.get(fontStyle);
		if (flm != null) {
			return flm;
		}
		int count = 1;
		FontSource[] fonts1;
		if (this.localdb != null) {
			fonts1 = this.localdb.lookup(fontStyle);
			count += fonts1.length;
		} else {
			fonts1 = null;
		}
		FontSource[] fonts2 = this.globaldb.lookup(fontStyle);
		count += fonts2.length;
		FontMetrics[] fms = new FontMetrics[count];
		int j = 0;
		if (fonts1 != null) {
			for (int i = 0; i < fonts1.length; ++i) {
				fms[j++] = new FontMetricsImpl(this.fontStore, fonts1[i],
						fontStyle);
			}
		}
		for (int i = 0; i < fonts2.length; ++i) {
			fms[j++] = new FontMetricsImpl(this.fontStore, fonts2[i], fontStyle);
		}
		if (fontStyle.getDirection() == FontStyle.DIRECTION_TB) {
			fms[fms.length - 1] = new FontMetricsImpl(this.fontStore,
					MissingCIDFontSource.INSTANCES_TB, fontStyle);
		} else {
			fms[fms.length - 1] = new FontMetricsImpl(this.fontStore,
					MissingCIDFontSource.INSTANCES_LTR, fontStyle);
		}
		flm = new FontListMetrics(fms);
		this.fontListMetricsCache.put(fontStyle, flm);
		return flm;
	}

	public Glypher getGlypher() {
		return new CharacterHandler();
	}

	protected class CharacterHandler implements Glypher {
		private GlyphHandler glyphHandler;

		private final char[] ch = new char[3];

		private int off;

		private byte len;

		private int gid;

		private FontListMetrics fontListMetrics = null;

		private int fontBound = 0;

		private FontStyle fontStyle;

		private FontMetricsImpl fontMetrics = null;

		private int pgid = -1;

		private boolean outOfRun = true;

		public CharacterHandler() {
			// ignore
		}

		public void setGlyphHander(GlyphHandler glyphHandler) {
			this.glyphHandler = glyphHandler;
		}

		public void fontStyle(FontStyle fontStyle) {
			this.glyphBreak();
			this.fontListMetrics = null;
			this.fontStyle = fontStyle;
		}

		private void glyphBreak() {
			if (!this.outOfRun) {
				this.glyph();
				this.endRun();
			}
		}

		public void characters(char[] ch, int off, int len) {
			if (this.fontListMetrics == null) {
				this.fontListMetrics = FontManagerImpl.this
						.getFontListMetrics(this.fontStyle);
				this.initFont();
			}

			for (int k = 0; k < len; ++k) {
				char c = ch[k + off];

				// \A0は空白に変換
				char cc = c == '\u00A0' ? '\u0020' : c;

				// ランの範囲を作成
				if (!this.fontMetrics.canDisplay(cc)) {
					this.glyphBreak();
					this.initFont();
				}

				// 利用可能な文字を検出
				for (int j = 0; j < this.fontBound; ++j) {
					FontMetricsImpl metrics = (FontMetricsImpl) this.fontListMetrics
							.getFontMetrics(j);
					if (metrics.canDisplay(cc)) {
						this.glyphBreak();
						this.fontMetrics = (FontMetricsImpl) this.fontListMetrics
								.getFontMetrics(j);
						this.fontBound = j;
						break;
					}
				}

				if (this.outOfRun) {
					this.outOfRun = false;
					this.gid = -1;
					this.pgid = -1;
					this.len = 0;
					this.glyphHandler.startTextRun(this.fontStyle,
							this.fontMetrics);
				}

				// 通常の文字として処理
				Font font = this.fontMetrics.getFont();
				int gid = font.toGID(cc);

				// 連字のチェック
				int lgid = this.fontMetrics.getLigature(this.gid, gid);
				if (lgid != -1) {
					// 連字にできる
					this.gid = this.pgid;
					gid = lgid;
					this.ch[this.len] = c;
					++this.len;
				} else {
					// 連字にできない
					if (this.gid != -1) {
						this.glyph();
					}
					this.ch[0] = c;
					this.len = 1;
				}
				this.pgid = this.gid;
				this.gid = gid;
			}
		}

		public void quad(Quad quad) {
			this.glyphBreak();
			this.glyphHandler.quad(quad);
		}

		private void glyph() {
			this.glyphHandler.glyph(this.ch, this.off, this.len, this.gid);
		}

		private void endRun() {
			assert this.outOfRun == false;
			this.glyphHandler.endTextRun();
			this.outOfRun = true;
		}

		private void initFont() {
			this.fontBound = this.fontListMetrics.getLength();
			this.fontMetrics = (FontMetricsImpl) this.fontListMetrics
					.getFontMetrics(this.fontBound - 1);
		}

		public void flush() {
			this.glyphBreak();
			this.glyphHandler.flush();
		}
	}
}