package jp.cssj.sakae.g2d.util;

import java.awt.Canvas;
import java.awt.Font;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.MediaTracker;
import java.awt.TexturePaint;
import java.awt.Toolkit;
import java.awt.font.TextAttribute;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import jp.cssj.sakae.g2d.gc.G2dGC;
import jp.cssj.sakae.g2d.image.RasterImageImpl;
import jp.cssj.sakae.g2d.image.png.PNGDecodeParam;
import jp.cssj.sakae.g2d.image.png.PNGImage;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.font.FontFamily;
import jp.cssj.sakae.gc.font.FontFamilyList;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.font.util.FontUtils;
import jp.cssj.sakae.gc.image.Image;
import jp.cssj.sakae.gc.paint.Color;
import jp.cssj.sakae.gc.paint.LinearGradient;
import jp.cssj.sakae.gc.paint.Paint;
import jp.cssj.sakae.gc.paint.Pattern;
import jp.cssj.sakae.gc.paint.RGBColor;
import jp.cssj.sakae.gc.paint.RadialGradient;

import org.apache.batik.ext.awt.LinearGradientPaint;
import org.apache.batik.ext.awt.MultipleGradientPaint;
import org.apache.batik.ext.awt.RadialGradientPaint;

/**
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: G2dUtils.java 763 2012-01-06 11:56:18Z miyabe $
 */
public final class G2dUtils {
	private static final Logger LOGGER = Logger.getLogger(G2dUtils.class
			.getName());

	private G2dUtils() {
		// unused
	}

	public static Paint fromAwtPaint(java.awt.Paint paint) {
		if (paint instanceof java.awt.Color) {
			return fromAwtColor((java.awt.Color) paint);
		}
		if (paint instanceof GradientPaint) {
			GradientPaint gpaint = (GradientPaint) paint;
			if (gpaint.isCyclic()) {
				return null;
			}
			return new LinearGradient(gpaint.getPoint1().getX(), gpaint
					.getPoint1().getY(), gpaint.getPoint2().getX(), gpaint
					.getPoint2().getY(), new double[] { 0, 1 }, new Color[] {
					fromAwtColor(gpaint.getColor1()),
					fromAwtColor(gpaint.getColor2()) }, null);
		}
		if (paint instanceof RadialGradientPaint) {
			RadialGradientPaint gpaint = (RadialGradientPaint) paint;
			float[] fs = gpaint.getFractions();
			double[] fractions = new double[fs.length];
			for (int i = 0; i < fs.length; ++i) {
				fractions[i] = fs[i];
			}
			java.awt.Color[] cs = gpaint.getColors();
			Color[] colors = new Color[cs.length];
			for (int i = 0; i < cs.length; ++i) {
				colors[i] = fromAwtColor(cs[i]);
			}
			return new RadialGradient(gpaint.getCenterPoint().getX(), gpaint
					.getCenterPoint().getY(), gpaint.getRadius(), gpaint
					.getFocusPoint().getX(), gpaint.getFocusPoint().getY(),
					fractions, colors, gpaint.getTransform());
		}
		if (paint instanceof LinearGradientPaint) {
			LinearGradientPaint gpaint = (LinearGradientPaint) paint;
			if (gpaint.getCycleMethod() != LinearGradientPaint.NO_CYCLE) {
				return null;
			}
			float[] fs = gpaint.getFractions();
			double[] fractions = new double[fs.length];
			for (int i = 0; i < fs.length; ++i) {
				fractions[i] = fs[i];
			}
			java.awt.Color[] cs = gpaint.getColors();
			Color[] colors = new Color[cs.length];
			for (int i = 0; i < cs.length; ++i) {
				colors[i] = fromAwtColor(cs[i]);
			}
			return new LinearGradient(gpaint.getStartPoint().getX(), gpaint
					.getStartPoint().getY(), gpaint.getEndPoint().getX(),
					gpaint.getEndPoint().getY(), fractions, colors, null);
		}
		if (paint instanceof TexturePaint) {
			TexturePaint tpaint = (TexturePaint) paint;
			Rectangle2D r = tpaint.getAnchorRect();
			AffineTransform at = AffineTransform.getTranslateInstance(r.getX(),
					r.getY());
			BufferedImage image = tpaint.getImage();
			at.scale(r.getWidth() / image.getWidth(),
					r.getHeight() / image.getHeight());
			return new Pattern(new RasterImageImpl(image), at);
		}
		return null;
	}

	public static java.awt.Paint toAwtPaint(LinearGradient gradient) {
		double[] fs = gradient.getFractions();
		float[] fractions = new float[fs.length];
		for (int i = 0; i < fs.length; ++i) {
			fractions[i] = (float) fs[i];
		}
		Color[] cs = gradient.getColors();
		java.awt.Color[] colors = new java.awt.Color[cs.length];
		for (int i = 0; i < cs.length; ++i) {
			colors[i] = toAwtColor(cs[i]);
		}
		return new LinearGradientPaint(new Point2D.Double(gradient.getX1(),
				gradient.getY1()), new Point2D.Double(gradient.getX2(),
				gradient.getY2()), fractions, colors,
				MultipleGradientPaint.NO_CYCLE, MultipleGradientPaint.SRGB,
				gradient.getTransform());
	}

	public static java.awt.Paint toAwtPaint(RadialGradient gradient) {
		double[] fs = gradient.getFractions();
		float[] fractions = new float[fs.length];
		for (int i = 0; i < fs.length; ++i) {
			fractions[i] = (float) fs[i];
		}
		Color[] cs = gradient.getColors();
		java.awt.Color[] colors = new java.awt.Color[cs.length];
		for (int i = 0; i < cs.length; ++i) {
			colors[i] = toAwtColor(cs[i]);
		}
		return new RadialGradientPaint(new Point2D.Double(gradient.getCX(),
				gradient.getCY()), (float) gradient.getRadius(),
				new Point2D.Double(gradient.getFY(), gradient.getFX()),
				fractions, colors, MultipleGradientPaint.NO_CYCLE,
				MultipleGradientPaint.SRGB, gradient.getTransform());
	}

	public static Color fromAwtColor(java.awt.Color color) {
		return RGBColor
				.create((short) color.getRed() / 255f,
						(short) color.getGreen() / 255f,
						(short) color.getBlue() / 255f);
	}

	public static java.awt.Color toAwtColor(Color color) {
		return new java.awt.Color(color.getRed(), color.getGreen(),
				color.getBlue());
	}

	public static java.awt.Paint toAwtPaint(Pattern pattern, GC gc) {
		Image image = pattern.getImage();
		double width = image.getWidth();
		double height = image.getHeight();
		BufferedImage bimage;
		if (image instanceof RasterImageImpl) {
			bimage = (BufferedImage) ((RasterImageImpl) image).getImage();
		} else {
			bimage = new BufferedImage((int) width, (int) height,
					BufferedImage.TYPE_INT_ARGB);
			Graphics2D bg = (Graphics2D) bimage.getGraphics();
			image.drawTo(new G2dGC(bg, gc.getFontManager()));
		}
		return new TexturePaint(bimage, new Rectangle2D.Double(0, 0, width,
				height));
	}

	public static void drawImage(Graphics2D g, BufferedImage image, double x,
			double y, double w, double h) {
		g.drawImage(image, new AffineTransform(w / image.getWidth(), 0, 0, h
				/ image.getHeight(), x, y), null);
	}

	public static final String toAwtFamilyName(FontFamily ffe) {
		if (ffe.isGenericFamily()) {
			switch (ffe.getGenericFamily()) {
			case FontFamily.CURSIVE:
				return "SansSerif";// AWT doesn't support logical font family
				// 'cursive'.
			case FontFamily.FANTASY:
				return "SansSerif";// AWT doesn't support logical font family
				// 'fantasy'.
			case FontFamily.MONOSPACE:
				return "Monospaced";
			case FontFamily.SANS_SERIF:
				return "SansSerif";
			case FontFamily.SERIF:
				return "Serif";

			default:
				throw new IllegalStateException();
			}
		}
		return ffe.getName();

	}

	public static final Font[] toFonts(FontStyle fontStyle) {
		Map atts = new HashMap();
		setFontAttributes(atts, fontStyle);
		Font[] fonts = new Font[fontStyle.getFamily().getLength()];
		for (int i = 0; i < fonts.length; ++i) {
			atts.put(TextAttribute.FAMILY,
					G2dUtils.toAwtFamilyName(fontStyle.getFamily().get(i)));
			fonts[i] = new Font(atts);
		}
		return fonts;
	}

	public static final FontFamilyList toFontFamilyList(String awtFamily,
			FontFamilyList defaultFamily) {
		if (awtFamily == null) {
			return defaultFamily;
		}
		return FontFamilyList.create(awtFamily);
	}

	public static final short toFontWeight(Float awtWeight, short defaultWeight) {
		if (awtWeight == null) {
			return defaultWeight;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_EXTRA_LIGHT) <= 0) {
			return FontStyle.FONT_WEIGHT_100;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_LIGHT) <= 0) {
			return FontStyle.FONT_WEIGHT_200;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_DEMILIGHT) <= 0) {
			return FontStyle.FONT_WEIGHT_300;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_REGULAR) <= 0) {
			return FontStyle.FONT_WEIGHT_400;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_SEMIBOLD) <= 0) {
			return FontStyle.FONT_WEIGHT_500;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_MEDIUM) <= 0) {
			return FontStyle.FONT_WEIGHT_500;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_DEMIBOLD) <= 0) {
			return FontStyle.FONT_WEIGHT_600;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_BOLD) <= 0) {
			return FontStyle.FONT_WEIGHT_700;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_HEAVY) <= 0) {
			return FontStyle.FONT_WEIGHT_700;
		} else if (awtWeight.compareTo(TextAttribute.WEIGHT_EXTRABOLD) <= 0) {
			return FontStyle.FONT_WEIGHT_800;
		}
		return FontStyle.FONT_WEIGHT_900;
	}

	public static final double toFontSize(Float awtSize, double defaultSize) {
		return awtSize != null ? awtSize.doubleValue() : defaultSize;
	}

	public static final byte toFontStyle(Float awtPosture, byte defaultStyle) {
		if (awtPosture == null) {
			return defaultStyle;
		} else if (awtPosture.equals(TextAttribute.POSTURE_OBLIQUE)) {
			return FontStyle.FONT_STYLE_OBLIQUE;
		}
		return FontStyle.FONT_STYLE_NORMAL;
	}

	public static final void setFontAttributes(Map atts, FontStyle fontStyle) {
		atts.put(TextAttribute.SIZE, new Float(fontStyle.getSize()));

		Float weight;
		switch (fontStyle.getWeight()) {
		case 100:
			weight = TextAttribute.WEIGHT_EXTRA_LIGHT;
			break;
		case 200:
			weight = TextAttribute.WEIGHT_LIGHT;
			break;
		case 300:
			weight = TextAttribute.WEIGHT_DEMILIGHT;
			break;
		case 400:
			weight = TextAttribute.WEIGHT_REGULAR;
			break;
		case 500:
			weight = TextAttribute.WEIGHT_SEMIBOLD;
			break;
		case 600:
			weight = TextAttribute.WEIGHT_DEMIBOLD;
			break;
		case 700:
			weight = TextAttribute.WEIGHT_BOLD;
			break;
		case 800:
			weight = TextAttribute.WEIGHT_EXTRABOLD;
			break;
		case 900:
			weight = TextAttribute.WEIGHT_ULTRABOLD;
			break;
		default:
			throw new IllegalStateException();
		}
		atts.put(TextAttribute.WEIGHT, weight);

		Float posture;
		switch (fontStyle.getStyle()) {
		case FontStyle.FONT_STYLE_NORMAL:
			posture = TextAttribute.POSTURE_REGULAR;
			break;
		case FontStyle.FONT_STYLE_ITALIC:
		case FontStyle.FONT_STYLE_OBLIQUE:
			posture = TextAttribute.POSTURE_OBLIQUE;
			break;
		default:
			throw new IllegalStateException();
		}
		atts.put(TextAttribute.POSTURE, posture);
	}

	private static Map normNameToAWTName = null;

	private static void buildNormNameToAWTName() {
		if (normNameToAWTName == null) {
			normNameToAWTName = new HashMap();
			Font[] fonts = GraphicsEnvironment.getLocalGraphicsEnvironment()
					.getAllFonts();
			for (int i = 0; i < fonts.length; ++i) {
				String name = fonts[i].getFontName();
				LOGGER.fine(name);
				normNameToAWTName.put(FontUtils.normalizeName(name), name);
			}
			normNameToAWTName = Collections.unmodifiableMap(normNameToAWTName);
		}
	}

	public static synchronized boolean isAvailable(String fontName) {
		buildNormNameToAWTName();
		return normNameToAWTName.containsKey(FontUtils.normalizeName(fontName));
	}

	public static synchronized String toAwtFontName(String fontName) {
		buildNormNameToAWTName();
		return (String) normNameToAWTName
				.get(FontUtils.normalizeName(fontName));
	}

	public static BufferedImage loadImage(ImageReader reader,
			ImageInputStream imageIn) throws IOException {
		try {
			String type = reader.getFormatName();
			BufferedImage buffer = null;
			if (type.equalsIgnoreCase("png")) {
				// 独自のPNGデコーダを使う
				try {
					RenderedImage rimage = new PNGImage(
							new ImageInputStreamWrapper(imageIn),
							new PNGDecodeParam());
					buffer = new BufferedImage(rimage.getWidth(),
							rimage.getHeight(), BufferedImage.TYPE_INT_ARGB);
					((Graphics2D) buffer.getGraphics()).drawRenderedImage(
							rimage, new AffineTransform());
				} catch (Exception e) {
					imageIn.seek(0);
				}
			}
			if (buffer == null) {
				try {
					// ImageIOを使う
					reader.setInput(imageIn);
					buffer = reader.read(0);
				} catch (Throwable e1) {
					e1.printStackTrace();
					// Toolkitを使う
					try {
						imageIn.seek(0);
						ByteArrayOutputStream out = new ByteArrayOutputStream();
						byte[] buff = new byte[8192];
						for (int len = imageIn.read(buff); len != -1; len = imageIn
								.read(buff)) {
							out.write(buff, 0, len);
						}
						java.awt.Image image = Toolkit.getDefaultToolkit()
								.createImage(out.toByteArray());
						MediaTracker tracker = new MediaTracker(new Canvas());
						tracker.addImage(image, 0);
						tracker.waitForAll();
						buffer = new BufferedImage(image.getWidth(null),
								image.getHeight(null),
								BufferedImage.TYPE_INT_ARGB);
						buffer.getGraphics().drawImage(image, 0, 0, null);
					} catch (IOException ioe) {
						throw ioe;
					} catch (Throwable e2) {
						IOException ioe = new IOException(e2.getMessage());
						ioe.initCause(e2);
						throw ioe;
					}
				}
			}
			return buffer;
		} finally {
			reader.dispose();
		}
	}
}

class ImageInputStreamWrapper extends InputStream {
	private final ImageInputStream in;

	public ImageInputStreamWrapper(ImageInputStream in) throws IOException {
		this.in = in;
		this.in.seek(0);
	}

	public int read() throws IOException {
		return this.in.read();
	}

	public int read(byte[] buff, int off, int len) throws IOException {
		return this.in.read(buff, off, len);
	}

	public int read(byte[] buff) throws IOException {
		return this.in.read(buff);
	}
}