package jp.cssj.sakae.g2d.gc.text;

import java.awt.font.TextAttribute;
import java.text.AttributedCharacterIterator;
import java.text.CharacterIterator;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import jp.cssj.sakae.g2d.gc.BridgeGraphics2D;
import jp.cssj.sakae.g2d.util.G2dUtils;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.font.FontFamilyList;
import jp.cssj.sakae.gc.font.FontListMetrics;
import jp.cssj.sakae.gc.font.FontPolicyList;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.font.FontStyleImpl;
import jp.cssj.sakae.gc.text.FilterGlyphHandler;
import jp.cssj.sakae.gc.text.GlyphHandler;
import jp.cssj.sakae.gc.text.Glypher;
import jp.cssj.sakae.gc.text.Quad;
import jp.cssj.sakae.gc.text.hyphenation.Hyphenation;
import jp.cssj.sakae.gc.text.layout.FilterCharacterHandler;
import jp.cssj.sakae.gc.text.layout.control.LineBreak;
import jp.cssj.sakae.gc.text.layout.control.Tab;

public class TextLayoutHandler extends FilterCharacterHandler {
	private final GC gc;

	private boolean styleChanged = true;

	private FontFamilyList fontFamilies = FontFamilyList.SERIF;

	private double size = 12.0;

	private byte style = FontStyle.FONT_STYLE_NORMAL;

	private short weight = FontStyle.FONT_WEIGHT_400;

	private byte direction = FontStyle.DIRECTION_LTR;

	public static final FontPolicyList DEFAULT_FONT_POLICY = new FontPolicyList(
			new byte[] { FontPolicyList.FONT_POLICY_CID_KEYED,
					FontPolicyList.FONT_POLICY_EMBEDDED,
					FontPolicyList.FONT_POLICY_CID_IDENTITY });

	private FontPolicyList fontPolicy = DEFAULT_FONT_POLICY;

	private FontStyle fontStyle;

	public TextLayoutHandler(GC gc, Hyphenation hyphenation,
			GlyphHandler glyphHandler) {
		this.gc = gc;
		FilterGlyphHandler textUnitizer = hyphenation.getTextUnitizer();
		textUnitizer.setGlyphHandler(glyphHandler);
		Glypher glypher = gc.getFontManager().getGlypher();
		glypher.setGlyphHander(textUnitizer);
		this.setCharacterHandler(glypher);
	}

	public void fontStyle(FontStyle fontStyle) {
		this.fontStyle = fontStyle;
		this.styleChanged = false;
		super.fontStyle(fontStyle);
	}

	private char[] ch = new char[10];
	private static final Set<TextAttribute> ATTRIBUTES = new HashSet<TextAttribute>(
			Arrays.asList(new TextAttribute[] { TextAttribute.FAMILY,
					TextAttribute.WEIGHT, TextAttribute.SIZE,
					TextAttribute.POSTURE }));

	public void characters(AttributedCharacterIterator aci) {
		FontFamilyList defaultFamily = this.fontFamilies;
		double defaultSize = this.size;
		short defaultWeight = this.weight;
		byte defaultStyle = this.style;
		byte defaultDirection = this.direction;
		while (aci.current() != CharacterIterator.DONE) {
			this.setFontFamilies(G2dUtils.toFontFamilyList(
					(String) aci.getAttribute(TextAttribute.FAMILY),
					defaultFamily));
			this.setFontWeight(G2dUtils.toFontWeight(
					(Float) aci.getAttribute(TextAttribute.WEIGHT),
					defaultWeight));
			this.setFontSize(G2dUtils.toFontSize(
					(Float) aci.getAttribute(TextAttribute.SIZE), defaultSize));
			this.setFontStyle(G2dUtils.toFontStyle(
					(Float) aci.getAttribute(TextAttribute.POSTURE),
					defaultStyle));
			Byte direction = (Byte) aci
					.getAttribute(BridgeGraphics2D.WRITING_MODE);
			this.setDirection(direction == null ? defaultDirection : direction
					.byteValue());

			int nextRun = aci.getRunLimit(ATTRIBUTES);
			int len = nextRun - aci.getIndex();
			if (len > this.ch.length) {
				this.ch = new char[len];
			}
			for (int i = 0; aci.getIndex() != nextRun; ++i) {
				this.ch[i] = aci.current();
				aci.next();
			}

			this.characters(this.ch, 0, len);
		}
		this.setFontFamilies(defaultFamily);
		this.setFontSize(defaultSize);
		this.setFontWeight(defaultWeight);
		this.setFontStyle(defaultStyle);
		this.setDirection(defaultDirection);
	}

	public void characters(char[] ch, int off, int len) {
		if (len == 0) {
			return;
		}
		if (this.styleChanged) {
			this.fontStyle(new FontStyleImpl(this.fontFamilies, this.size,
					this.style, this.weight, this.direction, this.fontPolicy));
		}
		int ooff = 0;
		for (int i = 0; i < len; ++i) {
			char c = ch[i + off];
			Quad quad = null;
			if (Character.isISOControl(c)) {
				FontListMetrics flm = this.gc.getFontManager()
						.getFontListMetrics(this.fontStyle);
				switch (c) {
				case '\n':
					quad = new LineBreak(flm);
					break;
				case '\t':
					quad = new Tab(flm);
					break;
				}
				if (quad != null) {
					if (i > ooff) {
						super.characters(ch, off + ooff, i - ooff);
					}
					ooff = i + 1;
					super.quad(quad);
					continue;
				}
			}
		}
		if (len > ooff) {
			super.characters(ch, off + ooff, len - ooff);
		}
	}
	
	private void changed() {
		this.flush();
		this.styleChanged = true;
	}

	public void setFontFamilies(FontFamilyList families) {
		if (this.fontFamilies.equals(families)) {
			return;
		}
		this.changed();
		this.fontFamilies = families;
	}

	public void setFontSize(double size) {
		if (this.size == size) {
			return;
		}
		this.changed();
		this.size = size;
	}

	public void setFontStyle(byte style) {
		if (this.style == style) {
			return;
		}
		this.changed();
		this.style = style;
	}

	public void setFontWeight(short weight) {
		if (this.weight == weight) {
			return;
		}
		this.changed();
		this.weight = weight;
	}

	public void setDirection(byte direction) {
		if (this.direction == direction) {
			return;
		}
		this.changed();
		this.direction = direction;
	}

	public void setFontPolicy(FontPolicyList fontPolicy) {
		if (this.fontPolicy.equals(fontPolicy)) {
			return;
		}
		this.changed();
		this.fontPolicy = fontPolicy;
	}

	public void characters(String text) throws GraphicsException {
		char[] ch = text.toCharArray();
		this.characters(ch, 0, ch.length);
	}
}
