package jp.cssj.sakae.g2d.gc;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.ArrayList;

import jp.cssj.sakae.font.Font;
import jp.cssj.sakae.g2d.image.RasterImageImpl;
import jp.cssj.sakae.g2d.util.G2dUtils;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.GroupImageGC;
import jp.cssj.sakae.gc.font.FontManager;
import jp.cssj.sakae.gc.image.Image;
import jp.cssj.sakae.gc.paint.Color;
import jp.cssj.sakae.gc.paint.LinearGradient;
import jp.cssj.sakae.gc.paint.Paint;
import jp.cssj.sakae.gc.paint.Pattern;
import jp.cssj.sakae.gc.paint.RadialGradient;
import jp.cssj.sakae.gc.text.Text;
import jp.cssj.sakae.pdf.font.FontMetricsImpl;

/**
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: G2dGC.java 759 2011-11-13 14:06:17Z miyabe $
 */

public class G2dGC implements GC {
	protected static class GraphicsState {
		public final Shape clip;

		public final AffineTransform transform;

		public final Stroke stroke;

		public final java.awt.Color strokeColor;

		public final java.awt.Paint fillPaint;

		public final AffineTransform fillAt, strokeAt;

		public final Composite composite;

		public GraphicsState(G2dGC gc) {
			Graphics2D g = gc.g;
			this.transform = g.getTransform();
			this.clip = g.getClip();
			this.stroke = g.getStroke();
			this.strokeColor = g.getColor();
			this.composite = g.getComposite();
			this.fillPaint = gc.fillPaint;
			this.fillAt = gc.fillAt;
			this.strokeAt = gc.strokeAt;
		}

		public void restore(G2dGC gc) {
			Graphics2D g = gc.g;
			g.setTransform(this.transform);
			g.setClip(this.clip);
			g.setStroke(this.stroke);
			g.setColor(this.strokeColor);
			g.setComposite(this.composite);
			gc.fillPaint = this.fillPaint;
			gc.fillAt = this.fillAt;
			gc.strokeAt = this.strokeAt;
		}
	}

	protected final Graphics2D g;

	protected boolean drewAnything = false;

	protected java.awt.Paint fillPaint;

	protected AffineTransform fillAt, strokeAt;

	protected ArrayList<GraphicsState> stack = new ArrayList<GraphicsState>();

	protected final FontManager fm;

	public G2dGC(Graphics2D g, FontManager fm) {
		this.g = g;
		this.g.setStroke(new BasicStroke(1f, BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_MITER));
		this.fillPaint = this.g.getPaint();
		this.fm = fm;
	}

	public FontManager getFontManager() {
		return this.fm;
	}

	public Graphics2D getGraphics2D() {
		return this.g;
	}

	public boolean drewAnything() {
		return this.drewAnything;
	}

	public void begin() {
		if (this.stack.isEmpty()) {
			this.drewAnything = false;
		}
		this.stack.add(new GraphicsState(this));
	}

	public void end() {
		GraphicsState state = (GraphicsState) this.stack.remove(this.stack
				.size() - 1);
		state.restore(this);
	}

	public void setLineWidth(double width) {
		BasicStroke stroke = (BasicStroke) this.g.getStroke();
		float fwidth = (float) width;
		this.g.setStroke(new BasicStroke(fwidth, stroke.getEndCap(), stroke
				.getLineJoin()));
	}
	
	public double getLineWidth() {
		return ((BasicStroke)this.g.getStroke()).getLineWidth();
	}

	public void setLinePattern(double[] pattern) {
		BasicStroke stroke = (BasicStroke) this.g.getStroke();
		float[] fpattern;
		if (pattern != null && pattern.length > 0) {
			fpattern = new float[pattern.length];
			for (int i = 0; i < pattern.length; ++i) {
				fpattern[i] = (float) pattern[i];
			}
		} else {
			fpattern = null;
		}
		this.g.setStroke(new BasicStroke(stroke.getLineWidth(), stroke
				.getEndCap(), stroke.getLineJoin(), stroke.getMiterLimit(),
				fpattern, stroke.getDashPhase()));
	}
	
	public double[] getLinePattern() {
		float[] da = ((BasicStroke)this.g.getStroke()).getDashArray();
		double[] pattern = new double[da.length];
		for (int i = 0; i < da.length; ++i) {
			pattern[i] = da[i];
		}
		return pattern;
	}

	public void setLineJoin(short style) {
		BasicStroke stroke = (BasicStroke) this.g.getStroke();
		this.g.setStroke(new BasicStroke(stroke.getLineWidth(), stroke
				.getEndCap(), style, stroke.getMiterLimit(), stroke
				.getDashArray(), stroke.getDashPhase()));
	}
	
	public short getLineJoin() {
		return (short)((BasicStroke)this.g.getStroke()).getLineJoin();
	}

	public void setLineCap(short style) {
		BasicStroke stroke = (BasicStroke) this.g.getStroke();
		this.g.setStroke(new BasicStroke(stroke.getLineWidth(), style, stroke
				.getLineJoin(), stroke.getMiterLimit(), stroke.getDashArray(),
				stroke.getDashPhase()));
	}
	
	public short getLineCap() {
		return (short)((BasicStroke)this.g.getStroke()).getEndCap();
	}

	protected void setPaint(Object _paint, boolean fill)
			throws GraphicsException {
		java.awt.Paint awtPaint;
		AffineTransform at;
		Paint paint = (jp.cssj.sakae.gc.paint.Paint)_paint;

		switch (paint.getPaintType()) {
		case Paint.COLOR:
			awtPaint = G2dUtils.toAwtColor((Color) paint);
			at = null;
			break;
			
		case Paint.PATTERN:
			Pattern pattern = (Pattern) paint;
			awtPaint = G2dUtils.toAwtPaint(pattern, this);
			at = pattern.getTransform();
			break;
			
		case Paint.LINEAR_GRADIENT:
			awtPaint = G2dUtils.toAwtPaint((LinearGradient) paint);
			at = null;
			break;
		case Paint.RADIAL_GRADIENT:
			awtPaint = G2dUtils.toAwtPaint((RadialGradient) paint);
			at = null;
			break;
			default:
				throw new IllegalStateException();
		}
		if (fill) {
			this.fillPaint = awtPaint;
			this.fillAt = at;
		} else {
			this.g.setPaint(awtPaint);
			this.strokeAt = at;
		}
	}

	public void setOpacity(float opacity) {
		if (opacity == 1) {
			this.g.setPaintMode();
			return;
		}
		this.g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
				opacity));
	}
	
	public float getOpacity() {
		return ((AlphaComposite)this.g.getComposite()).getAlpha();
	}

	public void setStrokePaint(Object paint) throws GraphicsException {
		this.setPaint(paint, false);
	}

	public void setFillPaint(Object paint) throws GraphicsException {
		this.setPaint(paint, true);
	}

	public void transform(AffineTransform at) {
		this.g.transform(at);
	}

	public AffineTransform getTransform() {
		return this.g.getTransform();
	}

	public void clip(Shape shape) {
		this.g.clip(shape);
	}

	public Shape getClip() {
		return this.g.getClip();
	}

	public void resetState() {
		GraphicsState state = (GraphicsState) this.stack
				.get(this.stack.size() - 1);
		state.restore(this);
	}

	public void drawImage(Image image) throws GraphicsException {
		this.drewAnything = true;
		image.drawTo(this);
	}

	public void fill(Shape shape) {
		this.drewAnything = true;
		java.awt.Paint paint = this.g.getPaint();
		this.g.setPaint(this.fillPaint);
		if (this.fillAt != null) {
			AffineTransform saveAt = g.getTransform();
			this.g.transform(this.fillAt);
			try {
				shape = this.fillAt.createInverse().createTransformedShape(
						shape);
			} catch (NoninvertibleTransformException e) {
				throw new RuntimeException(e);
			}
			this.g.fill(shape);
			this.g.setTransform(saveAt);
		} else {
			this.g.fill(shape);
		}
		this.g.setPaint(paint);
	}

	public void draw(Shape shape) {
		this.drewAnything = true;
		if (this.strokeAt != null) {
			AffineTransform saveAt = g.getTransform();
			this.g.transform(this.strokeAt);
			try {
				shape = this.strokeAt.createInverse().createTransformedShape(
						shape);
			} catch (NoninvertibleTransformException e) {
				throw new RuntimeException(e);
			}
			this.g.draw(shape);
			this.g.setTransform(saveAt);
		} else {
			this.g.draw(shape);
		}
	}

	public void fillDraw(Shape shape) {
		this.fill(shape);
		this.draw(shape);
	}

	public void drawText(Text text, double x, double y)
			throws GraphicsException {
		this.drewAnything = true;

		this.begin();
		this.transform(AffineTransform.getTranslateInstance(x, y));
		Font font = ((FontMetricsImpl) text.getFontMetrics()).getFont();
		try {
			font.drawTo(this, text);
		} catch (IOException e) {
			throw new GraphicsException(e);
		}
		this.end();
	}

	private static class G2dGroupImageGC extends G2dGC implements GroupImageGC {
		final BufferedImage image;
		
		G2dGroupImageGC(Graphics2D g2d, FontManager fm, BufferedImage image) {
			super(g2d, fm);
			this.image = image;
		}

		public Image finish() throws GraphicsException {
			 return new RasterImageImpl(this.image);
		}
	}
	
	public GroupImageGC creatgeGroupImage(double width, double height)
			throws GraphicsException {
		BufferedImage image = new BufferedImage((int)width, (int)height, BufferedImage.TYPE_INT_ARGB);
		Graphics2D g2d = (Graphics2D)image.getGraphics();
		G2dGroupImageGC gc = new G2dGroupImageGC(g2d, this.getFontManager(), image);
		GraphicsState state = new GraphicsState(this);
		state.restore(gc);
		gc.stack = (ArrayList<GraphicsState>)this.stack.clone();
		return gc;
	}
}