package jp.cssj.sakae.gc.font.util;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;

import jp.cssj.sakae.font.BBox;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.font.ShapedFont;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.font.FontMetrics;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.text.Text;

/**
 * テキスト描画・フォント関連のユーティリティです。
 * 
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: FontUtils.java 764 2012-02-05 10:31:00Z miyabe $
 */
public final class FontUtils {
	private FontUtils() {
		// unused
	}

	/**
	 * フォント名を大文字に統一し、ハイフン、スペースを除去します。
	 * 
	 * @param fontName
	 * @return
	 */
	public static String normalizeName(String fontName) {
		StringBuffer buff = new StringBuffer();
		for (int i = 0; i < fontName.length(); ++i) {
			char ch = fontName.charAt(i);
			if (Character.isWhitespace(ch) || ch == '-' || ch == '_') {
				continue;
			}
			buff.append(Character.toUpperCase(ch));
		}
		return buff.toString();
	}

	public static boolean equals(FontStyle a, FontStyle b) {
		return a.getFamily().equals(b.getFamily())
				&& a.getSize() == b.getSize() && a.getStyle() == b.getStyle()
				&& a.getWeight() == b.getWeight()
				&& a.getDirection() == b.getDirection()
				&& a.getPolicy().equals(b.getPolicy());
	}

	public static int hashCode(FontStyle fontStyle) {
		int h = fontStyle.getFamily().hashCode();
		long a = Double.doubleToLongBits(fontStyle.getSize());
		h = 31 * h + (int) (a ^ (a >>> 32));
		h = 31 * h + fontStyle.getStyle();
		h = 31 * h + fontStyle.getWeight();
		h = 31 * h + fontStyle.getDirection();
		h = 31 * h + fontStyle.getPolicy().hashCode();
		return h;
	}

	/**
	 * フォントを描画します。
	 * 
	 * @param gc
	 * @param font
	 * @param text
	 */
	public static void drawPDFFont(GC gc, ShapedFont font, Text text)
			throws GraphicsException {
		FontStyle fontStyle = text.getFontStyle();
		byte direction = fontStyle.getDirection();
		double fontSize = fontStyle.getSize();
		int goff = text.getGOff();
		int glen = text.getGLen();
		int[] gids = text.getGIDs();
		double letterSpacing = text.getLetterSpacing();
		double[] xadvances = text.getXAdvances(false);
		FontMetrics fm = text.getFontMetrics();
		AffineTransform at;
		{
			double s = fontSize / FontSource.UNITS_PER_EM;
			at = AffineTransform.getScaleInstance(s, s);
		}

		if (direction == FontStyle.DIRECTION_TB) {
			// 縦書きモード
			if (font.getFontSource().getDirection() == direction) {
				// 縦書き対応フォント
				gc.transform(AffineTransform.getTranslateInstance(
						-fontSize / 2.0, fontSize * 0.88));
				GeneralPath path = new GeneralPath();
				int pgid = 0;
				for (int i = 0; i < glen; ++i) {
					AffineTransform at2 = at;
					int gid = gids[goff + i];
					if (i > 0) {
						double dy = fm.getAdvance(pgid) + letterSpacing;
						dy -= fm.getKerning(pgid, gid);
						if (xadvances != null) {
							dy += xadvances[i];
						}
						at.preConcatenate(AffineTransform.getTranslateInstance(
								0, dy));
					}
					pgid = gid;
					Shape shape = font.getShapeByGID(gid);
					if (shape != null) {
						double width = (fontSize - fm.getWidth(gid)) / 2.0;
						if (width != 0) {
							at2 = AffineTransform
									.getTranslateInstance(width, 0);
							at2.concatenate(at2);
						}
						path.append(shape.getPathIterator(at2), false);
					}
				}
				gc.fill(path);
				return;
			} else {
				// 横倒し
				gc.transform(AffineTransform.getRotateInstance(Math.PI / 2.0));
				BBox bbox = font.getFontSource().getBBox();
				double dy = ((bbox.lly + bbox.ury) * fontSize / FontSource.UNITS_PER_EM) / 2.0;
				gc.transform(AffineTransform.getTranslateInstance(0, dy));
			}
		}
		// 横書き
		int pgid = 0;
		final GeneralPath path = new GeneralPath();
		for (int i = 0; i < glen; ++i) {
			final int gid = gids[goff + i];
			if (i > 0) {
				double dx = fm.getAdvance(pgid) + letterSpacing;
				if (i > 0) {
					dx -= fm.getKerning(pgid, gid);
				}
				if (xadvances != null) {
					dx += xadvances[i];
				}
				at.preConcatenate(AffineTransform.getTranslateInstance(dx, 0));
			}
			Shape shape = font.getShapeByGID(gid);
			if (shape != null) {
				path.append(shape.getPathIterator(at), false);
			}
			pgid = gid;
		}
		gc.fill(path);
	}
}