package jp.cssj.sakae.font.otf;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.sakae.font.AbstractFontSource;
import jp.cssj.sakae.font.BBox;
import jp.cssj.sakae.font.Font;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.font.Panose;
import net.zamasoft.font.FontFile;
import net.zamasoft.font.Glyph;
import net.zamasoft.font.table.CmapFormat;
import net.zamasoft.font.table.CmapTable;
import net.zamasoft.font.table.HeadTable;
import net.zamasoft.font.table.HheaTable;
import net.zamasoft.font.table.NameRecord;
import net.zamasoft.font.table.NameTable;
import net.zamasoft.font.table.Os2Table;
import net.zamasoft.font.table.Table;
import net.zamasoft.font.table.XmtxTable;

/**
 * @author <a href="mailto:tatsuhiko at miya dot be">MIYABE Tatsuhiko </a>
 * @version $Id: OpenTypeFontSource.java 764 2012-02-05 10:31:00Z miyabe $
 */
public class OpenTypeFontSource extends AbstractFontSource {
	private static final Logger LOG = Logger.getLogger(OpenTypeFontSource.class
			.getName());

	private static final long serialVersionUID = 4L;

	protected static Map<File, net.zamasoft.font.OpenTypeFont[]> fileToFont = new WeakHashMap<File, net.zamasoft.font.OpenTypeFont[]>();

	protected final File file;

	protected final int index;

	protected final short upm;

	protected String fontName;

	protected final BBox bbox;

	protected final short ascent, descent, xHeight, capHeight, spaceAdvance,
			stemH, stemV;

	protected Panose panose;

	protected final byte direction;

	protected final CmapFormat cmap;

	public OpenTypeFontSource(File file, int index, byte direction)
			throws IOException {
		this.index = index;
		this.file = file;
		net.zamasoft.font.OpenTypeFont ttFont = this.getOpenTypeFont();

		// フォントメトリック情報
		{
			// long time = System.currentTimeMillis();
			HeadTable head = (HeadTable) ttFont.getTable(Table.head);
			this.upm = head.getUnitsPerEm();
			short llx = (short) (head.getXMin() * FontSource.UNITS_PER_EM / this.upm);
			short lly = (short) (head.getYMin() * FontSource.UNITS_PER_EM / this.upm);
			short urx = (short) (head.getXMax() * FontSource.UNITS_PER_EM / this.upm);
			short ury = (short) (head.getYMax() * FontSource.UNITS_PER_EM / this.upm);
			BBox bbox = new BBox(llx, lly, urx, ury);
			this.bbox = bbox;
			this.setItalic((head.getMacStyle() & 2) != 0);
		}

		List<String> aliases = new ArrayList<String>();
		String fontName = null;
		{
			// long time = System.currentTimeMillis();
			NameTable name = (NameTable) ttFont.getTable(Table.name);
			for (int i = 0; i < name.size(); ++i) {
				NameRecord record = name.get(i);
				short nameId = record.getNameId();
				if (nameId == 1 || nameId == 3 || nameId == 4) {
					aliases.add(record.getRecordString());
				} else if (nameId == 6) {
					fontName = record.getRecordString();
				}
			}
		}
		this.aliases = (String[]) aliases.toArray(new String[aliases.size()]);

		if (fontName == null) {
			throw new NullPointerException();
		}
		this.fontName = fontName;

		{
			// long time = System.currentTimeMillis();
			Os2Table os2 = (Os2Table) ttFont.getTable(Table.OS_2);
			short weight = (short) os2.getWeightClass();
			this.setWeight(weight);
			short cFamilyClass = os2.getFamilyClass();
			net.zamasoft.font.table.Panose panose = os2.getPanose();
			this.panose = new Panose(cFamilyClass, panose.code);
		}

		{
			HheaTable hhea = (HheaTable) ttFont.getTable(Table.hhea);
			this.ascent = (short) (hhea.getAscender() * FontSource.UNITS_PER_EM / this.upm);
			this.descent = (short) (-hhea.getDescender()
					* FontSource.UNITS_PER_EM / this.upm);
		}

		this.cmap = ((CmapTable) ttFont.getTable(Table.cmap)).getCmapFormat(
				Table.platformMicrosoft, Table.encodingUGL);
		{
			int gid = this.cmap.mapCharCode(' ');
			XmtxTable hmtx = (XmtxTable) ttFont.getTable(Table.hmtx);
			this.spaceAdvance = (short) (hmtx.getAdvanceWidth(gid)
					* FontSource.UNITS_PER_EM / this.upm);
		}

		{
			net.zamasoft.font.OpenTypeFont font = this.getOpenTypeFont();
			Glyph gx = font.getGlyph(this.cmap.mapCharCode('x'));
			this.xHeight = gx.getPath() == null ? DEFAULT_X_HEIGHT : (short) gx
					.getPath().getBounds().height;
			Glyph gh = font.getGlyph(this.cmap.mapCharCode('H'));
			this.capHeight = gh.getPath() == null ? DEFAULT_CAP_HEIGHT
					: (short) gh.getPath().getBounds().height;
		}

		this.stemH = 0;
		this.stemV = 0;

		if (LOG.isLoggable(Level.FINE)) {
			LOG.fine("new font: " + this.getFontName());
		}

		this.direction = direction;
	}

	public net.zamasoft.font.OpenTypeFont getOpenTypeFont() {
		return getOpenTypeFont(this.file, this.index);
	}

	public static synchronized net.zamasoft.font.OpenTypeFont getOpenTypeFont(
			File file, int index) {
		net.zamasoft.font.OpenTypeFont[] fonts = (net.zamasoft.font.OpenTypeFont[]) fileToFont
				.get(file);
		if (fonts != null && fonts[index] != null) {
			return fonts[index];
		}

		try {
			FontFile fontFile = FontFile.getFontFile(file);
			if (fonts == null) {
				fonts = new net.zamasoft.font.OpenTypeFont[fontFile
						.getNumFonts()];
			}
			fonts[index] = fontFile.getFont(index);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		fileToFont.put(file, fonts);
		return fonts[index];
	}

	public byte getDirection() {
		return this.direction;
	}

	public void setFontName(String fontName) {
		this.fontName = fontName;
	}

	public Panose getPanose() {
		return this.panose;
	}

	public void setPanose(Panose panose) {
		this.panose = panose;
	}

	public BBox getBBox() {
		return this.bbox;
	}

	public String getFontName() {
		return this.fontName;
	}

	public short getXHeight() {
		return this.xHeight;
	}

	public short getCapHeight() {
		return this.capHeight;
	}

	public short getSpaceAdvance() {
		return this.spaceAdvance;
	}

	public short getAscent() {
		return this.ascent;
	}

	public short getDescent() {
		return this.descent;
	}

	public short getStemH() {
		return this.stemH;
	}

	public short getStemV() {
		return this.stemV;
	}

	public short getUnitsPerEm() {
		return this.upm;
	}

	public CmapFormat getCmapFormat() {
		return this.cmap;
	}

	public boolean canDisplay(int c) {
		if (this.getDirection() == FontStyle.DIRECTION_TB) {
			if (c <= 0xFF || (c >= 0xFF60 && c <= 0xFFDF)) {
				return false;
			}
		}
		return this.cmap.mapCharCode(c) != 0;
	}

	public Font createFont() {
		return new OpenTypeFontImpl(this);
	}
}
